/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <gdk/gdkx.h>
#include <gnome.h>

#include <libxklavier/xklavier_config.h>

#include "xkb_capplet.h"

#include "../common/gnome-startup.h"
#include "../common/switchcuts.h"

//#define XK_XKB_KEYS
//#include <X11/keysymdef.h>

static GnomeClient *client;
static char *argv0;
static int initOnly = 0;

static struct poptOption opts[] = {
  {
   "init-session-settings",
   '\0',
   POPT_ARG_NONE,
   &initOnly,
   0,
   N_( "Apply XKB configuration and quit" ),
   NULL},
  {
   NULL,
   '\0',
   0,
   NULL,
   0,
   NULL,
   NULL}
};

static void XkbCappletUpdateSession( GSwitchItXkbConfig * gswic );
static void XkbCappletInitSessionSettingsOnly( GSwitchItXkbConfig * gswic );

void XkbCappletUpdateGuiFromConfig( XkbCapplet * gswic )
{
//model
  if( gswic->xkbConfig.model != NULL )
  {
    GtkWidget *menu =
      g_object_get_data( G_OBJECT( gswic->capplet ), "modelsMenu_menu" );
    GtkWidget *theItem = g_object_get_data( G_OBJECT( gswic->capplet ),
                                            XkbModelName2Id( gswic->xkbConfig.
                                                             model ) );
    gtk_menu_shell_select_item( GTK_MENU_SHELL( menu ), theItem );
    gtk_menu_shell_activate_item( GTK_MENU_SHELL( menu ), theItem, TRUE );
  }
//layout
  XkbCappletUpdateLayoutButtonsFromConfig( gswic );
//options...
  {
    // first - clear all!
    GSList *optionsCtls = g_object_get_data( G_OBJECT( gswic->capplet ),
                                             OPTION_CONTROLS_LIST_PROPERTY );
    GSList *theOption = gswic->xkbConfig.options;

    while( optionsCtls != NULL )
    {
      GtkWidget *optionsCtl = GTK_WIDGET( optionsCtls->data );
      if( GTK_IS_TREE_VIEW( optionsCtl ) )      // multi
      {
        GtkTreeSelection *selection =
          gtk_tree_view_get_selection( GTK_TREE_VIEW( optionsCtl ) );
        gtk_tree_selection_unselect_all( selection );
      } else if( GTK_IS_OPTION_MENU( optionsCtl ) )     // single
      {
        gtk_option_menu_set_history( GTK_OPTION_MENU( optionsCtl ), 0 );
      }
      optionsCtls = optionsCtls->next;
    }

    // now - fill
    while( theOption != NULL )
    {
      char *group, *option;
      if( GSwitchItConfigSplitItems
          ( theOption->data, &group, &option ) && option != NULL )
      {
        GtkWidget *omenu = g_object_get_data( G_OBJECT( gswic->capplet ),
                                              XkbOptionGroupName2Id( group,
                                                                     "omenu" ) );
        if( omenu != NULL )
        {
          GtkWidget *menuItem = g_object_get_data( G_OBJECT( gswic->capplet ),
                                                   XkbOptionName2Id( group,
                                                                     option ) );
          GtkWidget *menu =
            gtk_option_menu_get_menu( GTK_OPTION_MENU( omenu ) );
          gtk_menu_shell_select_item( GTK_MENU_SHELL( menu ), menuItem );
          gtk_menu_shell_activate_item( GTK_MENU_SHELL( menu ), menuItem,
                                        TRUE );
        } else
        {
          GtkWidget *listView = g_object_get_data( G_OBJECT( gswic->capplet ),
                                                   XkbOptionGroupName2Id
                                                   ( group, "list" ) );
          GtkTreeModel *listModel =
            gtk_tree_view_get_model( GTK_TREE_VIEW( listView ) );
          GtkTreeViewColumn *firstColumn =
            gtk_tree_view_get_column( GTK_TREE_VIEW( listView ), 0 );
          GtkTreeIter iter;
          gboolean found = FALSE;

          if( gtk_tree_model_get_iter_first( listModel, &iter ) )
          {
            do
            {
              GValue *val = g_new0( GValue, 1 );
              gtk_tree_model_get_value( listModel, &iter, 1, val );
              if( !strcmp( g_value_get_string( val ), option ) )
              {
                GtkTreeSelection *selection =
                  gtk_tree_view_get_selection( GTK_TREE_VIEW( listView ) );
                gtk_tree_selection_select_iter( selection, &iter );
                found = TRUE;
              }
              g_value_unset( val );
              g_free( val );
            }
            while( !found && gtk_tree_model_iter_next( listModel, &iter ) );
          }
        }
      } else
        XklDebug( 150, "Could not split [%s]\n", theOption->data );
      theOption = theOption->next;
    }

  }

  gtk_option_menu_set_history( GTK_OPTION_MENU
                               ( g_object_get_data
                                 ( G_OBJECT( gswic->capplet ),
                                   "switchcutsOMenu" ) ),
                               gswic->xkbConfig.switchcutId );
}

static void XkbCappletUpdateAddOptionFromTree( GtkTreeModel * listModel,
                                               GtkTreePath * path,
                                               GtkTreeIter * iter,
                                               XkbCapplet * gswic )
{
  GValue *val = g_new0( GValue, 1 );
  const char *optionName;
  const char *groupName = g_object_get_data( G_OBJECT( gswic->capplet ),
                                             CURRENT_OPTION_GROUP_NAME_PROPERTY );

  gtk_tree_model_get_value( listModel, iter, 1, val );
  optionName = g_value_get_string( val );
  GSwitchItXkbConfigOptionsAdd( &gswic->xkbConfig, groupName, optionName );
  g_value_unset( val );
  g_free( val );
}

void XkbCappletUpdateConfigFromGui( XkbCapplet * gswic )
{
// model
  GtkWidget *modelMenu = g_object_get_data( G_OBJECT( gswic->capplet ),
                                            "modelsMenu_menu" );
  GtkWidget *theModelItem = gtk_menu_get_active( GTK_MENU( modelMenu ) );
  const gchar *modelName = gtk_widget_get_name( theModelItem );
  int i, j = 0;
  Switchcut *sc;
  GtkWidget *theItem;
  GSList *optionsCtls = g_object_get_data( G_OBJECT( gswic->capplet ),
                                           OPTION_CONTROLS_LIST_PROPERTY );
  GSwitchItXkbConfigModelSet( &gswic->xkbConfig, modelName );
// layouts
  GSwitchItXkbConfigLayoutsReset( &gswic->xkbConfig );
  for( i = XkbNumKbdGroups; --i >= 0; )
  {
    const char *id = XkbCappletLayout2Id( j++, "btn" );
    GtkWidget *btn = CappletGetGladeWidget( gswic, id );
    const char *fullLayoutName = g_object_get_data( G_OBJECT( btn ),
                                                    FULL_LAYOUT_NAME_BTN_PROPERTY );
    char *layoutName, *variantName;
    if( GSwitchItConfigSplitItems
        ( fullLayoutName, &layoutName, &variantName ) )
      GSwitchItXkbConfigLayoutsAdd( &gswic->xkbConfig, layoutName,
                                    variantName );
  }

// options
  GSwitchItXkbConfigOptionsReset( &gswic->xkbConfig );
  while( optionsCtls != NULL )
  {
    GtkWidget *optionsCtl = GTK_WIDGET( optionsCtls->data );
    const char *groupName = gtk_widget_get_name( optionsCtl );
    if( GTK_IS_TREE_VIEW( optionsCtl ) )        // multisel
    {
      GtkTreeSelection *selection =
        gtk_tree_view_get_selection( GTK_TREE_VIEW( optionsCtl ) );
      g_object_set_data( G_OBJECT( gswic->capplet ),
                         CURRENT_OPTION_GROUP_NAME_PROPERTY,
                         ( gpointer ) groupName );
      gtk_tree_selection_selected_foreach( selection,
                                           ( GtkTreeSelectionForeachFunc )
                                           XkbCappletUpdateAddOptionFromTree,
                                           gswic );
      g_object_set_data( G_OBJECT( gswic->capplet ),
                         CURRENT_OPTION_GROUP_NAME_PROPERTY, NULL );
    } else if( GTK_IS_OPTION_MENU( optionsCtl ) )       // singlesel
    {
      const gchar *optionName;
      theItem =
        gtk_menu_get_active( GTK_MENU
                             ( gtk_option_menu_get_menu
                               ( GTK_OPTION_MENU( optionsCtl ) ) ) );
      optionName = gtk_widget_get_name( theItem );
      XklDebug( 200, "Option [%s:%s] selected \n", groupName, optionName );
      if( optionName != NULL && *optionName != '\0' )
        GSwitchItXkbConfigOptionsAdd( &gswic->xkbConfig, groupName,
                                      optionName );
    }
    optionsCtls = optionsCtls->next;
  }

  theItem = gtk_menu_get_active( GTK_MENU
                                 ( gtk_option_menu_get_menu
                                   ( GTK_OPTION_MENU
                                     ( g_object_get_data
                                       ( G_OBJECT( gswic->capplet ),
                                         "switchcutsOMenu" ) ) ) ) );

  sc = g_object_get_data( G_OBJECT( theItem ), "switchcut" );
  gswic->xkbConfig.switchcutId = sc - switchcuts;
}

void XkbCappletActivationError( GtkWindow * w )
{
  char *vendor = ServerVendor( GDK_DISPLAY(  ) );
  int release = VendorRelease( GDK_DISPLAY(  ) );
  gboolean badXFree430Release = ( !strcmp( vendor,
                                           "The XFree86 Project, Inc" ) ) &&
    ( release == 40300000 );

  GtkWidget *msg = gtk_message_dialog_new( w,
                                           GTK_DIALOG_MODAL |
                                           GTK_DIALOG_DESTROY_WITH_PARENT,
                                           GTK_MESSAGE_ERROR,
                                           GTK_BUTTONS_OK,
                                           _
                                           ( "Error activating XKB configuration.\n"
                                             "Probably internal X server problem.\n\nX server version data:\n%s\n%d%s" ),
                                           vendor,
                                           release, badXFree430Release ?
                                           _
                                           ( "You are using XFree 4.3.0.\n"
                                             "There are known problems with complex XKB configurations.\n"
                                             "Try using simpler configuration or taking more fresh version of XFree software." )
                                           : "" );
  gtk_dialog_run( GTK_DIALOG( msg ) );
  gtk_widget_destroy( msg );
}

void XkbCappletOk( GtkWidget * w, XkbCapplet * gswic )
{
  XkbCappletUpdateConfigFromGui( gswic );
  GSwitchItXkbConfigSave( &gswic->xkbConfig );

  if( !GSwitchItXkbConfigActivate( &gswic->xkbConfig ) )
    XkbCappletActivationError( GTK_WINDOW( gswic->capplet ) );

  XkbCappletUpdateSession( &gswic->xkbConfig );
  gtk_main_quit(  );
}

void XkbCappletCancel( GtkWidget * w, XkbCapplet * gswic )
{
  gtk_main_quit(  );
}

void XkbCappletHelp( GtkWidget * w, XkbCapplet * gswic )
{
  GSwitchItHelp( GTK_WINDOW( gswic->capplet ), "xkbPropsCapplet" );
}

void XkbCappletSetStateToChanged( GObject * o, XkbCapplet * gswic )
{
  gswic->stateChanged = TRUE;
  gtk_widget_set_sensitive( CappletGetGladeWidget( gswic, "btnOk" ), TRUE );
}

void
  XkbCappletUseCustomXkbSettingsButtonToggled
  ( GtkToggleButton * togglebutton, XkbCapplet * gswic )
{
  GtkToggleButton *btnc = GTK_TOGGLE_BUTTON( CappletGetGladeWidget( gswic,
                                                                    "rbUseCustomXkbSettings" ) );
  GtkToggleButton *btng = GTK_TOGGLE_BUTTON( CappletGetGladeWidget( gswic,
                                                                    "rbUseGlobalXkbSettings" ) );
  gboolean enablec = gtk_toggle_button_get_active( btnc );
  gboolean enableg = gtk_toggle_button_get_active( btng );
  if( enableg == enablec )
    gtk_toggle_button_set_active( btng, !enablec );

  XkbCappletUseSomeXkbSettingsButtonToggled( togglebutton, gswic, enablec );
}

void
  XkbCappletUseGlobalXkbSettingsButtonToggled
  ( GtkToggleButton * togglebutton, XkbCapplet * gswic )
{
  GtkToggleButton *btnc = GTK_TOGGLE_BUTTON( CappletGetGladeWidget( gswic,
                                                                    "rbUseCustomXkbSettings" ) );
  GtkToggleButton *btng = GTK_TOGGLE_BUTTON( CappletGetGladeWidget( gswic,
                                                                    "rbUseGlobalXkbSettings" ) );
  gboolean enablec = gtk_toggle_button_get_active( btnc );
  gboolean enableg = gtk_toggle_button_get_active( btng );
  if( enableg == enablec )
    gtk_toggle_button_set_active( btnc, !enableg );
}

void
XkbCappletUseSomeXkbSettingsButtonToggled( GtkToggleButton *
                                           togglebutton,
                                           XkbCapplet * gswic,
                                           gboolean overrideSettings )
{
  GtkWidget *ctl;
  GSList *optionsList = g_object_get_data( G_OBJECT( gswic->capplet ),
                                           OPTION_CONTROLS_LIST_PROPERTY );
  int i;
  if( overrideSettings )
  {
    // at startup, togglebutton == NULL, we don't need to load the data...
    if( togglebutton != NULL )
      GSwitchItXkbConfigLoad( &gswic->xkbConfig );
  } else
  {
    GSwitchItXkbConfigLoadInitial( &gswic->xkbConfig );
  }
  gswic->xkbConfig.overrideSettings = overrideSettings;

  ctl = CappletGetGladeWidget( gswic, "modelsMenu" );
  gtk_widget_set_sensitive( ctl, overrideSettings );
  ctl = CappletGetGladeWidget( gswic, "switchcutsOMenu" );
  gtk_widget_set_sensitive( ctl, overrideSettings );

  for( i = XkbNumKbdGroups; --i >= 0; )
  {
    const char *btnId = XkbCappletLayout2Id( i, "btn" );
    GtkWidget *btn = CappletGetGladeWidget( gswic, btnId );
    const char *btnClearId = XkbCappletLayout2Id( i, "btnClear" );
    GtkWidget *btnClear = CappletGetGladeWidget( gswic, btnClearId );
    GtkWidget *oldXWarning = CappletGetGladeWidget( gswic, "oldXWarning" );
    if( XklMultipleLayoutsSupported(  ) )
    {
      gtk_widget_set_sensitive( btn, overrideSettings );
      gtk_widget_set_sensitive( btnClear, overrideSettings
                                &&
                                ( g_object_get_data
                                  ( G_OBJECT( btn ),
                                    FULL_LAYOUT_NAME_BTN_PROPERTY ) !=
                                  NULL ) );
    } else
    {
      gtk_widget_show( oldXWarning );
      gtk_widget_set_sensitive( btn, ( i == 0 ) && overrideSettings );
      gtk_widget_set_sensitive( btnClear, ( i == 0 ) && overrideSettings
                                &&
                                ( g_object_get_data
                                  ( G_OBJECT( btn ),
                                    FULL_LAYOUT_NAME_BTN_PROPERTY ) !=
                                  NULL ) );
    }
  }

  while( optionsList != NULL )
  {
    gtk_widget_set_sensitive( GTK_WIDGET( optionsList->data ),
                              overrideSettings );
    optionsList = g_slist_next( optionsList );
  }

  if( togglebutton != NULL )
    XkbCappletSetStateToChanged( G_OBJECT( togglebutton ), gswic );

  XkbCappletUpdateGuiFromConfig( gswic );
}

void XkbCappletUpdateSession( GSwitchItXkbConfig * gswic )
{
  GnomeClientFlags flags;
  client = gnome_master_client(  );
  flags = gnome_client_get_flags( client );
  if( flags & GNOME_CLIENT_IS_CONNECTED )
  {
    gchar *session_args[3];
    XklBackupNamesProp(  );

    if( gswic->overrideSettings )
    {
      gboolean firstLaunch =
        gnome_startup_acquire_token( "GNOME_XKB_PROPERTIES",
                                     gnome_client_get_id( client ) );
      if( firstLaunch )
      {
        session_args[0] = argv0;
        session_args[1] = "--init-session-settings";
        session_args[2] = NULL;
        /* We use a priority of 5 so that we start before
         * session-managed WM's (priority 10), for safety.
         * */
        gnome_client_set_priority( client, 5 );
        gnome_client_set_restart_style( client, GNOME_RESTART_ANYWAY );
        gnome_client_set_restart_command( client, 2, session_args );
        XklDebug( 100,
                  "Now the capplet will be called at startup, thanks to %p/%d\n",
                  client, gnome_client_get_id( client ) );
      } else
      {
        gnome_client_set_restart_style( client, GNOME_RESTART_NEVER );
        XklDebug( 100,
                  "The token is already locked, can do nothing about it from %p/%d\n",
                  client, gnome_client_get_id( client ) );
      }
    } else
    {
      gnome_client_set_restart_style( client, GNOME_RESTART_NEVER );
      XklDebug( 100,
                "Now the capplet will NOT be called at startup, thanks to %p/%d\n",
                client, gnome_client_get_id( client ) );
    }
    gnome_client_flush( client );
  }
}

void XkbCappletInitSessionSettingsOnly( GSwitchItXkbConfig * gswic )
{
  GConfClient *confClient;
  XklDebug( 100, "Just setting XKB and nothing more\n" );

  confClient = gconf_client_get_default(  );
  GSwitchItXkbConfigInit( gswic, confClient );
  g_object_unref( confClient );
  GSwitchItXkbConfigLoad( gswic );
  XkbCappletUpdateSession( gswic );

  if( !gswic->overrideSettings )
    GSwitchItXkbConfigLoadInitial( gswic );

  if( !GSwitchItXkbConfigActivate( gswic ) )
    XkbCappletActivationError( NULL );

  GSwitchItXkbConfigTerm( gswic );
}

extern void XkbDescDump( FILE * fs, int level, XkbDescPtr kbd );

int main( int argc, char **argv )
{
  XkbCapplet gswic;
  GError *gconf_error = NULL;
  GConfClient *confClient;

  bindtextdomain( PACKAGE, GNOMELOCALEDIR );
  bind_textdomain_codeset( GETTEXT_PACKAGE, "UTF-8" );
  textdomain( PACKAGE );
  memset( &gswic, 0, sizeof( gswic ) );

  //gnome_init_with_popt_table( "xkb-properties", VERSION, argc, argv,
  //                            opts, 0, NULL );
  gnome_program_init( "xkb-properties", VERSION, LIBGNOMEUI_MODULE, argc,
                      argv, GNOME_PARAM_POPT_TABLE, opts, GNOME_PARAM_NONE );
  argv0 = argv[0];

  if( !gconf_init( argc, argv, &gconf_error ) )
  {
    g_warning( _( "Failed to init GConf: %s\n" ), gconf_error->message );
    g_error_free( gconf_error );
    return 1;
  }
  gconf_error = NULL;

  //GSwitchItInstallGlibLogAppender(  );

  //!! Check the result - can be useful
  XklInit( GDK_DISPLAY(  ) );

  if( initOnly )
  {
    // --init-session-settings
    XkbCappletInitSessionSettingsOnly( &gswic.xkbConfig );
    XklTerm(  );
    return 0;
  }

  if( bonobo_init( &argc, argv ) == FALSE )
    g_error( _( "Could not initialize Bonobo" ) );
  XklConfigInit(  );
  if( !XklConfigLoadRegistry(  ) )
  {
    g_warning( _( "Failed to init XklConfigRegistry\n" ) );
    XklConfigTerm(  );
    XklTerm(  );
    return 1;
  }

  confClient = gconf_client_get_default(  );
  GSwitchItXkbConfigInit( &gswic.xkbConfig, confClient );
  g_object_unref( confClient );
  GSwitchItXkbConfigLoad( &gswic.xkbConfig );

  XkbCappletUpdateSession( &gswic.xkbConfig );
  XkbCappletSetup( &gswic );
  gtk_main(  );
  // game over!
  XklTerm(  );
  GSwitchItXkbConfigTerm( &gswic.xkbConfig );
  XklConfigFreeRegistry(  );
  XklConfigTerm(  );
  return 0;
}
