/* gt_gtk.c - Gtk display driver
 *
 * Copyright (C) 1998, 1999 Free Sorftware Foundation
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.org.
 * 
 * Description:
 * 
 *  This file handles the GTK interface for gtalk, and keyboard input.
 * 
 * $Log: gt_gtk.c,v $
 * Revision 1.12  1999/08/26 11:57:49  zappo
 * Added Ctxt to fix_user_windows.  Fixed all occurances of XRead to pass
 * down Ctxt.  X_read tracks the eightbit flag.
 *
 * Revision 1.11  1999/06/06 13:02:24  zappo
 * Fixed GNOME initialization for the latest version of gnome.
 * Use GTK_TEXT_INDEX to reference characters.
 *
 * Revision 1.10  1999/02/27 14:39:05  zappo
 * Upgtaded to handle gnome 0.99.8
 * (gnome_init fixed, and gnome_dialog_set_modal removed.)
 * Updated to handle gtk 1.1.15.
 * (Fixed all text indexing to use GTK_TEXT_INDEX which fixes gap problems.)
 *
 * Revision 1.9  1998/10/29 17:02:00  zappo
 * Added a scrollbar to the text widgets.
 *
 * Revision 1.8  1998/10/17 01:52:40  zappo
 * Enabled GTK_ONLY flag to work w/ new configure script.
 * Fixed the gnome about box.
 *
 * Revision 1.7  1998/07/04 16:22:38  zappo
 * Added gnome features.
 *
 * Revision 1.6  1998/04/30 02:56:23  zappo
 * Added some GNOME support where applicable.
 *
 * Revision 1.5  1998/04/29 01:55:08  zappo
 * Made all widgets use my fixed-width font.  Choose courier first, then fixed.
 * Popup text widgets now have auto-wrap turned off.
 * Fixed resizing width of talk window problem.
 * Fixed minimum width of main window problem.
 * Added menu item help in minibuffer. (Added menu_show_help)
 * Moved gtk_main_iteration_do to end of _xread function.
 *
 * Revision 1.4  1998/04/28 22:33:34  zappo
 * Rearranged all the geometry management so we have a reasonable layout.
 * Updated popups so all text is inserted before the popup is displayed.
 * Updated the event loop to use a new features called a quick-timout.
 * This timeout catches the fast GTK timouts used for poping up menus.
 *
 * Revision 1.3  1998/04/27 17:44:09  zappo
 * Fixed a delete-line problem.. Went 1 character too far.
 *
 * Revision 1.2  1998/04/26 15:17:38  zappo
 * fleshed out the entire GTK interface.  It has vague resemblances of
 * working, but I have some bad refresh bugs, bad geometry management,
 * and bad event handling.
 *
 * Revision 1.1  1998/01/29 17:26:43  zappo
 * Initial revision
 *
 * Tokens: ::Header:: gtalkc.h
 */
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>

#include "gtalklib.h"
#include "gtalkc.h"

#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#ifndef GTK_ONLY
#include <gnome.h>
#endif

char Xinputbuffer[1024];

/* We need this to correctly handle writing into a window with a
 * border.  In addition, BSD gtk gets maxx wrong.
 */
static GtkWidget *popuptext, *popup = NULL;
static int popup_width = 0;

static GtkWidget *echo_area = NULL;
static GtkWidget *talk_pane = NULL;

struct InputDevice *Xio;
GtkWidget *toplevel, *gnomeapp;
GtkWidget *lastmenu;

GdkFont *fixed_font = NULL;
int      font_width, font_height;

static void no_action_here() {};

void X_xread();
static void GTK_close();
static char GTK_getch();
#ifdef PROTOTYPES
static void GTK_iconify(struct TalkContext *Ctxt);
static void GTK_fix_user_windows(struct TalkContext *Ctxt, struct WindowList *list);
static void *GTK_popup(int width, int height);
static void showabout(GtkWidget *key, gpointer data);
static void menu_show_help(GtkWidget *key, GdkEventCrossing *event, gpointer data);
void simulate_key_callback(GtkWidget *w, gpointer key);
static int GTK_winwidth(void *window);
static void GTK_clearwin(void *window);
static void GTK_puttext(void *window, char *text, int size);
static void GTK_refresh(void *window, int force);
static void GTK_delchar(void *window);
static void GTK_delword(void *window);
static void GTK_delline(void *window);
static int GTK_fillcolumn(void *window);
static void GTK_bell(struct TalkContext *Ctxt);
static void GTK_setlabel(GtkWidget *win, char *label);
static void GTK_refresh_screen(struct WindowList *wlist);
static void GTK_map_popup(void *window);
static void GTK_set_echo_area(char *label, int showGTKor);
static void GTK_delwin(void *window);
#else
static void GTK_iconify();
static void GTK_fix_user_windows();
static void *GTK_popup();
static void showabout();
static void menu_show_help();
void simulate_key_callback();
static int GTK_winwidth();
static void GTK_clearwin();
static void GTK_puttext();
static void GTK_refresh();
static void GTK_delchar();
static void GTK_delword();
static void GTK_delline();
static int GTK_fillcolumn();
static void GTK_bell();
static void GTK_setlabel();
static void GTK_refresh_screen();
static void GTK_map_popup();
static void GTK_set_echo_area();
static void GTK_delwin();
#endif

static struct IfaceMethods GTKMethods =
{
  GTK_close,
  GTK_bell,
  GTK_iconify,
  /* Management */
  GTK_fix_user_windows,
  GTK_clearwin,
  GTK_winwidth,
  GTK_setlabel,
  /* editing commands */
  GTK_delchar,
  GTK_delline,
  GTK_delword,
  GTK_fillcolumn,
  GTK_puttext,
  GTK_refresh,
  GTK_refresh_screen,
  /* IO stuff */
  no_action_here,
  GTK_getch,
  GTK_popup,
  GTK_map_popup,
  GTK_delwin,
  GTK_set_echo_area
};

static struct TalkContext *Ctxt;

#ifndef GTK_ONLY
static GnomeClient *client;
static gint save_state();
#endif


/*
 * Function: X_init, GTK_close
 *
 *   Initialization routines.
 *     init - the whole system:
 *     close - close the system
 *
 * Returns:     IfaceMethods - GTK display methods
 * Parameters:  Ctxt - Context
 *
 * History:
 * zappo   8/30/95    Created
 */
struct IfaceMethods *X_init(CtxtIn)
     struct TalkContext *CtxtIn;
{
  extern void X_make_menus();

  GtkWidget *vbox1;
  GtkWidget *menubar;
  GtkWidget *about;
  GtkStyle *style;

   
#ifdef GTK_ONLY
  int fake_argc = 0;
  gtk_init (&fake_argc, NULL);
#else
  static char *args[1] = { "gtalk" };

  /* REQUIRES GNOME 0.99.x */
  gnome_init("GNU talk", VERSION, 1, args);

  client = gnome_master_client ();
  
  /* Arrange to be told when something interesting happens.  */
  gtk_signal_connect (GTK_OBJECT (client), "save_yourself",
		      GTK_SIGNAL_FUNC (save_state), (gpointer) "gtalk");
#endif

  /* This is evil.  Fix it someday */
  Ctxt = CtxtIn;

  /* Did it succeed? */
  if(gdk_display == NULL)
    return;

  Xio = GTL_x(GDK_DISPLAY());
  Ctxt->tty = Xio;		/* we get user input here */

  /* Setup management hooks so our new device does stuff */
  Xio->readme  = X_xread;
  Xio->timeout = 1;
  Xio->timefn  = NULL;
  Xio->timemsg = NULL;

  /* Now that things are going well, lets make the fixed font.
     We should probably allow user specified fonts, but I'm not sure
     how to do that yet. */
  fixed_font = gdk_font_load("-*-courier-medium-r-*-*-120-*");
  if(fixed_font == NULL)
    {
      fixed_font = gdk_font_load("fixed");
    }
  font_height = fixed_font->ascent + fixed_font->descent;
  /* Using subtraction lets us have the inter char space included. */
  font_width = gdk_text_measure(fixed_font, "MM", 2) -
    gdk_text_measure(fixed_font, "M", 1);

#ifndef GTK_ONLY
  /* Create one of those gnome app thingies... */
  gnomeapp = gnome_app_new("gtalk", "GNU Talk");

  toplevel = gnomeapp;

#else
  toplevel = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(toplevel), "GNU Talk");
#endif

  gtk_signal_connect(GTK_OBJECT(toplevel), "destroy",
		     GTK_SIGNAL_FUNC(GTK_close), NULL);

  vbox1 = gtk_vbox_new(False, 0);

#ifdef GTK_ONLY
  gtk_container_add(GTK_CONTAINER(toplevel), vbox1);
#endif

  menubar = gtk_menu_bar_new();

#ifdef GTK_ONLY
  gtk_box_pack_start(GTK_BOX(vbox1), menubar, FALSE, FALSE, 0);
#else
  gnome_app_set_menus(GNOME_APP(gnomeapp), GTK_MENU_BAR(menubar));
  gnome_app_set_contents(GNOME_APP(gnomeapp), GTK_WIDGET(vbox1));
#endif

  X_make_menus(menubar);

#ifndef GTK_ONLY
  /* Create an extra menu item: start with HELP.  How to do this? */

  /* The about box */
  about = gtk_menu_item_new_with_label("About...");
  gtk_menu_append(GTK_MENU(lastmenu), about);
  gtk_signal_connect(GTK_OBJECT(about), "activate",
		     GTK_SIGNAL_FUNC(showabout), NULL);
  gtk_signal_connect(GTK_OBJECT(about), "enter_notify_event",
		     GTK_SIGNAL_FUNC(menu_show_help),
		     (gpointer)"About GTK Gtalk.");

  gtk_widget_show(about);
#endif

  /* A pane is just the line with the grip, so make a special box. */
  talk_pane = gtk_vbox_new(True, 0);
  gtk_widget_set_usize(GTK_WIDGET(talk_pane), 80*font_width, 12*font_height);
  gtk_box_pack_start(GTK_BOX(vbox1), talk_pane, TRUE, TRUE, 0);

  echo_area = gtk_entry_new();
  gtk_widget_set_usize(GTK_WIDGET(echo_area), 80*font_width, 0);
  gtk_box_pack_end(GTK_BOX(vbox1), echo_area, FALSE, FALSE, 0);

  style = GTK_WIDGET(echo_area)->style;
  /* Copied this from gtkstyle.c */
  gdk_font_unref (style->font);
  style->font = fixed_font;
  gdk_font_ref (style->font);
  
  gtk_widget_show (menubar);
  gtk_widget_show (echo_area);
  gtk_widget_show (talk_pane);
  gtk_widget_show (vbox1);
  gtk_widget_show (toplevel);

  XFlush(GDK_DISPLAY());
     
  return &GTKMethods;
}

#ifndef GTK_ONLY
static gint
save_state (GnomeClient        *client,
	    gint                phase,
	    GnomeRestartStyle   save_style,
	    gint                shutdown,
	    GnomeInteractStyle  interact_style,
	    gint                fast,
	    gpointer            client_data)
{
  gchar *session_id;
  gchar *argv[3];
  gint x, y, w, h;

  session_id= gnome_client_get_id (client);

  /* The only state that cac has is the window geometry. 
     Get it. */
  gdk_window_get_geometry (gnomeapp->window, &x, &y, &w, &h, NULL);

  /* Save the state using gnome-config stuff. */
  gnome_config_push_prefix (gnome_client_get_config_prefix (client));
  
  gnome_config_set_int ("Geometry/x", x);
  gnome_config_set_int ("Geometry/y", y);
  gnome_config_set_int ("Geometry/w", w);
  gnome_config_set_int ("Geometry/h", h);

  gnome_config_pop_prefix ();
  gnome_config_sync();

  /* Here is the real SM code. We set the argv to the parameters needed
     to restart/discard the session that we've just saved and call
     the gnome_session_set_*_command to tell the session manager it. */
  argv[0] = (char*) client_data;
  argv[1] = "--discard-session";
  argv[2] = gnome_client_get_config_prefix (client);
  gnome_client_set_discard_command (client, 3, argv);

  /* Set commands to clone and restart this application.  Note that we
     use the same values for both -- the session management code will
     automatically add whatever magic option is required to set the
     session id on startup.  */
  gnome_client_set_clone_command (client, 1, argv);
  gnome_client_set_restart_command (client, 1, argv);

  return TRUE;
}
#endif

static void GTK_close()
{
  gtk_exit(0);

  Xio->fd = 1; /* to prevent clean_dev from erroring out */
  GT_clean_dev(Xio, False);  
}


/*
 * Function: GTK_iconify
 *
 *   Locally defined function which will suspect gtalk via iconization.
 *
 * Returns:     static void - 
 * Parameters:  None
 *
 * History:
 * zappo   3/11/97    Created
 */
static void GTK_iconify(Ctxt)
     struct TalkContext *Ctxt;
{ 
  GtkWindow *win = GTK_WINDOW(toplevel);
  Display *disp = GDK_DISPLAY();

  /* Iconify whatever we have */
  XIconifyWindow(disp, GDK_WINDOW_XWINDOW(win), DefaultScreen(disp));
}


/*
 * Function: GTK_fix_user_windows
 *
 *   Takes the window list pointer, and redoes all the windows to make
 * sure that they are the right size, and that all windows fit on the
 * display.
 *
 * Returns:     struct  - 
 * Parameters:  Ctxt - Current context.
 *              list - Pointer toWindow list
 *
 * History:
 * zappo   8/30/95    Created
 */
static void GTK_fix_user_windows(Ctxt, list)
     struct TalkContext *Ctxt;
     struct WindowList *list;
{
  struct WindowList *loop;
  gint winsize;
  gint lheight;
  int count, done, uncount = 0;
  
  /* count up the number of windows. In this loop, LOCAL won't count,
   * so start at 1. */
  for(loop = NextElement(list), count = 1; loop; loop=NextElement(loop))
    {
      if(DISP_window_displayed(loop))
	count++;
      else if(loop->group)
	uncount++;
    }

  /* Now, lets see how much space we have */
  winsize = GTK_WIDGET(talk_pane)->allocation.height;
  

  if(winsize/count < 100)
    {
      winsize = 100*count;
    }

  /* this sets the minimum size, but won't resize if we are to go
   * smaller for some reason.
   */
  gtk_widget_set_usize(GTK_WIDGET(talk_pane), 80*font_width, 100*count);

  /* rebuild the window list */
  for(loop = list, done = 0; loop; loop = NextElement(loop))
    {
      /* c short circuit protects us from a seg-fault */
      /* a non-user window is local guy, and all others have user pointers */
      /* if a user is cleaned, windows will have been cleaned */
      if(DISP_window_displayed(loop))
	{
	  if(loop->window)
	    {
	      /* Set the height of the window to this new size.*/
	      lheight = (font_height + 
			 GTK_LABEL(loop->statusline)->misc.ypad * 2);

	      /* Fit the text in here perfectly. */
	      gtk_widget_set_usize(GTK_WIDGET(loop->window),
				   0, winsize/count - lheight);
	      gtk_widget_set_usize(GTK_WIDGET(loop->group), 0, winsize/count);

	    }
	  else
	    {
	      GtkWidget *vscrollbar;
	      GtkWidget *hbox;
	      /* Create a group which will contain the text widget
	       * and the label widget.
	       */
	      loop->group = gtk_vbox_new(False, 0);
	      if(loop->user == NULL)
		{
		  /* The first pane (localpane) always goes on top! */
		  gtk_box_pack_start(GTK_BOX(talk_pane), loop->group,
				     TRUE, TRUE, 0);
		}
	      else
		{
		  gtk_box_pack_end(GTK_BOX(talk_pane), loop->group,
				   TRUE, TRUE, 0);
		}
	      hbox = gtk_hbox_new(False, 0);
	      gtk_box_pack_start(GTK_BOX(loop->group), hbox,
				 TRUE, TRUE, 0);
	      
	      loop->window = gtk_text_new(NULL, NULL);
	      vscrollbar = gtk_vscrollbar_new (GTK_TEXT(loop->window)->vadj);
	      gtk_box_pack_start(GTK_BOX(hbox), loop->window,
				 TRUE, TRUE, 0);
	      gtk_box_pack_start(GTK_BOX(hbox), vscrollbar,
				 FALSE, FALSE, 0);
	      /* This is the label under each user's text showing
		 their status. */
	      loop->statusline = gtk_label_new("----Status---");
	      gtk_box_pack_end(GTK_BOX(loop->group), loop->statusline,
			       FALSE, FALSE, 0);
	      lheight = (font_height + 
			 GTK_LABEL(loop->statusline)->misc.ypad * 2);

	      /* Fit the text in here perfectly. */
	      gtk_widget_set_usize(GTK_WIDGET(loop->window),
				   0, winsize/count - lheight);
	      gtk_widget_set_usize(GTK_WIDGET(loop->group), 0, winsize/count);

	      gtk_widget_show (GTK_WIDGET(loop->statusline));

	      gtk_widget_show (GTK_WIDGET(hbox));
	      gtk_widget_show (GTK_WIDGET(loop->window));
	      gtk_widget_show (GTK_WIDGET(vscrollbar));
	      gtk_widget_realize (GTK_WIDGET(loop->window));
	      gtk_widget_show (GTK_WIDGET(loop->group));
	    }
	}
      else
	{
	  /* We must free up the windows if they exist! */
	  if(loop->group)
	    {
	      GTK_delwin(loop->group);
	      loop->group = NULL;
	    }
	}
    }

  /* This routine is oft called from deep within other things which do
   * not call the X read routine, so call it here just to be sure */
  X_xread(Ctxt, NULL);
}


/*
 * Function: GTK_winwidth
 *
 *   Return width of the gtalk window in characters displayed.
 *
 * Returns:     Nothing
 * Parameters:  window - EmptyPointer to window
 *
 * History:
 * zappo   8/30/95    Created
 */
static int GTK_winwidth(window)
     void *window;
{ 
  gint winsize, winwidth;

  gdk_window_get_size(GTK_WIDGET(talk_pane)->window, &winwidth, &winsize);

  return winwidth / font_width;
}


/*
 * Function: GTK_clearwin, puttext, refresh, refresh_screen
 *
 *   Window contents editing routines
 *
 * Returns:     Nothing
 * Parameters:  win - EmptyPointer to win
 *
 * History:
 * zappo   8/30/95    Created
 */
static void GTK_delete_bunch_o_chars (window, num)
     GtkWidget *window;
     int num;
{
  int len = gtk_text_get_length(GTK_TEXT(window));

  gtk_text_set_point(GTK_TEXT(window), len);

  gtk_text_set_editable(GTK_TEXT(window), TRUE);

  gtk_text_backward_delete(GTK_TEXT(window), num);

  /* Always false, otherwise we can do stuff w/ the mouse. */
  gtk_text_set_editable(GTK_TEXT(window), TRUE);  

  X_xread(Ctxt, NULL);
}
static void GTK_clearwin(window)
     void *window;
{
  int len = gtk_text_get_length(GTK_TEXT(window));
  /* Delete's all text in the text control WINDOW */

  GTK_delete_bunch_o_chars(GTK_WIDGET(window), len);
}

static void GTK_delchar(window)
     void *window;
{
  GTK_delete_bunch_o_chars(GTK_WIDGET(window), 1);
}

static void GTK_delword(window)
     void *window;
{
  int len = gtk_text_get_length(GTK_TEXT(window));
  int count = 0;

  gtk_text_set_point(GTK_TEXT(window), len);

  /* Skip over whitespace first */
  while (((len - count) != 0) && 
	 ((GTK_TEXT_INDEX(GTK_TEXT(window), len-count) == ' ') ||
	  (GTK_TEXT_INDEX(GTK_TEXT(window), len-count) == '\t')))
    {
      count++;
    }

  /* Skip over the text chars. */
  while (((len - count) != 0) && 
	 (GTK_TEXT_INDEX(GTK_TEXT(window), len-count) != ' ') && 
	 (GTK_TEXT_INDEX(GTK_TEXT(window), len-count) != '\t') && 
	 (GTK_TEXT_INDEX(GTK_TEXT(window), len-count) != '\n'))
    {
      count++;
    }
  GTK_delete_bunch_o_chars(GTK_WIDGET(window), count);
}

static int GTK_line_length(window)
     GtkWidget *window;
{
  int len = gtk_text_get_length(GTK_TEXT(window));
  int count = 0;

  gtk_text_set_point(GTK_TEXT(window), len);

  while (((len - count) != 0) && (GTK_TEXT_INDEX(GTK_TEXT(window), len-count) != '\n'))
    count++;

  /* This will skip the CR character.*/
  if((count > 0) && (GTK_TEXT_INDEX(GTK_TEXT(window), len-count) == '\n'))
    count--;

  return count;
}

static void GTK_delline(window)
     void *window;
{
  GTK_delete_bunch_o_chars(GTK_TEXT(window),
			   GTK_line_length(GTK_WIDGET(window)));
}

static int GTK_fillcolumn(window)
     void *window;
{
  /* Return True when the cursor position has past the fill column */
  return ((GTK_line_length(GTK_WIDGET(window)) + 10) >
	  GTK_winwidth(window));
}


/*
 * Function: GTK_puttext
 *
 *   Locally defined function which 
 *
 * Returns:     static void - 
 * Parameters:  window - EmptyPointer  window
 *              text   - String Pointer  text
 *              size   - Number of size
 * History:
 * zappo   4/26/98    Created
 */
static void GTK_puttext(window, text, size)
     void *window;
     char *text;
     int   size;
{
  GtkWidget *textw = NULL;
  int len;

  if(window != NULL)
    {
      if(window == popup)
	textw = popuptext;
      else
	textw = window;
    }
  else return;

  /* add text to the end of WINDOW's text */
  gtk_text_set_editable(GTK_TEXT(textw), TRUE);

  len = gtk_text_get_length(GTK_TEXT(textw));
  
  gtk_text_set_point(GTK_TEXT(textw), len);
  
  gtk_text_insert(GTK_TEXT(textw), fixed_font, NULL, NULL, text, size);
  if(textw == popuptext)
    {
      /* When in popup mode, we have to insert a CR */
      gtk_text_insert(GTK_TEXT(textw), fixed_font, NULL, NULL, "\012", 1);
    }

  /* Always false, otherwise we can do stuff w/ the mouse. */
  gtk_text_set_editable(GTK_TEXT(textw), FALSE);

  X_xread(Ctxt, NULL);
}

static void GTK_refresh(window, force)
     void *window;
     int force;
{
  /* Called on C-l.  X interfaces ignore this */
}

static void GTK_refresh_screen(wlist)
     struct WindowList *wlist;
{
}


/*
 * Function: GTK_getch
 *
 *   Reads from the Xinput buffer created from the X event reader.
 * This is used to simulate a stream environment for the text reading
 * functions.
 *
 * Returns:     unsigned char  - The desired character on 0.
 * Parameters:  None
 *
 * History:
 * zappo   8/31/95    Created
 */
static char GTK_getch()
{
  unsigned char r;
  int i;

  r = Xinputbuffer[0];

  /* Copy everything backwards... This will usually be a 1 character
     string, so lets just see what happens in terms of performance. */
  for(i = 0; Xinputbuffer[i]; Xinputbuffer[i] = Xinputbuffer[i+1], i++);

  return r;
}


/*
 * Function: GTK_setlabel
 *
 *   Sets the label under a talk window
 *
 * Returns:     Nothing
 * Parameters:  win   - EmptyPointer to win
 *              label - Pointer toCharacter of label
 * History:
 * zappo   1/23/96    Created
 */
static void GTK_setlabel(win, label)
     GtkWidget *win;
     char *label;
{
  int len;

  /* Set the label on the text under a window */
  gtk_label_set(GTK_LABEL(win), label);
  gtk_label_set_justify(GTK_LABEL(win), GTK_JUSTIFY_LEFT);
}
static void GTK_set_echo_area(label, showcursor)
     char *label;
     int showcursor;
{
  /* The bolow is OK if the echo area widget is compatible with the
   * label widgets.
   */
  if(echo_area) {
    /* Set the text for a label window.  This includes
     * the labels associated w/ user windows
     */
    gtk_entry_set_text(GTK_ENTRY(echo_area), label);
    gtk_entry_set_position(GTK_ENTRY(echo_area), strlen(label));
    
    XFlush(GDK_DISPLAY());
  }
}

/*
 * Function: *_callback
 *
 *   Locally defined functions which are the callbacks bound to menu items,
 * and key presses found in the widget version of the etalk interface
 *
 * Returns:     static void  - 
 * Parameters:  w         - Widget w
 *              XCtxt     - Context
 *              call_data - 
 * History:
 * zappo   9/17/95    Created
 */
void simulate_key_callback(w, key)
     GtkWidget *w;
     gpointer key;
{
  /* This prevents spurious outputs */
  if(popup == NULL) DISP_reset_any_prompts(Ctxt);

  strcpy(Xinputbuffer+strlen(Xinputbuffer), (char *)key);

  while(strlen(Xinputbuffer) > 0) /* run new input */
    {
      DISP_suppress_feedback(Ctxt, 1);
      DISP_input(Ctxt, NULL);
      DISP_suppress_feedback(Ctxt, 0);
    }
}

/*
 * Function: showabout
 *
 *   Show an about box for GNU talk.
 *
 * Returns:     Nothing
 * Parameters:  w   - Pointer Widget w
 *              key - Number of key
 * History:
 * zappo   4/29/98    Created
 */
#ifndef GTK_ONLY
static void showabout(w, data)
     GtkWidget *w;
     gpointer data;
{
  const gchar *authors[] = { "Eric M. Ludlam", NULL };
  const gchar *chatty = "GTK front end for the GNU talk client.";

  GtkWidget *about =
    gnome_about_new("GNU Talk", VERSION, GTALK_C,
		    authors, chatty, NULL);
  gtk_widget_show(about);
}
#endif

/*
 * Function: menu_show_help
 *
 *   Locally defined function which provides help for menu items
 *
 * Returns:     static void - 
 * Parameters:  w       - Pointer Widget w
 *              message - The message text to display
 * History:
 * zappo   4/28/98    Created
 */
static void menu_show_help(w, event, message)
     GtkWidget *w;
     GdkEventCrossing *event;
     gpointer message;
{
  GTK_set_echo_area((char *)message, False);
}

/*
 * Function: GTK_popup
 *
 *   Popup window routines
 *
 * Returns:     Nothing
 * Parameters:  width  - Width of the popup window to create.
 *              height - Height of the popup window to create.
 * History:
 * zappo   8/30/95    Created
 */
static void *GTK_popup(width, height)
     int width;
     int height;
{
  GtkWidget *vbox;
  GtkWidget *aa;
  GtkWidget *ok;

  /* Create but do not MAP a new widget which acts as an info dialog. */
#ifdef GTK_ONLY
  popup = gtk_dialog_new();
  gtk_window_position(GTK_WINDOW(popup), GTK_WIN_POS_MOUSE);
  gtk_window_set_policy(GTK_WINDOW(popup), FALSE, FALSE, TRUE);
  gtk_window_set_title(GTK_WINDOW(popup), "Talk Info");
  gtk_signal_connect(GTK_OBJECT(popup), "delete_event",
		     GTK_SIGNAL_FUNC(simulate_key_callback),
		     "\033");

  aa = GTK_DIALOG(popup)->action_area;

  ok = gtk_button_new_with_label("Ok");
  gtk_signal_connect(GTK_OBJECT(ok), "clicked",
		     GTK_SIGNAL_FUNC(simulate_key_callback),
		     "\033");
  gtk_box_pack_end(GTK_BOX(aa), ok, FALSE, FALSE, 0);

  gtk_widget_show(ok);

  vbox = GTK_DIALOG(popup)->vbox;
#else

  popup = gnome_dialog_new("Talk Info", GNOME_STOCK_BUTTON_CANCEL, NULL);

  gnome_dialog_button_connect(GNOME_DIALOG(popup), 0,
			      GTK_SIGNAL_FUNC(simulate_key_callback),
			      "\033");

  vbox = GNOME_DIALOG(popup)->vbox;
#endif

  popuptext = gtk_text_new(NULL, NULL);
  gtk_widget_set_usize(GTK_WIDGET(popuptext),
		       width*font_width, height*font_height);
  GTK_TEXT(popuptext)->line_wrap = FALSE;
  gtk_box_pack_start(GTK_BOX(vbox), popuptext, TRUE, TRUE, 0);

  gtk_widget_show(popuptext);
  gtk_widget_realize(popuptext);

  return popup;
}

static void GTK_map_popup(window)
     void *window;
{
  /* Maps WINDOW into the display.  WINDOW will have been created with
   * the GTK_popup command */
  gtk_widget_show(popup);
}


/*
 * Function: GTK_delwin
 *
 *   Deletes a gtk window.  Additionally checks for popup so we can
 * delete our version of popup as well.
 *
 * Returns:     Nothing
 * Parameters:  window - EmptyPointer to window
 *
 * History:
 * zappo   8/31/95    Created
 */
static void GTK_delwin(window)
     void *window;
{
  if(window == popup)
    {
      gtk_widget_destroy(popup);
      popup = NULL;
    }
  else
    {
      gtk_widget_destroy(window);
    }
}


/*
 * Function: GTK_bell
 *
 *   Rings the bell via gtk.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *
 * History:
 * zappo   9/24/95    Created
 */
static void GTK_bell(Ctxt)
     struct TalkContext *Ctxt;
{
  XBell(GDK_DISPLAY(), 100);
  XFlush(GDK_DISPLAY());
}


/*
 * Function: XW_build_menu
 *
 *   Use this to build one widget menu with callbacks.  All callbacks
 * are given data 
 *
 * Returns:     Nothing
 * Parameters:  parent   - Widget parent
 *              name     - Name of
 *              items    -  items
 *              numitems - Number of number
 * History:
 * zappo   9/16/95    Created
 */
struct MenuItemConstructor {
  char           *button_name;
  char           *describe;
  GtkSignalFunc  *callback;
  void           *data;
};

void XW_build_menu(parent, name, items, numitems)
     GtkWidget *parent;
     char *name;
     struct MenuItemConstructor *items;
     int numitems;
{
  GtkWidget *menubutton;
  GtkWidget *submenu;
  GtkWidget *item;
  int i;

  menubutton = gtk_menu_item_new_with_label(name);
  gtk_menu_bar_append(GTK_MENU_BAR(parent), menubutton);

  submenu = gtk_menu_new();

  gtk_menu_item_set_submenu(GTK_MENU_ITEM(menubutton),
			    submenu);

  for(i=0; i<numitems; i++)
    {
      item = gtk_menu_item_new_with_label(items[i].button_name);
      gtk_menu_append(GTK_MENU(submenu), item);

      gtk_signal_connect(GTK_OBJECT(item), "activate",
			 GTK_SIGNAL_FUNC(simulate_key_callback),
			 (gpointer)items[i].data);

      gtk_signal_connect(GTK_OBJECT(item), "enter_notify_event",
			 GTK_SIGNAL_FUNC(menu_show_help),
			 (gpointer)items[i].describe);
      
      /* Add description for minibuffer when this item is selected here. */

      gtk_widget_show(item);
    }
  gtk_widget_show(menubutton);

  /* This is so we can append extra thingies if we feel like it. */
  lastmenu = submenu;
}


/*
 * Function: X_xread
 *
 *   Locally defined function which reads from the X fid and does
 * useful things with it.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *              dev  - Pointer to device
 * History:
 * zappo   9/17/95    Created
 */
Time mouse_time;

void X_xread(Ctxt, dev)
     struct TalkContext *Ctxt;	/* NOT USED */
     struct InputDevice *dev;
{
  XEvent event;

  /* Our IO looping is costly, handle X events in big chunks! In
     addition, the pending fn flushes all our X io, as a result, we
     needn't worry about things not being completed nicely. */
  while(gdk_events_pending())
    {
      XPeekEvent(GDK_DISPLAY(), &event);
  
      /* We could hang a callback on every text widget, but I feel
	 like cheating since I have this handy location where I can
	 grab all keyboard IO. */
      switch(event.type)
	{
	case KeyPress:
	  /* The following bit of genius was taken from the emacs
	     source tree: xterm.c */
	  {
	    XComposeStatus compose_status;
	    KeySym keysym;
	    int modifiers;
	    int nbytes;
	    char *bufferpos = Xinputbuffer+strlen(Xinputbuffer);
	    
	    /* Pull the event. */
	    XNextEvent(GDK_DISPLAY(), &event);

	    modifiers = event.xkey.state;

	    event.xkey.state &= ~ControlMask;
	    nbytes = XLookupString (&event.xkey, bufferpos,
				    80, &keysym, &compose_status);
	    bufferpos[nbytes] = 0; /* terminate it just in case */

	    switch(keysym)
	      {
		/* Read some of the keysyms which might not directly 
		 * translate themselves, and force a translation.  The
		 * previous command *might* handle some of these, but
		 * I just want to make sure there are no mistakes
		 */
	      case XK_BackSpace:
	      case XK_Delete:
#ifdef XK_KP_Delete
	      case XK_KP_Delete:
#endif
		sprintf(bufferpos, "%c", Ctxt->editkeys[0]);
		break;
	      case XK_Clear:
		sprintf(bufferpos, "%c", Ctxt->editkeys[1]);
		break;
		/* Here are some keysyms with no normal mapping.  Translate
		 * them into the keycodes used in etalk to perform some 
		 * useful functions.
		 */
	      case XK_Home:
#ifdef XK_KP_Home
	      case XK_KP_Home:
#endif
		/* M-f = Fix the windows */
		sprintf(bufferpos, "%c", 'f' | 1<<7);
		break;
		/* Let everything else just flow through */
	      case XK_End:
#ifdef XK_KP_End
	      case XK_KP_End:
#endif
		/* M-h = hangup on somebody */
		sprintf(bufferpos, "%c", 'h' | 1<<7);
		break;
	      case XK_Begin:
#ifdef XK_KP_Begin
	      case XK_KP_Begin:
#endif
		/* M-c = call on somebody */
		sprintf(bufferpos, "%c", 'c' | 1<<7);
		break;
	      case XK_Cancel:
	      case XK_Break:
		/* C-g = Abort key in most instances */
		sprintf(bufferpos, "%c", 7);
		break;
	      case XK_Help:
		/* M-H = GUI help */
		sprintf(bufferpos, "%c", 'h' | 1<<7);
		break;
		
	      default:
		if(modifiers & ControlMask)
		  {
		    /* If it's upper case, downshift it! */
		    if((bufferpos[0] > 'A') && (bufferpos[0] < 'Z'))
		      bufferpos[0] -= 'A' - 'a';
		    
		    if(bufferpos[0] == ' ')
		      bufferpos[0] = 0;
		    else
		      /* Now turn it into a control character */
		      /* Ctrl-a = 1, so subtract 'a' and + 1 */
		      bufferpos[0] -= ('a' - 1);
		  }

		/* Now lets turn it into a META command */
		if((modifiers & Mod1Mask) || (modifiers & Mod2Mask))
		  {
		    /* bufferpos[0] |= 1<<7; */
		    bufferpos[2] = 0;
		    bufferpos[1] = bufferpos[0];
		    bufferpos[0] = 27;  /* META prefix */
		  }
		else if(! Ctxt->eight_bit)
		  {
		    /* Only clear high bit if eight_bit display is off.*/
		    bufferpos[0] &= ~(1<<7);
		  }
	      }
#ifdef X_DUMP
	    printf("Read in [%c] total %d chars\n", bufferpos[0], nbytes);
#endif
	    /* this is ok because I terminated the string by hand */
	    while(strlen(Xinputbuffer) > 0)
	      {
		/* CRs are interpreted this way.. */
		if(Xinputbuffer[0] == 13)
		  Xinputbuffer[0] = 10;
		DISP_input(Ctxt, NULL);
	      }
	  }
	  break;
#ifdef DO_SELECTIONS
	case SelectionNotify:
	  X_do_selection(&event.xselection);
	  break;	  
	case ButtonPress:
	case ButtonRelease:
	  mouse_time = event.xbutton.time;
	  /* Store the time, and then fall through to dispatching 
	   * this event. */
#endif
	default:
	  /* Now we must figure out how to make GTK take over 
	   * event processing for this single event.
	   * I cheated and looked at the source.  I think
	   * this will do it.
	   */
	  gtk_main_iteration();

	  XFlush(GDK_DISPLAY());

	  /* This is probably repetative, but that's ok. */
	  GT_set_quicktimer(X_xread, Xio);
	  break;
	}

    }

  /* This will clean up timers and the like... */
  gtk_main_iteration_do(FALSE);
}
