/* gt_scan.c - Scan input from other talk clients
 *
 * Copyright (C) 1997, 1998, 1999 Free Software Foundation
 * Copyright (C) 1995, 1996 Eric M. Ludlam
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.org.
 * 
 * 
 * Description:
 * 
 *   These routines will manage the scanning of an input stream from a
 * remote talk client.  Text will be scanned for codes described in
 * EMBEDED.TXT, which will be removed from the input stream.
 * 
 * $Log: gt_scan.c,v $
 * Revision 1.15  1999/11/29 17:05:16  zappo
 * Converted old crypt code into generic filter code.
 *
 * Revision 1.14  1998/10/11 11:35:54  zappo
 * Changed GTALK_VERSION to VERSION for automake.
 *
 * Revision 1.13  1998/01/04 13:32:13  zappo
 * Fixed warnings
 *
 * Revision 1.12  1997/12/14 19:16:57  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.11  1997/10/17 02:26:17  zappo
 * Added new answer protocol section for dealing w/ out-of-sync answers in
 * a tcp stream.  Also free scanbuff before calling any function that might
 * in turn recurse into scan_evaluate, thus protecting from some messes.
 *
 * Revision 1.10  1997/10/14 02:13:30  zappo
 * Fixed so that unknown scanned messages are assumed to be text minibuffer
 * messages.
 *
 * Revision 1.9  1997/10/03 23:21:53  zappo
 * Fixed SCAN_send_version to use My delete-line char, not remotes.
 *
 * Revision 1.8  1997/03/12 00:27:38  zappo
 * Placed a print statement into a if(verbose) block.
 *
 * Revision 1.7  1997/02/01 14:32:28  zappo
 * Changed USER_send and USER_read to also take Ctxt
 *
 * Revision 1.6  1997/01/26  15:30:35  zappo
 * Changed what was put into the version send string for use with updated
 * configure script
 *
 * Revision 1.5  1996/07/27  21:29:22  zappo
 * Replaced send routines with USER_send so encryption is maintained
 * and added the ENCRYPT_CONNECTION message manager.
 *
 * Revision 1.4  1996/03/02  03:19:40  zappo
 * Fixed some warnings
 *
 * Revision 1.3  1995/12/10  00:15:27  zappo
 * Added check for etalk 0.10 codes
 *
 * Revision 1.2  1995/11/26  23:09:49  zappo
 * Use DISP_yorn in Socket mode cleans up emacs interface
 *
 * Revision 1.1  1995/09/29  08:37:25  zappo
 * Initial revision
 *
 * History:
 * zappo   9/22/95    Created
 *
 * Tokens: ::Header:: gtalkc.h
 */

#include "gtalklib.h"
#include "gtalkc.h"


/*
 * Function: scanfor
 *
 *   Locally defined function which scans for CH in BUFF bound by SIZE
 *
 * Returns:     static char * - 
 * Parameters:  buff - Pointer toCharacter of buffer
 *              ch   - Character of ch
 *              size - Number of size
 * History:
 * zappo   9/23/95    Created
 */
static char *scanfor(buff, ch, size)
     char *buff;
     char  ch;
     int   size;
{
  int i;
  
  for(i=0; (i<size) && (buff[i]!=ch); i++);

  if(i == size) return NULL;

  return &buff[i];
}

/*
 * Function: SCAN_send_version
 *
 *   Sends the encoded version to the remote, if necessary.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *              uo   - Pointer to uo
 * History:
 * zappo   9/24/95    Created
 */
void SCAN_send_version(Ctxt, uo)
     struct TalkContext *Ctxt;
     struct UserObject  *uo;
{
  if(Ctxt->runstate != Socket)
    {
      char buffer[40];			   /* shouldn't be more that 25 */
      
      sprintf(buffer, "%c%c%s %d GNUTALK %s%c\n",
	      ETALK_ESCAPE,
	      TCP_VERSION_MESSAGE,
	      Ctxt->myname,	/* our username */
	      htons(Ctxt->remote_connect->raddr.sin_port), /* our connect port # */
	      VERSION,	/* our version */
	      Ctxt->editkeys[1]); /* delete-line, for non-awaretalk programs */
      /* version and delete  */
      USER_send(Ctxt, uo, buffer, strlen(buffer));

      if(verbose)
	printf("Sent VERSION information\n");
    }
}

/*
 * Function: SCAN_remote_io
 *
 *   Scans the input string from user USER for embedded codes.  These
 * codes include communication for specific types of operations
 * including opening of extra channels, reporting of extra users which
 * may be connected, and most importantly, transferring the version
 * numbers between clients.
 *
 * Returns:     Nothing
 * Parameters:  uo    -  uo
 *              input - Pointer toCharacter of input
 * History:
 * zappo   9/22/95    Created
 */
void SCAN_remote_io(Ctxt, uo, input, size)
     struct TalkContext *Ctxt;
     struct UserObject  *uo;
     char               *input;
     int                *size;
{
  char *tmptxt, *tmp, *new;
  int newsize, i, j, tmpsize;

  i = 0;
  tmptxt = input;
  
  do{
    tmpsize = *size - (tmptxt - input);

    /* Look to see if we are collecting, and add data up to the EOL */
    if(uo->scanbuff)
      {	
	/* Create the new scanbuff from the existing characters */
	tmp = scanfor(tmptxt, '\n' /* 10: LF */, tmpsize);
	if(tmp)
	  {
	    /* Advance tmp to point PAST the CR */
	    tmp++;
	    newsize = strlen(uo->scanbuff) + (tmp - tmptxt) + 1;
	  }
	else
	  newsize = strlen(uo->scanbuff) + tmpsize + i;
	new = (char *)malloc(newsize);
	strcpy(new, uo->scanbuff);
	if(tmp)
	  strncat(new, input, tmp - input); /* this will terminate it with 0 */
	else
	  strncat(new, input, tmpsize);
	free(uo->scanbuff);
	uo->scanbuff = new;
	
	/* Modify the input buffer */
	if(tmp)
	  {
	    for(j = (tmp - tmptxt), i=0; j < tmpsize; j++, i++)
	      tmptxt[i] = tmptxt[j];
	    *size -= (tmp - tmptxt);	   /* update size           */
	  }
	else
	  {
	    *size -= tmpsize;		   /* update size           */
	  }

	/* If we found a \n, then lets do something based on that */
	if(tmp) 
	  {
	    SCAN_evaluate(Ctxt, uo);	    
	  }
      }

    if(*size)
      {
	tmpsize = *size - (tmptxt - input);

	tmp = scanfor(tmptxt, 3, tmpsize);

	if(tmp)
	  {
	    /* Initialize the message scan buffer */
	    uo->scanbuff = (char *)malloc(2);
	    strcpy(uo->scanbuff, "\03");
	    
	    /* Move tmptxt up to this pointer */
	    tmptxt = tmp;

	    /* remove the C-c */
	    for(j = 1, i=0; j < tmpsize; j++, i++)
	      tmptxt[i] = tmptxt[j];
	    *size -= 1;			   /* removed one char */
	  }
      }
  } while((uo->scanbuff) && *size);

  /* Here we are done scanning and input is suitably modified without
     the embeded codes */
}

/*
 * Function: SCAN_evaluate
 *
 *   Evaluates the scanbuffer belonging to UO, and then frees it.  See the
 * file EMBEDED.TXT for details on the embeded codes, and what they mean.
 *
 * Returns:     Nothing
 * Parameters:  uo -  uo
 *
 * History:
 * zappo   9/23/95    Created
 */
void SCAN_evaluate(Ctxt, uo)
     struct TalkContext *Ctxt;
     struct UserObject *uo;
{

  if(uo->scanbuff)
    {
      if(verbose)
	printf("Scanned (%d)\"%s\"\n", uo->scanbuff[1], uo->scanbuff + 2);
      
      switch(uo->scanbuff[1])	   /* scan the code character */
	{
	case 1:			/* multi-link data */
	case 4:			/* y/n quesion */
	case 5:			/* user list request, or response */
	case 7:			/* file to buffer send */
	  if(Ctxt->runstate == Socket)
	    {
	      GT_send(uo->local, uo->scanbuff, strlen(uo->scanbuff));
	    }
	  else
	    DISP_message(Ctxt, "Unsuppoted message type received!");
	  break;
	case 2:			/* query answer */
	  if(Ctxt->runstate == Socket)
	    {
	      GT_send(uo->local, uo->scanbuff, strlen(uo->scanbuff));
	    }
	  else if(!uo->query_fn)
	    {
	      char *b = (char *)malloc(strlen(uo->scanbuff) + 25);
	      sprintf(b, "No query got answer [%s]", uo->scanbuff);
	      DISP_message(Ctxt, b);
	      free(b);
	    }
	  else
	    {
	      /* Call the query callback.  Callback must look at
		 scanbuff on it's own! */
	      uo->query_fn(Ctxt, uo);
	    }
	  break;
	case TCP_PERSONAL_NAME:
	  {
	    char *tok;
		
	    /* Also send this to interface */
	    if(Ctxt->runstate == Socket)
	      {
		GT_send(uo->local, uo->scanbuff, strlen(uo->scanbuff));
	      }

	    tok = strtok(uo->scanbuff+2, "\n");
	    if(tok)
	      {
		uo->longname = (char *)malloc(strlen(tok)+1);
		if(!uo->longname)
		  gtalk_shutdown("Error allocing space for preferred name\n");
		strcpy(uo->longname, tok);
		DISP_update_user(Ctxt, uo);
	      }
	  }
	  break;
	case TCP_CONNECT_REQUEST:
	  {
	    struct UserObject *fu;
	    char *user, *host, *port;
	    int p;
	    
	    user = strtok(&uo->scanbuff[2], "@");
	    if(user)
	      {
		host = strtok(NULL, " ");
		
		fu = USER_namefind(user, host);
		
		if(!fu)
		  {
		    port = strtok(NULL, "\n");
		    p = atoi(port);
		    
		    if(p)
		      {
			struct UserObject *new;
			int doit = 0;
			/* Make sure the user wants to connect to
			   this person */
			if(Ctxt->querynewtcp == AlwaysConnect) doit = 1;
			if(Ctxt->querynewtcp == QueryConnect)
			  {
			    char qbuff[100];
			    sprintf(qbuff, "%s is talking to %s. Attach?",
				    uo->name, user);
			    if(DISP_yes_or_no(Ctxt, qbuff))
			      doit = 1;
			  }
			/* Allocate windows as necessary */
			new = USER_alloc();
			if(!new)
			  {
			    gtalk_shutdown("Error creating new user!\n");
			  }
			new->name = strdup(user);
			if(!new->name)
			  gtalk_shutdown("strdup failed in scanner!");
			
			/* Setup windows */
			if(Ctxt->runstate == Socket)
			  {
			    /* Tell emacs about the window we need by */
			    /* Sending unsolicited new tcp connection request*/
			    printf("%c%c%d %s@%s\n", 
				   ETALK_ESCAPE, TTY_NEW_USER,
				   new->id, new->name, host);
			    /* Place this new socket directly into our
			       new user struct, and set it up as a
			       local port */
			    new->local     = TCP_accept(Ctxt->emacs_connect);
			    new->local->state   = WAITING;
			    new->local->readme  = STREAM_local_read;
			    new->local->timeout = 0;
			    new->local->timefn  = NULL;
			  }
#ifndef NO_IFACE
			else
			  DISP_new_user(Ctxt, new);			
#endif	
			/* Make the connection */
			PROTOCOL_connect(Ctxt, new->id, p, new->name,
					 host, NULL);
		      }
		    else
		      DISP_message(Ctxt, "Bad data transfer in connection request");
		  }
		else
		  {
		    if(verbose)
		      printf("Duplicate user: %s@%s found\n", user, host);
		  }
	      }
	    else
	      DISP_message(Ctxt, "Bad data transfer in connection request");
	  }
	  break;
	case TCP_VERSION_MESSAGE:
	  {
	    char          *tok;
	    char          *buffer;

	    /* Send interface this data */
	    if(Ctxt->runstate == Socket)
	      {
		GT_send(uo->local, uo->scanbuff, strlen(uo->scanbuff));
	      }
	    /* Also grab this info for ourselves, AND respond */
	    /* First grab the user name (in case of unsolicited connect) */
	    tok = strtok(uo->scanbuff+2, " \n");
	    if(tok)
	      {
		if(strcmp(tok, "ETALK") == 0)
		  {
		    /* In this case we have an OLD version of etalk!
		     * Parse correctly for these older versions.
		     */
		    uo->type = ETALK;
		    tok = strtok(NULL, " .");
		    if(tok)
		      {
			uo->ver = atoi(tok);
			tok = strtok(NULL, " \n\025");
			if(tok)
			  uo->num = atoi(tok);
		      }
		  }
		else 
		  {
		    char *free_me = NULL;

		    if(strcmp(tok, uo->name))
		      {
			/* It's different, so make a new name! */
			if(uo->name)
			  free(uo->name);
			uo->name = strdup(tok);
			if(!uo->name)
			  gtalk_shutdown("Error creating strduping username");
		      }

		    /* Now grab to port connection number (for future reference) */
		    tok = strtok(NULL, " ");
		    if(tok)
		      {
			uo->connect = atoi(tok); /* connect port number */
		      }
		    /* Now grab the talk version */
		    tok = strtok(NULL, " ");
		    if(!strcmp(tok, "ETALK"))
		      uo->type = ETALK;
		    else if(!strcmp(tok, "GNUTALK"))
		      uo->type = GNUTALK;
		    else
		      uo->type = Unknowntalk;

		    /* now grab major/minor version of remote client */
		    if(uo->type)
		      {
			tok = strtok(NULL, ".");
			if(tok)
			  {
			    uo->ver = atoi(tok);
			    tok = strtok(NULL, "\025\n");
			    if(tok)
			      uo->num = atoi(tok);
			  }
#ifndef NO_IFACE
			/* Only do this ourselves if there is no interface */
			if(Ctxt->runstate != Socket)
			  {
			    buffer = (char *)malloc(4 + 5 + 
						    strlen(Ctxt->myprefname));

			    /* Send the connect port followed by the preferred name */
			    sprintf(buffer, "%c%c%s\n",
				    ETALK_ESCAPE,
				    TCP_PERSONAL_NAME,
				    Ctxt->myprefname);
			    USER_send(Ctxt, uo, buffer, strlen(buffer));

			    free(buffer);
			  }
#endif
		      }

#ifndef NO_IFACE
		    /* Only update display if not in socket mode */
		    if(Ctxt->runstate != Socket)
		      DISP_update_user(Ctxt, uo);
#endif
		    free_me = uo->scanbuff;
		    uo->scanbuff = NULL; /* Prevent annoying appending */


		    /* Last but not least, send connect requests to all our
		     * friends who might not be talking to this individual
		     * Do this here even for interfaces so the interface
		     * doesn't have to worry about it.
		     */
		    USER_send_link_commands(Ctxt, uo);

		    free(free_me);
		  }
		/* Here we decide if there is a default filter to use... */
		if((Ctxt->default_filter != 0) &&
		   ((uo->type == ETALK) || (uo->type == GNUTALK)))
		  {
		    FILTER_init(Ctxt, uo, Ctxt->default_filter);
		  }
	      }
	    break;
	  }
	case TCP_FILTER_CONNECTION:
	  {
	    char buffer[100];
	    char *filter_type, *extra_data;
	    char *free_me = NULL;

	    filter_type = strtok(uo->scanbuff + 2, " ");
	    if(filter_type)
	      {
		extra_data = strtok(NULL, " \n");
	      }
	    else
	      {
		DISP_message(Ctxt, "Invalid Filter request.");
		break;
	      }

	    sprintf(buffer, "Filter requeset type [%s] Verifying...", 
		    filter_type);
	    DISP_message(Ctxt, buffer);

	    free_me = uo->scanbuff;
	    uo->scanbuff = NULL; /* Prevent annoying appending */
	    FILTER_respond(Ctxt, uo, filter_type, extra_data);
	    free(free_me);

	    break;
	  }
	case TCP_ANSWER:
	  {
	    char *msg = NULL;
	    if(uo->last_answer != TCP_ANSWER_INVALID) {
	      DISP_message(Ctxt, "Nasty last_answer error.  Continuing anyway.\n");
	    }
	    
	    uo->last_answer = uo->scanbuff[2]; /* The answer code is here */

	    if(uo->scanbuff[2] != '\n') 
	      msg = strtok(uo->scanbuff+3, "\n");

	    if(msg && msg[0]) {
	      if(uo->last_msg) free(uo->last_msg);
	      uo->last_msg = strdup(msg);
	    }
	    break;
	  }
	default:
	  /* Here we just pass the unknown mesasge type upwards, or
	   * ignore it (depending) */
	  if(Ctxt->runstate == Socket)
	    {
	      GT_send(uo->local, uo->scanbuff, strlen(uo->scanbuff));
	    }
	  else
	  {
	      /* Zap the CR at the end */
	      uo->scanbuff[strlen(uo->scanbuff)-1] = 0;
	      DISP_message(Ctxt, uo->scanbuff + 1);
	    }
	  break;
	}

      /* Free up the scanned buffer */
      if(uo->scanbuff) {
	free(uo->scanbuff);
	uo->scanbuff = NULL;
      }
    }
}

