/*
 * Copyright (C) 1997, 1999 Free Software Foundation
 * Copyright (C) 1994, 1995, 1996 Eric M. Ludlam
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 *
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 *
 * Please send bug reports, etc. to zappo@gnu.org.
 *
 * Purpose:
 *   This file creates/delets and otherwise manages user structures
 * which are run-time objects used while dealing with live connections
 * to other people.
 * 
 * $Log: gt_user.c,v $
 * Revision 1.28  1999/11/29 17:05:31  zappo
 * Converted old crypt code into generic filter code.
 *
 * Revision 1.27  1998/01/04 13:32:28  zappo
 * Fixed warnings
 *
 * Revision 1.26  1997/12/14 19:17:24  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.25  1997/11/05 03:46:30  zappo
 * Use return sizes from crypt commands in _read and _write functions.
 * Safer than using strlen for large requests.
 *
 * Revision 1.24  1997/10/17 02:26:56  zappo
 * Added USER_get_answer which scans a remote user until an answer comes
 * back in last_answer.
 *
 * Revision 1.23  1997/10/07 00:06:26  zappo
 * Added USER_OnlyOne, which returns status of how many people are currently
 * hooked in.
 *
 * Revision 1.22  1997/10/04 12:34:05  zappo
 * Fixed user lookup by name
 *
 * Revision 1.21  1997/10/03 23:20:39  zappo
 * Fixed iofind function.
 *
 * Revision 1.20  1997/07/22 13:22:00  zappo
 * Changed to use the list library.
 *
 * Revision 1.19  1997/03/12 00:25:46  zappo
 * Added new window part to user struct.  Set this to null when hanging up on
 * someone.  Also added AutoClean checks.
 *
 * Revision 1.18  1997/02/23 03:25:16  zappo
 * API change on GT_clean_dev changes.
 *
 * Revision 1.17  1997/02/01 14:22:08  zappo
 * Changed USER_send and USER_read to also take Ctxt
 *
 * Revision 1.16  1997/01/28  03:21:30  zappo
 * Fixed core dump during print, and change return of USER_send
 *
 * Revision 1.15  1996/07/28  18:02:53  zappo
 * Added encryption fields to data, added USER_send and USER_read so that
 * output is encrypted, or decryped as needed automatically.  Fixed print
 * command so that encryption information is displayed, and fixed a
 * search routine.
 *
 * Revision 1.14  1996/03/03  15:46:53  zappo
 * Fixed finduser logic problems
 *
 * Revision 1.13  1996/02/26  00:19:48  zappo
 * Added finddata (to find users who have associated datalinks) and
 * updated duplicate finder code, and warnings
 *
 * Revision 1.12  1996/02/01  02:32:31  zappo
 * Updated to handle pipe parts, and a new search fn.
 *
 * Revision 1.11  1995/12/10  00:16:03  zappo
 * Fixed some printing stuff, and checks for etalk 0.10
 *
 * Revision 1.10  1995/11/27  00:04:55  zappo
 * When hanging up on a user: use _abort, not _delete_all
 *
 * Revision 1.9  1995/11/22  03:40:20  zappo
 * In USER_hangup discovered use of u->remote after it was cleaned.  Also
 * fixed check against wrong state befure _delete_all was called.
 *
 * Revision 1.8  1995/09/29  08:40:08  zappo
 * Updated to handle new fields in the user structure, added namefind
 * function, and send_link_commands (to send commands to other users to
 * contact UO), and handle new talk type values
 *
 * Revision 1.7  1995/09/20  23:29:33  zappo
 * Made updates needed for use with new structure names, and for use with
 * new dialog box stuff, as well as adding a new user state UNREACHABLE
 *
 * Revision 1.6  1995/07/16  16:12:06  zappo
 * Added a new state for a user which us UNREACHABLE meaning in the
 * process of being deleted after a failed attempt to connect.
 *
 * Revision 1.5  1995/04/03  00:58:37  zappo
 * Cast needed for -traditional to be happy in gcc.
 *
 * Revision 1.4  1995/03/25  03:57:30  zappo
 * Updateded copyright.
 *
 * Revision 1.3  1995/03/08  02:43:30  zappo
 * Fixed some printing troubles for user objects
 *
 * Revision 1.2  1995/01/29  14:25:03  zappo
 * Fixed some -Wall warnings
 *
 * Revision 1.1  1994/08/29  23:30:37  zappo
 * Initial revision
 *
 * ::Header:: gtalkc.h 
 */
#include "gtalklib.h"
#include "gtalkc.h"
#include "alloca.h"

static MakeList(list);
/*
 * Start our ids at 1 so 0 can be an unfilled flag...
 */
static int user_id = 1;


/*
 * Function: USER_alloc
 *
 * Create and add one user to the user structure list.  There are no
 * parameters since those parts are added later on...
 * 
 * Parameters: None
 *
 * History:
 * eml  4/11/94
 */
struct UserObject *USER_alloc()
{
  struct UserObject *new;

  new = (struct UserObject *)LIST_alloc(&list, sizeof(struct UserObject));

  if(!new)
    return NULL;

  new->name        	    = NULL;
  new->longname    	    = NULL;
  new->tty         	    = NULL;
  new->id          	    = user_id;
  new->type        	    = VanillaTalk;
  new->state       	    = USER_NEW;
  new->window               = NULL;
  new->outbound_filter      = 0;
  new->inbound_filter       = 0;
  new->outbound_filter_data = NULL;
  new->inbound_filter_data  = NULL;
  new->last_answer 	    = TCP_ANSWER_INVALID;
  new->last_msg    	    = NULL;
  new->scanbuff    	    = NULL;
  new->query_fn    	    = NULL;
  new->remote      	    = NULL;
  new->local       	    = NULL;
  new->outpipe     	    = NULL;
  new->inpipe      	    = NULL;
  new->datauser    	    = NULL;
  user_id++;			/* setup for next user id. */

  return new;
}

/*
 * Function: USER_clean, IsDead
 *
 * Clean out all user structs which are labelled as CLOSED
 * 
 * Parameters:
 *
 * History:
 * eml 4/21/94
 */
static unsigned char IsDead(u, criteria)
     struct UserObject *u;
     void *criteria;
{
  return ((u->state == USER_CLOSED) || (u->state == USER_UNREACHABLE));
}

void USER_clean()
{
  struct UserObject *u;

  while((u = (struct UserObject *)LIST_find(&list, IsDead, NULL)) != NULL)
    {
      if(verbose)
	printf("Freeing user %s ...\n", u->name?u->name:"Unknown");

      if(u->window) u->window->user = NULL;/* Remove the cross-reference */

      if(u->name) free(u->name); /* free the name field */
      if(u->outbound_filter_data) free(u->outbound_filter_data);
      if(u->inbound_filter_data) free(u->inbound_filter_data);

      LIST_deallocate(&list, u);
    }  
}


/*
 * Function: USER_send
 *
 *   Sends to UOs socket and fills in BUFF.  Performs filtering if necessary.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *              uo   - Pointer  uo
 *              buff - Buffer to send
 *              size - Size or buffer
 * History:
 * zappo   5/18/96    Created
 */
void USER_send(Ctxt, uo, buff, size)
     struct TalkContext *Ctxt;
     struct UserObject  *uo;
     char               *buff;
     int                 size;
{
  if(uo->remote == NULL) return; /* The user isn't connected yet */

  if(uo->outbound_filter != 0)
    {
      /* Always make 1 extra so there is room for NULL */
      char *outbuff = (char *)alloca(size+1);
      int rsize;

      if(outbuff == NULL) gtalk_shutdown("alloca failed!");

      FILTER_outbound(Ctxt, uo, buff, outbuff, size, &rsize);

      GT_send(uo->remote, outbuff, rsize);
    }
  else
    {
      GT_send(uo->remote, buff, size);
    }
}


/*
 * Function: USER_read
 *
 *   Reads from user's remote port and performs filtering if necessary
 *
 * Returns:     int  - 
 * Parameters:  Ctxt - Context
 *              uo   - Pointer  uo
 *              buff - Buffer to read into
 *              size - Size or buffer
 * History:
 * zappo   5/18/96    Created
 */
int USER_read(Ctxt, uo, buff, size)
     struct TalkContext *Ctxt;
     struct UserObject  *uo;
     char               *buff;
     int                 size;
{
  int rsize;

  if(uo->inbound_filter)
    {
      char *buffer = (char *)alloca(size);

      rsize = GT_recv(uo->remote, buffer, size);

      FILTER_inbound(Ctxt, uo, buffer, buff, rsize, &rsize);
    }
  else
    {
      rsize = GT_recv(uo->remote, buff, size);
    }

  return rsize;
}


/*
 * Function: USER_get_answer
 *
 *   Scan for an answer for the user.  Other data might be in the pipe
 * before the answer reaches us, so take care of that gracefully.
 *
 * Returns:     int - 
 * Parameters:  Ctxt - Context
 *              uo   - Pointer a user.
 * History:
 * zappo   10/16/97   Created
 */
int USER_get_answer(Ctxt, uo)
     struct TalkContext *Ctxt;
     struct UserObject  *uo;
{
  int retval;

  while(uo->last_answer == TCP_ANSWER_INVALID) {
    GT_select_all(Ctxt, uo->remote);
  }

  if(verbose) {
    printf("Got answer %d: %s\n", uo->last_answer,
	   uo->last_msg?uo->last_msg:"<None>");
  }

  retval = uo->last_answer;
  uo->last_answer = -1;

  if(uo->last_msg) {
    DISP_message(Ctxt, uo->last_msg);
    free(uo->last_msg);
    uo->last_msg = NULL;
  }

  return retval;
}


/*
 * Function: USER_find
 *
 * When given the id, return a pointer to the structure containing that users
 * information
 * 
 * Parameters: id - the id of the user
 *
 * History:
 * eml 4/11/94
 */
static unsigned char MatchId(uo, c)
     struct UserObject  *uo;
     int *c;
{
  return (uo->id == *c);
}

struct UserObject *USER_find(id)
     int id;
{
  return (struct UserObject *)LIST_find(&list, MatchId, &id);
}

/*
 * Function: USER_iofind
 *
 * Find a user struct based on an io device.
 * 
 * Parameters: io - the io device
 *
 * History:
 * eml 4/12/94
 */
static unsigned char MatchIO(uo, io)
     struct UserObject *uo;
     struct InputDevice *io;
{
  return ((uo->remote == io) || (uo->local == io) || (uo->inpipe == io));
}
struct UserObject *USER_iofind(io)
     struct InputDevice *io;
{
  return (struct UserObject *)LIST_find(&list, MatchIO, io);
}

/*
 * Function: USER_namefind
 *
 *   Finds name user object based on a name and host address
 *
 * Returns:     struct UserObject * - 
 * Parameters:  name - Name of
 *              host - Pointer toCharacter of host
 * History:
 * zappo   9/25/95    Created
 */
struct UserObject *USER_namefind(name, host)
     char *name, *host;
{
  struct UserObject *u;
  struct HostObject *h;

  u = FirstElement(list);
  h = HOST_gen_host(host);	/* this should be recycled */

  /* Look at both local and remotes.  We should never get
   * multiple instances of these things...
   */
  while(u)
    {
      if((u->state != USER_UNREACHABLE) && (u->state != USER_CLOSED) &&
	 !strcmp(name, u->name) && (h == u->remote->host))
	return u;
      u = NextElement(u);
    }

  return u;
}


/*
 * Function: USER_finddata
 *
 *   Finds the user object which is really a DATALINK bound to USER.
 *
 * Returns:     struct UserObject * - 
 * Parameters:  user - Pointer to user
 *
 * History:
 * zappo   2/25/96    Created
 */
struct UserObject *USER_finddata(user)
     struct UserObject *user;
{
  struct UserObject *u;

  u = FirstElement(list);

  while(u)
    {
      if((u->state != USER_UNREACHABLE) && (u->state != USER_CLOSED) &&
	 (u->datauser == user))
	return u;
      u = NextElement(u);
    }

  return u;
}


/*
 * Function: USER_finduser
 *
 *   Finds user user object based on NAME, where NAME is a character
 * string representing either a number encoded UID, or the given login
 * name of that user.
 *
 * Returns:     struct UserObject * - 
 * Parameters:  Ctxt - Context
 *              name - Name of
 * History:
 * zappo   1/6/96     Created
 */
static unsigned char MatchUsername(uo, name)
     struct UserObject *uo;
     char *name;
{
  return ((uo->state == USER_CONNECTED) && !strcmp(uo->name, name)); 
}
struct UserObject *USER_finduser(name)
     char               *name;
{
  int uid;
  struct UserObject *u;

  if(!name) return NULL;

  uid = atoi(name);

  if(uid != 0)
    {
      u = USER_find(uid);
    }
  else
    {
      struct UserObject *v;

      u = (struct UserObject *)LIST_find(&list, MatchUsername, name);
      /* exit if there was no initial match */
      if(!u) return NULL;

      v = (struct UserObject *)LIST_find_next(&u->link, MatchUsername, name);

      /* Flush duplicates upon multiple hosts */
      if(u && v) return NULL;
    }
  return u;
}


/*
 * Function: MatchConnected,USER_OnlyOne
 *
 *   If there is only one choice for a currently active user, return
 * that user object.  If there are _NO_ user objects, return 1.  If
 * there is more than one, return NULL.
 *
 * Returns:     struct UserObject * - 
 * Parameters:  None
 *
 * History:
 * zappo   10/5/97    Created
 */
static unsigned char MatchConnected(uo, name)
     struct UserObject *uo;
     char *name;
{
  return (uo->state == USER_CONNECTED);
}
struct UserObject *USER_OnlyOne()
{
  struct UserObject *uo;
  
  uo = (struct UserObject *)LIST_find(&list, MatchConnected, NULL);

  if(uo == NULL)
    {
      uo = (struct UserObject *)1;
    }
  else if(LIST_find_next(&uo->link, MatchConnected, NULL))
    {
      uo = NULL;
    }
  return uo;
}


/*
 * Function: USER_hangup
 *
 * Hangup on a selected user.
 * 
 * Parameters: id - user id or -1 for everyone.
 *
 * History:
 * eml 4/14/94
 */
void USER_hangup(Ctxt, id)
     struct TalkContext *Ctxt;
     int                 id;
{
  struct UserObject *u;
  int start, end, i;

  if(id == -1) {
    start = 1;
    end = user_id;
  } else {
    start = id;
    end = id + 1;
  }

  for(i = start; i < end; i++)
    {
      u = USER_find(i);		/* get the user struct      */

      if(!u) continue;		/* if error, keep on trukin */

      if(u->state == USER_CLOSED) continue;

      if(Ctxt->runstate == Socket)
	printf("\03%c%d\n", TTY_DELETED, u->id);

      GT_clean_dev(u->local, Ctxt->cleanup);
      u->local = NULL;

      if(u->state == USER_CALLING)
	PROTOCOL_abort(Ctxt);	/* we need to abort */

      GT_clean_dev(u->remote, Ctxt->cleanup);
      u->remote = NULL;

      u->state = USER_CLOSED;

    }
  /* Do this last.  Even if we didn't get anyone, this won't take long */
  if(Ctxt->cleanup == AutoClean) USER_clean();
}

/*
 * Function: USER_send_link_commands
 *
 *   Sends a CONNECT command to all users currently being displayed.
 * This command requests that that individual connect to UO.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *              uo   - Pointer to uo
 * History:
 * zappo   9/27/95    Created
 */
void USER_send_link_commands(Ctxt, uo)
     struct TalkContext *Ctxt;
     struct UserObject  *uo;
{
  struct UserObject *loop;
  char               port[10];	/* shouldn't be more that 6 */
  char               codes[3];

  /* We cannot process ETALK version less than 0,10 */
  if((uo->type == ETALK) && (uo->ver == 0) && (uo->num < 10)) return;

  /* Do not check type!  We want others to link in with their own names */

  sprintf(codes, "%c%c", ETALK_ESCAPE, TCP_CONNECT_REQUEST);

  for(loop=FirstElement(list); loop; loop=NextElement(loop))
    {
      if(loop && (uo != loop) &&
	 (loop->state != USER_UNREACHABLE) &&
	 (loop->state != USER_CLOSED) &&
	 ((loop->type == ETALK) || (loop->type == GNUTALK)))
	{
	  /* Send this user the command */
	  USER_send(Ctxt, loop, codes, 2);
	  USER_send(Ctxt, loop, uo->name, strlen(uo->name));
	  USER_send(Ctxt, loop, "@", 1);
	  USER_send(Ctxt, loop, uo->remote->host->name, 
		  strlen(uo->remote->host->name));
	  USER_send(Ctxt, loop, " ", 1);
	  sprintf(port, "%d", uo->connect);
	  USER_send(Ctxt, loop, port, strlen(port));
	  /* Woah!  is this a bug? */
	  USER_send(Ctxt, loop, "\n", 1);
	}
    }
}


/*
 * Function: USER_print
 *
 * Print out a formatted list of the users we are keeping track of,
 * including all new and all dead.
 * 
 * Parameters: Ctxt - talk context
 *
 * History:
 * eml 4/14/94
 * eml 8/30/95    added compatibility with curses display module
 */
static void USER_print_node(uo, Ctxt)
     struct UserObject *uo;
     struct TalkContext *Ctxt;
{
  static char *us[] = { "NEW", "CONNECT", "UNREACHABLE", "DEAD", "CALLING" };
  static char *tp[] = { "Unknown", "DataLnk", "Subproc", "Vanilla", 
			  "ETALK", "GNUtalk", "YTALK" };
  char buffer[200];

  sprintf(buffer, "%d\t%-7s\t%s\t%-8s%s\t%s\t%d\t%d\t%s\t%s",
	  uo->id,
	  uo->name?uo->name:"Unknown", 
	  (uo->state<(sizeof(us)/sizeof(char *)))?us[uo->state]:"bad val",
	  (uo->type<(sizeof(tp)/sizeof(char *)))?tp[uo->type]:"bad val",
	  FILTER_name(uo->outbound_filter),
	  FILTER_name(uo->inbound_filter),
	  (uo->local)?uo->local->fd:-1,
	  (uo->remote)?uo->remote->fd:-1,
	  (uo->outpipe)?"Yes":"No",
	  (uo->remote)?uo->remote->host->name:"Undetermined");
  DISP_message(Ctxt, buffer);
}
void USER_print(Ctxt)
     struct TalkContext *Ctxt;
{
  DISP_message(Ctxt,"Id Num\tName\tState\tType\t>Filter\t<Filter\tSend D\tRecv D\tFilter\tHost");

  if(FirstElement(list) != NULL)
    {
      LIST_map(&list, USER_print_node, Ctxt);
    }
  else
    {
      DISP_message(Ctxt, "No users currently active.");
    }
}


/*
 * Function: USER_number
 *
 *   Returns the number of user structures
 *
 * Returns:     int  - 
 * Parameters:  None
 *
 * History:
 * zappo   8/25/95    Created
 */
int USER_number()
{
  return LIST_map(&list, NULL, NULL);
}
