// GTK Viewer for Vrml 97 library
//
// Copyright (C) 1998 by Erik Andersen <andersee@debian.org>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA  02111-1307, USA.


#include "ViewerGtk.h"

#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>

#define VIEW_ASPECT 1.3

using namespace OpenVRML;
using namespace OpenVRML::GL;

ViewerGtk::ViewerGtk(Browser& browser,
		     const char *,
		     GtkWidget *parent) :
  ViewerOpenGL(browser),
  d_timer(0),
  d_Stop(false),
  d_redrawNeeded(false),
  d_glarea(NULL)
{
  createWidget( parent );
}

ViewerGtk::~ViewerGtk()
{
  /*
    if (d_glarea && GTK_WIDGET_REALIZED(d_glarea))
    {
    gtk_object_unref(GTK_OBJECT(d_glarea));
    d_glarea = 0;
    }
  */
}

void ViewerGtk::handleInput( ViewerOpenGL::EventInfo *e )
{
  if (d_glarea)
    if (gtk_gl_area_make_current(GTK_GL_AREA(d_glarea)))
      input(e);
}


void ViewerGtk::handleRedraw()
{
  if (d_glarea && d_redrawNeeded)
  {
    if (gtk_gl_area_make_current(GTK_GL_AREA(d_glarea)))
    {
      redraw();
      d_redrawNeeded = false;
    }
  }
}


//  Do expose/redraw callback.

static gint gtk_expose( GtkWidget* widget,
		 GdkEventExpose* event,
		 ViewerGtk* viewer )
{
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_GL_AREA(widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);
  g_return_val_if_fail (viewer != NULL, FALSE);

  /* draw only on the last expose event */
  if ( event->count > 0)
    return TRUE;

  viewer->handleRedraw();

  return TRUE;
}

void ViewerGtk::handleResize(int w, int h)
{
  if (d_glarea)
    if (gtk_gl_area_make_current(GTK_GL_AREA(d_glarea)))
      resize(w, h);
}

static gint gtk_reshape (GtkWidget* widget,
	                 GdkEventConfigure* event, 
	                 ViewerGtk* viewer)
{
  int w = widget->allocation.width;
  int h = widget->allocation.height;
 
  if(viewer)
    viewer->handleResize(w,h);

  return TRUE;
 }

//  Input callback

static gint gtk_button_press(GtkWidget*,
                             GdkEventButton* event, 
                             ViewerGtk* viewer)
{
  ViewerOpenGL::EventInfo e;
  e.event = ViewerOpenGL::EVENT_MOUSE_CLICK;
  switch (event->button)
  {
    case 1: e.what = 0; break;
    case 2: e.what = 1; break;
    case 3: e.what = 2; break;
  }
  e.x = (int)event->x;
  e.y = (int)event->y;
  
  viewer->handleInput( &e );

  return TRUE;
}


static gint gtk_button_release(GtkWidget*,
                               GdkEventButton* event, 
                               ViewerGtk* viewer)
{
  ViewerOpenGL::EventInfo e;
  e.event = ViewerOpenGL::EVENT_MOUSE_RELEASE;
  switch (event->button)
  {
    case 1: e.what = 0; break;
    case 2: e.what = 1; break;
    case 3: e.what = 2; break;
  }
  e.x = (int)event->x;
  e.y = (int)event->y;
  viewer->handleInput( &e );

  return TRUE;
}


static gint gtk_motion_notify(GtkWidget*,
                              GdkEventMotion* event, 
                              ViewerGtk* viewer)
{
  ViewerOpenGL::EventInfo e;
  e.event = ViewerOpenGL::EVENT_MOUSE_DRAG;
  e.what = 0;
  if (event->state & GDK_BUTTON1_MASK)
    e.what = 0;
  else if (event->state & GDK_BUTTON2_MASK)
    e.what = 1;
  else if (event->state & GDK_BUTTON3_MASK)
    e.what = 2;
  else
    e.event = ViewerOpenGL::EVENT_MOUSE_MOVE;
  e.x = (int)event->x;
  e.y = (int)event->y;

  viewer->handleInput( &e );
  return TRUE;
}


static gint gtk_key_press(GtkWidget* widget,
                          GdkEventKey* event,
                          ViewerGtk* viewer)
{
  ViewerOpenGL::EventInfo e;
  e.event = ViewerOpenGL::EVENT_KEY_DOWN;
  switch (event->keyval)
  {
    case GDK_Home:	 e.what = ViewerOpenGL::KEY_HOME; break;
    case GDK_Left:	 e.what = ViewerOpenGL::KEY_LEFT; break;
    case GDK_Up:	 e.what = ViewerOpenGL::KEY_UP; break;
    case GDK_Right:	 e.what = ViewerOpenGL::KEY_RIGHT; break;
    case GDK_Down:	 e.what = ViewerOpenGL::KEY_DOWN; break;
    case GDK_Page_Up:	 e.what = ViewerOpenGL::KEY_PAGE_UP; break;
    case GDK_Page_Down: e.what = ViewerOpenGL::KEY_PAGE_DOWN; break;

    default:
      if (event->length <= 0) return TRUE; // Unhandled non-printable key
        e.what = event->string[0];
      break;
  }

  viewer->handleInput( &e );
  /* prevent the default handler from being run */
  gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
  return TRUE;
}

void ViewerGtk::handleDestroy()
{
  if (d_glarea)
  {
    gtk_object_unref(GTK_OBJECT(d_glarea));
    d_glarea = 0;
  }
}

static gint gtk_destroy(GtkWidget* widget,
                        ViewerGtk* viewer)
{
  ViewerOpenGL::EventInfo e;
  e.event = ViewerOpenGL::EVENT_KEY_DOWN;
  e.what = 'q';
  viewer->handleInput( &e );

  viewer->handleDestroy();

  return TRUE;
}

void ViewerGtk::wsPostRedraw()
{
  if ( d_glarea && ! d_redrawNeeded )
  {
    d_redrawNeeded = true;
    gtk_widget_queue_draw(GTK_WIDGET(d_glarea));
  }

  // give GTK a chance to redraw if animations are running
  gtk_main_iteration_do(FALSE);
}



void ViewerGtk::wsSetCursor( CursorStyle c )
{
  if (d_glarea)
  {
    GdkCursor* cursor;
    switch(c)
    {
      case CURSOR_INHERIT:
        cursor = NULL;
	return;
      case CURSOR_INFO:
	cursor = gdk_cursor_new ( GDK_HAND1 );
	break;
      case CURSOR_CYCLE:
	cursor = gdk_cursor_new ( GDK_EXCHANGE );
	break;
      case CURSOR_UP_DOWN:
	cursor = gdk_cursor_new ( GDK_SB_V_DOUBLE_ARROW );
	break;
      case CURSOR_CROSSHAIR:
	cursor = gdk_cursor_new ( GDK_CROSSHAIR );
	break;
      default:
	cursor = gdk_cursor_new ( GDK_ARROW);
	break;
    }

    gdk_window_set_cursor (d_glarea->window, cursor );

    if(cursor != NULL)
      gdk_cursor_destroy(cursor);
  }
}

void ViewerGtk::wsSwapBuffers()
{
  if (d_glarea)
  {
    GtkGLArea *area = GTK_GL_AREA(d_glarea);
    gtk_gl_area_swap_buffers(area);
  }
}

// Timer callback calls the base class viewer update() method via
// the public timerUpdate method.

static gint timeout_callback(gpointer data)
{
  ViewerGtk *viewer = (ViewerGtk*)data;
  if (viewer) viewer->timerUpdate();
  return FALSE;
}

void ViewerGtk::timerUpdate()
{
  d_timer = 0;
  update( 0.0 );	// No gl calls should be made from update()
}

void ViewerGtk::SetStop( bool state)
{
  d_Stop=state;
  if (state==true)
    timerUpdate();
}

// 
void ViewerGtk::wsSetTimer( double t ) 
{
  if (! d_timer && ! d_Stop )
    {
      unsigned int millis = (unsigned int) (1000.0 * t);
      d_timer = gtk_timeout_add(millis, GtkFunction(timeout_callback), this);
    }
}


void ViewerGtk::createWidget( GtkWidget *window )
{
  /* Attribute list for gtkglarea widget. Specifies a
     list of Boolean attributes and enum/integer
     attribute/value pairs. The last attribute must be
     GDK_GL_NONE. See glXChooseVisual manpage for further
     explanation.
  */
  int attrlist[] = {
    //GDK_GL_ALPHA_SIZE,1,
    GDK_GL_DOUBLEBUFFER,
    GDK_GL_RGBA,
    GDK_GL_RED_SIZE,1,
    GDK_GL_GREEN_SIZE,1,
    GDK_GL_BLUE_SIZE,1,
    GDK_GL_NONE  /* last argument must be GDK_GL_NONE */
  };  

  /* create new OpenGL widget */
  d_glarea = gtk_gl_area_new(attrlist);

  if (d_glarea == NULL || !GTK_IS_GL_AREA(d_glarea) ) {
    g_error("Can't create GtkGLArea widget\n");
    return;
  }

  /* set up events and signals for GTK OpenGL widget */
  gtk_widget_set_events(GTK_WIDGET(d_glarea),
                        GDK_EXPOSURE_MASK|
                        GDK_BUTTON_PRESS_MASK|
                        GDK_BUTTON_RELEASE_MASK|
                        GDK_KEY_PRESS_MASK|
                        GDK_POINTER_MOTION_MASK);

  gtk_signal_connect(GTK_OBJECT(d_glarea), "expose_event",
                     GTK_SIGNAL_FUNC(gtk_expose), this);
  gtk_signal_connect(GTK_OBJECT(d_glarea), "key_press_event",
                     GTK_SIGNAL_FUNC(gtk_key_press), this);
  gtk_signal_connect(GTK_OBJECT(d_glarea), "motion_notify_event",
                     GTK_SIGNAL_FUNC(gtk_motion_notify), this);
  gtk_signal_connect(GTK_OBJECT(d_glarea), "button_press_event",
                     GTK_SIGNAL_FUNC(gtk_button_press), this);
  gtk_signal_connect(GTK_OBJECT(d_glarea), "button_release_event",
                     GTK_SIGNAL_FUNC(gtk_button_release), this);
  gtk_signal_connect(GTK_OBJECT(d_glarea), "destroy",
                     GTK_SIGNAL_FUNC (gtk_destroy), this);
  gtk_signal_connect(GTK_OBJECT(d_glarea), "configure_event",
                     GTK_SIGNAL_FUNC(gtk_reshape), this);
  /* minimum size */
  gtk_widget_set_usize(d_glarea, 400, (gint)(400/VIEW_ASPECT) ); 


  /* destroy this window when exiting from gtk_main() */
  gtk_quit_add_destroy(1, GTK_OBJECT(d_glarea));

  /* put glarea into window and show it */
  gtk_container_add(GTK_CONTAINER(window), d_glarea);
  gtk_widget_show(d_glarea);

  /* set focus to glarea widget */
  GTK_WIDGET_SET_FLAGS(d_glarea, GTK_CAN_FOCUS);
  gtk_widget_grab_focus(GTK_WIDGET(d_glarea));
}
