/* ups_canvas.c - Routines for handling the battery/status canvas

   Copyright (C) 2000  Henning Kulander <hennikul@ifi.uio.no>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include "support.h"
#include "ups_canvas.h"

#define PIXMAP_FILE "gupsc/ups.png"
#define PIXMAP_WIDTH 191.0
#define PIXMAP_HEIGHT 256.0

/* Coordinates for the battery meter box */
#define BATTERY_METER_X1 27.0
#define BATTERY_METER_Y1 72.0
#define BATTERY_METER_X2 53.0
#define BATTERY_METER_Y2 227.0

/* Coordinates for the low battery light */
#define BATTERY_LOW_X1 28.0
#define BATTERY_LOW_Y1 28.0
#define BATTERY_LOW_X2 53.0
#define BATTERY_LOW_Y2 37.0

/* Coordinates for the overload light */
#define BATTERY_OVERLOAD_X1 85.0
#define BATTERY_OVERLOAD_Y1 28.0
#define BATTERY_OVERLOAD_X2 111.0
#define BATTERY_OVERLOAD_Y2 37.0

/* Coordinates for the online light */
#define BATTERY_ONLINE_X1 139.0
#define BATTERY_ONLINE_Y1 28.0
#define BATTERY_ONLINE_X2 166.0
#define BATTERY_ONLINE_Y2 37.0

#define GREEN (guint32)(0xff)
#define RED (guint32)(0xff)
#define ALPHA (guint32)(0xb0)
#define GET_GREEN(p) ((p) < (0.5) ? \
		      ((guint32)((GREEN)*2*p)) : \
		      ((guint32)(GREEN)))
#define GET_RED(p) ((p) > (0.5) ? \
		      ((guint32)((RED)*2*(1-p))) : \
		      ((guint32)(RED)))

GtkWidget *canvas;
GnomeCanvasItem *battery_meter;
GnomeCanvasItem *battery_meter_value;
GnomeCanvasItem *battery_low;
GnomeCanvasItem *battery_overload;
GnomeCanvasItem *battery_online;
GnomeCanvasGroup *group;

/* Set the low battery light to status given by value */
gboolean 
ups_battery_low_set (gboolean value)
{
	if (value == TRUE) /* Show lowbattery light */
		gnome_canvas_item_set (battery_low, 
				       "fill_color_rgba",
				       GNOME_CANVAS_COLOR_A (RED, 0x00, 0x00,
							     ALPHA),
                                       /* Partial trancparancy */
				       NULL);
	else /* Hide lowbattery light */
		gnome_canvas_item_set (battery_low, 
				       "fill_color_rgba",
				       GNOME_CANVAS_COLOR_A (0x00, 0x00, 0x00, 
							     0x00), 
                                       /* Full Trancparancy */
				       NULL);
	return TRUE ;
}

/* Set the overload light to status given by value */
gboolean 
ups_battery_overload_set (gboolean value)
{
	if (value == TRUE) /* Show overload light */
		gnome_canvas_item_set (battery_overload, 
				       "fill_color_rgba",
				       GNOME_CANVAS_COLOR_A (RED, GREEN, 0x00,
							     ALPHA),
                                       /* Partial trancparancy */
				       NULL);
	else /* Hide overload light */
		gnome_canvas_item_set (battery_overload, 
				       "fill_color_rgba",
				       GNOME_CANVAS_COLOR_A (0x00, 0x00, 0x00,
							     0x00),  
                                       /* Full Trancparancy */
				       NULL);
	return TRUE;
}

/* Set the online light to status given by value */
gboolean 
ups_battery_online_set (gboolean value)
{
	if (value == TRUE) /* Show AC-Online light */
		gnome_canvas_item_set (battery_online, 
				       "fill_color_rgba",
				       GNOME_CANVAS_COLOR_A (0x00, GREEN, 0x00,
							     ALPHA),
                                       /* Partial trancparancy */
				       NULL );
	else /* Hide AC-Online light */
		gnome_canvas_item_set (battery_online, 
				       "fill_color_rgba",
				       GNOME_CANVAS_COLOR_A (0x00, 0x00, 0x00,
							     0x00), 
                                       /* Full Trancparancy */
				       NULL);
	return TRUE;
}

/* Set the batterymeter to value given by percent */
gboolean 
ups_battery_meter_set (gfloat percent)
{
	gchar *percent_string;
	gfloat p;
	
	p = (percent>100.0 ? 100.0 : percent)/100.0; /* perc/100,maxval=1*/
	
	/* Update batterymeter */
	gnome_canvas_item_set (battery_meter, 
			       "y1", BATTERY_METER_Y2
			       -(p) * (BATTERY_METER_Y2-BATTERY_METER_Y1),
			       "fill_color_rgba", 
			       GNOME_CANVAS_COLOR_A (GET_RED (p), 
						     GET_GREEN (p), 
						     0x00, ALPHA),
			       NULL);

	/* Update value string */
	percent_string = g_strdup_printf ("%3.0f%%", percent); 
	gnome_canvas_item_set (battery_meter_value,
			       "text", percent_string,
			       NULL);
	g_free (percent_string);
	return TRUE;
}

/* Set fontname for percent number */
void 
ups_battery_set_font (gchar *fontname)
{
	gnome_canvas_item_set( battery_meter_value,
			       "font_gdk", gdk_font_load(fontname),
			       NULL);     
}

/* Initialize the canvas */
void
ups_canvas_init (GtkWidget *canvas, gchar *fontname)
{
	GdkImlibImage *im=NULL;
	gchar *im_filename;
	GnomeCanvasItem *image;
	GdkPixmap *mask;
	gchar *percent_string;
	gfloat percent = 0.0;
	gfloat p;

	g_return_if_fail (canvas != NULL);
	g_return_if_fail (fontname != NULL);

	/* Make the bitmap appear in the canvas */
	p = percent/100;
	group = gnome_canvas_root (GNOME_CANVAS (canvas));
	im_filename = gnome_pixmap_file (PIXMAP_FILE);
	if (im_filename)
		im = gdk_imlib_load_image (im_filename);
	else 
		g_warning (_("Couldn't find pixmap file: %s"), PIXMAP_FILE);
	g_free (im_filename);
	if (!im)
		printf ("Couldn't open file...\n");
	else {
		gdk_imlib_render (im, PIXMAP_WIDTH, PIXMAP_HEIGHT);
		image = gnome_canvas_item_new (group, 
					       gnome_canvas_image_get_type (),
					       "image", im,
					       "x", PIXMAP_WIDTH/2,
					       "y", PIXMAP_HEIGHT/2,
					       "width", PIXMAP_WIDTH,
					       "height", PIXMAP_HEIGHT,
					       NULL);
		mask = gdk_imlib_move_mask (im); /* Get the mask */
		gtk_widget_shape_combine_mask (canvas, mask, 0, 0);
	}

	/* Draw batterypercent bar */
	battery_meter = gnome_canvas_item_new (group,
					       gnome_canvas_rect_get_type (),
					       "x1", BATTERY_METER_X1,
					       "y1", BATTERY_METER_Y2,/*h=0*/ 
					       "x2", BATTERY_METER_X2,
					       "y2", BATTERY_METER_Y2,
					       "fill_color_rgba", 
					       GNOME_CANVAS_COLOR_A (
						       GET_RED (p),
						       GET_GREEN (p),
						       0x00,
						       ALPHA),
					       "outline_color", NULL,
					       "width_units", 0.0,
					       NULL);
	
	/* Make text object */
	g_snprintf (percent_string = g_malloc (8), 8, "%3.0f%%", 100.0); 
	battery_meter_value =
		gnome_canvas_item_new (group,
				       gnome_canvas_text_get_type (),
				       "text", percent_string,
				       "x", (BATTERY_METER_X1+BATTERY_METER_X2)
				            /2.0,
				       "y", BATTERY_METER_Y2 - 6.0,
				       "anchor", GTK_ANCHOR_CENTER,
				       "font_gdk", gdk_font_load (fontname),
				       "clip_width", BATTERY_METER_X2
				                      -BATTERY_METER_X1,
				       "clip_height", 20.0,
				       "fill_color_rgba", 0xffffffff,
				       NULL);
	g_free (percent_string);
	
	/* Make lowbattery light */
	battery_low = gnome_canvas_item_new (group,
					     gnome_canvas_rect_get_type (),
					     "x1", BATTERY_LOW_X1,
					     "y1", BATTERY_LOW_Y1,
					     "x2", BATTERY_LOW_X2,
					     "y2", BATTERY_LOW_Y2,
					     "fill_color_rgba", 
					     GNOME_CANVAS_COLOR_A (
						     RED, 0x00, 0x00, ALPHA),
					     "outline_color", NULL,
					     "width_units", 0.0,
					     NULL);
	
	/* Make overload light */
	battery_overload = gnome_canvas_item_new (group,
						  gnome_canvas_rect_get_type(),
						  "x1", BATTERY_OVERLOAD_X1,
						  "y1", BATTERY_OVERLOAD_Y1,
						  "x2", BATTERY_OVERLOAD_X2,
						  "y2", BATTERY_OVERLOAD_Y2,
						  "fill_color_rgba", 
						  GNOME_CANVAS_COLOR_A (
							  RED, GREEN, 0x00, 
							  ALPHA),
						  "outline_color", NULL,
						  "width_units", 0.0,
						  NULL);
	
	/* Make AC-Online light */
	battery_online = gnome_canvas_item_new (group,
						gnome_canvas_rect_get_type (),
						"x1", BATTERY_ONLINE_X1,
						"y1", BATTERY_ONLINE_Y1,
						"x2", BATTERY_ONLINE_X2,
						"y2", BATTERY_ONLINE_Y2,
						"fill_color_rgba", 
						GNOME_CANVAS_COLOR_A (
							0x00,GREEN,0x00,ALPHA),
						"outline_color", NULL,
						"width_units", 0.0,
						NULL);
}
