/*
 *  JLib - Jacob's Library.
 *  Copyright (C) 2003, 2004  Juan Carlos Seijo Prez
 * 
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 * 
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 * 
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  Juan Carlos Seijo Prez
 *  jacob@mainreactor.net
 */

///////////////////////////////////////////////////////////////////////////////
// @author: Juan Carlos Seijo Prez
// @date: 19/04/2003
// @description: Clase para cadenas de texto.
// @notes: Compilar con soporte de funciones inline (MSVC: /Ob[1|2])
///////////////////////////////////////////////////////////////////////////////

#ifndef _JSTRING_INCLUDED
#define _JSTRING_INCLUDED

#include <JLib/Util/JTypes.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <JLib/Util/JLoadSave.h>

class JString : public JLoadSave
{
  friend bool operator==(const char *s1, JString& s2);
  friend bool operator==(JString& s1, const char *s2);
  friend bool operator!=(const char *s1, JString& s2);
  friend bool operator!=(JString& s1, const char *s2);
  friend bool operator>(const char *s1, JString& s2);
  friend bool operator<(const char *s1, JString& s2);
  friend bool operator>=(const char *s1, JString& s2);
  friend bool operator<=(const char *s1, JString& s2);

protected:
  char *data;           // Caracteres
  u32 length;  // Longitud

public:

  inline JString();                             // Constructor
  inline JString(const char *s);                // Constructor
  inline JString(JString& s, u32 start, u32 end = (u32)-1);    // Constructor de subcadena
  inline JString(const char *s, u32 start, u32 end = (u32)-1); // Constructor de subcadena
  inline JString(JString& s);                   // Constructor copia

  inline virtual ~JString();                    // Destructor
  
  inline void Clear();                          // Borra la cadena
  inline bool Equals(JString& s);               // Son iguales?
  inline bool operator==(JString& s);           // Son iguales?
  inline s32 CompareTo(JString& s);             // Comparacin de cadenas
  inline s32 CompareTo(const char *s);          // Comparacin de cadenas
  inline bool operator>(JString& s);            // Es la primera mayor?
  inline bool operator<(JString& s);            // Es la primera menor?
  inline bool operator>=(JString& s);           // Es la primera mayor o igual?
  inline bool operator<=(JString& s);           // Es la primera menor o igual?
  inline bool operator!=(JString& s);           // Son diferentes?
  inline char& operator[](s32 pos);             // Devuelve el caracter en la posicin dada
  inline void operator=(const char *s);         // Asignacin
  inline void operator=(const JString &s);      // Asignacin
  inline const char* Str();                     // Devuelve el array de caracteres
  
  inline void Append(JString& s);               // Concatena la cadena
  inline void Append(const char * s);           // Concatena la cadena
  inline void operator+=(JString& s);           // Concatena la cadena
  inline void operator+=(const char *s);        // Concatena la cadena
  inline JString & operator+(JString &s);       // Concatena la cadena
  inline JString & operator+(const char *s);    // Concatena la cadena
  inline void Append(char c);                   // Concatena un caracter
  inline void operator+=(char c);               // Concatena un caracter
  inline void Format(const char *fmt, ...);     // Da formato tipo printf

  inline operator const char *() const {return data;}  // Cast a cadena

  inline u32 Length() const {return length;} // Devuelve la longitud

  // Lee la cadena de un fichero (longitud + cadena)
  inline u32 Load(JFile &f);

  // Escribe la cadena a un fichero (longitud + cadena)
  inline u32 Save(JFile &f);
};

// Son iguales?
inline bool operator==(const char *s1, JString& s2)
{
  return (strcmp(s1, s2.data) == 0);
}

// Son iguales?
inline bool operator==(JString& s1, const char *s2)
{
  return (strcmp(s1.data, s2) == 0);
}

// Son diferentes?
inline bool operator!=(const char *s1, JString& s2)
{
  return (strcmp(s1, s2.data) != 0);
}

// Son diferentes?
inline bool operator!=(JString& s1, const char *s2)
{
  return (strcmp(s1.data, s2) != 0);
}

inline bool operator>(const char *s1, JString& s2)    // Es la primera mayor?
{
  return (strcmp(s1, s2.data) > 0);
}

inline bool operator<(const char *s1, JString& s2)    // Es la primera menor?
{
  return (strcmp(s1, s2.data) < 0);
}

inline bool operator>=(const char *s1, JString& s2)   // Es la primera mayor o igual?
{
  return (strcmp(s1, s2.data) >= 0);
}

inline bool operator<=(const char *s1, JString& s2)   // Es la primera menor o igual?
{
  return (strcmp(s1, s2.data) <= 0);
}

// Constructor
JString::JString() : length(0)
{
  data = new char[1];
  data [0] = '\0';
}

// Constructor
JString::JString(const char *s)
{
  if (s)
  {
    length = (u32)strlen(s);
    data = new char[length + 1];
    strcpy(data, s);
  }
  else
  {
    data = new char[1];
    data [0] = '\0';
    length = 0;
  }
}

// Constructor de subcadena
inline JString::JString(JString& s, u32 start, u32 end)
{
  if (end == -1)
    end = s.Length();

  if (start >= 0 && start < s.Length() && end > start && end <= s.Length())
  {
    length = end - start;
    data = new char[length + 1];
    strncpy(data, s.data + start, length);
    data[length] = '\0';
  }
  else
  {
    data = new char[1];
    data [0] = '\0';
    length = 0;
  }
}

// Constructor de subcadena
inline JString::JString(const char *s, u32 start, u32 end)
{
  if (s)
  {
    u32 len = (u32)strlen(s);
    if (end == -1)
      end = len;

    if (start >= 0 && start < len && end > start && end <= len)
    {
      length = end - start;
      data = new char[length + 1];
      strncpy(data, s + start, length);
      data[length] = '\0';
    }
    else
    {
      data = new char[1];
      data [0] = '\0';
      length = 0;
    }
  }
  else
  {
    data = new char[1];
    data [0] = '\0';
    length = 0;
  }
}

// Constructor copia
JString::JString(JString &s)
{
  length = s.length;
  data = new char[length + 1];
  strcpy(data, s.data);
}

// Destructor
JString::~JString()
{
  delete[] data;
}

// Borra la cadena
inline void JString::Clear()
{
  delete[] data;
  data = new char[1];
  
  // La cadena no tiene estado 'nulo', como mucho estar vaca
  data [0] = '\0';
  length = 0;
}

// Son iguales?
inline bool JString::Equals(JString& s)
{
  return (strcmp(data, s.data) == 0);
}

// Son iguales?
inline bool JString::operator==(JString& s)
{
  return (strcmp(data, s.data) == 0);
}

// Comparacin de cadenas
inline s32 JString::CompareTo(JString &s)
{
  return (strcmp(data, s.data));
}

// Comparacin de cadenas
inline s32 JString::CompareTo(const char *s)
{
  return (strcmp(data, s));
}

// Es la primera mayor?
inline bool JString::operator>(JString& s)
{
  return (strcmp(data, s.data) > 0);
}

// Es la primera menor?
inline bool JString::operator<(JString& s)
{
  return (strcmp(data, s.data) < 0);
}

// Es la primera mayor o igual?
inline bool JString::operator>=(JString& s)
{
  return (strcmp(data, s.data) >= 0);
}

// Es la primera menor o igual?
inline bool JString::operator<=(JString& s)
{
  return (strcmp(data, s.data) <= 0);
}

// Son diferentes?
inline bool JString::operator!=(JString& s)
{
  return (strcmp(data, s.data) != 0);
}

// Devuelve el caracter en la posicin dada
inline char& JString::operator[](s32 pos)
{
  return data[pos];
}

// Asignacin
inline void JString::operator=(const char *s)
{
  if (s)
  {
    u32 len;
    
    len = (u32)strlen(s);
    if (len > length)
    {
      delete[] data;
      data = new char[len + 1];
    }

    length = len;
    strcpy(data, s);
  }
  else
  {
    delete[] data;
    data = new char[1];
    data[0] = '\0';
    length = 0;
  }
}

// Asignacin
inline void JString::operator=(const JString &s)
{
  if (s.Length() > length)
  {
    delete[] data;
    data = new char[s.length + 1];
  }
  length = s.length;
  strcpy(data, s.data);
}

// Devuelve el array de caracteres
inline const char* JString::Str()
{
  return ((const char*)data);
}

// Concatenacin de cadenas
inline void JString::Append(JString& s)
{
  if (s.length)
  {
    u32 len;
    len = s.length;

    char *strAux = new char[length + len + 1];
    strAux[0] = 0;
    
    strcpy(strAux, data);
    delete[] data;
    
    // Evita contar los caracteres en strcat(), ya que sabemos la longitud
    strcpy(strAux + length, s.data);
    
    data = strAux;
    length += len;
  }
}

// Concatenacin de cadenas
inline void JString::Append(const char *s)
{
  u32 len = (u32)strlen(s);

  if (len)
  {
    char *strAux = new char[length + len + 1];
    strAux[0] = 0;
    
    strcpy(strAux, data);
    delete[] data;
    
    // Evita contar los caracteres en strcat(), ya que sabemos la longitud
    strcpy(strAux + length, s);
    
    data = strAux;
    length += len;
  }
}

// Concatenacin de cadenas
inline void JString::operator+=(JString& s)
{
  if (s.length)
  {
    u32 len;
    len = s.length;

    char *strAux = new char[length + len + 1];
    strAux[0] = 0;
    
    strcpy(strAux, data);
    
    // Evita contar los caracteres en strcat(), ya que sabemos la longitud
    strcpy(strAux + length, s.data);
    
    delete[] data;
    data = strAux;
    length += len;
  }
}

// Concatenacin de cadenas
inline void JString::operator+=(const char *s)
{
  u32 len = (u32)strlen(s);

  if (len)
  {
    char *strAux = new char[length + len + 1];
    strAux[0] = 0;
    
    strcpy(strAux, data);
    delete[] data;
    
    // Evita contar los caracteres en strcat(), ya que sabemos la longitud
    strcpy(strAux + length, s);
    
    data = strAux;
    length += len;
  }
}

// Concatena la cadena
inline JString & JString::operator+(JString &s)
{
  if (s.length)
  {
    u32 len;
    len = s.length;

    char *strAux = new char[length + len + 1];
    strAux[0] = 0;
    
    strcpy(strAux, data);
    
    // Evita contar los caracteres en strcat(), ya que sabemos la longitud
    strcpy(strAux + length, s.data);
    
    delete[] data;
    data = strAux;
    length += len;
  }

	return *this;
}

inline JString & JString::operator+(const char *s)
{
  u32 len = (u32)strlen(s);

  if (len)
  {
    char *strAux = new char[length + len + 1];
    strAux[0] = 0;
    
    strcpy(strAux, data);
    delete[] data;
    
    // Evita contar los caracteres en strcat(), ya que sabemos la longitud
    strcpy(strAux + length, s);
    
    data = strAux;
    length += len;
  }

  return *this;
}


// Concatena un caracter
inline void JString::Append(char c)
{
  if (c)
  {
    char *strAux = new char[length + 2];
    strAux[0] = 0;
    
    if (data)
    {
      strcpy(strAux, data);
      delete[] data;
    }
    
    strAux[length] = c;
    data = strAux;
    length += 1;
    strAux[length] = 0;
  }
}

// Concatena un caracter
inline void JString::operator+=(char c)
{
  if (c)
  {
    char *strAux = new char[length + 2];
    strAux[0] = 0;
    
    if (data)
    {
      strcpy(strAux, data);
      delete[] data;
    }
    
    strAux[length] = c;
    data = strAux;
    length += 1;
    strAux[length] = 0;
  }
}

// Devuelve la longitud
inline void JString::Format(const char *fmt, ...)
{
  va_list vlist;
  s32 i;
  char str[1024];
  memset(str, 0, 1024);

  va_start(vlist, fmt);
  i = vsprintf(str, fmt, vlist);
  va_end(vlist);

#ifdef DEBUG
  if (i > 1024)
  {
    printf("JString no admite Format de ms de 1024 bytes!!\n");
  }
#endif  // DEBUG

  operator=(str);
}

// Lee la cadena de un fichero (longitud + cadena)
inline u32 JString::Load(JFile &f)
{
  f.Read(&length, sizeof(length));
  delete[] data;
  data = new char[length + 1];
  f.Read(data, length);
  data[length] = 0;

  return 0;
}

// Escribe la cadena a un fichero (longitud + cadena)
inline u32 JString::Save(JFile &f)
{
  f.Write(&length, sizeof(length));
  f.Write(data, length);

  return 0;
}

#endif // _JSTRING_INCLUDED
