/* The GPL applies to this program.
  In addition, as a special exception, the copyright holders give
  permission to link the code of portions of this program with the
  OpenSSL library under certain conditions as described in each
  individual source file, and distribute linked combinations
  including the two.
  You must obey the GNU General Public License in all respects
  for all of the code used other than OpenSSL.  If you modify
  file(s) with this exception, you may extend this exception to your
  version of the file(s), but you are not obligated to do so.  If you
  do not wish to do so, delete this exception statement from your
  version.  If you delete this exception statement from all source
  files in the program, then also delete it here.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <sys/time.h>
#ifndef NO_SSL
#include <openssl/ssl.h>
#include "mssl.h"
#endif

#include "gen.h"
#include "http.h"
#include "io.h"
#include "str.h"
#include "tcp.h"
#include "utils.h"
#include "error.h"

static volatile int stop = 0;

void version(void)
{
	fprintf(stderr, "HTTPing v" VERSION ", (C) 2003-2005 folkert@vanheusden.com\n");
#ifndef NO_SSL
	fprintf(stderr, "SSL support included\n");
#endif
}

void usage(void)
{
	fprintf(stderr, "\n-g url		url (e.g. -g http://localhost/)\n");
	fprintf(stderr, "-h hostname	hostname (e.g. localhost)\n");
	fprintf(stderr, "-p portnr	portnumber (e.g. 80)\n");
	fprintf(stderr, "-x host:port	hostname+portnumber of proxyserver\n");
	fprintf(stderr, "-c count	how many times to connect\n");
	fprintf(stderr, "-i interval	delay between each connect\n");
	fprintf(stderr, "-t timeout	timeout (default: 30s)\n");
	fprintf(stderr, "-s		show statuscodes\n");
#ifndef NO_SSL
	fprintf(stderr, "-l		connect using SSL\n");
#endif
	fprintf(stderr, "-f		flood connect (no delays)\n");
	fprintf(stderr, "-m		give machine parseable output (see\n");
	fprintf(stderr, "		also -o and -e)\n");
	fprintf(stderr, "-o rc,rc,...	what http results codes indicate 'ok'\n");
	fprintf(stderr, "		coma seperated WITHOUT spaces inbetween\n");
	fprintf(stderr, "		default is 200, use with -e\n");
	fprintf(stderr, "-e str		string to display when http result code\n");
	fprintf(stderr, "		doesn't match\n");
	fprintf(stderr, "-q		quiet, only returncode\n");
	fprintf(stderr, "-V		show the version\n\n");
}

void handler(int sig)
{
	stop = 1;
}

int main(int argc, char *argv[])
{
	char *hostname = NULL;
	char *proxy = NULL, *proxyhost = NULL;
	int proxyport = 8080;
	int portnr = 80;
	char *get = NULL, *request;
	int req_len;
	int c;
	int count = -1, curncount = 0;
	int wait = 1, quiet = 0;
	int ok = 0, err = 0;
	double min = 999999999999999.0, avg = 0.0, max = 0.0;
	int timeout=30;
	char show_statuscodes = 0;
	char use_ssl = 0;
	SSL_CTX *client_ctx = NULL;
	char machine_readable = 0;
	char *ok_str = "200";
	char *err_str = "-1";

        while((c = getopt(argc, argv, "g:h:p:c:i:x:t:o:e:flqsmV?")) != -1)
        {
                switch(c)
                {
		case 'e':
			err_str = optarg;
			break;

		case 'o':
			ok_str = optarg;
			break;

		case 'x':
			proxy = optarg;
			break;

		case 'g':
			get = optarg;
			break;

		case 'h':
			hostname = optarg;
			break;

		case 'p':
			portnr = atoi(optarg);
			break;

		case 'c':
			count = atoi(optarg);
			break;

		case 'i':
			wait = atoi(optarg);
			break;

		case 't':
			timeout = atoi(optarg);
			break;

		case 'f':
			wait = 0;
			break;

#ifndef NO_SSL
		case 'l':
			use_ssl = 1;
			break;
#endif

		case 'm':
			machine_readable = 1;
			break;

		case 'q':
			quiet = 1;
			break;

		case 's':
			show_statuscodes = 1;
			break;

		case 'V':
			version();
			return 0;

		case '?':
		default:
			usage();
			return 1;
		}
	}

#ifndef NO_SSL
	if (use_ssl && portnr == 80)
		portnr = 443;
#endif

	if (get != NULL && hostname == NULL)
	{
		char *slash, *colon;
		char *getcopy = mystrdup(get);
		char *http_string = "http://";
		int http_string_len = 7;

#ifndef NO_SSL
		if (use_ssl)
		{
			http_string_len = 8;
			http_string = "https://";
		}
#endif

		if (strncasecmp(getcopy, http_string, http_string_len) != 0)
		{
			fprintf(stderr, "'%s' is a strange URL\n", getcopy);
			fprintf(stderr, "Expected: %s...\n", http_string);
			return 2;
		}

		slash = strchr(&getcopy[http_string_len], '/');
		if (slash)
			*slash = 0x00;

		colon = strchr(&getcopy[http_string_len], ':');
		if (colon)
		{
			*colon = 0x00;
			portnr = atoi(colon + 1);
		}

		hostname = &getcopy[http_string_len];
	}

	if (hostname == NULL)
	{
		fprintf(stderr, "No hostname/getrequest given\n");
		usage();
		return 3;
	}

	if (get == NULL)
	{
#ifndef NO_SSL
		if (use_ssl)
		{
			get = mymalloc(8 /* http:// */ + strlen(hostname) + 1 /* colon */ + 5 /* portnr */ + 1 /* / */ + 1 /* 0x00 */, "get");
			sprintf(get, "https://%s:%d/", hostname, portnr);
		}
		else
		{
#endif
			get = mymalloc(7 /* http:// */ + strlen(hostname) + 1 /* colon */ + 5 /* portnr */ + 1 /* / */ + 1 /* 0x00 */, "get");
			sprintf(get, "http://%s:%d/", hostname, portnr);
#ifndef NO_SSL
		}
#endif
	}

	if (proxy)
	{
		char *dummy = strchr(proxy, ':');
		proxyhost = proxy;
		if (dummy)
		{
			*dummy=0x00;
			proxyport = atoi(dummy + 1);
		}

		fprintf(stderr, "Using proxyserver: %s:%d\n", proxyhost, proxyport);
	}

#ifndef NO_SSL
	if (use_ssl)
	{
		client_ctx = initialize_ctx();
		if (!client_ctx)
			error_exit("problem creating SSL context\n");
	}
#endif

	request = mymalloc(strlen(get) + 4096, "request");
	sprintf(request, "HEAD %s HTTP/1.0\r\nUser-Agent: HTTPing v" VERSION "\r\n\r\n", get);
	req_len = strlen(request);

	if (!quiet && !machine_readable)
		printf("PING %s:%d (%s):\n", hostname, portnr, get);

	signal(SIGINT, handler);
	signal(SIGTERM, handler);

	timeout *= 1000;	/* change to ms */

	while((curncount < count || count == -1) && stop == 0)
	{
		double ms;
		double dstart, dend;
		struct timeval start, end;
        	struct timezone tz;
		char *reply;
		int fd;

		if (gettimeofday(&start, &tz) == -1)
		{
			perror("gettimeofday");
			break;
		}

		for(;;)
		{
			int rc;
			char *sc = NULL;
			SSL *ssl_h = NULL;
			BIO *s_bio;

			if (proxyhost)
				fd = connect_to(proxyhost, proxyport, timeout);
			else
				fd = connect_to(hostname, portnr, timeout);

			if (fd == -3)	/* ^C pressed */
				break;

			if (fd >= 0)
			{
				/* set fd blocking */
				if (set_fd_blocking(fd) == -1)
					break;

#ifndef NO_SSL
				if (use_ssl)
				{
					int rc;

					rc = connect_ssl(fd, client_ctx, &ssl_h, &s_bio, timeout);
					if (rc != 0)
						fd = rc;
				}
#endif
			}

			curncount++;

			if (fd < 0)
			{
				if (fd == -1)
					printf("error connecting to host\n");
				else if (fd == -2)
					printf("timeout connecting to host\n");

				err++;

				break;
			}

#ifndef NO_SSL
			if (use_ssl)
				rc = WRITE_SSL(ssl_h, request, req_len);
			else
#endif
				rc = mywrite(fd, request, req_len, timeout);
			if (rc != req_len)
			{
				if (rc == -1)
					printf("error sending request to host\n");
				else if (rc == -2)
					printf("timeout sending to host\n");
				else if (rc == -3)
				{/* ^C */}
				else if (rc == 0)
					printf("connection prematurely closed by peer\n");

				close(fd);
				err++;

				break;
			}

			rc = get_HTTP_headers(fd, ssl_h, &reply, timeout);

			if (show_statuscodes || machine_readable)
			{
				/* statuscode is in first line behind
				 * 'HTTP/1.x'
				 */
				char *dummy = strchr(reply, ' ');

				if (dummy)
				{
					sc = strdup(dummy + 1);

					/* lines are normally terminated with a
					 * LF
					 */
					dummy = strchr(sc, '\n');
					if (dummy)
						*dummy = 0x00;
				}
			}

			free(reply);

			if (rc < 0)
			{
				if (rc == RC_SHORTREAD)
					printf("error receiving reply from host\n");
				else if (rc == RC_TIMEOUT)
					printf("timeout receiving reply from host\n");

				close(fd);
				err++;

				break;
			}

			ok++;

			if (gettimeofday(&end, &tz) == -1)
			{
				perror("gettimeofday");
				break;
			}

#ifndef NO_SSL
			if (use_ssl)
			{
				if (close_ssl_connection(ssl_h, fd) == -1)
					error_exit("error shutting down ssl");
			}
#endif
			close(fd);

			dstart = (((double)start.tv_sec) + ((double)start.tv_usec)/1000000.0);
			dend = (((double)end.tv_sec) + ((double)end.tv_usec)/1000000.0);
			ms = (dend - dstart) * 1000.0;
			avg += ms;
			min = min > ms ? ms : min;
			max = max < ms ? ms : max;

			if (machine_readable)
			{
				char *dummy = strchr(sc, ' ');

				if (dummy) *dummy = 0x00;

				if (strstr(ok_str, sc))
					printf("%f", ms);
				else
					printf("%s", err_str);

				if (show_statuscodes)
					printf(" %s", sc);

				printf("\n");
			}
			else
			{
				printf("connected to %s:%d, seq=%d time=%.2f ms %s\n", hostname, portnr, curncount-1, ms, sc?sc:"");
			}

			free(sc);

			break;
		}

		if (curncount != count && !stop)
			sleep(wait);
	}

	if (!quiet && !machine_readable)
	{
		printf("--- %s ping statistics ---\n", get);
		if (count == -1)
			printf("%d connects, %d ok, %3.2f%% failed\n", curncount, ok, (((double)err) / ((double)curncount)) * 100.0);
		else
			printf("%d connects, %d ok, %3.2f%% failed\n", curncount, ok, (((double)err) / ((double)count)) * 100.0);

		if (ok > 0)
			printf("round-trip min/avg/max = %.1f/%.1f/%.1f ms\n", min, avg / (double)ok, max);
	}

	if (ok)
		return 0;
	else
		return 127;
}
