/***************************************************************************
                          kfocustask.cpp  -  description
                             -------------------
    begin                : Sun Aug 12 2001
    copyright            : (C) 2001 by Jeffrey Yu
    email                : jeffyu@cs.stanford.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "kfocustask.h"

#include "kfmenddate.h"
#include "kfxenddate.h"

KFMTask::KFMTask() : QObject() {
	title_ = QString("Task");
	comments_ = QString("");
	startDate_ = QDate::currentDate();
	
	endDate_ = NULL;
	endDate(new KFMEndDate());
}


KFMTask::KFMTask(const KFMTask* task) : QObject() {
	title_ = task->title();
	comments_ = task->comments();
	startDate_ = task->startDate();
	
	endDate_ = NULL;
	endDate(new KFMEndDate(task->endDate()));
}


KFMTask::~KFMTask(){
	if (endDate_ != NULL) {
		delete endDate_;
	}
}

int KFMTask::operator<(const KFMTask& other){
	// Check started states
	if (started() && !other.started())
		return 1;
	if (!started() && other.started())
		return 0;
	
	// From this point on, the task and the other are either
	// both started or both not started.

	if ( *(endDate()) < *(other.endDate()) ) {
		return 1;
	}
	else if ( *(endDate()) == *(other.endDate()) ) {
		if ( position() < other.position() ) {
			return 1;
		}
		else {
			return 0;
		}
	}
	else {
		return 0;
	}
}

int KFMTask::operator==(const KFMTask& other){
	if (started() != other.started())
		return 0;

	if ( *(endDate()) == *(other.endDate()) ) {
		if ( position() == other.position() ) {
			return 1;
		}
		else {
			return 0;
		}
	}
	else {
		return 0;
	}
}


QDomElement KFMTask::domElement(QDomDocument* dDoc){
	QDomElement eTask = dDoc->createElement("task");
	
		// title tag
		QDomElement eTitle = dDoc->createElement("title");
			QDomText titleTextNode = dDoc->createTextNode( title() );
			eTitle.appendChild(titleTextNode);
		eTask.appendChild(eTitle);
		
		// comments tag
		QDomElement eComments = dDoc->createElement("comments");
			QDomText commentsTextNode = dDoc->createTextNode( this->comments() );
			eComments.appendChild(commentsTextNode);
		eTask.appendChild(eComments);
		
		// startdate element
		QDomElement eStartDate = dDoc->createElement("startdate");
			eStartDate.setAttribute("year", startDate().year());
			eStartDate.setAttribute("month", startDate().month());
			eStartDate.setAttribute("day", startDate().day());
			eTask.appendChild(eStartDate);
		
		// enddate element
		KFXEndDate xmlEndDate(dDoc, endDate());
		eTask.appendChild(xmlEndDate.domElement());
		
	return eTask;	
}

/** Setup the task with data from a DOM element */
void KFMTask::domElement(QDomElement eTask){
	QDomNode n = eTask.firstChild();
	while (!n.isNull()) {
		QDomElement e = n.toElement();
		if (!e.isNull()) {
			if (e.tagName() == "title") {
				QDomText t = e.firstChild().toText();
				if (!t.isNull())
					title(t.data());
			}
			else if (e.tagName() == "comments") {
				QDomText t = e.firstChild().toText();
				if (!t.isNull())
					comments(t.data());
			}
			else if (e.tagName() == "startdate") {
				QString year = e.attribute("year","1970");
				QString month = e.attribute("month", "12");
				QString day = e.attribute("day", "26");
				startDate(QDate(year.toInt(), month.toInt(), day.toInt()));
			}
			else if (e.tagName() == KFXEndDate::xmlTag()) {
				KFXEndDate xmlEndDate(NULL, endDate());
				xmlEndDate.domElement(e);
			}
		}
		n = n.nextSibling();
	}
}


/** Unique identifier for the task */
uint KFMTask::id() const{
	return id_;
}


/** Retrieve the task's title */
QString KFMTask::title() const{
	return title_;
}

/** Set the task's title */
void KFMTask::title(QString title) {
	if (title != title_) {
		title_ = title;
		emit titleChanged();
		emit contentsChanged();
	}
}


/** Get the task's comments */
QString KFMTask::comments() const{
  return comments_;
}

/** Set the task's commencts */
void KFMTask::comments(QString comments){
	if (comments != comments_) {
		comments_ = comments;
		emit commentsChanged();
		emit contentsChanged();
	}
}


void KFMTask::startDate(const QDate startDate){
	if (startDate != startDate_) {
		startDate_ = startDate;
		emit startDateChanged();
		emit contentsChanged();
	}
}


KFMEndDate* KFMTask::endDate() const{
	return endDate_;
}

void KFMTask::endDate(const KFMEndDate* srcEndDate){
	if (srcEndDate != NULL) {
		if (endDate() == NULL) {
			// Construct a new enddate from the prototype
			endDate(new KFMEndDate(srcEndDate));
		}
		else {
			// Remodel the existing enddate based on the prototype
			endDate()->endDate(srcEndDate);
			emit contentsChanged();
		}
	}
}

void KFMTask::endDate(KFMEndDate* newEndDate){
	if (newEndDate != endDate_) {
		if (endDate() != NULL) {
			KFMEndDate* oldEndDate = endDate();
			endDate_ = NULL;
			
			// Clearing operations go here
			disconnect(oldEndDate, 0, this, 0);
			
			delete oldEndDate;
		}
		
		endDate_ = newEndDate;
		
		if (endDate() != NULL) {
			
			// Setting operations go here
			connect(endDate(), SIGNAL(contentsChanged()), this,
					SLOT(onEndDateContentsChanged()));		
		}
		
		emit contentsChanged();
	}
}


void KFMTask::onEndDateContentsChanged(){
	emit endDateChanged();
	emit contentsChanged();
}
