/*******************************************************************************
**                            backup.cpp
**                konserve -- A small backup application
**			 -------------------
**  copyright: (C) 2002 - 2004 by Florian Simnacher
**  email    : simnacher AT gmx DOT de
*******************************************************************************/

/*******************************************************************************
**  This program is free software; you can redistribute it and/or             **
**  modify it under the terms of the GNU General Public License               **
**  as published by the Free Software Foundation; either version 2            **
**  of the License, or (at your option) any later version.                    **
**                                                                            **
**  This program is distributed in the hope that it will be useful,           **
**  but WITHOUT ANY WARRANTY; without even the implied warranty of            **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             **
**  GNU General Public License for more details.                              **
**                                                                            **
**  You shougld have received a copy of the GNU General Public License        **
**  along with this program; if not, write to the Free Software               **
**  Foundation, Inc., 59 Temple Place - Suite 330, Boston,                    **
**  MA  02111-1307, USA.                                                      **
*******************************************************************************/

// Qt includes
#include <qfile.h>
#include <qfileinfo.h>

// KDE includes
#include <karchive.h>
#include <kdebug.h>
#include <ktempfile.h>
#include <ktar.h>

// My includes
#include "konservedebug.h"
#include "backup.h"
#include "helper.h"

/*******************************************************************************
void __debug_print_kurl( KURL url, QString obj )
{
    kdDebug() << obj << "." << "directory(false, false): "
              << url.directory(false, false)
              << endl;
    kdDebug() << obj << "." << "fileName(false): " << url.fileName(false)
              << endl;
    kdDebug() << obj << "." << "path(): " << url.path() << endl;
    kdDebug() << obj << "." << "url(): " << url.url() << endl << endl;
}
*******************************************************************************/

////////////////////////////////////////////////////////////////////////////////
///////                  class Backup                                    ///////
////////////////////////////////////////////////////////////////////////////////

Backup::Backup( KURL surl,
                KURL aurl,
                bool readonly )
    : mSourceUrl( surl ),
      mTmpSourceUrl(),
      mArchiveUrl( aurl ),
      mTmpArchiveUrl(),
      mbReadOnly( readonly ),
      mpArchive( 0L )
{
    TRACE();
}

Backup::~Backup()
{
    TRACE();
}

void Backup::doBackup()
{
    TRACE();
    Q_ASSERT( mSourceUrl.isValid() );
    Q_ASSERT( mArchiveUrl.isValid() );
    Q_ASSERT( ! mbReadOnly );

    KTempFile tmpFile;
    mTmpSourceUrl.setPath( tmpFile.name() );
    tmpFile.unlink();

    if ( mSourceUrl.isLocalFile() )
    {
        mTmpSourceUrl = mSourceUrl;
        slotStartListing( 0L );
    }
    else
    {
        KIO::Job* job = KIO::copy( mSourceUrl, mTmpSourceUrl );
        connect(
            dynamic_cast<const QObject*>(job), SIGNAL( result( KIO::Job * ) ),
            SLOT( slotStartListing( KIO::Job *) )
            );
    }
}

void Backup::slotStartListing( KIO::Job* aJob)
{
    TRACE();
    Q_ASSERT( 0L == mpArchive );
    Q_ASSERT( mTmpSourceUrl.isValid() );
    Q_ASSERT( mTmpSourceUrl.isLocalFile() );

    if ( 0L != aJob )
    {
        if ( aJob->error() )
        {
            aJob->showErrorDialog();
            emit finishedBackup(false);
            return;
        }
    }
    KTempFile tmpFile;
    mTmpArchiveUrl.setPath( tmpFile.name() );
    tmpFile.unlink();

    mpArchive = new KTar( mTmpArchiveUrl.path(), "application/x-gzip" );
    mpArchive->open( IO_WriteOnly );

    if ( QFileInfo( mTmpSourceUrl.path() ).isFile() ||
         QFileInfo( mTmpSourceUrl.path() ).isSymLink())
    {
        KFileItem f( const_cast<const KURL&>( mTmpSourceUrl ),
                     KMimeType::defaultMimeType(),
                     KFileItem::Unknown );
        this->writeToArchive( f );
        mpArchive->close();
        delete mpArchive;
        mpArchive = 0L;
        copyArchive();
    }
    else if ( QFileInfo( mTmpSourceUrl.path() ).isDir() )
    {
        KIO::Job* job = KIO::listRecursive( mTmpSourceUrl, false, true );
        connect(
            job, SIGNAL( entries( KIO::Job*, const KIO::UDSEntryList&)),
            SLOT( slotGetEntriesAndPackEm( KIO::Job*, const KIO::UDSEntryList&))
            );
        connect(
            job, SIGNAL( result( KIO::Job* ) ),
            SLOT( slotFinishedPacking( KIO::Job* ) )
            );
    }
    else
    {
        Q_ASSERT( false );
    }

    Q_ASSERT( mTmpArchiveUrl.isValid() );
    Q_ASSERT( mTmpArchiveUrl.isLocalFile() );
}

void Backup::slotGetEntriesAndPackEm( KIO::Job*, const KIO::UDSEntryList& list )
{
    TRACE();
    Q_ASSERT( mTmpSourceUrl.isValid() );
    Q_ASSERT( mTmpSourceUrl.isLocalFile() );

    KIO::UDSEntryListConstIterator it=list.begin();
    for (; it != list.end(); ++it)
    {
        KFileItem f( *it, mTmpSourceUrl );

        if ( f.text() == "." || f.text() == ".." )
            continue;
        if ( f.isDir() )
            continue;
        else if ( f.isFile() )
            writeToArchive( f, true );
        else
            Q_ASSERT( false );
    }
}

void Backup::slotFinishedPacking( KIO::Job* aJob )
{
    TRACE();
    Q_ASSERT( QFileInfo( mTmpArchiveUrl.path() ).exists() );
    if ( aJob->error() )
    {
        aJob->showErrorDialog();
        emit finishedBackup(false);
        return;
    }
    mpArchive->close();
    // who has invented the protected base class Dtor???
    delete dynamic_cast<KTar*>(mpArchive);
    mpArchive = 0L;
    copyArchive();
}

void Backup::copyArchive()
{
    TRACE();
    Q_ASSERT( mArchiveUrl.isValid() );
    Q_ASSERT( QFileInfo( mTmpArchiveUrl.path() ).exists() );
    Q_ASSERT( mTmpArchiveUrl.isValid() );
    Q_ASSERT( mTmpArchiveUrl.isLocalFile() );

    KURL destUrl;
    if ( mArchiveUrl.fileName(false) == QString::null )
        destUrl = dotCheck( defaultArchiveName( mSourceUrl, mArchiveUrl ) );
    else
        destUrl = dotCheck( mArchiveUrl );

    KIO::Job* job = KIO::file_move( mTmpArchiveUrl, destUrl, -1, true );
    connect(
        job, SIGNAL( result( KIO::Job * ) ),
        SLOT( slotFinished( KIO::Job *) )
        );
}

void Backup::slotFinished( KIO::Job* aJob )
{
    TRACE();
    if ( aJob->error() )
    {
        aJob->showErrorDialog();
        emit finishedBackup(false);
        return;
    }
    emit finishedBackup(true);

    Q_ASSERT( !QFileInfo( mTmpArchiveUrl.path() ).exists() );
}

KURL  Backup::defaultArchiveName( KURL source, KURL backup )
{
    TRACE();
    QDateTime qdt = QDateTime::currentDateTime();

    return KURL(
        backup.url(1) + source.fileName() + "-" +
        qdt.toString( "yyyyMMddhhmmss" ) + ".tar.gz"
        );
}

KURL Backup::dotCheck( KURL inUrl )
{
    TRACE();
    QString url_string = inUrl.url();
    int index = url_string.findRev( "/" );

    if ( -1 == index )
    {
        if ( url_string.startsWith( "." ) )
            return KURL( "dot" + url_string );
        else
            return inUrl;
    }
    if ( -1 < index )
    {
        QString rightstring = url_string.right( url_string.length() - index - 1 );
        QString leftstring  = url_string.left( index + 1 );

        if ( rightstring.startsWith( "." ) )
            return KURL( leftstring + "dot" + rightstring );
    }
    return inUrl;
}

void Backup::writeToArchive( KFileItem f, bool usetext )
{
    TRACE();
    if ( f.isLink() )
    {
        QFile *file;

        if (usetext)
            file = new QFile( f.url().path() + f.name() );
        else
            file = new QFile( f.url().path() );

        Q_CHECK_PTR( file );
        Q_ASSERT( file->exists() );

        char  *c = new char[file->size()];

        if ( !file->open( IO_ReadOnly ) )
            return;
        file->readBlock( c, file->size() );

        if (usetext)
            mpArchive->writeFile(
                mSourceUrl.path() + f.text(), Helper::userName(),
                Helper::userGroup(), (unsigned int) file->size(), c
                );
        else
            mpArchive->writeFile(
                mSourceUrl.path(), Helper::userName(),
                Helper::userGroup(), (unsigned int) file->size(), c
                );
        file->close();
        delete[] c;
        delete file;
    }
    else if ( f.isFile() )
    {
        QFile *file;
        if (usetext)
            file = new QFile( f.url().path() + f.text() );
        else
            file = new QFile( f.url().path() );

        Q_CHECK_PTR( file );
        Q_ASSERT( file->exists() );
        char  *c = new char[file->size()];

        if ( !file->open( IO_ReadOnly ) )
            return;
        file->readBlock( c, file->size() );

        if (usetext)
            mpArchive->writeFile(
                mSourceUrl.path() + f.text(), Helper::userName(),
                Helper::userGroup(), (unsigned int) file->size(), c
                );
        else
            mpArchive->writeFile(
                mSourceUrl.path(), Helper::userName(),
                Helper::userGroup(), (unsigned int) file->size(), c
                );
        file->close();
        delete[] c;
        delete file;
    }
    else
    {
        Q_ASSERT( false ); // Should not happen
    }
}
