// =============================================================================
//
//      --- kvi_irc_userlist.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviIrcUserList"

#include "kvi_debug.h"
#include "kvi_irc_user.h"
#include "kvi_irc_userlist.h"
#include "kvi_string.h"

// TODO: We want a hash table here for 2.1.0!!!

// ===========================================================================
// Master list
//
KviIrcUserList::KviIrcUserList()
{
	m_pHead = 0;
	m_pTail = 0;
}

KviIrcUserList::~KviIrcUserList()
{
	// All items should have been removed before this list is destroyed.
	__range_invalid(m_pHead);
	clearList();
}

void KviIrcUserList::clearList()
{
	while( m_pHead ) removeNode(m_pHead);
	__range_valid(m_pTail == 0);
}

void KviIrcUserList::removeNode(KviIrcUserListNode *node)
{
	// Internal: removes a node from the list
	// and deletes it!
	__range_valid(node);
	__range_valid(node->pUser->hasNick());
	__range_valid(findNode(node->pUser->nick()));

	if( node == m_pHead ) {
		// Head node
		if( node->next ) {
			// Has a next one, it becomes the new head node.
			node->next->prev = 0;
			m_pHead = node->next;
		} else {
			// The only node in the list (must be the tail node too!)
			__range_valid(m_pTail == node);
			m_pTail = 0;
			m_pHead = 0;
		}
	} else {
		// In the middle, or the last one
		__range_valid(node->prev);
		if( node->next ) {
			// Is in the middle, make necessary connections
			node->next->prev = node->prev;
			node->prev->next = node->next;
		} else {
			// The last one. Previous becomes new last
			m_pTail = node->prev;
			node->prev->next = 0;
		}
	}
	// Do not forget to delete it :)
	if( node->pUser ) delete node->pUser;
	if( node ) delete node;
}

void KviIrcUserList::insertNode(KviIrcUserListNode *node)
{
	// Insert a node in the list respecting the alphabetic order.
	__range_valid(node);
	__range_valid(node->pUser->hasNick());
	__range_valid(!findNode(node->pUser->nick()));

	if( !m_pHead ) {
		// First node in the list
		m_pHead = node;
		m_pTail = node;
		node->prev = 0;
		node->next = 0;
	} else {
		// There are other nodes, walk the list...
		KviIrcUserListNode *cur = m_pHead;

		do {
			// Walk until we find a greater one
			if( kvi_strcmpCI(node->pUser->m_nick_ptr, cur->pUser->m_nick_ptr) > 0 ) {
				// Cur is greater... insert before
				node->next = cur;
				node->prev = cur->prev;
				if( node->prev )
					node->prev->next = node;
				else
					m_pHead = node; // No prev item
				cur->prev = node;
				return;
			}
#ifdef _KVI_DEBUG_CLASS_NAME_
			if( !cur->next ) {
				__range_valid(cur == m_pTail);
			}
#endif
			cur = cur->next;
		} while( cur );

		// Ran to the end!
		m_pTail->next = node;
		node->prev = m_pTail;
		node->next = 0;
		m_pTail = node;
	}
}

KviIrcUserListNode *KviIrcUserList::findNode(const char *nick)
{
	// Look up a nickname in the list and find the related node.
	KviIrcUserListNode *cur = m_pHead;
	int result;
	while( cur ) {
		result = kvi_strcmpCI(nick, cur->pUser->m_nick_ptr);
		if( result == 0 ) return cur;
		if( result > 0 ) return 0; // Cannot be... the cur is greater...
		cur = cur->next;
	}
	return 0; // Not found
}

KviIrcUserListNode *KviIrcUserList::firstNode()
{
	m_pCur = m_pHead;
	return m_pHead;
}

KviIrcUserListNode *KviIrcUserList::lastNode()
{
	m_pCur = m_pTail;
	return m_pTail;
}

KviIrcUserListNode *KviIrcUserList::nextNode()
{
	if( m_pCur )
		m_pCur = m_pCur->next;
	return m_pCur;
}

KviIrcUserListNode *KviIrcUserList::prevNode()
{
	if( m_pCur )
		m_pCur = m_pCur->prev;
	return m_pCur;
}

KviIrcUser *KviIrcUserList::first()
{
	m_pCur = m_pHead;
	return m_pCur ? m_pCur->pUser : 0;
}

KviIrcUser *KviIrcUserList::last()
{
	m_pCur = m_pTail;
	return m_pCur ? m_pCur->pUser : 0;
}

KviIrcUser *KviIrcUserList::next()
{
	if( m_pCur )
		m_pCur = m_pCur->next;
	return m_pCur ? m_pCur->pUser : 0;
}

KviIrcUser *KviIrcUserList::prev()
{
	if( m_pCur )
		m_pCur = m_pCur->prev;
	return m_pCur ? m_pCur->pUser : 0;
}

void KviIrcUserList::getFirstFiveNicksToUpdate(KviStr &buffer)
{
	buffer = "";
	KviIrcUserListNode *cur = m_pHead;
	int iCount = 0;
	while( cur && iCount < 5 ) {
		if( !(cur->pUser->hasHost()) ) {
			buffer.append(cur->pUser->nick());
			if( iCount != 4 ) buffer.append(' ');
			iCount++;
		}
		cur = cur->next;
	}
}

KviIrcUser *KviIrcUserList::findUser(const KviIrcUser &user)
{
	// Same as above, but return the nick structure pointer
	KviIrcUserListNode *node = findNode(user.nick());
	if( node ) return node->pUser;
	return 0;
}

KviIrcUser *KviIrcUserList::findUser(const char *nick)
{
	// Same as above, but return the nick structure pointer (overload)
	KviIrcUserListNode *node = findNode(nick);
	if( node ) return node->pUser;
	return 0;
}

KviIrcUserListNode *KviIrcUserList::addUser(const KviIrcUser &user)
{
	// Add a nick to the list
	KviIrcUserListNode *node = findNode(user.nick());
	if( node ) {
		__range_valid(node->nRefs > 0);
		node->nRefs++;
		// Update user data
		if( !node->pUser->hasHost()     ) node->pUser->setHost(user.host());
		if( !node->pUser->hasUsername() ) node->pUser->setUsername(user.username());
	} else {
		node = new KviIrcUserListNode;
		node->pUser = new KviIrcUser(user);
		node->nRefs = 1;
		insertNode(node);
	}
	return node;
}

bool KviIrcUserList::updateUser(const KviIrcUser &user)
{
	// Update a user entry in the list
	KviIrcUserListNode *node = findNode(user.nick());
	if( node ) {
		if( !node->pUser->hasHost()     ) node->pUser->setHost(user.host());
		if( !node->pUser->hasUsername() ) node->pUser->setUsername(user.username());
	}
	return node != 0;
}

void KviIrcUserList::killUserByNode(KviIrcUserListNode *node)
{
	// Removes a node from the list
	// or decreases its reference count (if it is > 1)
	__range_valid(node);
	__range_valid(node->pUser);
	__range_valid(node->pUser->hasNick());
	if( node ) {
		if( node->nRefs < 2 ) {
			__range_valid(node->nRefs == 1);
			// Only one reference, remove the node
#ifdef _KVI_DEBUG_CHECK_RANGE_
			// Sanity check for the debug version
			KviStr nick = node->pUser->nick();
			removeNode(node);
			__range_invalid(findNode(nick.ptr()));
#else
			removeNode(node);
#endif
		} else node->nRefs--;
	}
}

