// =============================================================================
//
//      --- kvi_mimemanager.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviMimeManager"

#include <qfile.h>
#include <qregexp.h>

#include "kvi_config.h"
#include "kvi_mimemanager.h"
#include "kvi_mutex.h"
#include "kvi_string.h"

KviMutex *g_pMimeMutex = 0;

KviMimeManager::KviMimeManager()
{
	m_pMimeList = new QPtrList<KviMimeType>;
	m_pMimeList->setAutoDelete(true);
	m_pDefault = new KviMimeType;
	m_pDefault->mimeName    = "Unknown";
	m_pDefault->magicBytes  = "";
	m_pDefault->fileMask    = "*";
	m_pDefault->commandline = "xedit %";
	m_pDefault->savePath    = "";
	m_pDefault->iconPath    = "";
	if( g_pMimeMutex == 0 )
		g_pMimeMutex = new KviMutex();
}

KviMimeManager::~KviMimeManager()
{
	if( g_pMimeMutex ) {
		delete g_pMimeMutex;
		g_pMimeMutex = 0;
	}
	if( m_pMimeList ) { delete m_pMimeList; m_pMimeList = 0; }
	if( m_pDefault  ) { delete m_pDefault;  m_pDefault  = 0; }
}

void KviMimeManager::save(KviConfig *cfg)
{
	cfg->writeEntry("MimeCount", m_pMimeList->count());
	int i = 0;
	for( KviMimeType *m = m_pMimeList->first(); m; m = m_pMimeList->next() ) {
		KviStr key(KviStr::Format, "Mime_%d_Name", i);
		cfg->writeEntry(key.ptr(), m->mimeName.ptr());
		key.sprintf("Mime_%d_Magic", i);
		cfg->writeEntry(key.ptr(), m->magicBytes.ptr());
		key.sprintf("Mime_%d_FileMask", i);
		cfg->writeEntry(key.ptr(), m->fileMask.ptr());
		key.sprintf("Mime_%d_Commandline", i);
		cfg->writeEntry(key.ptr(), m->commandline.ptr());
		key.sprintf("Mime_%d_SavePath", i);
		cfg->writeEntry(key.ptr(), m->savePath.ptr());
		key.sprintf("Mime_%d_RemoteExecSafe", i);
		cfg->writeEntry(key.ptr(), m->remoteExecSafe);
		key.sprintf("Mime_%d_IconPath", i);
		cfg->writeEntry(key.ptr(), m->iconPath.ptr());
		i++;
	}
}

void KviMimeManager::load(KviConfig *cfg)
{
	int count = cfg->readIntEntry("MimeCount", 0);
	for( int i = 0; i < count; i++ ) {
		KviMimeType *t = new KviMimeType;
		KviStr key(KviStr::Format, "Mime_%d_Name", i);
		t->mimeName = cfg->readEntry(key.ptr(), "Unknown");
		key.sprintf("Mime_%d_Magic", i);
		t->magicBytes = cfg->readEntry(key.ptr(), "");
		key.sprintf("Mime_%d_FileMask", i);
		t->fileMask = cfg->readEntry(key.ptr(), "*");
		key.sprintf("Mime_%d_Commandline", i);
		t->commandline = cfg->readEntry(key.ptr(), "xedit %");
		key.sprintf("Mime_%d_SavePath", i);
		t->savePath = cfg->readEntry(key.ptr(), "");
		key.sprintf("Mime_%d_RemoteExecSafe", i);
		t->remoteExecSafe = cfg->readBoolEntry(key.ptr(), false);
		key.sprintf("Mime_%d_IconPath", i);
		t->iconPath = cfg->readEntry(key.ptr(), "");
		inSort(t);
	}
}


void KviMimeManager::inSort(KviMimeType *m)
{
	// Order the mimetypes in a way that we always find the correct match.
	// First sort by the number of wildcards inside: the one that has less wildcards comes first.
	// Then longer fileMasks (*.tar.gz)
	// Then shorter ones (*.gz) (*)
	// The mimes with a magic number go before similar mimes without one.
	uint idx = 0;
	m->fileMask.stripWhiteSpace();
	m->magicBytes.stripWhiteSpace();
	m->savePath.stripWhiteSpace();
	m->iconPath.stripWhiteSpace();

	KviStr strippedMaskM = m->fileMask;
	strippedMaskM.replaceAll('*', ""); // Remove all wildcards
	KviStr strippedMaskT;
	for( KviMimeType *t = m_pMimeList->first(); t; t = m_pMimeList->next() ) {
		if( m->fileMask.contains('*') <= t->fileMask.contains('*') ) {
			strippedMaskT = t->fileMask;
			strippedMaskT.replaceAll('*', "");
			// The item that we are inserting has less wildcards than the one in the list
			if( strippedMaskM.len() >= strippedMaskT.len() ) {
				// The item that we are inserting is longer than the one in the list
				if( m->magicBytes.len() >= t->magicBytes.len() ) {
					// The item that we are inserting has a longer magic than the one in the list
					m_pMimeList->insert(idx, m);
					return;
				} // else: has shorter magic... goes after... skip one item
			} // else: has shorter file mask... goes after... skip one item
		} // else: has more wildcards... goes after... skip one item
		idx++;
	}
	m_pMimeList->append(m);
}

bool KviMimeManager::checkMagicMatch(KviStr &magic, const char *fileName)
{
	QFile f(fileName);
	if( !f.exists() ) return true; // Matched
	if( f.size() <= 0 ) return false; // No way

	if( !f.open(IO_ReadOnly) ) {
		debug("Mime match: could not open file %s", fileName);
		return true; // !!!
	}

	char buffer[16];
	int toRead = (f.size() >= 15) ? 15 : f.size();
	int readLen = f.readBlock(buffer, toRead);

	f.close();

	if( readLen <= 0 ) return false;

	// Ugly hack
	for( int i = 0; i < readLen; i++ ) {
		if( buffer[i] == '\0' ) buffer[i] = '\01';
	}

	buffer[readLen - 1] = '\0';

	QRegExp rexp(_CHAR_2_QSTRING(magic.ptr()));
	return (rexp.search(buffer) == 0);
}

KviMimeType *KviMimeManager::findMatch(const char *fileName, bool bDoMagicCheck)
{
	KviMimeType *m;

	g_pMimeMutex->lock();

	KviStr fName(fileName);
	int lastIdx = fName.findLastIdx("/");
	if( lastIdx != -1 ) fName.cutLeft(lastIdx + 1);

	m = m_pMimeList->first();

	while( m ) {
		// If no magic checks have to be done, get rid of the
		// totally wild fileMasks (*)
		QRegExp rExp(_CHAR_2_QSTRING(m->fileMask.ptr()), false, true);
		if( rExp.search(_CHAR_2_QSTRING(fName.ptr())) == 0 ) {
			if( rExp.matchedLength() == fName.len() ) {
				// Matched...
				if( bDoMagicCheck ) {
					// Now check the first bytes of the file...
					m->magicBytes.stripWhiteSpace();
					if( m->magicBytes.len() <= 0 ) break;
					if( checkMagicMatch(m->magicBytes, fileName) ) break;
				} else break;
			}
		}
		m = m_pMimeList->next();
	}

	g_pMimeMutex->unlock();

	return m ? m : m_pDefault;
}
