// =============================================================================
//
//      --- kvi_app_fs.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviAppFS"

#include <qdir.h>
#include <qfileinfo.h>

#include "kvi_app.h"
#include "kvi_debug.h"
#include "kvi_fileutils.h"
#include "kvi_locale.h"
#include "kvi_mimemanager.h"
#include "kvi_options.h"

// ============ getGlobalKVIrcDirectory ================

void KviApp::getGlobalKVIrcDirectory(KviStr &szData, KVIrcSubdir dir, const char *appendFile)
{
	szData = m_szGlobalKVIrcDir.ptr();
	switch( dir ) {
		case None          :                            ; break;
		case Pics          : szData += "/pics"          ; break;
		case Plugins       : szData += "/plugins"       ; break;
		case ConfigPlugins : szData += "/config/plugins"; break;
		case Help          : szData += "/help";
			{
				// Localized help
				KviStr tmp = kvi_getLocaleName();
				tmp.prepend('/');
				tmp.prepend(szData);
				if( kvi_directoryExists(tmp.ptr()) )
					szData = tmp;
				else {
					tmp = szData;
					tmp.append("/en");
					if( kvi_directoryExists(tmp.ptr()) )
						szData = tmp;
				}
			}
			break;
		case HelpEN     : szData += "/help/en"  ; break;
		case HelpNoIntl : szData += "/help"     ; break;
		case Log        : __debug("WARNING Global log directory requested!");      break;
		case Incoming   : __debug("WARNING Global incoming directory requested!"); break;
		case Trash      : __debug("WARNING Global trash directory requested!");    break;
		case Config     : szData += "/config"   ; break; // Read only...!!!
		case Audio      : szData += "/audio"    ; break;
		case Scripts    : szData += "/scripts"  ; break;
		case MsgColors  : szData += "/msgcolors"; break;
		case Charmaps   : szData += "/charmaps" ; break;
	}
	if( appendFile ) {
		szData.ensureLastCharIs('/');
		szData.append(appendFile);
	}
}

// =============== getLocalKVIrcDirectory ================

void KviApp::getLocalKVIrcDirectory(KviStr &szData, KVIrcSubdir dir, const char *appendFile, bool bCreateIfNeeded)
{
	szData = m_szLocalKVIrcDir.ptr();
	szData.ensureLastCharIs('/');
	switch( dir ) {
		case None          :                             break;
		case Pics          : szData += "pics";           break;
		case Config        : szData += "config";         break;
		case ConfigPlugins : szData += "config/plugins"; break;
		case Log           : szData += "log";            break;
		case Incoming      :
			if( g_pOptions->m_szLocalIncomingDirectory.hasData() ) {
				if( g_pOptions->m_szLocalIncomingDirectory.firstCharIs('/') )
					szData = g_pOptions->m_szLocalIncomingDirectory;
				else {
					szData.append(g_pOptions->m_szLocalIncomingDirectory);
				}
			} else {
				szData += "incoming";
			}
			g_pOptions->m_szLocalIncomingDirectory = szData;
			break;
		case Help       : szData += "help";      break;
		case HelpEN     : szData += "help";      break;
		case HelpNoIntl : szData += "help";      break;
		case Audio      : szData += "audio";     break;
		case Scripts    : szData += "scripts";   break;
		case Plugins    : szData += "plugins";   break;
		case Trash      : szData += "trash";     break;
		case MsgColors  : szData += "msgcolors"; break;
		case Charmaps   : szData += "charmaps";  break;
	}
	if( bCreateIfNeeded )
		kvi_makeDir(szData.ptr());
	if( appendFile ) {
		szData.ensureLastCharIs('/');
		szData.append(appendFile);
	}
}

// ====================== trashFile ====================
/**
 * Moves a file to the local trash directory.
 */
bool KviApp::trashFile(const char *filename)
{
	// Exists?
	QFileInfo fi(filename);
	if( !fi.exists() ) return false; // Nothing to trash

	// Build the filename
	KviStr lastPart = filename;
	lastPart.cutToLast('/'); // Need only the filename
	lastPart.append(".bak_");
	QDateTime tm(QDateTime::currentDateTime());
	lastPart.append(tm.toString());
	lastPart.replaceAll(' ', "_");
	// Ok... have lastPart.bak_Dec_11_31_1999
	// Find the thrash directory (make it if necessary)
	KviStr trashFile;
	getLocalKVIrcDirectory(trashFile, Trash, lastPart.ptr(), true);
	// Check if a such file already exists
	fi.setFile(trashFile.ptr());
	while( fi.exists() ) {
		trashFile.append(".rnm");
		fi.setFile(trashFile.ptr());
	}
	// Rename the file
	return kvi_renameFile(filename, trashFile.ptr());
}

// ============= getChannelDumpLogFileName ==============
/**
 * Returns a valid filename for the channel dump log
 */
void KviApp::getChannelDumpLogFileName(KviStr &str)
{
	unsigned int logN = 0;
	KviStr fPrefix = "kick_";
	do {
		KviStr fName(KviStr::Format, "%s%u.log", fPrefix.ptr(), logN);
		getLocalKVIrcDirectory(str, Log, fName.ptr());
		logN++;
		if( logN > 9999 ) {
			// Time to clear logs! :)
			fPrefix += "x";
			logN = 0;
		}
	} while( kvi_fileExists(str.ptr()) );
}

// ===================== findUserFile =====================
/**
 * Looks for a user file in the standard directories
 */
bool KviApp::findUserFile(KviStr &szRetPath, const char *filename)
{
	// Absolute paths cannot be "found"...
	if( kvi_isAbsolutePath(filename) ) {
		szRetPath = filename;
		return kvi_fileExists(filename);
	}

	KviStr szP;
	// Check in the mime save directory
	KviMimeType *m = g_pOptions->m_pMimeManager->findMatch(filename, false);

	if( m ) {
		if( m->savePath.hasData() ) {
			szP = m->savePath;
			szP.ensureLastCharIs('/');
			szP.append(filename);
			if( kvi_fileExists(szP.ptr()) ) {
				szRetPath = szP.ptr();
				return true;
			}
		}
	}

	if( g_pOptions->m_szLastScriptPath.hasData() ) {
		szP = g_pOptions->m_szLastScriptPath;
		szP.ensureLastCharIs('/');
		szP.append(filename);
		if( kvi_fileExists(szP.ptr()) ) {
			szRetPath = szP;
			return true;
		}
	}

	getLocalKVIrcDirectory(szP, Incoming, filename);
	if( kvi_fileExists(szP.ptr()) ) {
		szRetPath = szP.ptr();
		return true;
	}
	getLocalKVIrcDirectory(szP, Pics, filename);
	if( kvi_fileExists(szP.ptr()) ) {
		szRetPath = szP.ptr();
		return true;
	}
	getLocalKVIrcDirectory(szP, Audio, filename);
	if( kvi_fileExists(szP.ptr()) ) {
		szRetPath = szP.ptr();
		return true;
	}
	getLocalKVIrcDirectory(szP, Log, filename);
	if( kvi_fileExists(szP.ptr()) ) {
		szRetPath = szP.ptr();
		return true;
	}
	getGlobalKVIrcDirectory(szP, Pics, filename);
	if( kvi_fileExists(szP.ptr()) ) {
		szRetPath = szP.ptr();
		return true;
	}
	getGlobalKVIrcDirectory(szP, Audio, filename);
	if( kvi_fileExists(szP.ptr()) ) {
		szRetPath = szP.ptr();
		return true;
	}
	// Last resort...
	szP = QDir::homeDirPath();
	szP.ensureLastCharIs('/');
	szP.append(filename);
	if( kvi_fileExists(szP.ptr()) ) {
		szRetPath = szP.ptr();
		return true;
	}

	szRetPath = filename;
	return false;
}

// ========================= findImage ===========================
/**
 * Find a file inside the default "Image" directories
 */
bool KviApp::findImage(KviStr &szRetPath, const char *filename)
{
	// Find a user file...
	if( kvi_isAbsolutePath(filename) ) {
		szRetPath = filename;
		return kvi_fileExists(filename);
	}

	KviStr szP;
	if( g_pOptions->m_szLastScriptPath.hasData() ) {
		szP = g_pOptions->m_szLastScriptPath;
		szP.ensureLastCharIs('/');
		szP.append(filename);
		if( kvi_fileExists(szP.ptr()) ) {
			szRetPath = szP;
			return true;
		}
		szP = g_pOptions->m_szLastScriptPath;
		szP.ensureLastCharIs('/');
		szP.append("pics/");
		szP.append(filename);
		if( kvi_fileExists(szP.ptr()) ) {
			szRetPath = szP;
			return true;
		}
	}

	getLocalKVIrcDirectory(szP, Pics, filename);
	if( kvi_fileExists(szP.ptr()) ) {
		szRetPath = szP;
		return true;
	}
	getGlobalKVIrcDirectory(szP, Pics, filename);
	if( kvi_fileExists(szP.ptr()) ) {
		szRetPath = szP;
		return true;
	}
	// Last resort
	szP = QDir::homeDirPath();
	szP.ensureLastCharIs('/');
	szP.append(filename);
	if( kvi_fileExists(szP.ptr()) ) {
		szRetPath = szP;
		return true;
	}
	return false;
}

// =================== getDefaultDccSaveFilePath =================
/**
 * Return the best place to save the file
 */
void KviApp::getDefaultDccSaveFilePath(KviStr &path, const char *filename)
{
	KviMimeType *m = g_pOptions->m_pMimeManager->findMatch(filename, false);
	// The mime manager always returns something
	path = "";
	if( m->savePath.hasData() ) {
		// Check the path validity
		if( *(m->savePath.ptr()) == '/' ) {
			// At least a valid dir...
			if( kvi_makeDir(m->savePath.ptr()) ) {
				// Ok... the directory was created
				path = m->savePath;
			}
		}
	}
	if( path.isEmpty() ) getLocalKVIrcDirectory(path, Incoming, 0, true);
	if( path.isEmpty() ) path = QDir::homeDirPath(); // Last resource...
}

// ================== getReadOnlyConfigPath ====================
/**
 * We want to READ a config file: find it
 */
bool KviApp::getReadOnlyConfigPath(KviStr &buffer, const char *config_name)
{
	// Take a look in the local directory
	getLocalKVIrcDirectory(buffer, Config, config_name);
	if( !kvi_fileExists(buffer.ptr()) ) {
		// No saved config yet... check for defaults
		KviStr tmp;
		getGlobalKVIrcDirectory(tmp, Config, config_name);
		if( !kvi_fileExists(tmp.ptr()) ) {
			// No defaults... no such config file at all.
			return false;
		} else buffer = tmp; // Get the defaults this time
	} // else: file exists...
	return true;
}

