// =============================================================================
//
//      --- kvi_options_plugins.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviOptionsPlugins"

#include <qcheckbox.h>
#include <qlayout.h>

#include "kvi_app.h"
#include "kvi_defines.h"
#include "kvi_filedialog.h"
#include "kvi_frame.h"
#include "kvi_label.h"
#include "kvi_listbox.h"
#include "kvi_locale.h"
#include "kvi_options_plugins.h"
#include "kvi_pushbutton.h"
#ifdef COMPILE_PLUGIN_SUPPORT
	#include "kvi_console.h"
	#include "kvi_error.h"
	#include "kvi_plugin.h"
	extern KviPluginManager *g_pPluginManager;
#endif

/*
	@quickhelp: KviOptionsPlugins
	@widget: Plugin manager dialog
		From this dialog you can load and unload plugins.<br>
		You can obtain help about one of the loaded plugins
		by clicking the "plugin help" button after selecting
		the plugin in the list box.<br>
		This function will be available only if the plugin
		exports a help routine.<br>
		If the plugin has a configuration dialog, it will be
		accessible by clicking the "Configure plugin" button.<br>
		The "autoload on startup" option makes the plugin be
		loaded at every KVIrc startup until is explicitly unloaded.<br>
		This dialog will not be available if KVIrc has been compiled
		without the plugin support.<br>
*/
KviOptionsPlugins::KviOptionsPlugins(QWidget *parent)
	: KviOptionsWidget(parent, KVI_GENERAL_OPTIONS_WIDGET_ID_PLUGINS)
{
#ifdef COMPILE_PLUGIN_SUPPORT
	QGridLayout *g = new QGridLayout(this, 8, 2, KVI_GENERAL_OPTIONS_GRIDLAYOUT_BORDER, 1);

	m_pPluginBox = new KviListBox(this);
	m_pPluginBox->setMultiSelection(false);
	connect(m_pPluginBox, SIGNAL(highlighted(const QString &)), this, SLOT(pluginListItemSelected(const QString &)));
	g->addMultiCellWidget(m_pPluginBox, 0, 5, 0, 0);

	KviPushButton *load = new KviPushButton(_i18n_("&Load Plugin"), this);
	connect(load, SIGNAL(clicked()), this, SLOT(pluginLoad()));
	g->addWidget(load, 6, 0);

	m_pUnloadPlugin = new KviPushButton(_i18n_("&Unload Plugin"), this);
	connect(m_pUnloadPlugin, SIGNAL(clicked()), this, SLOT(pluginUnload()));
	g->addWidget(m_pUnloadPlugin, 7, 0);

	KviLabel *l = new KviLabel(_CHAR_2_QSTRING(_i18n_("Module name:")), this);
	g->addWidget(l, 0, 1);
	m_pPluginModuleNameLabel = new KviLabel(this);
	m_pPluginModuleNameLabel->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	g->addWidget(m_pPluginModuleNameLabel, 0, 2);

	l = new KviLabel(_CHAR_2_QSTRING(_i18n_("Service name:")), this);
	g->addWidget(l, 1, 1);
	m_pPluginServiceNameLabel = new KviLabel(this);
	m_pPluginServiceNameLabel->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	g->addWidget(m_pPluginServiceNameLabel, 1, 2);

	l = new KviLabel(_CHAR_2_QSTRING(_i18n_("Author:")), this);
	g->addWidget(l, 2, 1);
	m_pPluginAuthorLabel = new KviLabel(this);
	m_pPluginAuthorLabel->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	g->addWidget(m_pPluginAuthorLabel, 2, 2);

	l = new KviLabel(_CHAR_2_QSTRING(_i18n_("Version:")), this);
	g->addWidget(l, 3, 1);
	m_pPluginVersionLabel = new KviLabel(this);
	m_pPluginVersionLabel->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	g->addWidget(m_pPluginVersionLabel, 3, 2);

	l = new KviLabel(_CHAR_2_QSTRING(_i18n_("Description:")), this);
	l->setAlignment(AlignTop);
	g->addWidget(l, 4, 1);
	m_pPluginDescriptionLabel = new KviLabel(this);
	m_pPluginDescriptionLabel->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	m_pPluginDescriptionLabel->setAlignment(AlignLeft | AlignTop | WordBreak);
	g->addWidget(m_pPluginDescriptionLabel, 4, 2);

	g->setRowStretch(4, 2);

	m_pPluginAutoloadCheck = new QCheckBox(_i18n_("Automatically load on startup"), this);
	m_pPluginAutoloadCheck->setMinimumSize(m_pPluginAutoloadCheck->sizeHint());
	connect(m_pPluginAutoloadCheck, SIGNAL(toggled(bool)), this, SLOT(pluginAutoloadToggled(bool)));
	g->addMultiCellWidget(m_pPluginAutoloadCheck, 5, 5, 1, 2);

	m_pConfigurePlugin = new KviPushButton(_i18n_("&Configure Plugin"), this);
	connect(m_pConfigurePlugin, SIGNAL(clicked()), this, SLOT(pluginConfigure()));
	g->addMultiCellWidget(m_pConfigurePlugin, 6, 6, 1, 2);

	m_pHelpOnPlugin = new KviPushButton(_i18n_("Plugin &Help"), this);
	connect(m_pHelpOnPlugin, SIGNAL(clicked()), this, SLOT(pluginHelp()));
	g->addMultiCellWidget(m_pHelpOnPlugin, 7, 7, 1, 2);

	fillPluginBox();
#endif
}

KviOptionsPlugins::~KviOptionsPlugins()
{
	// Nothing here
}

void KviOptionsPlugins::commitChanges()
{
	// Nothing here
}

#ifdef COMPILE_PLUGIN_SUPPORT
void KviOptionsPlugins::fillPluginBox(unsigned int selectIdx)
{
	m_pPluginBox->clear();
	QPtrList<KviPluginData> *l = g_pPluginManager->pluginList();
	for( KviPluginData *d = l->first(); d; d = l->next() ) {
		m_pPluginBox->insertItem(_CHAR_2_QSTRING(d->filename.ptr()));
	}
	if( m_pPluginBox->count() == 0 ) {
		m_pPluginModuleNameLabel->setText("");
		m_pPluginVersionLabel->setText("");
		m_pPluginServiceNameLabel->setText("");
		m_pPluginDescriptionLabel->setText("");
		m_pPluginAuthorLabel->setText("");
		m_pUnloadPlugin->setEnabled(false);
		m_pConfigurePlugin->setEnabled(false);
		m_pPluginAutoloadCheck->setEnabled(false);
		m_pHelpOnPlugin->setEnabled(false);
	} else m_pPluginBox->setCurrentItem(selectIdx);
}
#endif

void KviOptionsPlugins::pluginListItemSelected(const QString &str)
{
#ifdef COMPILE_PLUGIN_SUPPORT
	KviStr tmp = str;
	KviPluginData *d = g_pPluginManager->findPlugin(tmp.ptr());
	if( !d ) return;

	m_pPluginModuleNameLabel->setText(d->description->module_name);
	m_pPluginServiceNameLabel->setText(d->description->service_name);
	m_pPluginAuthorLabel->setText(d->description->author);
	m_pPluginVersionLabel->setText(d->description->version);
	m_pPluginDescriptionLabel->setText(d->description->description);
	m_pUnloadPlugin->setEnabled(true);
	m_pConfigurePlugin->setEnabled(d->description->config_routine);
	m_pPluginAutoloadCheck->setEnabled(true);
	m_pPluginAutoloadCheck->setChecked(d->bAutoload);
	m_pHelpOnPlugin->setEnabled(d->description->help_routine);
#endif
}

void KviOptionsPlugins::pluginAutoloadToggled(bool bActive)
{
#ifdef COMPILE_PLUGIN_SUPPORT
	KviStr tmp = m_pPluginBox->text(m_pPluginBox->currentItem());
	if( tmp.isEmpty() )
		return;

	KviPluginData *d = g_pPluginManager->findPlugin(tmp.ptr());
	if( !d ) return;

	d->bAutoload = bActive;
#endif
}

void KviOptionsPlugins::pluginUnload()
{
#ifdef COMPILE_PLUGIN_SUPPORT
	KviStr tmp = m_pPluginBox->text(m_pPluginBox->currentItem());
	if( tmp.isEmpty() )
		return;

	if( !g_pPluginManager->unloadPlugin(tmp.ptr()) )
		return;
	fillPluginBox();
#endif
}

void KviOptionsPlugins::pluginLoad()
{
#ifdef COMPILE_PLUGIN_SUPPORT
	KviStr tmp;
	g_pApp->getGlobalKVIrcDirectory(tmp, KviApp::Plugins);
	KviStr fName = KviFileDialog::getOpenFileName(tmp.ptr(), "*.so");
	if( fName.hasData() ) {
		KviStr error;
		KviPluginData *d = g_pPluginManager->loadPlugin(fName.ptr(), true, &error);
		if( !d ) {
			g_pApp->warningBox(_i18n_("Could not load plugin %s\n%s"), fName.ptr(), error.ptr());
			return;
		}

		KviPluginCommandStruct plgcmd;
		plgcmd.app     = g_pApp;
		plgcmd.handle  = d->handle;
		plgcmd.params  = 0;
		plgcmd.frame   = g_pApp->m_pFrameList->first();
		plgcmd.error   = KVI_ERROR_NoError;
		plgcmd.console = plgcmd.frame->m_pConsole;
		plgcmd.window  = plgcmd.console;
		plgcmd.sock    = plgcmd.frame->m_pSocket;

		if( !g_pPluginManager->executeInitRoutine(d, &plgcmd) ) {
			g_pApp->warningBox(
				_i18n_("Plugin initialization failed\n%s\n%s"),
				fName.ptr(),
				plgcmd.errorstr.hasData()
					? plgcmd.errorstr.ptr()
					: "Unspecified error\nMore info may be available in the shell output"
			);
			g_pPluginManager->unloadPlugin(d->filename.ptr());
			return;
		}
		fillPluginBox(m_pPluginBox->count());
	}
#endif
}

void KviOptionsPlugins::pluginConfigure()
{
#ifdef COMPILE_PLUGIN_SUPPORT
	KviStr tmp = m_pPluginBox->text(m_pPluginBox->currentItem());
	if( tmp.isEmpty() )
		return;

	KviPluginData *d = g_pPluginManager->findPlugin(tmp.ptr());
	if( !d ) return;

	if( d->description->config_routine ) {
		d->description->config_routine();
	}
#endif
}

void KviOptionsPlugins::pluginHelp()
{
#ifdef COMPILE_PLUGIN_SUPPORT
	KviStr tmp = m_pPluginBox->text(m_pPluginBox->currentItem());
	if( tmp.isEmpty() )
		return;

	KviPluginData *d = g_pPluginManager->findPlugin(tmp.ptr());
	if( !d ) return;

	if( d->description->help_routine )
		d->description->help_routine();
#endif
}

#include "m_kvi_options_plugins.moc"
