// =============================================================================
//
//      --- kvi_aliaseditor.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//   Copyright (C) 1999-2000 Till Busch (buti@geocities.com)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviAliasEditor"

#include <qhbox.h>
#include <qheader.h>
#include <qsplitter.h>
#include <qvbox.h>

#include "kvi_alias.h"
#include "kvi_aliaseditor.h"
#include "kvi_aliasnamevalidator.h"
#include "kvi_lineedit.h"
#include "kvi_locale.h"
#include "kvi_messagebox.h"
#include "kvi_pushbutton.h"
#include "kvi_script_datalistview.h"
#include "kvi_script_editor.h"

// TODO: Totally missing quick help!

// Declared in kvi_app.cpp:
extern KviAliasManager *g_pAliasManager;

KviStr g_szLastSelectedAlias = "";

/*
	@quickhelp: KviAliasEditor
	@widget: Aliases tab
		Here, you can edit the aliases (custom commands).<br>
		To create a new alias, click on "Add Alias;" a new "unnamed" alias will be created
		with an empty data buffer.<br>
		Click on "Rename" to set the desired name, and edit your alias body.<br>
		Please note that while you CAN create an alias with a name that
		is already used by a KVIrc command, you will NOT be able to use it.<br>
*/

/**
 *
 * ALIAS EDITOR
 *
 * This widget allows editing of the aliases.
 * Each alias is kept inside a KviScriptDataListViewItem (KviListViewItem)
 * child of the m_pListView.
 * Aliases are saved by calling "commit"
 *
 */
KviAliasEditor::KviAliasEditor(QWidget *parent, const char *name)
	: QWidget(parent, name)
{
	m_pCurItem  = 0;
	m_pSplitter = new QSplitter(this);
	m_pSplitter->setOpaqueResize();

	QVBox *left_box = new QVBox(m_pSplitter);
	m_pListView = new KviListView(left_box);
	m_pListView->addColumn(_i18n_("Alias"));
	m_pListView->header()->hide();
	m_pListView->setSelectionMode(QListView::Single);
	m_pListView->setAutoResize(false);
	m_pNew = new KviPushButton(_i18n_("&Add Alias"), left_box);
	m_pDelete = new KviPushButton(_i18n_("&Remove Alias"), left_box);
	m_pDelete->setEnabled(false);
	m_pClear = new KviPushButton(_i18n_("&Clear Aliases"), left_box);

	QVBox *right_box = new QVBox(m_pSplitter);
	QHBox *hbox = new QHBox(right_box);
	m_pLineEdit = new KviLineEdit(hbox);
	m_pLineEdit->setEnabled(false);
	m_pLineEdit->installEventFilter(this);
	m_pLineEdit->setValidator(new KviAliasNameValidator(m_pLineEdit));
	m_pEditName = new KviPushButton(_i18n_("Re&name"), hbox);
	m_pEditName->setEnabled(false);

	m_pEditor = new KviScriptEditor(right_box);
	m_pEditor->setEnabled(false);

	fillListView();
	connect(
		m_pListView, SIGNAL(selectionChanged(KviListViewItem *)),
		this,  SLOT(listViewSelectionChanged(KviListViewItem *))
	);
	connect(m_pEditName, SIGNAL(clicked()), this, SLOT(editAliasName()));
	connect(m_pNew,      SIGNAL(clicked()), this, SLOT(makeNewAlias()));
	connect(m_pDelete,   SIGNAL(clicked()), this, SLOT(deleteCurrentAlias()));
	connect(m_pClear,    SIGNAL(clicked()), this, SLOT(clearAllAliases()));

	KviScriptDataListViewItem *it = findAlias(g_szLastSelectedAlias.ptr());
	if( it )
		m_pListView->setSelected(it, true);
}

KviAliasEditor::~KviAliasEditor()
{
	// Nothing here
}

void KviAliasEditor::commitAliasName()
{
	if( !m_pLineEdit->isEnabled() )
		return;

	m_pLineEdit->setEnabled(false);
	if( !m_pCurItem ) return;

	KviStr txt = m_pLineEdit->text();
	if( txt.isEmpty() ) {
		KviMessageBox::error(
			_i18n_("Alias name is empty"),
			_i18n_("Invalid alias name"),
			this
		);
		m_pLineEdit->setText(m_pCurItem->text(0));
		return;
	}
	KviListViewItem *it = m_pListView->firstChild();
	while( it ) {
		if( it != m_pCurItem ) {
			KviStr tmp = it->text(0);
			if( kvi_strEqualCI(txt.ptr(), tmp.ptr()) ) {
				KviMessageBox::error(
					_i18n_("An alias with that name already exists"),
					_i18n_("Invalid alias name"),
					this
				);
				m_pLineEdit->setText(m_pCurItem->text(0));
				return;
			}
		}
		it = it->nextSibling();
	}
	if( m_pCurItem ) {
		m_pCurItem->setText(0, m_pLineEdit->text());
		return;
	}
	m_pEditor->setFocus();
}

bool KviAliasEditor::eventFilter(QObject *o, QEvent *e)
{
	if( (o == this) || (o == m_pLineEdit) ) {
		if( (e->type() == QEvent::Accel) || (e->type() == QEvent::KeyPress) ) {
			if( m_pLineEdit->isEnabled() ) {
				if( ((QKeyEvent *) e)->key() == Qt::Key_Return ) {
					((QKeyEvent *) e)->accept();
					commitAliasName();
					return true;
				}
			}
		}
	}
	if( o == m_pLineEdit ) {
		if( m_pLineEdit->isEnabled() ) {
			if( e->type() == QEvent::FocusOut ) {
				commitAliasName();
				return true;
			}
		}
	}
	return QWidget::eventFilter(o, e);
}

void KviAliasEditor::saveCurrentItem()
{
	if( m_pCurItem ) {
		m_pCurItem->m_szBuffer = m_pEditor->text();
	}
}

void KviAliasEditor::editAliasName()
{
	m_pLineEdit->setEnabled(true);
	m_pLineEdit->setFocus();
}

void KviAliasEditor::deleteCurrentAlias()
{
	if( m_pCurItem ) {
		KviScriptDataListViewItem *it = m_pCurItem;
		m_pCurItem = 0;
		delete it;
		it = (KviScriptDataListViewItem *) m_pListView->currentItem();
		if( it )
			m_pListView->setSelected(it, true);
	}
}

void KviAliasEditor::clearAllAliases()
{
	if( KviMessageBox::warningYesNo(
		_i18n_("Are you sure you want to remove all aliases?"),
		_i18n_("Clear Aliases"),
		this) == KviMessageBox::Yes
	) {
		m_pCurItem = 0;
		m_pListView->clear();
	}
}

KviScriptDataListViewItem *KviAliasEditor::findAlias(const char *name)
{
	for( KviListViewItem *it = m_pListView->firstChild(); it; it = it->nextSibling() ) {
		KviStr tmp = it->text(0);
		if( kvi_strEqualCI(tmp.ptr(), name) )
			return (KviScriptDataListViewItem *) it;
	}
	return 0;
}

void KviAliasEditor::makeNewAlias()
{
	KviStr tmp = _i18n_("unnamed");
	int i = 1;
	while( findAlias(tmp.ptr()) ) {
		tmp.sprintf(_i18n_("unnamed%d"), i);
		i++;
	}
	KviScriptDataListViewItem *it = new KviScriptDataListViewItem(m_pListView, "", tmp.ptr());
	m_pListView->setSelected(it, true);
	m_pEditor->setFocus();
}

void KviAliasEditor::listViewSelectionChanged(KviListViewItem *it)
{
	saveCurrentItem();
	m_pCurItem = 0;
	if( it ) {
		m_pEditor->setText(((KviScriptDataListViewItem *) it)->m_szBuffer.ptr());
		m_pLineEdit->setText(((KviListViewItem *) it)->text(0));
		m_pEditor->setFocus();
	} else {
		m_pEditor->setText("");
		m_pLineEdit->setText("");
	}
	m_pCurItem = (KviScriptDataListViewItem *) it;
	m_pEditor->setEnabled(it);
	m_pEditName->setEnabled(it);
	m_pDelete->setEnabled(it);
}

void KviAliasEditor::fillListView()
{
	KviScriptDataListViewItem *it = 0;
	for( KviAlias *a = g_pAliasManager->m_pAliasList->first(); a; a = g_pAliasManager->m_pAliasList->next() ) {
		it = new KviScriptDataListViewItem(m_pListView, a->szBuffer.ptr(), a->szName.ptr());
	}
}

void KviAliasEditor::commit()
{
	if( m_pCurItem )
		g_szLastSelectedAlias = m_pCurItem->text(0);
	saveCurrentItem();
	g_pAliasManager->clearAll();
	for( KviScriptDataListViewItem *it = (KviScriptDataListViewItem *)m_pListView->firstChild()
	   ; it
	   ; it = (KviScriptDataListViewItem *) it->nextSibling()
	) {
		KviAlias *a = new KviAlias();
		a->szName = it->text(0);
		a->szBuffer = it->m_szBuffer;
		g_pAliasManager->addAlias(a);
	}
}

void KviAliasEditor::resizeEvent(QResizeEvent *)
{
	m_pSplitter->setGeometry(0, 0, width(), height());
}

#include "m_kvi_aliaseditor.moc"
