// =============================================================================
//
//      --- kvi_objectviewer.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//   Copyright (C) 1999-2000 Till Busch (buti@geocities.com)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviObjectViewer"

#include <qlayout.h>
#include <qsplitter.h>

#include "kvi_app.h"
#include "kvi_frame.h"
#include "kvi_listview.h"
#include "kvi_locale.h"
#include "kvi_objectviewer.h"
#include "kvi_script_object.h"
#include "kvi_script_objectcontroller.h"
#include "kvi_script_objectclassdefinition.h"
#include "kvi_userparser.h"

// TODO: Totally missing quick help!

// Global class definitions (kvi_script_object.cpp)
extern KviScriptObjectClassDefinition *g_pClassDefinitions;

/*
	@quickhelp: KviObjectViewer
	@widget: Object viewer
		This tab displays the tree of currently existing <a href="syntax_objects.kvihelp">object classes</a>,
		and the tree of existing object instances.<br>
*/

/**
 * OBJECT VIEWER
 */
KviObjectViewer::KviObjectViewer(QWidget *parent, const char *name)
	: QWidget(parent, name)
{
	QGridLayout *g = new QGridLayout(this, 1, 1, 4, 2);
	QSplitter *s = new QSplitter(QSplitter::Horizontal, this);
	s->setOpaqueResize();
	g->addWidget(s, 0, 0);
	QSplitter *v = new QSplitter(QSplitter::Vertical, s);
	v->setOpaqueResize();

	m_pClassView = new KviListView(v);
	m_pClassView->addColumn(_i18n_("Class"));
	m_pClassView->addColumn(_i18n_("Definition"));
	m_pClassView->setRootIsDecorated(true);
	connect(m_pClassView, SIGNAL(currentChanged(KviListViewItem *)), this, SLOT(classViewCurrentChanged(KviListViewItem *)));
	m_pFncView = new KviListView(v);
	m_pFncView->addColumn(_i18n_("Function"));
	m_pFncView->addColumn(_i18n_("Comments"));
	m_pObjectView = new KviListView(s);
	m_pObjectView->addColumn(_i18n_("Object"));
	m_pObjectView->addColumn(_i18n_("Name"));
	m_pObjectView->addColumn(_i18n_("Class"));
	m_pObjectView->addColumn(_i18n_("Comments"));
	m_pObjectView->setRootIsDecorated(true);
	fillViews();
}

KviObjectViewer::~KviObjectViewer()
{
	// Nothing here
}

void KviObjectViewer::classViewCurrentChanged(KviListViewItem *it)
{
	m_pFncView->clear();
	KviStr cl = it->text(0);
	KviScriptObjectClassDefinition *def = 0;
	if( kvi_strEqualCI(cl.ptr(), g_pClassDefinitions->getClass()) )
		def = g_pClassDefinitions;
	else
		def = g_pClassDefinitions->lookupChildClassDefinition(cl.ptr());
	if( !def ) {
		debug("Oops... %s: no such class definition", cl.ptr());
		return;
	}
	QAsciiDictIterator<KviScriptObjectFunctionStruct> iter(*(def->functionDict()));
	while( iter.current() ) {
		KviStr tmp;
		if( iter.current()->fncHandler )
			tmp = _i18n_("Core code handler");
		else
			tmp = _i18n_("Script handler");
		if( iter.current()->flags & KVI_SCRIPTOBJECTFUNCTION_FLAG_INHERITED ) tmp.append(_i18n_(", Inherited"));
		if( iter.current()->flags & KVI_SCRIPTOBJECTFUNCTION_FLAG_OVERRIDE  ) tmp.append(_i18n_(", Override"));
		if( iter.current()->flags & KVI_SCRIPTOBJECTFUNCTION_FLAG_BUILTIN   ) tmp.append(_i18n_(", Built-in"));
		else tmp.append(_i18n_(", User-defined"));
		it = new KviListViewItem(m_pFncView, iter.currentKey(), tmp.ptr());
		++iter;
	}
}

void KviObjectViewer::fillViews()
{
	m_pClassView->clear();
	m_pObjectView->clear();
	m_pFncView->clear();
	fillClassDefinition(0, g_pClassDefinitions);
	for( KviFrame *f = g_pApp->m_pFrameList->first(); f; f = g_pApp->m_pFrameList->next() ) {
		fillObjectItem(0, f->m_pUserParser->m_pScriptObjectController->topLevelObject());
	}
}

void KviObjectViewer::fillObjectItem(KviListViewItem *p, KviScriptObject *o)
{
	if( !o )return;

	KviListViewItem *it = 0;
	if( p )
		it = new KviListViewItem(p, o->id(), o->getName(), o->getClass(), _i18n_("-"));
	else
		it = new KviListViewItem(m_pObjectView, o->id(), o->getName(), o->getClass(), _i18n_("Built-in root object"));
	it->setOpen(true);
	QPtrList<KviScriptObject> *l = o->childrenList();
	for( KviScriptObject *obj = l->first(); obj; obj = l->next() ) {
		fillObjectItem(it, obj);
	}
}

void KviObjectViewer::fillClassDefinition(KviListViewItem *p, KviScriptObjectClassDefinition *d)
{
	if( !d ) return;

	KviListViewItem *it;
	if( p ) {
		it = new KviListViewItem(p, d->getClass(), d->isBuiltin()
			? _i18n_("Built-in")
			: _i18n_("User-defined")
		);
	} else {
		it = new KviListViewItem(m_pClassView, d->getClass(), d->isBuiltin()
			? _i18n_("Built-in")
			: _i18n_("User-defined")
		);
	}
	it->setOpen(true);
	QPtrList<KviScriptObjectClassDefinition> *l = d->childClasses();
	if( l ) {
		for( KviScriptObjectClassDefinition *def = l->first(); def; def = l->next() ) {
			fillClassDefinition(it, def);
		}
	}
}

#include "m_kvi_objectviewer.moc"
