// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptButton"

#include "kvi_error.h"
#include "kvi_pushbutton.h"
#include "kvi_script_button.h"
#include "kvi_script_objectclassdefinition.h"

// TODO: put a dialog example in the button class documentation

/*
	@class: button
	@short:
		A push button widget
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@functions:
		!fn: $text()
		Returns the text displayed on the button.<br>

		!fn: $setText(&lt;text&gt;)
		Sets the text to be displayed on the button.<br>

	@events:
		!ev: OnClick()
		Triggered when the user clicks the button.<br>
		<b>This signal has a default event handler that emits the [classsignal:button]clicked[/classsignal] signal</b>.
		If you override the event handler, and still want the signal to be emitted, you have to emit
		it by yourself!.<br>
		The default event handler is the following one:<br>
		<example>
			[fnc]$this[/fnc]->[classfnc:object]$emit[/classfnc]([classsignal:button]clicked[/classsignal])
		</example>

	@signals:
		!sg: clicked()
		Emitted when the button has been clicked by the user.<br>
		<b>This signal is emitted by the default [classevent:button]OnClick[/classevent] event handler!<br>
		If you define a new event handler, and still want the signal to be emitted,
		you have to emit it by yourself!

	@description:
		This is a simple GUI button class.<br>
		It shows a simple text label and reacts to the user clicks.<br>
		The constructor for this widget accepts an optional additional parameter:
		the text to be displayed.<br>
		When the user clicks the button, the [classevent:button]OnClick[/classevent]
		event is triggered. The [classevent:button]OnClick[/classevent] event has a default
		event handler that emits the [classsignal:button]clicked[/classsignal] signal.<br>
		If you define a new event handler, and still want the signal to be emitted,
		you have to emit it by yourself!

	@examples:
		Simple usage:<br><br>
		Construct a simple (toplevel) button displaying the text "Hello".<br>
		<example>
			%btn = [fnc]$new[/fnc]([class]button[/class], [fnc]$root[/fnc], the_button, Hello)
			%btn->[classfnc:widget]$show[/classfnc]()
		</example>
		React to its click by setting a handler to the OnClick event.<br>
		<example>
			%btn = [fnc]$new[/fnc]([class]button[/class], [fnc]$root[/fnc], the_button, Hello)
			[cmd]obj_setevent[/cmd](%btn, [classevent:button]OnClick[/classevent])
			{
			&nbsp;	[cmd]echo[/cmd] Button clicked!
			}
			%btn->[classfnc:widget]$show[/classfnc]()
		</example>
		Please note that in the example above the signal clicked is no longer emitted!.<br>
		In a simple usage case it is not useful.<br>
		If you still want the signal to be emitted when the user clicks the button:<br>
		<example>
			%btn = [fnc]$new[/fnc]([class]button[/class], [fnc]$root[/fnc], the_button, Hello)
			[cmd]obj_setevent[/cmd](%btn, [classevent:button]OnClick[/classevent])
			{
			&nbsp;	[cmd]echo[/cmd] Button clicked!
			&nbsp;  [fnc]$this[/fnc]->[classfnc:object]$emit[/classfnc]([classsignal:button]clicked[/classsignal])
			}
			%btn->[classfnc:widget]$show[/classfnc]()
		</example>

	@seealso:
		[class]object[/class] class</a>,
		[class]widget[/class] class</a>,
		<a href="syntax_objects.kvihelp">Objects documentation</a>
*/

/**
 * BUTTON class
 */
void KviScriptButton::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("setText", (scriptObjectFunction) &KviScriptButton::builtinFunction_SETTEXT);
	d->addBuiltinFunction("text",    (scriptObjectFunction) &KviScriptButton::builtinFunction_TEXT);

	KviScriptEventStruct *s = new KviScriptEventStruct();
	s->szName   = "OnClick";
	s->szBuffer = "$this->$emit(clicked)";
	d->addDefaultEvent(s);
}

KviScriptButton::KviScriptButton(
	KviScriptObjectController *cntrl, KviScriptObject *p, const char *name, KviScriptObjectClassDefinition *pDef)
	: KviScriptWidget(cntrl, p, name, pDef)
{
	// Nothing here
}

KviScriptButton::~KviScriptButton()
{
	// Nothing here
}

bool KviScriptButton::init(QPtrList<KviStr> *params)
{
	if( parent() ) {
		if( parent()->inherits("KviScriptWidget") ) {
			m_pWidget = new KviPushButton(((KviScriptWidget *) parent())->m_pWidget, name());
		}
	}
	if( !m_pWidget )
		m_pWidget = new KviPushButton(0, name());
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget, SIGNAL(clicked()),   this, SLOT(buttonClicked()));
	connect(m_pWidget, SIGNAL(destroyed()), this, SLOT(widgetDestroyed()));
	if( params ) {
		KviStr *pS = params->first();
		if( pS )
			((KviPushButton *) m_pWidget)->setText(pS->ptr());
	}
	return true;
}

int KviScriptButton::builtinFunction_TEXT(QPtrList<KviStr> *, KviStr &buffer)
{
	buffer.append(((KviPushButton *) m_pWidget)->text());
	return KVI_ERROR_Success;
}

int KviScriptButton::builtinFunction_SETTEXT(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			((KviPushButton *) m_pWidget)->setText(pS->ptr());
			return KVI_ERROR_Success;
		}
	}
	((KviPushButton *) m_pWidget)->setText("");
	return KVI_ERROR_Success;
}

void KviScriptButton::buttonClicked()
{
	// Trigger clicked()
	KviStr params;
	triggerEvent("OnClick", params);
}

#include "m_kvi_script_button.moc"
