// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptPopupMenu"

#include <qcursor.h>

#include "kvi_defines.h"
#include "kvi_error.h"
#include "kvi_popupmenu.h"
#include "kvi_script_objectclassdefinition.h"
#include "kvi_script_objectcontroller.h"
#include "kvi_script_popupmenu.h"

// From kvi_app.cpp
extern QPixmap *g_pixViewOut[KVI_OUT_NUM_IMAGES];

/*
	@class: popupmenu
	@short:
		A popup menu widget
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@functions:
		!fn: $insertItem(&lt;text&gt;[,image_id])
		Inserts an item with &lt;text&gt; and an optional image (see [fnc]$icon[/fnc]()).<br>
		Returns the item id.<br>

		!fn: $insertSeparator()
		Inserts a separator

		!fn: $insertMenu(&lt;text&gt;,&lt;menu_object_id&gt;[,image_id])
		Inserts a submenu ite with text &lt;text&gt; and an optional image.<br>
		The &lt;menu_object_id&gt; is the id of the inserted popupmenu class object.<br>

		!fn: $popup();
		Shows the popup and allows the user to select one of the items.<br>
		When the user has clicked on an item the popup will be hidden and the OnItemClicked
		event will be fired.<br>

		!fn: $text(&lt;id&gt;)
		Returns the text displayed on the item with the specified &lt;id&gt;

		!fn: $clear()
		Removes all the items from this popup.<br>

		!fn: $setItemEnabled(&lt;item_id&gt;, &lt;bEnabled&gt)
		Enables or disables the specified item.<br>

		!fn: $setItemChecked(&lt;item_id&gt;, &lt;bChecked&gt)
		Checks or unchecks the specified item.<br>

		!fn: $removeItem(&lt;item_id&gt;)
		Removes the specified item.<br>

		!fn: $isItemEnabled(&lt;item_id&gt;)
		Returns '1' if the specified item is enabled, '0' otherwise.<br>

		!fn: $isItemChecked(&lt;item_id&gt;)
		Returns '1' if the specified item is checked, '0' otherwise.<br>

	@events:
		!ev: OnItemClicked($1 = &lt;itemId&gt;)
		Triggered when the user selects one of the items of the menu.<br>
		&lt;itemId&gt; contains the id of the item that has been selected.<br>
		The default event handler emits the [classsignal:popupmenu]itemClicked[/classsignal] signal.<br>
		If you provide a new handler for this event, and still want the signal
		to be emitted, you must emit it by yourself:<br>
		<example>
			[fnc]$this[/fnc]->[classfnc:object]$emit[/classfnc]([classsignal:popupmenu]itemClicked[/classsignal], $1)
		</example>

	@signals:
		!sg: itemClicked($1 = &lt;itemId&gt;)
		Emitted when the user selects one of the menu items.<br>
		&lt;itemId&gt; contains the id of the item that has been selected.<br>
		<b>This signal is emitted by the default [classevent:popupmenu]OnItemClicked[/classevent] event handler!<br>
		If you define a new event handler, and still want the signal to be emitted,
		you have to emit it by yourself!</b>

	@description:
		A popup menu widget.<br>
		The managing of this class is not obvious:<br>
		You need to call $insertItem() for each item that you want to add.<br>
		$insertItem() will return a unique id that you can use in the OnItemClicked
		event to check which item has been selected by the user.<br>
		Better take a look at the examples :)<br>

	@examples:
		The simplest (and most useless) way to use the menu:<br>
		<example>
		%menu = [fnc]$new[/fnc]([class]popupmenu[/class], [fnc]$root[/fnc], mypopup)
		%menu->$insertItem(Item 1, $icon(echo))
		%menu->$insertItem(Item 2, $icon(privmsg))
		%menu->$insertItem(Item 3, $icon(kill))
		obj_setevent(%menu, OnItemClicked)echo You have clicked the "$this->$text($1)"
		%menu->$popup()
		</example>
		This is kewl! :)<br>
		Try to guess what it does.<br>
		<example>
		[cmd]option[/cmd] szPrivmsgFormatNickLinkCommand "[dbl][cmd]whois[/cmd] \$1[rbt]builtin_nick_popup \$1"
		[cmd]alias[/cmd] builtin_nick_popup {
		&nbsp;	%p = [fnc]$root[/fnc]->[classfnc:object]$findChild[/classfnc](builtin_nick_popup)
		&nbsp;	[cmd]if[/cmd]("%p" != "0")%p->[classfnc:popupmenu]$clear[/classfnc]()
		&nbsp;	[cmd:if]else[/cmd] %p = [fnc]$new[/fnc]([class]popupmenu[/class], [fnc]$root[/fnc], builtin_nick_popup)
		&nbsp;	%p->%nick = $1
		&nbsp;	%p->%idWhois = %p->[classfnc:popupmenu]$insertItem[/classfnc](whois $1, [fnc]$icon[/fnc](whois))
		&nbsp;	%p->%idQuery = %p->[classfnc:popupmenu]$insertItem[/classfnc](query $1, [fnc]$icon[/fnc](query))
		&nbsp;	%p->%idKill  = %p->[classfnc:popupmenu]$insertItem[/classfnc](kill $1, [fnc]$icon[/fnc](kill))
		&nbsp;	[cmd]obj_setevent[/cmd](%p, [classevent:popupmenu]OnItemClicked[/classevent]){
		&nbsp;		[cmd]switch[/cmd]($1){
		&nbsp;			case([fnc]$this[/fnc]->%idWhois)[cmd]whois[/cmd] [fnc]$this[/fnc]->%nick
		&nbsp;			case([fnc]$this[/fnc]->%idQuery)[cmd]query[/cmd] [fnc]$this[/fnc]->%nick
		&nbsp;			case([fnc]$this[/fnc]->%idKill)[cmd]kill[/cmd] [fnc]$this[/fnc]->%nick
		&nbsp;		}
		&nbsp;	}
		&nbsp;%p->[classfnc:popupmenu]$popup[/classfnc]()
		}
		</example>
	@seealso:
		class [class]object[/class], <br>
		class [class]widget[/class], <br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/

/**
 * POPUP MENU class
 */
void KviScriptPopupMenu::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("insertItem",      (scriptObjectFunction) &KviScriptPopupMenu::builtinFunction_INSERTITEM);
	d->addBuiltinFunction("removeItem",      (scriptObjectFunction) &KviScriptPopupMenu::builtinFunction_REMOVEITEM);
	d->addBuiltinFunction("setItemEnabled",  (scriptObjectFunction) &KviScriptPopupMenu::builtinFunction_SETITEMENABLED);
	d->addBuiltinFunction("setItemChecked",  (scriptObjectFunction) &KviScriptPopupMenu::builtinFunction_SETITEMCHECKED);
	d->addBuiltinFunction("isItemEnabled",   (scriptObjectFunction) &KviScriptPopupMenu::builtinFunction_ISITEMENABLED);
	d->addBuiltinFunction("isItemChecked",   (scriptObjectFunction) &KviScriptPopupMenu::builtinFunction_ISITEMCHECKED);
	d->addBuiltinFunction("insertMenu",      (scriptObjectFunction) &KviScriptPopupMenu::builtinFunction_INSERTMENU);
	d->addBuiltinFunction("insertSeparator", (scriptObjectFunction) &KviScriptPopupMenu::builtinFunction_INSERTSEPARATOR);
	d->addBuiltinFunction("popup",           (scriptObjectFunction) &KviScriptPopupMenu::builtinFunction_POPUP);
	d->addBuiltinFunction("clear",           (scriptObjectFunction) &KviScriptPopupMenu::builtinFunction_CLEAR);
	d->addBuiltinFunction("text",            (scriptObjectFunction) &KviScriptPopupMenu::builtinFunction_TEXT);
	KviScriptEventStruct *s = new KviScriptEventStruct();
	s->szName   = "OnItemClicked";
	s->szBuffer = "$this->$emit(itemClicked, $1)";
	d->addDefaultEvent(s);
}

KviScriptPopupMenu::KviScriptPopupMenu(
	KviScriptObjectController *cntrl, KviScriptObject *p, const char *name, KviScriptObjectClassDefinition *pDef)
	: KviScriptWidget(cntrl, p, name, pDef)
{
	// Nothing here
}

KviScriptPopupMenu::~KviScriptPopupMenu()
{
	// Nothing here
}

bool KviScriptPopupMenu::init(QPtrList<KviStr> *params)
{
	if( parent() ) {
		if( parent()->inherits("KviScriptWidget") ) {
			m_pWidget = new KviPopupMenu(((KviScriptWidget *) parent())->m_pWidget, name());
		}
	}
	if( !m_pWidget )
		m_pWidget = new KviPopupMenu(0, name());
	connect(m_pWidget, SIGNAL(activated(int)), this, SLOT(itemActivated(int)));
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget, SIGNAL(destroyed()), this, SLOT(widgetDestroyed()));
	return true;
}

int KviScriptPopupMenu::builtinFunction_INSERTITEM(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			KviStr *pS2 = params->next();
			KviStr tmp;
			if( pS2 ) {
				bool bOk = false;
				int id   = pS2->toInt(&bOk) % KVI_OUT_NUM_IMAGES;
				if( bOk )
					tmp.setNum(((KviPopupMenu *) m_pWidget)->insertItem(*(g_pixViewOut[id]), pS->ptr()));
				else {
					QPixmap thePix(pS2->ptr());
					if( thePix.isNull() )
						tmp.setNum(((KviPopupMenu *) m_pWidget)->insertItem(*(g_pixViewOut[id]), pS->ptr()));
					else
						tmp.setNum(((KviPopupMenu *) m_pWidget)->insertItem(thePix, pS->ptr()));
				}
			} else tmp.setNum(((KviPopupMenu *) m_pWidget)->insertItem(pS->ptr()));
			buffer.append(tmp);
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptPopupMenu::builtinFunction_REMOVEITEM(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			bool bOk = false;
			int id   = pS->toInt(&bOk);
			if( bOk )
				((KviPopupMenu *) m_pWidget)->removeItem(id);
			else
				return KVI_ERROR_InvalidParameter;
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptPopupMenu::builtinFunction_SETITEMENABLED(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			KviStr *pS2 = params->next();
			if( pS2 ) {
				bool bOk = false;
				int id   = pS->toInt(&bOk);
				if( !bOk )
					return KVI_ERROR_InvalidParameter;
				int bEnabled = pS2->toInt(&bOk);
				if( !bOk )
					return KVI_ERROR_InvalidParameter;
				((KviPopupMenu *) m_pWidget)->setItemEnabled(id, bEnabled);
				return KVI_ERROR_Success;
			}
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptPopupMenu::builtinFunction_SETITEMCHECKED(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			KviStr *pS2 = params->next();
			if( pS2 ) {
				bool bOk = false;
				int id   = pS->toInt(&bOk);
				if( !bOk )
					return KVI_ERROR_InvalidParameter;
				int bChecked = pS2->toInt(&bOk);
				if( !bOk )
					return KVI_ERROR_InvalidParameter;
				((KviPopupMenu *) m_pWidget)->setItemChecked(id, bChecked);
				return KVI_ERROR_Success;
			}
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptPopupMenu::builtinFunction_INSERTMENU(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			KviStr *pS2 = params->next();
			if( pS2 ) {
				KviScriptObject *o = controller()->findObjectById(pS2->ptr());
				if( !o )
					return KVI_ERROR_ObjectNotFound;
				if( !o->inherits("KviScriptPopupMenu") )
					return KVI_ERROR_ObjectIsNotAPopup;
				KviStr *pS3 = params->next();
				KviStr tmp;
				if( pS3 ) {
					bool bOk = false;
					int id   = pS3->toInt(&bOk) % KVI_OUT_NUM_IMAGES;
					if( bOk ) {
						tmp.setNum(
							((KviPopupMenu *) m_pWidget)->insertItem(*(g_pixViewOut[id]), pS->ptr(),
							((KviPopupMenu *) (((KviScriptPopupMenu *) o)->m_pWidget)))
						);
					} else {
						QPixmap thePix(pS3->ptr());
						if( thePix.isNull() ) {
							tmp.setNum(
								((KviPopupMenu *) m_pWidget)->insertItem(*(g_pixViewOut[id]), pS->ptr(),
								((KviPopupMenu *) (((KviScriptPopupMenu *) o)->m_pWidget)))
							);
						} else {
							tmp.setNum(
								((KviPopupMenu *) m_pWidget)->insertItem(thePix, pS->ptr(),
								((KviPopupMenu *) (((KviScriptPopupMenu *) o)->m_pWidget)))
							);
						}
					}
				} else {
					tmp.setNum(
						((KviPopupMenu *) m_pWidget)->insertItem(pS->ptr(),
						((KviPopupMenu *) (((KviScriptPopupMenu *) o)->m_pWidget)))
					);
				}
				buffer.append(tmp);
				return KVI_ERROR_Success;
			}
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptPopupMenu::builtinFunction_TEXT(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			bool bOk = false;
			int id   = pS->toInt(&bOk);
			if( !bOk )
				return KVI_ERROR_InvalidParameter;
			KviStr tmp = ((KviPopupMenu *) m_pWidget)->text(id);
			buffer.append(tmp);
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptPopupMenu::builtinFunction_ISITEMENABLED(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			bool bOk = false;
			int id   = pS->toInt(&bOk);
			if( !bOk )
				return KVI_ERROR_InvalidParameter;
			buffer.append(((KviPopupMenu *) m_pWidget)->isItemEnabled(id) ? '1' : '0');
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptPopupMenu::builtinFunction_ISITEMCHECKED(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			bool bOk = false;
			int id   = pS->toInt(&bOk);
			if( !bOk )
				return KVI_ERROR_InvalidParameter;
			buffer.append(((KviPopupMenu *) m_pWidget)->isItemChecked(id) ? '1' : '0');
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptPopupMenu::builtinFunction_POPUP(QPtrList<KviStr> *params, KviStr &buffer)
{
	((KviPopupMenu *) m_pWidget)->popup(QCursor::pos());
	return KVI_ERROR_Success;
}

int KviScriptPopupMenu::builtinFunction_INSERTSEPARATOR(QPtrList<KviStr> *params, KviStr &buffer)
{
	((KviPopupMenu *) m_pWidget)->insertSeparator();
	return KVI_ERROR_Success;
}

int KviScriptPopupMenu::builtinFunction_CLEAR(QPtrList<KviStr> *params, KviStr &buffer)
{
	((KviPopupMenu *) m_pWidget)->clear();
	return KVI_ERROR_Success;
}

void KviScriptPopupMenu::itemActivated(int id)
{
	KviStr parms(KviStr::Format, "%d", id);
	triggerEvent("OnItemClicked", parms);
}

#include "m_kvi_script_popupmenu.moc"
