// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptQtWrapper"

#include <qobjectlist.h>
#include <qwidget.h>
#include <qwidgetlist.h>

#include "kvi_app.h"
#include "kvi_frame.h"
#include "kvi_script_objectcontroller.h"
#include "kvi_script_qtwrapper.h"

/*
	@class: qtwrapper
	@short:
		A Qt widget wrapper
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@description:
		This is a class for advanced KVIrc scripting.<br>
		It can wrap any existing KVIrc widget.<br>

		The KVIrc Qt widgets are arranged in trees (just as the objects).<br>
		The difference is that there are more toplevel widgets (and so more than one tree).<br>
		You can use [cmd]debug[/cmd] -w dump_objects to take a look at the KVIrc Qt objects tree.<br>
		Here is a part of the tree:<br><br>
		<p>
		[...]<br>
		Ptr 136544360: Top level object : kvirc_main_frame, class KviFrame, visible, rect = 4, 0, 788, 568<br>
		&nbsp;&nbsp;&nbsp;&nbsp;Ptr 136554184: Object : recent_servers_popup, class KviPopupMenu<br>
		&nbsp;&nbsp;&nbsp;&nbsp;Ptr 136555360: Object : recent_channels_popup, class KviPopupMenu<br>
		&nbsp;&nbsp;&nbsp;&nbsp;Ptr 137333152: Object : recent_nicknames_popup, class KviPopupMenu<br>
		&nbsp;&nbsp;&nbsp;&nbsp;Ptr 137336648: Object : main_splitter, class QSplitter<br>
		&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Ptr 137337464: Object : mdi_manager, class KviMdiManager<br>
		&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Ptr 137340656: Object : window_list_popup, class KviPopupMenu<br>
		&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Ptr 137514256: Object : mdi_child, class KviMdiChild<br>
		[...]<br>
		</p>
		<br><br>
		Every listed widget has a "name", a "class" and a set of properties.<br>
		The first entry in this example is a KVIrc server window, class "KviFrame" and name "kvirc_main_frame":
		it is a toplevel widget.<br>
		The "recent_server_popup", "recent_channels_popup", "recent_nicknames_popup" and the "main_splitter"
		are direct children of that widget.<br>
		The "mdi_manager" (that is the largest visible widget in each KVIrc main frame) is
		a direct child of the main_splitter.<br>
		To indicate a specific widget we will use the "class::name" form.<br>
		So to indicate the main KVIrc frame you will use "KviFrame::kvirc_main_frame",
		to indicate the MDI manager widget you will user "KviMdiManager::mdi_manager".<br>
		The special cases are:<br>
		"::mdi_manager" (::name) that designates the first widget that is named "mdi_manager" and has ANY class.<br>
		or "KviPopupMenu::" (class[::]) that designates the first widget with class "KviPopupMenu" and has ANY name.<br>
		The widget to wrap is defined by the constructor arguments of this class:<br>
		the fourth argument indicates the toplevel widget.<br>
		If there are arguments after the fourth, these indicate successive child widgets
		to be looked in the children sub-tree of the previous one.<br>
		To wrap the main KVIrc frame you will use:<br>
		<example>
		%tmp = [fnc]$new[/fnc](qtwrapper, [fnc]$root[/fnc], a_name, KviFrame::kvirc_main_frame)
		</example>
		To wrap the MDI manager you will use:<br>
		<example>
		%tmp = [fnc]$new[/fnc](qtwrapper, [fnc]$root[/fnc], a_name, KviFrame::kvirc_main_frame, KviMdiManager::mdi_manager)
		</example>
		The example above will first find the toplevel KviFrame then look in its sub-tree for the KviMdiManager::mdi_manager
		widget.<br>
		It could be done also in the following way:<br>
		<example>
		%tmp = [fnc]$new[/fnc](qtwrapper, [fnc]$root[/fnc], a_name, KviFrame::kvirc_main_frame, QSplitter::main_splitter, KviMdiManager::mdi_manager)
		</example>
		Since KVIrc can have more than one main frame, there is a special identifier for the
		main frame where the command is executed : '!'.<br>
		So again, the example above can be rewritten as:<br>
		<example>
		%tmp = [fnc]$new[/fnc](qtwrapper, [fnc]$root[/fnc], a_name, !, QSplitter::main_splitter, KviMdiManager::mdi_manager)
		</example>
		If there is exactly one KviFrame::kvirc_main_frame, the last two examples do exactly the
		same job, otherwise the first one wraps the "first frame found", the second one
		wraps the "current frame".<br>
		If there is exactly one mdi_manager in the specified main frame, the example above can
		be rewritten as follows:<br>
		<example>
		%tmp = [fnc]$new[/fnc](qtwrapper, [fnc]$root[/fnc], a_name, !, ::mdi_manager)
		</example>
		The constructor returns the <b>string</b> "0" if the widget to wrap cannot be found,
		and thus the class instance is not created.<br>
		You will find more examples of wrapping later in this documentation.<br>
		Once you have wrapped a widget, you will receive its events (look at the events
		for the [class]widget[/class] class), you will be able to
		use the [class]widget[/class] class functions, set its qt properties and add children
		widgets to it.<br>
		When the wrapped widget is destroyed, this class instance is destroyed too.<br>
		When this class instance is destroyed, the wrapped widget is NOT (you cannot destroy the wrapped widget).<br>
		Widget names and class names ARE case sensitive.<br>
		<b>NOTE:</b><br>
		This class is a *hack* :)<br>
		It allows to do a lot of strange things with the KVIrc executable.<br>
		It <b>IS</b> possible to make a royal mess in a way that KVIrc will be not able to recover.<br>
		The only way that you have to learn to use it well is <b>trial and error</b>.<br>
		In the really worst case the "error" will lead to a segfault :)<br>
		But this should not make you give up: while writing KVIrc I fight with SIGSEGV every day...<br>
		so just restart again and do not make the error a second time.<br>
		Just be sure that the scripts that use this class are widely tested if you are going
		to distribute it.<br>

	@examples:
		Wrap the Qt desktop widget, get its dimensions and set its color to dark gray.<br>
		<example>
		%desktop = [fnc]$new[/fnc](qtwrapper, [fnc]$root[/fnc], just_a_name, QWidget::desktop)
		[cmd]echo[/cmd] Desktop width : %desktop->[classfnc:widget]$width[/classfnc]()
		[cmd]echo[/cmd] Desktop height : %desktop->[classfnc:widget]$height[/classfnc]()
		%desktop->[classfnc:widget]$setBackgroundColor[/classfnc](303030)
		</example>
		Change the color of the status bar of the current KVIrc main frame.<br>
		<example>
		%bar =  [fnc]$new[/fnc](qtwrapper, [fnc]$root[/fnc], just_a_name, !, KviStatusBar::status_bar)
		%bar->[classfnc:widget]$setBackgroundColor[/classfnc](000000)
		</example>
		React to right clicks on the empty part of the toolbar in the current KVIrc main frame.<br>
		You could install a right click popup there :).<br>
		<example>
		%bar =  [fnc]$new[/fnc](qtwrapper, [fnc]$root[/fnc], just_a_name, !, KviMainToolBar::main_tool_bar)
		[cmd]obj_setevent[/cmd](%bar, [classevent:widget]OnMouseButtonPress[/classevent])[cmd]echo[/cmd] Toolbar clicked!
		# Mess with it just to show it better :)
		%bar->[classfnc:widget]$setBackgroundColor[/classfnc](303030)
		%bar->[classfnc:widget]$setQtProperty[/classfnc](frameShadow, Sunken)
		%bar->[classfnc:widget]$setQtProperty[/classfnc](frameShape, Panel)
		</example>
		Wrap the console window splitter and add a child widget to it.<br>
		<example>
		%spl = [fnc]$new[/fnc](qtwrapper, [fnc]$root[/fnc], just_a_name, !, KviConsole::@CONSOLE, QSplitter::console_splitter)
		%child = [fnc]$new[/fnc]([class]multilineedit[/class], %spl, child_editor)
		%child->[classfnc:widget]$show[/classfnc]()
		%child->[classfnc:multilineedit]$setText[/classfnc](Hello world!)
		</example>
		Do the same on the main frame splitter<br>
		<example>
		%spl = [fnc]$new[/fnc](qtwrapper, [fnc]$root[/fnc], just_a_name, !, QSplitter::main_splitter)
		%child = [fnc]$new[/fnc]([class]multilineedit[/class], %spl, child_editor)
		%child->[classfnc:widget]$show[/classfnc]()
		%child->[classfnc:multilineedit]$setText[/classfnc](Hi :)...[fnc]$lf[/fnc]\I could be replaced with any other widget)
		</example>
		You can add any widget instead of a multilineedit.<br>
		This could be useful for script configuration dialogs.<br>
		<br><br>
		Wrap the main window and list its qt properties<br>
		<example>
		%Main = [fnc]$new[/fnc](qtwrapper, [fnc]$root[/fnc], just_a_name, !)
		%Main->[classfnc:widget]$listQtProperties[/classfnc]()
		</example>
		Continuing the example above.<br>
		Set the toolbars drag-movement to opaque mode.<br>
		<example>
		%Main->[classfnc:widget]$setQtProperty[/classfnc](opaqueMoving, 1)
		</example>
		Continuing the example above.<br>
		Set the main window geometry and caption.<br>
		<example>
		%Main->[classfnc:widget]$setQtProperty[/classfnc](geometry, 30, 10, 600, 460)
		%Main->[classfnc:widget]$setQtProperty[/classfnc](caption, Messing with KVIrc !)
		</example>
		The beginning of a royal mess...<br>
		Wrap the main window splitter and change its orientation.<br>
		<example>
		%spl = [fnc]$new[/fnc](qtwrapper, [fnc]$root[/fnc], just_a_name, !, ::main_splitter)
		%spl->[classfnc:widget]$setQtProperty[/classfnc](orientation, Vertical)
		</example>
		Wrap a hidden popup menu and show it :)<br>
		<example>
		%pop = [fnc]$new[/fnc](qtwrapper, [fnc]$root[/fnc], just_a_name, !, ::window_list_popup)
		%pop->[classfnc:widget]$show[/classfnc]()
		</example>
	@seealso:
		class [class]object[/class], <br>
		class [class]widget[/class], <br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/

/**
 * QTWRAPPER class
 */
void KviScriptQtWrapper::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	// Nothing here
}

KviScriptQtWrapper::KviScriptQtWrapper(
	KviScriptObjectController *cntrl, KviScriptObject *p, const char *name, KviScriptObjectClassDefinition *pDef)
	: KviScriptWidget(cntrl, p, name, pDef)
{
	m_pWidget = 0;
}

KviScriptQtWrapper::~KviScriptQtWrapper()
{
	// Nothing here
}

bool KviScriptQtWrapper::init(QPtrList<KviStr> *params)
{
	if( !params ) return false;

	m_pWidget = 0;
	for( KviStr *s = params->first(); s; s = params->next() ) {
		KviStr szClass;
		KviStr szName;
		int idx = s->findFirstIdx("::");
		if( idx != -1 ) {
			szClass = s->left(idx);
			szName  = (s->ptr() + idx + 2);
		} else {
			szClass = s->ptr();
			szName  = "";
		}
		if( m_pWidget ) {
			m_pWidget = findWidgetToWrap(
				szClass.hasData() ? szClass.ptr() : 0, szName.hasData() ? szName.ptr() : 0, m_pWidget
			);
		} else {
			m_pWidget = kvi_strEqualCS(szClass.ptr(), "!")
				? controller()->mainFrame()
				: findTopLevelWidgetToWrap(szClass.hasData() ? szClass.ptr() : 0, szName.hasData() ? szName.ptr() : 0
			);
		}
		if( !m_pWidget ) return false;
	}
	if( !m_pWidget ) return false;

	m_bAutoDestroyControlledWidget = false;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget, SIGNAL(destroyed()), this, SLOT(widgetDestroyed()));
	return true;
}

QWidget *KviScriptQtWrapper::findTopLevelWidgetToWrap(const char *szClass, const char *szName)
{
	QWidgetList *list = g_pApp->topLevelWidgets();
	if( !list ) return 0;

	QWidgetListIt it(*list);
	while( it.current() ) {
		bool bNameMatch  = false;
		bool bClassMatch = false;
		if( szName )
			bNameMatch = kvi_strEqualCS(it.current()->name(), szName);
		else
			bNameMatch = true;
		if( szClass )
			bClassMatch = kvi_strEqualCS(it.current()->className(), szClass);
		else
			bClassMatch = true;
		if( bNameMatch && bClassMatch ) {
			QWidget *w = it.current();
			delete list;
			return w;
		}
		++it;
	}
	delete list;
	return 0;
}

QWidget *KviScriptQtWrapper::findWidgetToWrap(const char *szClass, const char *szName, QWidget *childOf)
{
	QObjectList *list = childOf->queryList(szClass ? szClass : 0, szName ? szName : 0, false, true);
	if( !list ) return 0;

	QObjectListIt it(*list);
	while( it.current() ) {
		if( it.current()->isWidgetType() ) {
			QWidget *w = (QWidget *) it.current();
			delete list;
			return w;
		}
		++it;
	}
	delete list;
	return 0;
}

#include "m_kvi_script_qtwrapper.moc"
