// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptTimer"

#include "kvi_error.h"
#include "kvi_script_objectclassdefinition.h"
#include "kvi_script_timer.h"

/*
	@class: timer
	@short:
		A timer object
	@inherits:
		[class]object[/class]<br>

	@functions:
		!fn: $start(&lt;interval&gt;)
		Starts this timer with a specified interval (timeout) in milliseconds.<br>
		If the timer was previously running, it is first stopped.<br>
		This function returns '1' if successful, or '0' if
		the underlying system was not able to provide a timer service:<br>
		usually you can avoid the return value of this function.<br>

		!fn: $stop()
		Stops this timer.<br>

		!fn: $singleShot(&lt;interval&gt;, [bAutodestroy]);
		Starts this timer with a specified interval (timeout) in milliseconds.<br>
		The timer will "fire" only once.<br>
		If the &lt;bAutodestroy&gt; parameter is provided, and is '1'
		this timer object will automatically destroy itself, otherwise
		it will just return in stopped state.<br>
		If the timer was previously running, it is first stopped.<br>
		This function returns '1' if successful, or '0' if
		the underlying system was not able to provide a timer service:<br>
		usually you can avoid the return value of this function.<br>

		!fn: $isRunning()
		Returns '1' if the timer is currently running, '0' if not.<br>

		!fn: $interval()
		Returns the interval between the timer 'shots' if the timer
		is currently running, '0' otherwise.<br>

	@events:
		!ev: OnTimeout()
		Triggered by this timer object when the interval of time has passed.<br>
		The default event handler for this event emits the [classsignal:timer]timeout[/classsignal].<br>
		If you provide a new event handler and still want the signal to be emitted
		you must emit it by yourself.<br>
		<example>
		[fnc]$this[/fnc]->[classfnc:object]$emit[/classfnc]([classsignal:timer]timeout[/classsignal])
		</example>

	@signals:
		!sg: timeout()
		This signal is emitted by the default handler for the [classevent:timer]OnTimeout[/classevent].
		event handler.<br>
		If you provide a new event handler and still want the signal to be emitted
		you must emit it by yourself.<br>
		<example>
		[fnc]$this[/fnc]->[classfnc:object]$emit[/classfnc]([classsignal:timer]timeout[/classsignal])
		</example>

	@description:
		A simple timer object with a millisecond precision.<br>
		It fires the [classevent:timer]OnTimeout[/classevent] event after the
		specified delay.<br> The defaul event handler for that event
		emits the [classsignal:timer]timeout[/classsignal] signal.<br>
		The timer can be continous (started by [classfnc:timer]$start[/classfnc]())
		or "single shot" (started by [classfnc:timer]$singleShot[/classfnc]()).<br>
		In the single shot mode the timer has also the ability of autodestroy itself.<br>

	@examples:
		A class for saying text line by line to a channel (or another window).<br>
		It is useful when you have bots on the channel and they will likely kick
		you if you flood the channel with a long text.<br>
		(If the bots are not your problem, the problem may be the server
		flood protection that disconnects you for 'Excess Flood').<br>
		So here we go: we derive a class from [class]timer[/class].<br>
		<example>
		&nbsp;[cmd]class[/cmd](asciitimer, [class]timer[/class])
		&nbsp;{
		&nbsp;	[cmd:class]function[/cmd](run)
		&nbsp;	{
		&nbsp;      # The second param is the window to say the text to
		&nbsp;		[fnc]$this[/fnc]->%window = $2
		&nbsp;      # The remaining params are the "text" to say
		&nbsp;		[fnc]$this[/fnc]->%text = "$3-"
		&nbsp;      # The first parameter is the timeout in msecs
		&nbsp;		[cmd]setreturn[/cmd] [fnc]$this[/fnc]->[classfnc:timer]$start[/classfnc]($1)
		&nbsp;	}
		&nbsp;
		&nbsp;	[cmd:class]event[/cmd]([classevent:timer]OnTimeout[/classevent])
		&nbsp;	{
		&nbsp;		%line = [fnc]$strLeftToFirstCS[/fnc]([fnc]$lf[/fnc], "[fnc]$this[/fnc]->%text")
		&nbsp;		[fnc]$this[/fnc]->%text = [fnc]$strRightFromFirstCS[/fnc]([fnc]$lf[/fnc], "[fnc]$this[/fnc]->%text")
		&nbsp;		[cmd]say[/cmd] [fnc]$this[/fnc]->%window %line
		&nbsp;		[cmd]if[/cmd]("[fnc]$this[/fnc]->%text" == "")[cmd]destroy[/cmd] [fnc]$this[/fnc]
		&nbsp;	}
		&nbsp;}
		</example>
		Now you can use the following code to run it (maybe place it in an
		appropriate alias)...<br>
		<example>
		%test = [fnc]$new[/fnc](asciitimer, [fnc]$root[/fnc], test)
		%test->$run(1000, #channel, "This[fnc]$lf[/fnc]\Is[fnc]$lf[/fnc]\A[fnc]$lf[/fnc]\Test[fnc]$lf[/fnc]\String[fnc]$lf[/fnc]\With[fnc]$lf[/fnc]\Newlines")
		</example>
	@seealso:
		class [class]object[/class], <br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/

/**
 * TIMER class
 */
void KviScriptTimer::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("start",      (scriptObjectFunction) &KviScriptTimer::builtinFunction_START);
	d->addBuiltinFunction("stop",       (scriptObjectFunction) &KviScriptTimer::builtinFunction_STOP);
	d->addBuiltinFunction("isRunning",  (scriptObjectFunction) &KviScriptTimer::builtinFunction_ISRUNNING);
	d->addBuiltinFunction("interval",   (scriptObjectFunction) &KviScriptTimer::builtinFunction_INTERVAL);
	d->addBuiltinFunction("singleShot", (scriptObjectFunction) &KviScriptTimer::builtinFunction_SINGLESHOT);

	KviScriptEventStruct *s = new KviScriptEventStruct();
	s->szName   = "OnTimeout";
	s->szBuffer = "$this->$emit(timeout)";
	d->addDefaultEvent(s);
}

KviScriptTimer::KviScriptTimer(
	KviScriptObjectController *cntrl, KviScriptObject *p, const char *name, KviScriptObjectClassDefinition *pDef)
	: KviScriptObject(cntrl, p, name, pDef)
{
	m_iTimerId = -1;
	m_iTimeout = 0;
}

KviScriptTimer::~KviScriptTimer()
{
	if( m_iTimerId != -1 )
		killTimer(m_iTimerId);
}

void KviScriptTimer::timerEvent(QTimerEvent *e)
{
	if( e->timerId() == m_iTimerId ) {
		KviStr params;
		triggerEvent("OnTimeout", params);
		if( m_bSingleShot ) {
			killTimer(m_iTimerId);
			m_iTimerId    = -1;
			m_iTimeout    = 0;
			m_bSingleShot = false;
			if( m_bAutoDestroyAfterSingleShot )
				dieOutOfThisEventStep();
		}
	}
}

int KviScriptTimer::builtinFunction_START(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			bool bOk = false;
			int i    = pS->toInt(&bOk);
			if( bOk && (i >= 0) ) {
				if( m_iTimerId != -1 )
					killTimer(m_iTimerId);
				m_iTimerId    = startTimer(i);
				m_iTimeout    = i;
				m_bSingleShot = false;
				buffer.append((m_iTimerId != -1) ? '1' : '0');
				return KVI_ERROR_Success;
			}
			return KVI_ERROR_InvalidParameter;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptTimer::builtinFunction_SINGLESHOT(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			bool bOk = false;
			int i    = pS->toInt(&bOk);
			if( bOk && (i >= 0) ) {
				if( m_iTimerId != -1 )
					killTimer(m_iTimerId);
				m_iTimerId    = startTimer(i);
				m_iTimeout    = i;
				m_bAutoDestroyAfterSingleShot = false;
				m_bSingleShot = true;
				KviStr *pS2 = params->next();
				if( pS2 ) {
					if( kvi_strEqualCI(pS2->ptr(), "1") )
						m_bAutoDestroyAfterSingleShot = true;
				}
				buffer.append((m_iTimerId != -1) ? '1' : '0');
				return KVI_ERROR_Success;
			}
			return KVI_ERROR_InvalidParameter;
		}
	}
	return KVI_ERROR_Success;
}

int KviScriptTimer::builtinFunction_STOP(QPtrList<KviStr> *, KviStr &buffer)
{
	if( m_iTimerId == -1 )
		return KVI_ERROR_Success;
	killTimer(m_iTimerId);
	m_iTimerId    = -1;
	m_iTimeout    = 0;
	m_bSingleShot = false;
	return KVI_ERROR_Success;
}

int KviScriptTimer::builtinFunction_ISRUNNING(QPtrList<KviStr> *, KviStr &buffer)
{
	buffer.append((m_iTimerId != -1) ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptTimer::builtinFunction_INTERVAL(QPtrList<KviStr> *, KviStr &buffer)
{
	KviStr timeout(KviStr::Format, "%d", m_iTimeout);
	buffer.append(timeout);
	return KVI_ERROR_Success;
}

#include "m_kvi_script_timer.moc"
