// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptToolButton"

#include "kvi_app.h"
#include "kvi_error.h"
#include "kvi_options.h"
#include "kvi_script_objectclassdefinition.h"
#include "kvi_script_toolbutton.h"
#include "kvi_toolbarbutton.h"

/*
	@class: toolbutton
	@short:
		A tool button widget (mainly for toolbars)
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]

	@functions:
		!fn: $setText(&lt;text&gt;)
		Sets the text to be used as the "text label" and as the tooltip for the button.<br>

		!fn: $text()
		Returns the text label of the button.<br>

		!fn: $setImage(&lt;image_path&gt;)
		Sets the image that should be displayed on the button.<br>
		The &lt;image_path&gt; can be absolute or a simple file name
		(in this case KVIrc will look for the file in a set of predefined
		directories including the local KVIrc pics directory, the global one, the
		user home directory)
		You should provide a 32x32 image : KVIrc will automatically
		resize it when oter formats will be requested.<br>
		Returns 1 if the image has been found and set sucesfully, 0 otherwise.<br>

	@events:
		!ev: OnClick()
		Triggered when the user clicks the button.<br>
		<b>This signal has a default event handler that emits the [classsignal:toolbutton]clicked[/classsignal] signal</b>.
		If you override the event handler, and still want the signal to be emitted, you have to emit
		it by yourself!.<br>
		The default event handler is the following one:<br>
		<example>
			[fnc]$this[/fnc]->[classfnc:object]$emit[/classfnc]([classsignal:toolbutton]clicked[/classsignal])
		</example>

	@signals:
		!sg: clicked()
		Emitted when the button has been clicked by the user.<br>
		<b>This signal is emitted by the default [classevent:toolbutton]OnClick[/classevent] event handler!<br>
		If you define a new event handler, and still want the signal to be emitted,
		you have to emit it by yourself!

	@description:
		This is a tool button class, to be used mainly with [class]toolbar[/class] objects as parent.<br>
		It usually shows an image that can be 32x32 or 16x16 pixels.<br>
		You should always provide a 32x32 image : KVIrc will resize it when needed.<br>
		The button can optionally display a text label.<br>
		The text label will be shown if KVIrc has this option enabled.<br>
		You can force the text label to be shown by setting the usesTextLabel qt property to
		true (see [classfnc:widget]$setQtProperty[/classfnc]()).<br>

		The constructor for this object accepts two additional (optional) parameters:<br>
		The path to the image to be displayed and the text label to be set.<br>

		<example>
		%btn = [fnc]$new[/fnc]([class]toolbutton[/class], [fnc]$root[/fnc], a_name, /usr/share/icons/quit.png, Quit the session)
		</example>

		When the user clicks the button, the [classevent:toolbutton]OnClick[/classevent]
		event is triggered. The [classevent:toolbutton]OnClick[/classevent] event has a default
		event handler that emits the [classsignal:toolbutton]clicked[/classsignal] signal.<br>
		If you define a new event handler, and still want the signal to be emitted,
		you have to emit it by yourself!

	@examples:
		A colorful (but useless) example.<br>
		A toolbar subclass with some buttons using the KVIrc file icons.<br>
		It will be docked initially on the top of the main KVIrc window
		and will NOT be dockable to the right edge.<br>

		<example>
		&nbsp;[cmd]class[/cmd] -o (mytoolbar, [class]toolbar[/class])
		&nbsp;{
		&nbsp;	[cmd:class]function[/cmd](constructor)
		&nbsp;	{
		&nbsp;		[fnc]$this[/fnc]->$newButton(html, html.png)
		&nbsp;		[fnc]$this[/fnc]->$newButton(h, source_h.png)
		&nbsp;		[fnc]$this[/fnc]->$newButton(c, source_c.png)
		&nbsp;		[fnc]$this[/fnc]->$newButton(exe, exec.png)
		&nbsp;		[fnc]$this[/fnc]->$newButton(image, image.png)
		&nbsp;		[fnc]$this[/fnc]->$newButton(lib, lib.png)
		&nbsp;		[fnc]$this[/fnc]->$newButton(rpm, rpm.png)
		&nbsp;		[fnc]$this[/fnc]->$newButton(shell, shellscript.png)
		&nbsp;		[fnc]$this[/fnc]->$newButton(sound, sound.png)
		&nbsp;		[fnc]$this[/fnc]->$newButton(text, txt.png)
		&nbsp;		[fnc]$this[/fnc]->$newButton(zip, zip.png)
		&nbsp;	}
		&nbsp;
		&nbsp;	[cmd:class]function[/cmd](newButton)
		&nbsp;	{
		&nbsp;		%btn = [fnc]$new[/fnc]([class]toolbutton[/class], [fnc]$this[/fnc], button_$1, $2, $1)
		&nbsp;		[cmd]connect[/cmd] %btn [classsignal:toolbutton]clicked[/classsignal] [fnc]$this[/fnc] buttonSlot
		&nbsp;	}
		&nbsp;
		&nbsp;	[cmd:class]function[/cmd](buttonSlot)
		&nbsp;	{
		&nbsp;		[cmd]echo[/cmd] "Clicked button \"[fnc]$signalSender[/fnc]->[classfnc:toolbutton]$text[/classfnc]()\""
		&nbsp;	}
		&nbsp;}
		&nbsp;
		&nbsp;%bar = [fnc]$new[/fnc](mytoolbar, [fnc]$root[/fnc], TestToolbar)
		&nbsp;
		&nbsp;%bar->[classfnc:toolbar]$setDockEnabled[/classfnc](right, 0)
		&nbsp;%bar->[classfnc:toolbar]$dock[/classfnc](top)
		&nbsp;%bar->[classfnc:widget]$show[/classfnc]()
		</example>
	@seealso:
		[class]object[/class] class</a>, <br>
		[class]widget[/class] class</a>, <br>
		[class]toolbar[/class] class</a>, <br>
		<a href="syntax_objects.kvihelp">Objects documentation</a>
*/

/**
 * BUTTON class
 */
void KviScriptToolButton::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("setText",  (scriptObjectFunction) &KviScriptToolButton::builtinFunction_SETTEXT);
	d->addBuiltinFunction("text",     (scriptObjectFunction) &KviScriptToolButton::builtinFunction_TEXT);
	d->addBuiltinFunction("setImage", (scriptObjectFunction) &KviScriptToolButton::builtinFunction_SETIMAGE);

	KviScriptEventStruct *s = new KviScriptEventStruct();
	s->szName   = "OnClick";
	s->szBuffer = "$this->$emit(clicked)";
	d->addDefaultEvent(s);
}

KviScriptToolButton::KviScriptToolButton(
	KviScriptObjectController *cntrl, KviScriptObject *p, const char *name, KviScriptObjectClassDefinition *pDef)
	: KviScriptWidget(cntrl, p, name, pDef)
{
	// Nothing here
}

KviScriptToolButton::~KviScriptToolButton()
{
	// Nothing here
}

bool KviScriptToolButton::init(QPtrList<KviStr> *params)
{
	if( parent() ) {
		if( parent()->inherits("KviScriptWidget") ) {
			m_pWidget = new KviToolBarButton(((KviScriptWidget *) parent())->m_pWidget, name());
		}
	}
	if( !m_pWidget )
		m_pWidget = new KviToolBarButton(0, name());
	((KviToolBarButton *) m_pWidget)->setUsesBigPixmap(g_pOptions->m_bUseBigToolbarPixmaps);
	((KviToolBarButton *) m_pWidget)->setUsesTextLabel(g_pOptions->m_bUseTextToolbarComment);

	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget, SIGNAL(clicked()),   this, SLOT(buttonClicked()));
	connect(m_pWidget, SIGNAL(destroyed()), this, SLOT(widgetDestroyed()));
	if( params ) {
		KviStr *pS1 = params->first();
		KviStr *pS2 = params->next();
		if( pS1 ) {
			KviStr path;
			if( g_pApp->findImage(path, pS1->ptr()) )
				((KviToolBarButton *) m_pWidget)->setPixmap(QPixmap(path.ptr()));
			if( pS2 ) {
				((KviToolBarButton *) m_pWidget)->setTextLabel(pS2->ptr(), true);
			}
		}
	}
	return true;
}

int KviScriptToolButton::builtinFunction_TEXT(QPtrList<KviStr> *, KviStr &buffer)
{
	buffer.append(((KviToolBarButton *) m_pWidget)->textLabel());
	return KVI_ERROR_Success;
}

int KviScriptToolButton::builtinFunction_SETTEXT(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			((KviToolBarButton *) m_pWidget)->setTextLabel(pS->ptr(), true);
			return KVI_ERROR_Success;
		}
	}
	((KviToolBarButton *) m_pWidget)->setTextLabel("", true);
	return KVI_ERROR_Success;
}

int KviScriptToolButton::builtinFunction_SETIMAGE(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			KviStr path;
			if( g_pApp->findImage(path, pS->ptr()) ) {
				((KviToolBarButton *) m_pWidget)->setPixmap(QPixmap(path.ptr()));
				buffer.append('1');
			} else {
				buffer.append('0');
			}
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

void KviScriptToolButton::buttonClicked()
{
	// Trigger clicked()
	KviStr params;
	triggerEvent("OnClick", params);
}

#include "m_kvi_script_toolbutton.moc"
