/*b
 * Copyright (C) 2001,2002  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <ncurses.h>
#include <panel.h>
#include <errno.h>
#include <time.h>
#include <sys/time.h>
#include <ctype.h>
#include "error.h"
#include "debug.h"
#include "rc.h"
#include "streamer.h"
#include "curse.h"
#include "linuxtrade.h"
#include "minihtml.h"
#include "bloomearn.h"

/*
 * &position=0050&pagenumber=000
 *
 * http://quote.bloomberg.com/apps/ecal?date=20030715&strtpt=0&endpt=500&c=US
 */
static WINDOW	*Win;
static WINDOW	*Subwin;
static PANEL	*Panel;

static int	Cursor;		// Article line number at top of display
static int	MaxCursor;	// Maximum value for above
static int	NumLines;	// Number of lines in the article
static time_t	LastDocPoll;	// Last time we grabbed the InPlay data
static time_t	LastLivePoll;	// Last time we grabbed the live quotes
static time_t	LastAlertPoll;	// Last time we checked for new document
static STREAMER	Sr = NULL;
static int	ValDocPoll;
static int	ValLivePoll;
static int	ValAlertPoll = -1;

static time_t	UrlTime;

#define	PADLINES	700

typedef struct
{
	char	sym[SYMLEN+1];
	int	x;
} QUOTELOC;

static QUOTELOC	Live[PADLINES];
static char	*Audio[PADLINES];

static void
get_live_quotes(void)
{
	int	y;
	int	cnt = 0;

	for (y = Cursor; y < Cursor+getmaxy(Win)-2; ++y)
		if (Live[y].sym[0])
		{
			(Sr->send_livequote)(Sr, Live[y].sym);
			++cnt;
		}
	if (cnt)
		(Sr->send_livequote_end)(Sr);

	wattrset(Subwin, A_BOLD);
	mvwprintw(Subwin, 2, getmaxx(Subwin)-15,
			"   Live Quotes ");
	mvwprintw(Subwin, 3, getmaxx(Subwin)-15,
			"  Price  Change");
	wattrset(Subwin, A_NORMAL);

	time(&LastLivePoll);
}

void
bloomearn_livequote(LIVEQUOTE *lq)
{
	int	y;

	if (!Win)
		return;

	for (y = Cursor; y < Cursor+getmaxy(Win)-2; ++y)
	{
		attr_t	attr;

		if (strcmp(lq->sym, Live[y].sym))
			continue;

		if (lq->last > lq->close)
			attr = COLOR_PAIR(1) | A_NORMAL;
		else if (lq->last < lq->close)
			attr = COLOR_PAIR(2) | A_NORMAL;
		else
			attr = A_NORMAL;

		wattrset(Subwin, attr);
		mvwprintw(Subwin, y, Live[y].x,
			"%7.2f %+7.2f", lq->last, lq->last - lq->close);
		wattrset(Subwin, A_NORMAL);
	}

	// TODO: optimize this to do 1 line and move into loop
	copywin(Subwin, Win,
			Cursor, 0,
			1, 1,
			getmaxy(Win)-2, getmaxx(Win)-2,
			FALSE);
	move(LINES-1, CursorX);
	update_panels();
	doupdate();
}

//
// TODO: make this asynchronous
//

static void
display_more(void)
{
	mvwaddch(Win, 1, getmaxx(Win)-1,
		Cursor ? ACS_UARROW : ACS_VLINE);

	mvwaddch(Win, getmaxy(Win)-2, getmaxx(Win)-1,
		(Cursor < MaxCursor) ? ACS_DARROW : ACS_VLINE);
}

static TBLCOL	Col[] =
{
	{  0, 99999},
	{  0, 11 },	//Company
	{ 13, 19 },	//Symbol
	{ 21, 25 },	//Ann Date
	{ 28, 34 },	//Period Ending
	{ 36, 50 },	//Est EPS
	{ 52, 78 },	//Audio
	{ 0, 99999 }
};
#define NUMCOL	asizeof(Col)

static int	Nstock0;

static void
postwordhook(int *tblcolp, char *wbuf, int wlen)
{
	int	x, y;
	static double	actual = 0, est = 0;
	static int	slash = 0;
	int	dolive = ValLivePoll > 1;
	int	tblcol = *tblcolp;

	// Hack: fix column header for est/actual
	if (tblcol == 6 && strcmp(wbuf, " AUDIO") == 0 )
	{
		x = getcurx(Subwin);
		y = getcury(Subwin);
		wmove(Subwin, y, Col[tblcol-1].sx);
		waddstr(Subwin, "EST / ACTUAL EPS ");
		wmove(Subwin, y, x);
	}

	// Colorize hits and misses
	if (tblcol == 4)
	{
	    slash = 0;
	    est = actual = 0;
	}
	if (tblcol == 5)
	{
		if (!slash && !est)
		    est = atof(wbuf);
		if (slash && !actual)
		    actual = atof(wbuf);
		if (strcmp(wbuf, "/") == 0)
		    slash = 1;
		if (est && actual)
		{
		    if (actual < est)
			minihtml_colorcol(Subwin, Col, 5, REDonBG);
		    else if (actual > est)
			minihtml_colorcol(Subwin, Col, 5, GREENonBG);
		}
	}

	// Add live quotes and add it to stocklist 0
	if (dolive && (wlen-0) < SYMLEN && tblcol == 2)
	{
		x = getcurx(Subwin);
		y = getcury(Subwin);

		if (strcmp(wbuf, "Symbol")
			&& strcmp(wbuf, "Page")
			&& strcmp(wbuf, "Forward"))
		{
			memcpy(Live[y].sym, wbuf, wlen);
			Live[y].sym[wlen] = 0;
			Live[y].x = getmaxx(Subwin) - 15;

			if (Nstock0)
				strcat(List0, " ");
			strcat(List0, Live[y].sym);
			++Nstock0;
			strcpy(StockList0Name, "Earnings");
		}
	}
	if (tblcol == 3 && strchr(wbuf, '/'))
	{
		strcat(List0Comments, wbuf);
		strcat(List0Comments, "\n");
	}
}

static void
anchorhook(char *parmp, int tblcol, int slash)
{
	char	audio[] = "href=\"rtsp:";
	int	y = getcury(Subwin);

	if (slash)
		return;
	if (strncmp(parmp, audio, sizeof(audio)-1))
		return;
	if (y < PADLINES)
		Audio[y] = strdup(parmp+5);
	wattrset(Subwin, A_BOLD|A_UNDERLINE);
	waddstr(Subwin, "Listen");
	wattrset(Subwin, A_NORMAL);
}

static void
get_bloomearn(void)
{
	FILE		*fp;
	int		y;
	char		buf[1024];
	struct tm	*tmp;

	tmp = localtime(&UrlTime);

	mvwcenter(Win, getmaxy(Win)/2, "...Please Wait...");
	update_panels(); doupdate();

	sprintf(buf, "bloomearn.helper -D%d %04d%02d%02d %s",
			Debug,
			tmp->tm_year + 1900,
			tmp->tm_mon + 1,
			tmp->tm_mday,
			"US");
	debug(1, "earnings: %s\n", buf);

	fp = popen(buf, "r");
	if (!fp)
	{
		mvwprintw(Subwin, getmaxy(Subwin)/2, 20,
				"Can't access bloomearn");
		touchwin(Win);
		LastAlertPoll = LastDocPoll = time(NULL);
		return;
	}

	/*
	 * Eat lines until we see the table
	 */
	if (0)
	    minihtml_skip_past_line(fp, "<table width=\"100%\"");

	/*
	 * Parse HTML
	 */
	Nstock0 = 0;
	List0[0] = 0;
	List0Comments[0] = 0;

	for (y = 0; y < PADLINES; ++y)
		Live[y].sym[0] = 0;

	minihtml_parse(Subwin, fp, Col, NUMCOL, MHP_DEFAULT,
			NULL, postwordhook, anchorhook, NULL);

	pclose(fp);

	LastAlertPoll = LastDocPoll = time(NULL);

	Cursor = 0;
	NumLines = getcury(Subwin) + 1;
	MaxCursor = NumLines - (getmaxy(Win) - 2);
	if (MaxCursor < 0)
		MaxCursor = 0;

	display_more();

	copywin(Subwin, Win,
			Cursor, 0,
			1, 1,
			getmaxy(Win)-2, getmaxx(Win)-2,
			FALSE);
	move(LINES-1, CursorX);
	update_panels();
	doupdate();
	get_live_quotes();
}

static void
check_bloomearn(void)
{
#if 0
	FILE		*fp;
	char		buf[BUFSIZ];
	static char	lastbuf[256];

	sprintf(buf, "%s \"%s\"", SUCKURL, Url);

	LastAlertPoll = time(NULL);

	fp = popen(buf, "r");
	if (!fp)
		return;

	/*
	 * Eat lines until we see the table
	 */
	while (fgets(buf, sizeof(buf), fp))
	{
		static char checkfor[] = "<H4>Updated:";

		if (strncmp(buf, checkfor, strlen(checkfor)-1))
			continue;

		if (strcmp(buf, lastbuf) == 0)
			break;

		memcpy(lastbuf, buf, sizeof(lastbuf));

		attrset(COLOR_PAIR(2));
		mvprintw(1, 14, "EARNINGS");
		attrset(A_NORMAL);
		break;
	}

	pclose(fp);
#endif
}

void
bloomearn_poll(void)
{
	time_t	now;

	if (!Win)
	{
		if (ValAlertPoll < 0)
			ValAlertPoll = 60 * atoi(get_rc_value(RcFile,
						"bloomearn_alert"));
		if (!ValAlertPoll)
			return;

		now = time(NULL);

		// Defer first poll for 35 seconds
		if (LastAlertPoll == 0)
			LastAlertPoll = now + 35 - ValAlertPoll;

		if (ValAlertPoll && now >= (LastAlertPoll + ValAlertPoll))
			check_bloomearn();

		return;
	}

	if (!ValLivePoll && !ValDocPoll)
		return;

	now = time(NULL);

	if (ValLivePoll && now >= (LastLivePoll + ValLivePoll))
		get_live_quotes();

	if (ValDocPoll && !Cursor && now >= (LastDocPoll + ValDocPoll))
		get_bloomearn();
}

static
void
display_popup_title(void)
{
	struct tm	*tm;
	char		buf[80];
	char		*fmt = "Bloomberg Earnings Calendar for %x";

	tm = localtime(&UrlTime);
	strftime(buf, sizeof(buf), fmt, tm);
	mvwcenter(Win, 0, buf);
}

void
bloomearn_popup(STREAMER sr)
{
	int	n;
	int	cols;

	// Clear alert, if any
	mvprintw(1, 7, "      ");

	n = LINES - 4 - 2 - NumStock - 12;
	if (n < 24)
		n = 24;
	Win = bestwin(n);
	if (!Win)
		error(1, "Can't create bloomearn window\n");

	cols = getmaxx(Win);

	wbkgd(Win, Reverse ? A_REVERSE : A_NORMAL);

	box(Win, 0, 0);
	// mvwprintw(Win, 0, (cols-17)/2, "Earnings Calendar");

	Subwin = newpad(PADLINES, cols - 2);
	if (!Subwin)
		error(1, "Can't create bloomearn pad\n");
	wbkgd(Subwin, Reverse ? A_REVERSE : A_NORMAL);

	Panel = new_panel(Win);

	time(&UrlTime);
	display_popup_title();

	ValDocPoll = 60 * atoi(get_rc_value(RcFile, "bloomearn_poll"));

	//
	// Lots of stocks on this list, limit poll period
	//
	ValLivePoll = atoi(get_rc_value(RcFile, "live_quotes"));
	if (ValLivePoll && ValLivePoll < 15)
		ValLivePoll = 15;

	// Paint empty frame to give user some feedback
	clearok(Win, TRUE);
	update_panels(); doupdate();

	memset(Audio, 0, sizeof(Audio));

	Sr = sr;
	get_bloomearn();
}

static void
popdown(void)
{
	int	i;

	for (i = 0; i < PADLINES; ++i)
		if (Audio[i])
		{
			free(Audio[i]);
			Audio[i] = 0;
		}
	hide_panel(Panel);
	update_panels();
	del_panel(Panel);
	delwin(Subwin);
	delwin(Win);
	Win = 0;
	Subwin = 0;
	Sr = NULL;
}

int
bloomearn_command(int c, STREAMER sr)
{
	struct tm	*tm;
	MEVENT		m;

	switch (c)
	{
	case '\f':
		move(LINES-1, CursorX);
		wrefresh(curscr);
		return -1;
	case 'j':
	case KEY_DOWN:
		if (++Cursor > MaxCursor)
		{
			--Cursor;
			beep();
			break;
		}
		break;
	case 'k':
	case KEY_UP:
		if (--Cursor < 0)
		{
			++Cursor;
			beep();
			break;
		}
		break;
	case CTRL('u'):
	case CTRL('b'):
	case KEY_PPAGE:
		if (Cursor == 0)
		{
			beep();
			break;
		}
		Cursor -= getmaxy(Win) - 2 - 1;
		if (Cursor < 0)
			Cursor = 0;
		break;
	case ' ':
	case CTRL('d'):
	case CTRL('f'):
	case KEY_NPAGE:
		if (Cursor == MaxCursor)
		{
			beep();
			break;
		}
		Cursor += getmaxy(Win) - 2 - 1;
		if (Cursor > MaxCursor)
			Cursor = MaxCursor;
		break;
	case '0':
	case KEY_HOME:
		Cursor = 0;
		break;
	case '$':
	case KEY_END:
		Cursor = MaxCursor;
		break;

	case 'l':
	case KEY_RIGHT:
		for (;;)
		{
			UrlTime += 24 * 60 *60;
			tm = localtime(&UrlTime);
			if (tm->tm_wday >= 1 && tm->tm_wday <= 5)
				break;
		}
		goto getnew;
	case 'h':
	case KEY_LEFT:
		for (;;)
		{
			UrlTime -= 24 * 60 *60;
			tm = localtime(&UrlTime);
			if (tm->tm_wday >= 1 && tm->tm_wday <= 5)
				break;
		}
		goto getnew;
	case '+':
		for (;;)
		{
			UrlTime += 24 * 60 *60;
			tm = localtime(&UrlTime);
			if (tm->tm_wday == 1)
				break;
		}
		goto getnew;
	case '-':
		for (;;)
		{
			UrlTime -= 24 * 60 *60;
			tm = localtime(&UrlTime);
			if (tm->tm_wday == 1)
				break;
		}
	getnew:
		Cursor = 0;
		werase(Subwin);
		copywin(Subwin, Win,
			Cursor, 0,
			1, 1,
			getmaxy(Win)-2, getmaxx(Win)-2,
			FALSE);
		display_popup_title();
		move(LINES-1, CursorX);
		update_panels();
		doupdate();
		get_bloomearn();
		break;

	case KEY_F(11):
		print_rect_troff(getbegy(Win), getbegx(Win),
				getmaxy(Win), getmaxx(Win),
				NULL, "screen.tr");
		break;
	case KEY_F(12):
	case CTRL('P'):
	case KEY_PRINT:
		print_window(Subwin, NumLines,
				get_rc_value(RcFile, "print_cmd"));
		break;

		// Change stocklist on main screen
	case '1': case '2': case '3': case '4': case '5':
	case '6': case '7': case '8': case '9':
		return 3;

	case KEY_MOUSE:
		if (getmouse(&m) != OK)
			break;

		// Clicks in our window ...
		if (m.y >= getbegy(Win)
			&& m.y < getbegy(Win) + getmaxy(Win))
		{
			int y = m.y - getbegy(Win) - 1 + Cursor;

			if (Audio[y])
			{
				char buf[BUFSIZ];
				sprintf(buf, "realplay %s &", Audio[y]);
				system(buf);
			}
			break;
		}

		// popdown and reprocess clicks in main window
		if (ungetmouse(&m) == OK)
			Ungetch = 1;
		popdown();
		return 2;

		// Quick switches to another popup
	case 'p':
	case 'u':
		popdown();
		return 1;

		// Regular exit
	case 033:
	case 'q':
	case 'x':
		popdown();
		return 2;
	default:
		beep();
		break;
	}

	display_more();
	copywin(Subwin, Win,
			Cursor, 0,
			1, 1,
			getmaxy(Win)-2, getmaxx(Win)-2,
			FALSE);
	move(LINES-1, CursorX);
	update_panels();
	doupdate();
	get_live_quotes();

	return (-1);
}
