/*
 * dlock.c - Solaris lslk lock functions
 *
 * Vic Abell
 * Purdue University Computing Center
 */


/*
 * Copyright 1996 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell.
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */
#ifndef lint
static char copyright[] =
"@(#) Copyright 1996 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dlock.c,v 1.15 2001/07/11 11:22:23 abe Exp $";
#endif


#include "lslk.h"
#include "kernelbase.h"


/*
 * Local definitions
 */

#if	solaris>=20501
#define	KVMHASHBN	8192		/* KVM hash bucket count -- MUST BE
					 * A POWER OF 2!!! */
#define	HASHKVM(va)	((int)((va * 31415) >> 3) & (KVMHASHBN-1))
					/* virtual address hash function */

# if	solaris<70000
#define	KAERR	(u_longlong_t)-1	/* kvm_physaddr() error return */
#define	KBUFT	char			/* kernel read buffer type */
#define	KPHYS	u_longlong_t		/* kernel physical address type */
#define	KVIRT	u_int			/* kernel virtual address type */
# else	/* solaris>=70000 */
#define	KAERR	(uint64_t)-1		/* kvm_physaddr() error return */
#define	KBUFT	void			/* kernel read buffer type */
#define	KPHYS	uint64_t		/* kernel physical address type */
#define	KVIRT	uintptr_t		/* kernel virtual address type */
# endif	/* solaris<70000 */
#endif	/* solaris>=20501 */


/*
 * Local structures
 */

#if	solaris>=20501
typedef struct kvmhash {
	KVIRT vpa;			/* virtual page address */
	KPHYS pa;			/* physical address */
	struct kvmhash *nxt;		/* next virtual address */
} kvmhash_t;
#endif	/* solaris>=20501 */


/*
 * Local variables
 */

#if	solaris>=20501
static struct as *Kas = (struct as *)NULL;
					/* pointer to kernel's address space
					 * map in kernel virtual memory */
static kvmhash_t **KVMhb = (kvmhash_t **)NULL;
					/* KVM hash buckets */
static int PageSz = 0;			/* page size */
static int PSMask = 0;			/* page size mask */
static int PSShft = 0;			/* page size shift */

# if	solaris<70000
static struct as Kam;			/* kernel's address space map */

static int Kmd = -1;			/* memory device file descriptor */
# endif	/* solaris<70000 */
#endif	/* solaris>=20501 */

#if	solaris>=20500
static KA_T Kb = (KA_T)NULL;		/* KERNELBASE for Solaris >= 2.5 */
#endif	/* solaris>=20500 */


/*
 * Local function prototypes
 */

#if	solaris>=20501
_PROTOTYPE(static void readkam,(KA_T addr));
#endif	/* solaris>=20501 */

_PROTOTYPE(static int readsysids,(void));
_PROTOTYPE(static void savelock,(struct inode *i, struct lock_descriptor *ld));


/*
 * gather_lock_info() -- gather lock information
 */

void
gather_lock_info()
{
	int hx;
	struct inode i, *ip;
	union ihead ih, *ihp;
	struct lock_descriptor ld;
	KA_T ldp;
	vnode_t *va;
/*
 * Loop through the inode hash buckets.
 */
	for (hx = 0, ihp = Ihdp; hx < Ihsz; hx++, ihp++) {
	    if (kread((KA_T)ihp, (char *)&ih, sizeof(ih))) {
		if (Owarn)
		    (void) fprintf(stderr, "%s: can't read ihead\n", Pn);
		return;
	    }
	/*
	 * Loop through the inodes of a hash bucket.
	 */
	    for (ip = ih.ih_chain[0];
	         ip && ip != (struct inode *)ihp;
	         ip = i.i_forw)
	    {
		if (kread((KA_T)ip, (char *)&i, sizeof(i))) {
		    if (Owarn)
			(void) fprintf(stderr,
			    "%s: premature exit from inode chain\n", Pn);
		    return;
		}
		if (!i.i_vnode.v_filocks || !i.i_vnode.v_count)
		    continue;
	    /*
	     * If the vnode, contained in the inode, has a non-NULL v_filocks
	     * pointer and a non-zero reference count, read and list the
	     * associated lock_descriptor structure chain.
	     *
	     * NB: this *seems* to work, mostly because we're looking
	     *     at inodes.  Under Solaris 2.3 (after patch 101318-45)
	     *     and all patch levels of 2.4, when the vnode refers to
	     *     an NFS file (an rnode), the v_filocks pointer addresses
	     *     a filock structure.
	     */
		va = (vnode_t *)((char *)ip + offsetof(struct inode, i_vnode));
		ldp = (KA_T)i.i_vnode.v_filocks;
		do {
		    if (kread((KA_T)ldp, (char *)&ld, sizeof(ld)))
			ldp = (KA_T)i.i_vnode.v_filocks;
		    else {
			ldp = (KA_T)ld.LOCK_NEXT;
			if ((ld.LOCK_FLAGS & ACTIVE_LOCK)
			&&  va == ld.LOCK_VNODE) {

	    		/*
	     		 * If the lock_descriptor references an active lock,
			 * save its information.
	     		 */
			    (void) savelock(&i, &ld);
			    if (is_lock_sel()) {
				NLockU++;
				Lp = (struct llock_info *)NULL;
			    }
			}
		    }
		} while (ldp && ldp != (KA_T)i.i_vnode.v_filocks);
	    }
	}
}


/*
 * get_cmdnm() -- get command name
 */

char *
get_cmdnm(lp)
	struct llock_info *lp;		/* local lock structure */
{
	struct proc *p;

/*
 * Get the proc structure for the lock info structure and return the
 * command name of its user structure.
 */
	if (lp->pid && (p = kvm_getproc(Kd, lp->pid)))
	    return(p->p_user.u_comm);
	return("(unknown)");
}


/*
 * get_kernel_access() -- get access to kernel information
 */

void
get_kernel_access()
{
	int err, i;

#if	solaris>=70000
	char *cp, isa[1024];
	short kbits = 32;

# if	defined(_LP64)
	short xkbits = 64;
# else	/* !defined(_LP64) */
	short xkbits = 32;
# endif	/* defined(_LP64) */

/*
 * Compare the Solaris 7 or higher lslk compilation bit size with the kernel
 * bit size.  Quit on a mismatch.
 */
	if (sysinfo(SI_ISALIST, isa, (long)sizeof(isa)) < 0) {
	    (void) fprintf(stderr, "%s: can't get ISA list: %s\n",
		Pn, strerror(errno));
	    Exit(1);
	}
	for (cp = isa; *cp;) {
	    if (strncmp(cp, "sparcv9", strlen("sparcv9")) == 0) {
		kbits = 64;
		break;
	    }
	    if (!(cp = strchr(cp, ' ')))
		break;
	    cp++;
	}
	if (kbits != xkbits) {
	    (void) fprintf(stderr,
		"%s: FATAL: lsof was compiled for a %d bit kernel,\n",
		Pn, (int)xkbits);
	    (void) fprintf(stderr,
		"      but this machine has booted a %d bit kernel.\n",
		(int)kbits);
	    Exit(1);
	}
#endif	/* solaris>=70000 */

/*
 * Convert kernel variable names to addresses and check the result.
 */
	if (Nmlst && !is_readable(Nmlst, 1))
	    Exit(1);
	if (nlist(Nmlst ? Nmlst : N_UNIX, Nl) < 0) {
	    (void) fprintf(stderr, "%s: nlist(%s) failed\n", Pn,
		Nmlst ? Nmlst : N_UNIX);
	    Exit(1);
	}
	for (err = i = 0; Nl[i].n_name[0]; i++) {
	  if (!Nl[i].n_value) {

#if	solaris>=20500
	    if (i == X_SYSIDS) {
	      if (Owarn) {
		(void) fprintf(stderr,
		  "%s: no system ID to IP address table found;\n", Pn);
		(void) fprintf(stderr,
		  "      Remote sources will be identified by system ID.\n");
	      }
	      continue;
	    }
#endif	/* solaris>=20500 */

	    (void) fprintf(stderr, "%s: no nlist address for %s\n",
		Pn, Nl[i].n_name);
	    err++;
	  }
	}
	if (err)
	    Exit(1);

#if	solaris>=20500
	Kb = (KA_T)Nl[X_KERNELBASE].n_value;
#endif	/* solaris>=20500 */

/*
 * Open access to kernel memory.
 */
	if (!(Kd = kvm_open(NULL, NULL, NULL, O_RDONLY, Pn)))
	    Exit(1);
/*
 * Drop setgid permission.
 */
	(void) dropgid();
/*
 * Read the inode hash table information.
 */
	if (kread((KA_T)Nl[X_IHEAD].n_value, (char *)&Ihdp, sizeof(Ihdp))) {
	    (void) fprintf(stderr, "%s: can't read *ihead\n", Pn);
	    Exit(1);
	}
	if (!Ihdp) {
	    (void) fprintf(stderr, "%s: *ihead is NUL\n", Pn);
	    Exit(1);
	}
	if (kread((KA_T)Nl[X_INOHSZ].n_value, (char *)&Ihsz, sizeof(Ihsz))) {
	    (void) fprintf(stderr, "%s: can't read inohsz\n", Pn);
	}

#if	solaris>=20501
/*
 * Get the kernel's virtual to physical map structure for Solaris 2.5.1 and
 * above.
 */
	if (Nl[X_KAS].n_value) {
	    PageSz = getpagesize();
	    PSMask = PageSz - 1;
	    for (i = 1, PSShft = 0; i < PageSz; i <<= 1, PSShft++)
		;
	    (void) readkam((KA_T)Nl[X_KAS].n_value);
	}
#endif	/* solaris>=20501 */

}


/*
 * initialize() -- initialize
 */

void
initialize()
{
	(void) get_kernel_access();

#if	solaris>=20500
/*
 * Read the Solaris 2.5 lock manager sysid information.
 */
	if (readsysids())
	    Exit(1);
#endif	/* solaris>=20500 */

}


/*
 * kread() -- read kernel memory
 */

int
kread(addr, buf, len)
	KA_T addr;			/* kernel address */
	char *buf;			/* local receiving buffer address */
	int len;			/* length to read */
{
	int br;
/*
 * Because lslk reads kernel data and follows pointers found there at a
 * rate considerably slower than the kernel, lslk sometimes acquires
 * invalid pointers.  If the invalid pointers are fed to kvm_[k]read(),
 * a segmentation violation may result, so legal kernel addresses are
 * limited by the value of the KERNELBASE symbol (Kb value from the
 * kernel's _kernelbase variable for Solaris 2.5 and above).
 */

#if	solaris>=20500
#define	KVMREAD	kvm_kread
# if	defined(sparc)
	if (addr < Kb)
# endif	/* defined(sparc) */
#else	/* solaris<20500 */
#define	KVMREAD kvm_read
# if	defined(sparc)
	if (addr < (KA_T)KERNELBASE)
# endif	/* defined(sparc) */
#endif	/* solaris>=20500 */

#if	defined(sparc)
	    return(1);
#endif	/* defined(sparc) */

#if	solaris>=20501

/*
 * Make sure the virtual address represents real physical memory by testing
 * it with kvm_physaddr().
 *
 * For Solaris below 7 read the kernel data with llseek() and read().  For
 * Solaris 7 and above use kvm_pread().
 */
	if (Kas) {

# if	solaris>20501
	    register int b2r;
	    register char *bp;
# endif	/* solaris>20501 */

	    register int h, ip, tb;
	    register kvmhash_t *kp;
	    KPHYS pa;
	    register KVIRT va, vpa;

# if	solaris<20600
	    for (tb = 0, va = (KVIRT)addr;
		 tb < len;
		 tb += br, va += (KVIRT)br)
# else	/* solaris>=20600 */
	    for (bp = buf, tb = 0, va = (KVIRT)addr;
		 tb < len;
		 bp += br, tb += br, va += (KVIRT)br)
# endif	/* solaris<20600 */

	    {
		vpa = (va & (KVIRT)~PSMask) >> PSShft;
		ip = (int)(va & (KVIRT)PSMask);
		h = HASHKVM(vpa);
		for (kp = KVMhb[h]; kp; kp = kp->nxt) {
		    if (kp->vpa == vpa) {
			pa = kp->pa;
			break;
		    }
		}
		if (!kp) {
		    if ((pa = kvm_physaddr(Kd, Kas, va)) == KAERR)
			return(1);
		    if (!(kp = (kvmhash_t *)malloc(sizeof(kvmhash_t)))) {
			(void) fprintf(stderr, "%s: no kvmhash_t space\n", Pn);
			Exit(1);
		    }
		    kp->nxt = KVMhb[h];
		    pa = kp->pa = (pa & ~(KPHYS)PSMask);
		    kp->vpa = vpa;
		    KVMhb[h] = kp;
		}

# if	solaris<20600
		br = (int)(len - tb);
		if ((ip + br) > PageSz)
		    br = PageSz - ip;
# else	/* solaris>=20600 */
		b2r = (int)(len - tb);
		if ((ip + b2r) > PageSz)
		    b2r = PageSz - ip;
		pa |= (KPHYS)ip;

#  if	solaris<70000
		if (llseek(Kmd, (offset_t)pa, SEEK_SET) == (offset_t)-1)
		    return(1);
		if ((br = (int)read(Kmd, (void *)bp, (size_t)b2r)) <= 0)
		    return(1);
#  else	/* solaris>=70000 */
		if ((br = kvm_pread(Kd, pa, (void *)bp, (size_t)b2r)) <= 0)
		    return(1);
#  endif	/* solaris<70000 */
# endif	/* solaris<20600 */

	    }

# if	solaris>=20600
	    return(0);
# endif	/* solaris>=20600 */

	}
#endif	/* solaris>=20501 */

/*
 * Use kvm_read for Solaris < 2.5; use kvm_kread() for Solaris >= 2.5.
 */
	br = KVMREAD(Kd, (u_long)addr, buf, len);
	return((br == len) ? 0 : 1);
}


/*
 * print_dev() -- print device number
 */

char *
print_dev(lp)
	struct llock_info *lp;		/* local lock structure */
{
	static char buf[128];

	(void) sprintf(buf, "%d,%d", (lp->dev >> L_BITSMINOR) & 0x3fff,
	    (lp->dev & L_MAXMIN));
	return(buf);
}


#if	solaris>=20501
/*
 * readkam() - read kernel's address map structure
 */

static void
readkam(addr)
	KA_T addr;			/* kernel virtual address */
{
	register int i;
	size_t sz;

#if	solaris<70000
	if (addr && !kread(addr, (char *)&Kam, sizeof(Kam)))
	    Kas = (KA_T)&Kam;
	else {
	    Kas = (struct as *)NULL;
	    return;
	}
#else	/* solaris>=70000 */
	Kas = (struct as *)addr;
#endif	/* solaris<70000 */

	if (!(KVMhb = (kvmhash_t **)calloc(KVMHASHBN, sizeof(kvmhash_t *)))) {
	    (void) fprintf(stderr,
		"%s: no space (%d) for KVM hash buckets\n",
		Pn, (KVMHASHBN * sizeof(kvmhash_t *)));
	    Exit(1);
	}
}
#endif	/* solaris>=20501 */


#if	solaris>=20500
/*
 * readsysids() -- read the lock manager's sysid information
 */

static int
readsysids()
{
	char buf[MAXPATHLEN], *cp, *hn, nm[MAXPATHLEN];
	int err;
	struct in_addr *ip;
	KA_T ka;
	MALLOC_S len;
	struct lhaddr *lh;
	int na = 0;
	struct lm_sysid s;
/*
 * Read the lm_sysids pointer from the kernel.  If none is available,
 * make sure the Sids table is empty.
 */
	if (!Nl[X_SYSIDS].n_value) {
	   Nsid = 0;
	   Sids = (struct SysIDs *)NULL;
	   return(0);
	}
	if (kread((KA_T)Nl[X_SYSIDS].n_value, (char *)&ka, sizeof(ka))) {
	    (void) fprintf(stderr, "%s: can't read %s\n", Pn,
		Nl[X_SYSIDS].n_name);
	    return(1);
	}
/*
 * Read the lm_sysid entries linked from the lm_sysids pointer.
 */
	for (; ka; ka = (KA_T)s.next) {
	    if (kread(ka, (char *)&s, sizeof(s)))
		break;
	/*
	 * Read the address buffer and see if it contains a host name.
	 */
	    if (s.addr.len < sizeof(buf) && s.addr.buf
	    &&  !kread((KA_T)s.addr.buf, buf, s.addr.len))
	    {

	    /*
	     * Truncate trailing blanks and a terminal dot from the address
	     * buffer.  See if the buffer contains printable characters that
	     * form a host name.
	     */
		buf[s.addr.len] = '\0';
		for (cp = &buf[s.addr.len] - 1; cp >= buf; cp--) {
		    if (!isprint(*cp)) {
			cp = (char *)NULL;
			break;
		    }
		    if (*cp != ' ' && *cp != '.')
			break;
		    *cp = '\0';
		}
		if (cp) {
		    if ((lh = get_haddr(0, buf, (unsigned long)0, &err))
		    && !err)
			cp = lh->hn;
		    else
			cp = (char *)NULL;
		}
	    } else
		cp = (char *)NULL;
	    if (!cp) {

	    /*
	     * If the address buffer didn't yield a host name, read the name
	     * buffer.  If it is non-empty and yields a host name, use it.
	     */
		if (!kread((KA_T)s.name, nm, sizeof(nm))) {
		    nm[sizeof(nm) - 1] = '\0';
		    if ((lh = get_haddr(0, nm, (unsigned long)0, &err))
		    && !err)
			cp = lh->hn;
		}
	    }
	    if (!cp)

	    /*
	     * If neither the address nor name buffer yielded a host name,
	     * skip this lm_sysid entry.
	     */
		continue;
	/*
	 * See if there's room for another local sysid entry.
	 */
	    if (Nsid >= na) {

	    /*
	     * Increase the size of the local sysid table and allocate space
	     * for it.
	     */
		na += 10;
		len = (MALLOC_S)(na * sizeof(struct SysIDs));
		if (!Sids)
		    Sids = (struct SysIDs *)malloc(len);
		else
		    Sids = (struct SysIDs *)realloc(Sids, len);
		if (!Sids) {
		    (void) fprintf(stderr,
			"%s: can't allocate %d SysIDs structures\n", Pn, na);
		    return(1);
		}
	    }
	/*
	 * Allocate space for the address name.
	 */
	    if (!(hn = (char *)malloc(strlen(cp)+1))) {
		(void) fprintf(stderr,
		    "%s: can't allocate space for ID %ld name: %s\n",
		    Pn, s.sysid, cp);
		return(1);
	    }
	    (void) strcpy(hn, cp);
	/*
	 * Create the local sysid entry.
	 */
	    Sids[Nsid].id = s.sysid;
	    Sids[Nsid].na = lh->na;
	    Sids[Nsid++].hn = hn;
	}
	return(0);
}
#endif	/* solaris>=20500 */


/*
 * savelock() -- save lock information
 */

static void
savelock(ip, lp)
	struct inode *ip;		/* inode structure pointer */
	struct lock_descriptor *lp;	/* lock_descriptor structure pointer */
{
	char *cp, *hn, *mn;
	struct hostent *he;
	int i, sel, src;
/*
 * Allocate a local lock structure.
 */
	(void) alloc_llock();
/*
 * Save: inode number; device; size; and process ID.
 */
	Lp->dev = ip->i_dev;
	Lp->inum = ip->i_number;
	Lp->pid = (unsigned long)lp->LOCK_PID;
	Lp->sz = (unsigned long)ip->i_size;
	Lp->szs = 1;
/*
 * Save the lock description: mandatory status; type; start; whence and
 * length or end.
 */
	Lp->mand = MANDLOCK(&ip->i_vnode, ip->i_mode) ? 1 : 0;
	switch (lp->LOCK_TYPE) {
	case F_RDLCK:
	    Lp->type = 1;
	    break;
	case F_WRLCK:
	    Lp->type = 2;
	    break;
	case (F_RDLCK|F_WRLCK):
	    Lp->type = 3;
	    break;
	default:
	    Lp->type = 0;
	}
	Lp->ss = 1;
	Lp->start = (unsigned long)lp->LOCK_START;

#if	defined(LOCK_WHENCE)
	Lp->ls = Lp->ws = 1;
	Lp->len = (unsigned long)lp->LOCK_LEN;
	Lp->whence = (unsigned long)lp->LOCK_WHENCE;
#else	/* !defined(LOCK_WHENCE) */
	Lp->es = 1;
	Lp->end = (unsigned long)lp->LOCK_END;
#endif	/* defined(LOCK_WHENCE) */

#if	solaris<20500
/*
 * If there's a sysid, indicating a remote lock, for Solaris 2.[34], save it.
 */
	if (lp->LOCK_SYSID) {
	    Lp->src = 1;
	    Lp->hn = (char *)NULL;
	    Lp->iap = alloc_in_addr();
	    Lp->iap->s_addr = (unsigned long)lp->LOCK_SYSID;
	}
#else	/* solaris>=20500 */
/*
 * If there's a sysid, indicating a remote lock, for Solaris 2.5 see if
 * there's an entry in the local sysid information table whose sysid
 * matches.  If there is, save the associated host name and address number,
 * otherwise set an "unknown-remote" source type and save the sysid as
 * the address number.
 */
	if (lp->LOCK_SYSID) {
	    Lp->iap = alloc_in_addr();
	    for (i = 0; i < Nsid; i++) {
		if (Sids[i].id == lp->LOCK_SYSID)
		    break;
	    }
	    if (i < Nsid) {
	        Lp->src = 1;
		Lp->hn = Sids[i].hn;
	        Lp->iap->s_addr = (unsigned long)Sids[i].na;
	    } else {
		Lp->src = 2;
		Lp->hn = (char *)NULL;
		Lp->iap->s_addr = lp->LOCK_SYSID;
	    }
	}
#endif	/* solaris<20500 */

	else
	    Lp->src = 0;
}
