/*
 * misc.c -- miscellaneous lslk functions
 *
 * V. Abell
 * Purdue University Computing Center
 */


/*
 * Copyright 1996 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell.
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */
#ifndef lint
static char copyright[] =
"@(#) Copyright 1996 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: misc.c,v 1.7 99/11/10 14:56:16 abe Exp $";
#endif


#include "lslk.h"


/*
 * alloc_in_addr() -- allocate in_addr structure space
 */

struct in_addr *
alloc_in_addr()
{
	struct in_addr *iap;

	if ((iap = (struct in_addr *)malloc(sizeof(struct in_addr))) == NULL) {
	    (void) fprintf(stderr, "%s: no space for in_addr structure\n", Pn);
	    Exit(1);
	}
	iap->s_addr = 0l;
	return(iap);
}


/*
 * alloc_llock() -- allocate local lock structure.
 */

void
alloc_llock()
{
	MALLOC_S len;

	if (!Lp) {

	/*
	 * If no current llock_info structure exists, create one.
	 */
	    if ((NLockU + 1) > NLockA) {
		NLockA += LOCKAU;
		len = (MALLOC_S)(NLockA * sizeof(struct llock_info));
		if (Lock) 
		    Lock = (struct llock_info *)realloc((MALLOC_P *)Lock, len);
		else
		    Lock = (struct llock_info *)malloc(len);
		if (!Lock) {
		    (void) fprintf(stderr, "%s: no space for local lock info\n",
			Pn);
		    Exit(1);
		}
	    }
	    Lp = &Lock[NLockU];
	} else {

	/*
	 * Release any previously malloc'd space this llock_info structure
	 * might have held.
	 */
	    if (Lp->iap)
		(void) free((MALLOC_P *)Lp->iap);
	    if (Lp->cmd && Lp->cs)
		(void) free((MALLOC_P *)Lp->cmd);
	}
/*
 * Initialize the local lock structure.
 */
	Lp->cmd = (char *)NULL;
	Lp->cs = 1;
	Lp->dev = (dev_t)0;
	Lp->end = Lp->len = Lp->start = Lp->sz = Lp->whence = 0L;
	Lp->es = Lp->ls = Lp->mand = Lp->src = Lp->ss = Lp->szs = Lp->ws = 0;
	Lp->hn = Lp->path = (char *)NULL;
	Lp->iap = (struct in_addr *)NULL;
	Lp->inum = (ino_t)0;
	Lp->pid = (unsigned long)0;
	Lp->sf = Lp->type = 0;
}

#if	defined(WILLDROPGID)


/*
 * dropgid() -- drop setgid permission
 */

void
dropgid()
{
	if (!Setuidroot && Setgid) {
	    if (setgid(Mygid) < 0) {
		(void) fprintf(stderr, "%s: can't setgid(%d): %s\n",
		    Pn, Mygid, strerror(errno));
		Exit(1);
	    }
	    Setgid = 0;
	}
}
#endif	/* defined(WILLDROPGID) */


/*
 * get_haddr() -- get host address
 */

struct lhaddr *
get_haddr(ty, hn, a, err)
	int ty;				/* request type: 0 = convert name
					 *		 1 = convert address */
	char *hn;			/* host name */
	unsigned long a;		/* address */
	int *err;			/* error reply -- non zero if lookup
					 * failed */
{
	static int aLhaddr = 0;
	struct hostent *he;
	int i;
	struct in_addr ia;
	MALLOC_S len;
	struct lhaddr *lh;
	unsigned long na;
	static int nLhaddr = 0;
	char *nhn;

	*err = 0;
/*
 * Search for a match in the cache.
 */
	for (i = 0, lh = Lhaddr; i < nLhaddr; i++, lh++) {
	    if (ty == 0) {
		if (strcasecmp(hn, lh->hn) == 0)
		return(lh);
	    } else if (lh->na == a)
		return(lh);
	}
/*
 * Make room for a new cache entry.
 */
	if (nLhaddr >= aLhaddr) {
	    aLhaddr += LHAAU;
	    len = (MALLOC_S)(aLhaddr * sizeof(struct lhaddr));
	    if (Lhaddr)
		Lhaddr = (struct lhaddr *)realloc(Lhaddr, len);
	    else
		Lhaddr = (struct lhaddr *)malloc(len);
	    if (!Lhaddr) {
		(void) fprintf(stderr,
		    "%s: no space for local host address cache\n", Pn);
		Exit(1);
	    }
	}
	lh = &Lhaddr[nLhaddr++];
	lh->hn = (char *)NULL;
	lh->na = (unsigned long)0;
/*
 * Fill in new cache entry.  Do name and address lookups as required.
 */
	if (ty == 0) {
	    nhn = hn;
	    if ((he = gethostbyname(hn))) {
		(void) lmemcpy((LMEMCPY_P *)&na, (LMEMCPY_P *)he->h_addr,
		    (MALLOC_S)sizeof(na));
		nhn = (char *)he->h_name;
	    } else
		*err = 1;
	} else {
	    na = (unsigned long)a;
	    if ((he = gethostbyaddr((char *)&na, sizeof(na), AF_INET)))
		nhn = (char *)he->h_name;
	    else {
		*err = 1;
		ia.s_addr = na;
		nhn = inet_ntoa(ia);
	    }
	}
/*
 * Allocate space for host name and copy it there.
 */
	if ((lh->hn = (char *)malloc(strlen(nhn)+1)) == NULL) {
	    (void) fprintf(stderr,
		"%s: no space for cached host name %s\n", Pn, nhn);
	    Exit(1);
	}
	(void) strcpy(lh->hn, nhn);
	lh->na = na;
	return(lh);
}


/*
 * is_lock_sel() -- is lock selected?
 */

int
is_lock_sel()
{
	int i;

	Lp->sf = 0;
	    if (Selopt == 0)
	return(1);
/*
 * Check for PID match.
 */
	if (Pid) {
	    for (i = 0; i < NPid; i++) {
		if (Pid[i] == Lp->pid) {
		    Lp->sf |= SELPID;
		    break;
		}
	    }
	}
/*
 * Check for file match.
 */
	if (Lfile) {
	    for (i = 0; i < NLfile; i++) {
		if (Lfile[i].dev == Lp->dev && Lfile[i].inum == Lp->inum) {
		    Lp->sf |= SELFILE;
		    Lp->path = Lfile[i].path;
		    break;
		}
	    }
	}
/*
 * Check for network match.
 */
	if (Nwad && Lp->src == 1 && (Lp->iap || Lp->hn)) {
	    for (i = 0; i < NNwad; i++) {
		if (Lp->iap && Nwad[i].na == (unsigned long)Lp->iap->s_addr) {
		    Lp->sf |= SELNWAD;
		    break;
		}
		if (!Lp->iap && Lp->hn && (Nwad[i].chn || Nwad[i].hn)) {
		    if ((Nwad[i].chn && strcmp(Nwad[i].chn, Lp->hn) == 0)
		    ||  (Nwad[i].hn && strcmp(Nwad[i].hn, Lp->hn) == 0)) {
			Lp->sf |= SELNWAD;
			break;
		    }
		}
	    }
	}
/*
 * Check the selection results.
 */
	if (Oand) {
	    if ((Lp->sf & Selopt) == Selopt)
		return(1);
	} else if (Lp->sf & Selopt)
	    return(1);
	return(0);
}


/*
 * is_readable() -- is file readable
 */

int
is_readable(path, msg)
	char *path;			/* file path */
	int msg;			/* issue warning message if 1 */
{
	if (access(path, R_OK) < 0) {
	    if (Owarn && msg == 1)
		(void) fprintf(stderr, ACCESSERRFMT, Pn, path, strerror(errno));
	    return(0);
	}
	return(1);
}


/*
 * print_lock_info() -- print lock info
 */

void
print_lock_info(sp)
	struct llock_info **sp;		/* sorted lock info pointers */
{
	char buf[128], *cp, *src;
	int devl = strlen(DEVTTL);
	int endl = strlen(ENDTTL);
	int err, i, j, k, l;
	int inodel = strlen(INODETTL);
	int lenl = strlen(LENTTL);
	struct lhaddr *lh;
	struct llock_info *lp;
	int mandl = strlen(MANDTTL);
	int pidl = strlen(PIDTTL);
	int srcl = strlen(SOURCETTL);
	int stl = strlen(STARTTTL);
	int szl = strlen(SIZETTL);
	int typel = strlen(TYPETTL);
	int whl = strlen(WHENCETTL);
/*
 * Make two passes through the lock information.
 * Calculate the maximum length for each output column during the first pass.
 * Print the lock information during the second pass.
 */
	for (i = 0; i < 2; i++) {
	    for (j = 0; j < NLockU; j++) {
		if (i && j == 0) {

		/*
		 * This is the first line to be printed during the second
		 * pass, so print a header line.
		 */
		    (void) printf("%-*.*s %*s %*s %*s %*s %*s %*s %*s %*s %*s",
			srcl, srcl, SOURCETTL,
			pidl, PIDTTL,
			devl, DEVTTL,
			inodel, INODETTL,
			szl, SIZETTL,
			typel, TYPETTL,
			mandl, MANDTTL,
			stl, STARTTTL,
			whl, WHENCETTL,
			endl, ENDTTL);
		    (void) printf(" %*s NAME\n", lenl, LENTTL);
		}
		lp = sp[j];
	    /*
	     * Determine the source column value: command name, host name,
	     * or network number.
	     */
		src = (char *)NULL;
		if (lp->src == 0)
		    src = get_cmdnm(lp);
		else if (lp->src == 1) {
		    if (!lp->hn && Oconv && lp->iap && lp->iap->s_addr) {
			lh = get_haddr(1, (char *)NULL,
				       (unsigned long)lp->iap->s_addr, &err);
			lp->hn = lh->hn;
		    }
		    if (lp->hn && (Oconv || !lp->iap || !lp->iap->s_addr))
			src = lp->hn;
		    else if (lp->iap && lp->iap->s_addr)
			src = inet_ntoa(*lp->iap);
		}
		if (!src)
		    src = "(unknown)";
		if (i == 0) {

		/*
		 * This is the first pass, so check for new maximum lengths
		 * for the device, inode number, PID, source, and file size
		 * columns.
		 */
		    l = strlen(print_dev(lp));
		    if (l > devl)
			devl = l;
		    (void) sprintf(buf, "%ld", lp->inum);
		    l = strlen(buf);
		    if (l > inodel)
			inodel = l;
		    (void) sprintf(buf, "%ld", lp->pid);
		    l = strlen(buf);
		    if (l > pidl)
			pidl = l;
		    l = strlen(src);
		    if (l > srcl)
			srcl = l;
		    if (lp->szs) {
			(void) sprintf(buf, "%lu", lp->sz);
			l = strlen(buf);
		    } else
			l = 1;
		    if (l > szl)
			szl = l;
		} else {

		/*
		 * The is the second pass, so print the source, PID, device,
		 * inode number, and size values.
		 */
		    (void) printf("%-*.*s %*d %*s %*ld ",
			srcl, srcl, src,
			pidl, lp->pid,
			devl, print_dev(lp),
			inodel, lp->inum);
		    if (lp->szs)
			(void) printf("%*lu", szl, lp->sz);
		    else
			(void) printf("%-*.*s", szl, szl, " ");
		}
	    /*
	     * Determine the lock type.
	     */
		switch (lp->type) {
		case 0:
		    cp = "";
		    break;
		case 1:
		    cp = "r";
		    break;
		case 2:
		    cp = "w";
		    break;
		case 3:
		    cp = "rw";
		    break;
		default:
		    cp = "?";
		}
		if (i == 0) {

		/*
		 * This is the first pass, so check for new maximum lengths
		 * for the type, mandatory lock status, relative offset,
		 * starting offset (whence), end, and length columns.
		 */
		    l = strlen(cp);
		    if (l > typel)
			typel = l;
		    (void) sprintf(buf, "%d", lp->mand);
		    l = strlen(buf);
		    if (l > mandl)
			mandl = l;
		    if (lp->ss) {
			(void) sprintf(buf, "%lu", lp->start);
			l = strlen(buf);
			if (l > stl)
			    stl = l;
		    }
		    if (lp->ws) {
			(void) sprintf(buf, "%lu", lp->whence);
			l = strlen(buf);
			if (l > whl)
				whl = l;
		    }
		    if (lp->es) {
			(void) sprintf(buf, "%lu", lp->end);
			l = strlen(buf);
			if (l > endl)
			    endl = l;
		    }
		    if (lp->ls) {
			(void) sprintf(buf, "%lu", lp->len);
			l = strlen(buf);
			if (l > lenl)
			    lenl = l;
		    }
		} else {

		/*
		 * This is the second pass, so print the type, mandatory
		 * lock status, relative offset, starting offset (whence),
		 * end, and length values.
		 */
		    (void) printf(" %*s %*d %*lu %*lu %*lu %*lu",
			typel, cp,
			mandl, lp->mand,
			stl, lp->ss ? lp->start : 0l,
			whl, lp->ws ? lp->whence : 0l,
			endl, lp->es ? lp->end : 0l,
			lenl, lp->ls ? lp->len : 0l);
		}
		if (i) {

		/*
		 * This is the second pass, so print the path name, or the
		 * mount point and device paths.
		 */
		    if (lp->path)
			(void) printf(" %s\n", lp->path);
		    else {
			for (k = 0; k < NMnt; k++) {
			    if (Mnt[k].dev == lp->dev)
				break;
			}
			if (k < NMnt)
			    (void) printf(" %s (%s)\n", Mnt[k].mntp, Mnt[k].sp);
			else
			    putchar('\n');
		    }
		}
	    }
	}
}
