/*****************************************************************************
 *                                                                           *
 * Widget:    gtkimrot                                                       *
 * Uses:      GTK+1.2, GdkImlib                                              *
 * Purpose:   Rotate images.                                                 *
 * Author:    Andreas Tille <tille@debian.org>                               *
 * Date:      February 27, 2000                                              *
 * Copyright: Andreas Tille, 1998, 1999, 2000                                *
 * License:   LGPL                                                           *
 * Algorithm: based on the paper "A Fast Algorithm for General Raster        *
 *            Rotatation" by Alan W. Paeth, Graphics Interface '86           *
 *            (Vancouver) and on a method written by Michael Halle           *
 *            of the Spatial Imaging Group, MIT Media Lab.                   *
 *            I used the ImageMagic 4.2.8 implementation of these as an      *
 *            example.                                                       *
 *                                                                           *
 *****************************************************************************/

#include <math.h>
#include <string.h>
#include <gdk_imlib.h>

#include "gtk/gtkbutton.h"
#include "gtk/gtkcontainer.h"
#include "gtk/gtkdialog.h"
#include "gtk/gtkdrawingarea.h"
#include "gtk/gtkeventbox.h"
#include "gtk/gtkframe.h"
#include "gtk/gtkhbbox.h"
#include "gtk/gtkhbox.h"
#include "gtk/gtklabel.h"
#include "gtk/gtksignal.h"
#include "gtk/gtkspinbutton.h"
#include "gtk/gtktable.h"
#include "gtk/gtkvbox.h"
#include "gtkimrot.h"

#include "gtkintl.h"

static void gtk_imrot_class_init     (GtkImRotClass *class);
static void gtk_imrot_init           (GtkImRot *imrot);
static void gtk_imrot_destroy        (GtkObject *object);

static void gtk_imrot_configure      (GtkWidget *widget, GdkEventConfigure *event, GtkImRot *imrot);
static void gtk_imrot_expose         (GtkWidget *widget, GdkEventExpose *event, GtkImRot *imrot);
static void gtk_imrot_value_changed  (GtkAdjustment *adj, GtkWidget *spinner);

/* internal helpers   */
static GtkAdjustment  *_create_adjustment  (GtkWidget *table, GtkImRot *imrot, gchar *name, 
                                            gint pos, gdouble *value);
static gint            _set_parameter_if_valid (gdouble *val2set, gdouble val, gint cond, 
						GtkAdjustment *adj);
static void            _draw_rot(GtkWidget *widget, GtkImRot *imrot);

static GtkWindowClass *parent_class = NULL;

GtkType
gtk_imrot_get_type (void)
{
  static GtkType IMROT_type = 0;

  if (!IMROT_type)
    {
      GtkTypeInfo imrot_info =
      {
	"GtkImRot",
	sizeof (GtkImRot),
	sizeof (GtkImRotClass),
	(GtkClassInitFunc) gtk_imrot_class_init,
	(GtkObjectInitFunc) gtk_imrot_init,
	/* reserved_1 */ NULL,
	/* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      IMROT_type = gtk_type_unique (GTK_TYPE_WINDOW, &imrot_info);
    }

  return IMROT_type;
}

static void
gtk_imrot_class_init (GtkImRotClass *class)
{
  GtkObjectClass *object_class;

  object_class = (GtkObjectClass*) class;

  parent_class = gtk_type_class (GTK_TYPE_DIALOG);

  object_class->destroy = gtk_imrot_destroy;
}

static void
gtk_imrot_init (GtkImRot *imrot)
{
  GtkWidget *confirm_area;
  /*  GtkWidget *event_box; */
  GtkWidget *hbox;
  GtkWidget *viewport;

  /* The dialog-sized vertical box  */
  imrot->main_vbox = gtk_vbox_new (FALSE, 10);
  gtk_container_set_border_width (GTK_CONTAINER (imrot), 10);
  gtk_container_add (GTK_CONTAINER (imrot), imrot->main_vbox);
  gtk_widget_show (imrot->main_vbox);

  /**********************************************************************************
   * The image to rotate                                                            *
   **********************************************************************************/

  viewport = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME(viewport), GTK_SHADOW_IN);
  gtk_box_pack_start (GTK_BOX(imrot->main_vbox), viewport, FALSE, FALSE, 0);
                                /* do not allow resizing! ^^^^^  ^^^^^  */
                                /* ... hmmm, we have to resize the viewport!!! */
  gtk_widget_show (viewport);
  /*  gtk_object_set_user_data (GTK_OBJECT(imrot->main_vbox), viewport); *
   *  Do I have to create a new viewport or hbox for resized images??    */

  hbox = gtk_hbox_new (FALSE, 0);  /* this seems to be neccessary to avoid resizing of drawing area */
  gtk_container_add (GTK_CONTAINER(viewport), hbox);
  gtk_widget_show (hbox);
  /*  gtk_object_set_user_data (GTK_OBJECT(viewport), hbox);             *
   *  Do I have to create a new viewport or hbox for resized images??    */

  imrot->draw = gtk_drawing_area_new();
  gtk_box_pack_start(GTK_BOX(hbox), imrot->draw, FALSE, FALSE, 0);
  gtk_widget_show (imrot->draw);

  /**********************************************************************************
   * all the signals and events of the pixmap area                                  *
   **********************************************************************************/
  gtk_widget_set_events(imrot->draw, GDK_EXPOSURE_MASK | GDK_LEAVE_NOTIFY_MASK | GDK_BUTTON_PRESS_MASK 
			 | GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK | GDK_ENTER_NOTIFY_MASK );
  /* Signals used to handle backing pixmap */
  gtk_signal_connect(GTK_OBJECT(imrot->draw), "expose_event", GTK_SIGNAL_FUNC(gtk_imrot_expose), imrot);
  gtk_signal_connect(GTK_OBJECT(imrot->draw), "configure_event", GTK_SIGNAL_FUNC(gtk_imrot_configure), imrot);

/*
  event_box = gtk_event_box_new ();
printf("%s(%i)\n", __FILE__, __LINE__);
  gtk_container_add ( GTK_CONTAINER(event_box), imrot->draw );
*/
  /**********************************************************************************
   * area for several parameters in the control part                                *
   **********************************************************************************/

  hbox = gtk_hbox_new(FALSE, 1);
  gtk_box_pack_start(GTK_BOX(imrot->main_vbox), hbox, TRUE, TRUE, 0);

  imrot->parms = gtk_hbox_new(FALSE, 1);
  gtk_box_pack_start(GTK_BOX(hbox), imrot->parms, TRUE, FALSE, 0);

  gtk_widget_show_all (hbox);
 
  /**********************************************************************************
   * The OK/Apply/Cancel button area                                                *
   **********************************************************************************/
   
  confirm_area = gtk_hbutton_box_new ();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(confirm_area), GTK_BUTTONBOX_END);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(confirm_area), 5);
  gtk_box_pack_end (GTK_BOX (imrot->main_vbox), confirm_area, FALSE, FALSE, 0);
  gtk_widget_show (confirm_area);

  /****************************************************************************
   * The OK button                                                            *
   ****************************************************************************/
  imrot->ok_button = gtk_button_new_with_label (_("OK"));
  GTK_WIDGET_SET_FLAGS (imrot->ok_button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (confirm_area), imrot->ok_button, TRUE, TRUE, 0);
  gtk_widget_grab_default (imrot->ok_button);
  gtk_widget_show (imrot->ok_button);

  /****************************************************************************
   * The Apply button                                                         *
   ****************************************************************************/
  imrot->apply_button = gtk_button_new_with_label (_("Apply"));
  GTK_WIDGET_SET_FLAGS (imrot->apply_button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (confirm_area), imrot->apply_button, TRUE, TRUE, 0);
  gtk_widget_show (imrot->apply_button);
  gtk_signal_connect(GTK_OBJECT(imrot->apply_button), "clicked", 
                     GTK_SIGNAL_FUNC(_draw_rot), imrot);

  /**********************************************************************************
   * The Cancel button                                                              *
   **********************************************************************************/
  imrot->cancel_button = gtk_button_new_with_label (_("Cancel"));
  GTK_WIDGET_SET_FLAGS (imrot->cancel_button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (confirm_area), imrot->cancel_button, TRUE, TRUE, 0);
  gtk_widget_show (imrot->cancel_button);

  /**********************************************************************************
   * Initialize structure value                                                     *
   **********************************************************************************/
  imrot->angle   = NULL;
  imrot->im      = NULL;
}

GtkWidget*
gtk_imrot_new (const gchar *title, GdkImlibImage *im, gdouble alpha)
/* --- Parameters: ---
 * const gchar   *title   : title of widget window
 * GtkImlibImage *im      : image to rotate
 * gdouble         alpha   : rotating angle
 */
{
  GtkImRot    *imrot;
  GtkWidget   *table;

  g_return_val_if_fail ( im != NULL, NULL );
  g_return_val_if_fail ( im->rgb_width > 0, NULL );
  g_return_val_if_fail ( im->rgb_height > 0, NULL );
  g_return_val_if_fail ( im->rgb_data , NULL ) ;

  imrot = gtk_type_new (GTK_TYPE_IMROT);
  g_return_val_if_fail ( imrot , NULL );

  if ( title ) gtk_window_set_title (GTK_WINDOW (imrot), title);
  else {
    gchar *ptr;
    
    ptr = g_strdup_printf (_("Rotation of %s"), im->filename ? im->filename : "???" );
    gtk_window_set_title (GTK_WINDOW (imrot), ptr);
    g_free (ptr) ;
  }

  gtk_drawing_area_size(GTK_DRAWING_AREA(imrot->draw), im->rgb_width, im->rgb_height);

  /**********************************************************************************
   * editables for horizontal and vertical movement                                 *
   **********************************************************************************/
  table = gtk_table_new (2, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_box_pack_start(GTK_BOX(imrot->parms), table, TRUE, FALSE, 0);
  
  imrot->angle = _create_adjustment(table, imrot, _("angle:"), 0, &(imrot->alpha));
  gtk_widget_show(table);
  imrot->im       = im;
  imrot->o_width  = im->rgb_width;
  imrot->o_height = im->rgb_height;
  imrot->o_data   = g_new (guchar, 3*imrot->o_width*imrot->o_height);
  memcpy(imrot->o_data, imrot->im->rgb_data, 3*imrot->o_width*imrot->o_height);
  
  return GTK_WIDGET (imrot);
}

gdouble
gtk_imrot_get_alpha (GtkImRot *imrot)
/* Get rotating angle */
{
  g_return_val_if_fail ( imrot, -1.0 );
  g_return_val_if_fail ( GTK_IS_IMROT (imrot), -1.0 );
 
  return imrot->alpha;
}

/**********************************************************************
 *	   internal functions                                         *
 **********************************************************************/

static void
gtk_imrot_destroy (GtkObject *object)
{
  GtkImRot *imrot;

  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_IMROT (object));

  imrot = GTK_IMROT (object);

  if ( imrot->pm ) gdk_imlib_free_pixmap(imrot->pm);
  imrot->pm = NULL;

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
gtk_imrot_configure(GtkWidget *widget, GdkEventConfigure *event, GtkImRot *imrot)
/* Create a new backing pixmap of the appropriate size 
 * --- Parameter: ---
 * GtkWidget         *widget : drawing area with rotating
 * GdkEventConfigure *event  : configure_event
 * GtkImRot          *imrot  : image rotate structure
 */
{
  gint w, h;

  g_return_if_fail ( widget );
  g_return_if_fail ( imrot );
  g_return_if_fail ( GTK_IS_DRAWING_AREA (widget) );
  g_return_if_fail ( GTK_IS_IMROT (imrot) );
  g_return_if_fail ( imrot->im );
  if ( event ) g_return_if_fail ( event->type == GDK_CONFIGURE );

  if ( !imrot->draw || !imrot->draw->window ) return ;

  gdk_window_get_size (imrot->draw->parent->window, &w, &h) ;
  if ( w != imrot->im->rgb_width || h != imrot->im->rgb_height ) {
    gdk_window_resize(imrot->draw->parent->window, imrot->im->rgb_width, imrot->im->rgb_height);
    gdk_window_resize(imrot->draw->window, imrot->im->rgb_width, imrot->im->rgb_height);
  }
    gtk_widget_show_all (imrot->draw->parent) ;

  if ( imrot->pm ) gdk_imlib_free_pixmap(imrot->pm); /* free the pixmap, not sure if this is necessary */

  g_return_if_fail ( gdk_imlib_render(imrot->im, imrot->im->rgb_width, imrot->im->rgb_height) ) ;
  imrot->pm = gdk_imlib_copy_image(imrot->im);

  gdk_window_set_back_pixmap(imrot->draw->window, imrot->pm, 0);
  gdk_window_clear(imrot->draw->window); 

  gdk_window_show(imrot->draw->window);
  gdk_flush();
  /* #endif */
  
  return;
}

static void
gtk_imrot_expose(GtkWidget *widget, GdkEventExpose *event, GtkImRot *imrot)
/* Redraw the screen from the backing pixmap 
 * --- Parameter: ---
 * GtkWidget      *widget : drawing area with image to cut and rubber box
 * GdkEventExpose *event  : expose_event
 * GtkImRot       *imrot  : image rotate structure
 */
{
  g_return_if_fail ( widget );
  g_return_if_fail ( imrot );
  g_return_if_fail ( GTK_IS_DRAWING_AREA (widget) );
  g_return_if_fail ( GTK_IS_IMROT (imrot) );
  if  ( !imrot->pm ) return ;
  g_return_if_fail ( event );
  g_return_if_fail ( event->type == GDK_EXPOSE );

  gdk_draw_pixmap(widget->window, widget->style->fg_gc[GTK_WIDGET_STATE(widget)], imrot->pm,
		  event->area.x, event->area.y, event->area.x, event->area.y,
		  event->area.width, event->area.height);
}

static GtkWidget *
_gtk_imrot_init_drawing_area(GtkImRot *imrot)
/* create drawing area inside event box and set up signal handlers
 * --- Parameter: ---
 * GtkImRot  *imrot                          : imrot widget to set up signal functions correctly
 * --- Return: ---
 * GtkWidget *_gtk_imrot_init_drawing_area() : new drawing area
 */
{
  GtkWidget *event_box, *draw;
   
  draw = gtk_drawing_area_new();

  /**********************************************************************************
   * all the signals and events of the pixmap area                                  *
   **********************************************************************************/
  gtk_widget_set_events(draw, GDK_EXPOSURE_MASK | GDK_LEAVE_NOTIFY_MASK | GDK_BUTTON_PRESS_MASK 
			 | GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK | GDK_ENTER_NOTIFY_MASK );
  /* Signals used to handle backing pixmap */
  gtk_signal_connect(GTK_OBJECT(draw), "expose_event", GTK_SIGNAL_FUNC(gtk_imrot_expose), imrot);
  gtk_signal_connect(GTK_OBJECT(draw), "configure_event", GTK_SIGNAL_FUNC(gtk_imrot_configure), imrot);

  event_box = gtk_event_box_new ();
  gtk_container_add ( GTK_CONTAINER(event_box), draw );
  if ( !(imrot->draw) ) imrot->draw = draw;
  return draw;
}


/**********************************************************************************
 *                    .............                                               *
 **********************************************************************************/

static GtkAdjustment*
_create_adjustment(GtkWidget *table, GtkImRot *imrot, gchar *name, gint pos, gdouble *value)
/* create spinners for coordinate input
 * --- Parameters: ---
 * GtkImRot *imrot  : image rotate structure
 * gchar    *name   : name of parameter
 * gint      pos    : position
 * gdouble    value  : initial value of parameter
 */
{
  GtkAdjustment *adj;
  GtkWidget     *label, *spinner;
   
  g_return_val_if_fail ( table, NULL );
  g_return_val_if_fail ( imrot, NULL );
  g_return_val_if_fail ( GTK_IS_TABLE (table), NULL );
  g_return_val_if_fail ( GTK_IS_IMROT (imrot), NULL );
  
  label   = gtk_label_new(name);
  gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, pos, pos+1);
  gtk_widget_show(label);

  adj     = GTK_ADJUSTMENT(gtk_adjustment_new(*value, 0.0, 360.0, 1, 0.0, 0.0));
  spinner = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 1, 0);
  gtk_widget_set_usize(spinner, 80, 0);
  gtk_spin_button_set_update_policy(GTK_SPIN_BUTTON(spinner), GTK_UPDATE_ALWAYS);
  gtk_object_set_user_data(GTK_OBJECT(adj), value);
  gtk_object_set_user_data(GTK_OBJECT(spinner), imrot);
  gtk_signal_connect(GTK_OBJECT(adj), "value_changed", 
                     GTK_SIGNAL_FUNC(gtk_imrot_value_changed), spinner);
  gtk_table_attach_defaults(GTK_TABLE(table), spinner, 1, 2, pos, pos+1);
  gtk_widget_show(spinner);
  
  return adj;
}

static void
_draw_rot(GtkWidget *widget, GtkImRot *imrot)
/* Call rotating algorithm and show new image
 * --- Parameter: ---
 * GtkWidget *widget: Apply-Button widget if called via button, else NULL
 * GtkImRot  *imrot : rotating structure
 */
{
  g_return_if_fail ( GTK_IS_IMROT (imrot) );

  if ( imrot->im ) gdk_imlib_kill_image(imrot->im);

  imrot->width  = imrot->o_width;
  imrot->height = imrot->o_height;
  if ( imrot->data ) g_free (imrot->data);
  imrot->data   = g_new (guchar, 3*imrot->width*imrot->height);
  memcpy(imrot->data, imrot->o_data, 3*imrot->width*imrot->height);
 
/*  printf("%s(%i) Drehen um %g Grad!! (%p)\n", __FILE__, __LINE__, imrot->alpha, widget); */
  
  gtk_imrot_rotate_image(imrot, imrot->alpha);
  
  imrot->im = gdk_imlib_create_image_from_data(imrot->data, NULL, imrot->width, imrot->height);
  
  gtk_imrot_configure(imrot->draw, NULL, imrot);
}

static void
gtk_imrot_value_changed(GtkAdjustment *adj, GtkWidget *spinner)
/* called while editing box parameters
 * --- Parameter: ---
 * GtkAdjustment *adj     : target for new value
 * GtkWidget     *spinner :
 */
{
  gdouble   *chg, new;
  GtkImRot *imrot;
  
  g_return_if_fail ( adj );
  g_return_if_fail ( spinner );
  g_return_if_fail ( GTK_IS_OBJECT (adj) );
  g_return_if_fail ( GTK_IS_SPIN_BUTTON (spinner) );
  imrot = gtk_object_get_user_data(GTK_OBJECT(spinner));
  g_return_if_fail ( imrot );
  g_return_if_fail ( GTK_IS_IMROT (imrot) );

  chg = gtk_object_get_user_data(GTK_OBJECT(adj));
  new = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spinner));

  if      ( chg == &(imrot->alpha) ) {
    if ( !_set_parameter_if_valid(&(imrot->alpha), new, 
                                  new >= 0.0 && new <= 360.0, adj) )
      return;
  } else return;
  _draw_rot ( NULL, imrot);
}

static gint 
_set_parameter_if_valid(gdouble *val2set, gdouble val, gint cond, GtkAdjustment *adj)
/* Set value in address if condition is true
 */
{
  g_return_val_if_fail ( adj , FALSE );
  g_return_val_if_fail ( GTK_IS_ADJUSTMENT (adj), FALSE );

  if ( cond ) {
    *val2set = val;
    return TRUE;
  }
  if ( *val2set == val ) return FALSE;  /* it's not necessary to set the value */
  gdk_beep();
  gtk_adjustment_set_value(adj, *val2set);
  return FALSE;
}

/*********************************************************************************
 *                                                                               *
 * Rotation algorithm                                                            *
 *                                                                               *
 *********************************************************************************/

static guchar *_gtk_imrot_90(guchar *image, gint *width, gint *height, gint storepix);
static guchar *_gtk_imrot_180(guchar *image, gint *width, gint *height, gint storepix);
static guchar *_gtk_imrot_270(guchar *image, gint *width, gint *height, gint storepix);
static gint    _gtk_imrot_border(GtkImRot *image, guint b_width, guint b_height);
static void    _gtk_imrot_crop(GtkImRot *image, const gdouble x_shear,
		    const gdouble y_shear, const guint width, const guint height);
static void    _gtk_imrot_shear_horizontally(GtkImRot *image, const gdouble degrees,
                    const guint width, const guint height,
                    const int x_offset, int y_offset, register guchar *range_limit);
static void    _gtk_imrot_shear_vertically(GtkImRot *image,const gdouble degrees,
                    const guint width,const guint height,int x_offset,
		    const int y_offset,register guchar *range_limit);

void
gtk_imrot_mirror_horizontally(guchar *data, gint width, gint height, gchar storepix)
/* mirror image horizontally (at a vertical axis)
 * --- Parameter: ---
 * guchar *data : image data to mirror
 * --- Return: ---
 * guchar *data : image with mirrored buffer
 */
{
  register guchar *ap, *ep;
  guchar          *zp, *fzp;
  register gint    linesize = storepix * width;
   
  g_return_if_fail( data && width && height );

  for ( fzp = (zp = data) + linesize * height; zp < fzp; zp += linesize )
    for ( ep = (ap = zp) + linesize - 1; ap < ep; ap++, ep-- ) {
      if ( storepix == 1 ) {
        register guchar zw = *ep;
        *ep = *ap;
        *ap =  zw;
      } else {
        register guchar zw = *(ep -= 2);
        *ep = *ap;
        *ap = zw;
        zw  = *++ep;
        *ep = *++ap;
        *ap = zw;
        zw  = *++ep;
        *ep = *++ap;
        *ap = zw;
        ep -= 2;
      }
    }
}

void
gtk_imrot_mirror_vertically(guchar *data, gint width, gint height, gchar storepix)
/* mirror image vertically (at a horizontal axis)
 * --- Parameter: ---
 * guchar *data : image data to mirror
 * --- Return: ---
 * guchar *data : image with mirrored buffer
 */
{
  register guchar *ap, *ep;
  guchar          *buf;
  register gint    linesize = storepix * width;
   
  g_return_if_fail( data && width && height );

  buf = g_malloc(linesize);
   
  for ( ep = (ap = data) + linesize * (height - 1); ap < ep; ap += linesize, ep -= linesize ) {
    memcpy(buf, ep, linesize);
    memcpy(ep,  ap, linesize);
    memcpy(ap, buf, linesize);
  }
  g_free(buf);
}

guchar *
gtk_imrot_crop_image_buffer(guchar *image, gint width, gint x, gint y, 
                                    gint w, gint h, gchar storepix)
/* Here the actual cropping is done ... just needed in the rotating function
 * no testing of valid crop coordinates is done
 * this might be useful for other programs
 * Note that original height is not needed, because no checks are done
 * --- Parameter: ---
 * guchar image   : image pixel data
 * gint   width   : original width
 * gint   x       : x coordinate of starting point
 * gint   y       : y coordinate of starting point
 * gint   w       : new width
 * gint   h       : new height
 * gchar  storepix: number of bytes representing one pixel
 * _-- Return: _--
 * guchar *gtk_imrot_crop_image_buffer() : buffer with cropped image data
 */
{
  register guchar *ap, *bp, *fap;
  register gint    sw, sW;
  guchar  *data;
   
  sw   = storepix * w;
  sW   = storepix * width;
  data = g_malloc(sw*h);
  
  for ( fap = (ap = image + storepix*(y*width + x)) + h*sW,
        bp  = data; ap < fap; ap += sW, bp += sw )
    memcpy(bp, ap, sw);
  g_free(image);
  return data;
}

gint
gtk_imgrot_quarter(GtkImRot *image, guint rotations)
/* rotation by
 *   0 if rotations % 4 == 0
 *  90 if rotations % 4 == 1
 * 180 if rotations % 4 == 2
 * 270 if rotations % 4 == 3
 * --- Parameter: ---
 * GtkImRot *image     : image structure
 * guint     rotations : number of quarters to rotate
 * --- Return: ---
 * GtkImRot *image     : structure with in place rotated image
 * gint gtk_imgrot_quarter() : 0 if OK
 */
{
  guchar *rotated;
  int     w, h;

  g_return_val_if_fail(image, -1);

  /* Initialize rotated image attributes. */
  rotations %= 4;

  rotated = g_new(guchar, image->width * image->height * 3);
  w = image->width;
  h = image->height;

  switch ( rotations ) {
  case 0: rotated = image->data;
          break;
  case 1: rotated = _gtk_imrot_90(image->data, &w, &h, 3);
          break;
  case 2: rotated = _gtk_imrot_180(image->data, &w, &h, 3);
          break;
  case 3: rotated = _gtk_imrot_270(image->data, &w, &h, 3);
          break;
  }
  image->width = w;
  image->height    = h;
  image->data    = rotated;

  return 0;
}

gint
gtk_imrot_sharpen_image(GtkImRot *image)
/* Apply sharpen algorithm using the matrix
 *     ( -1    -2    -1)
 *     ( -2    33    -2)
 *     ( -1    -2    -1)
 * This might be usefull also in other programs so make it externally
 * available
 * --- Parameter: ---
 * GtkImRot *image  : image structure
 * --- Return ---
 * GtkImRot *image  : sharpened image structure (sharpening is done inplace,
 *                    old data will be destroyed)
 * gint      gtk_imrot_sharpen_image(): 0 if OK
 */
{
#define Sharpen(weight) \
  total += (weight)*(gint) *s; \
  s+=3;

  guchar      *sharpened;
  gint         ic3, y;
  register int x;
  glong        total;
  register     guchar *q, *s, *s0, *s1, *s2;
  guchar *scanline;

  g_return_val_if_fail(image, -1);
  if ((image->width < 3) || (image->height < 3)) return -1;

  /* Initialize sharpened image attributes. */
  sharpened = g_new(guchar, image->width * image->height * 3);
  ic3 = 3*(image->width);

  /* Dump first scanline of image including the first pixel of the second */
  memcpy(sharpened, image->data, ic3 + 3);

  q = sharpened + ic3 + 3;
  scanline = image->data;

  /* Convolve each row. */
  for (y=1; y < (gint) (image->height-1); y++)
  {
    /* Initialize sliding window pointers. */
    s0 = scanline;
    s1 = scanline += ic3;     
    s2 = scanline +  ic3;
    for (x=0; x < (gint) (ic3-6); x++)
    {
      /* Compute weighted average of target pixel color components. */
      total = 0;
      s=s0;
      Sharpen(-1); Sharpen(-2); Sharpen(-1);
      s=s1;
      Sharpen(-2); Sharpen(33); Sharpen(-2);
      s=s2;
      Sharpen(-1); Sharpen(-2); Sharpen(-1);
      if (total < 0)
        *q = 0;
      else
        if (total > (gint) (255*21))
          *q = 255;
        else
          *q =(guchar) ((total+10)/21);
      q++;
      s0++;
      s1++;
      s2++;
    }
    /* Transfer last pixel of the scanline and the first from the next. */
    memcpy(q, s1+3, 6);
    q += 6;
  }
  /* Dump last scanline of pixels. */
  memcpy(q-3, scanline+ic3, ic3);
  g_free(image->data);
  image->data = sharpened;

  return 0;
#undef Sharpen
}

gint
gtk_imrot_rotate_image(GtkImRot *image, const gdouble degrees)
/* Rotate an image.  Positive angles rotate counter-clockwise (right-hand rule),
 * while negative angles rotate clockwise.  Rotated images are usually larger
 * than the originals and have black triangular corners.  
 * _gtk_imrot_rotate_image is based on the paper "A Fast Algorithm for General Raster
 * Rotatation" by Alan W. Paeth.  RotateImage is adapted from a similar method based on
 * the Paeth paper written by Michael Halle of the Spatial Imaging Group, MIT Media Lab.
 * I used ImageMagick 4.2.8 as an example for the implementation.
 * --- Parameters: ---
 * GtkImRot    *image:    address of an GtkImRot structure
 * const gdouble degrees:  shearing angle along the X axis
 */
{
  gdouble   angle;
  int      x_offset, y_offset;
  gdouble   shear_x, shear_y;

  guchar  *range_limit, *range_table;

  register int  i;

  guint height, rotations, width, y_width;

  /* Adjust rotation angle. */
  g_return_val_if_fail ( image, -1);
  angle = degrees;
  while ( angle < -45.0 ) angle += 360.0;
  for ( rotations = 0; angle > 45.0; rotations++ ) angle -= 90.0;
  rotations %= 4;

  /* Calculate shear equations. */
  shear_x = (-tan(angle*M_PI/360.0));
  shear_y = sin(angle*M_PI/180.0);

  gtk_imgrot_quarter(image, rotations);
  if ( (shear_x == 0.0) || (shear_y == 0.0) ) return 0;

  /* Initialize range table. */
  g_return_val_if_fail( range_table = g_new(guchar, 3*256), -1 );
  for ( i = 0; i < 256; i++ ) {
    range_table[i]       = 0;
    range_table[i + 256] = (guchar) i;
    range_table[i + 512] = 255;
  }
  range_limit = range_table + 256;

  /* Compute image size. */
  width  = image->width;
  height = image->height;
  
  y_width  =  width  +  (gint) ceil(fabs(shear_x)*(gdouble)(height-1));
  x_offset = (width  + ((gint) ceil(fabs(shear_y)*(gdouble)(height-1)) << 1)-width) >> 1;
  y_offset = (height +  (gint) ceil(fabs(shear_y)*(gdouble)(y_width-1))-height) >> 1;

  /* Surround image with border of background color. */
  g_return_val_if_fail (!_gtk_imrot_border(image, x_offset, y_offset + 1), -1);

  /* Perform a fractional rotation.  First, shear the image rows. */
  _gtk_imrot_shear_horizontally(image, shear_x, width, height, x_offset,
    ((gint) (image->height - height) >> 1),range_limit);

  /* Shear the image columns. */
  _gtk_imrot_shear_vertically(image, shear_y, y_width, height,
    ((gint) (image->width - y_width) >> 1), y_offset + 1, range_limit);

  /* Shear the image rows again. */
  _gtk_imrot_shear_horizontally(image, shear_x, y_width, image->height-2,
    ((gint) (image->width - y_width) >> 1), 1, range_limit);

  g_free(range_table);

  /* Crop image. */
  _gtk_imrot_crop(image,shear_x,shear_y,width,height);

  /* Sharpen image. */
  g_return_val_if_fail ( !gtk_imrot_sharpen_image(image), -1);
  
  return 0;
}

/*****************************************************************************
 *                                                                           *
 * internal functions                                                        *
 *                                                                           *
 *****************************************************************************/

#define DownShift(x) (((gulong) ((x)+0x2000)) >> 14)
#define UpShifted(x) ((gint) ((x)*0x4000+0.5))

static guchar *
_gtk_imrot_90(guchar *image, gint *width, gint *height, gint storepix)
/* rotate 90 degrees clockwise
 * --- Parameter: ---
 * guchar  *image  : image pixel buffer
 * gint    *width  : width of image
 * gint    *height : height of image
 * gint     pixsize: 1 for monochrome 3 for RGB
 * --- Return ---
 * guchar   gtk_imrot_90() : buffer with rotated image data or NULL if failed
 */
{
  gint             old_width, new_width, size;
  register guchar *op, *np, *fip;
  guchar          *data, *ozp, *nsp, *fzp;
   
  g_return_val_if_fail ( image && *width && *height, NULL);
  
  data      = g_new(guchar, (size = *width * *height)*storepix);
  old_width = *width * storepix;
  new_width = *height * storepix;
  for ( fzp = (ozp = image) + size * storepix, 
        nsp = data + (*height - 1) * storepix; ozp < fzp;
        ozp += old_width, nsp -= storepix ) {
    for ( fip = (op = ozp) + old_width, np = nsp; op < fip;
          op++, np += new_width ) {
      *np = *op;
      if ( storepix != 1 ) {
        *(np+1) = *++op;
        *(np+2) = *++op;
      }
    }
  }
  new_width = *height;
  *height   = *width;
  *width    = new_width;
  g_free(image);
  return data;
}

static guchar *
_gtk_imrot_180(guchar *image, gint *width, gint *height, gint storepix)
/* rotate 180 degrees
 * --- Parameter: ---
 * guchar  *image  : image pixel buffer
 * gint    *width  : width of image
 * gint    *height : height of image
 * gint     pixsize: 1 for monochrome 3 for RGB
 * --- Return ---
 * guchar   gtk_imrot_180() : buffer with rotated image data or NULL if failed
 */
{
  gtk_imrot_mirror_horizontally(image, *width, *height, storepix);
  gtk_imrot_mirror_vertically(image,   *width, *height, storepix);
  return image;
}

static guchar *
_gtk_imrot_270(guchar *image, gint *width, gint *height, gint storepix)
/* rotate 270 degrees clockwise
 * --- Parameter: ---
 * guchar  *image  : image pixel buffer
 * gint    *width  : width of image
 * gint    *height : height of image
 * gint     pixsize: 1 for monochrome 3 for RGB
 * --- Return ---
 * guchar   gtk_imrot_270() : buffer with rotated image data or NULL if failed
 */
{
  gint             old_width, new_width, size;
  register guchar *op, *np, *fip;
  guchar          *data, *ozp, *nsp, *fzp;
   
  g_return_val_if_fail ( image && *width && *height, NULL);
  

  data      = g_new(guchar, (size = *width * *height)*storepix);
  old_width = *width * storepix;
  new_width = *height * storepix;

  for ( fzp = (nsp = data) + size * storepix, 
        ozp = image + (*width - 1) * storepix; nsp < fzp;
        nsp += new_width, ozp -= storepix )
    for ( fip = (np = nsp) + new_width, op = ozp; np < fip;
          op += old_width, np++ ) {
      *np = *op;
      if ( storepix != 1 ) {
        *++np = *(op+1);
        *++np = *(op+2);
      }
    }
  new_width = *height;
  *height   = *width;
  *width    = new_width;

  g_free(image);
  return data;
}

static void 
_gtk_imrot_shear_horizontally(GtkImRot *image, const gdouble degrees,
                              const guint width, const guint height,
                              const int x_offset, int y_offset, register guchar *range_limit)
/* Shear the image in x-direction with given shear angle of 'degrees'.
 * Positive angles shear counter-clockwise (right-hand rule), and negative angles
 * shear clockwise.  Angles are measured relative to a vertical Y-axis.
 * X shears will widen an image creating black triangles on the left and right
 * sides of the source image.
 * --- Parameters: ---
 * GtkImRot    *image:    address of an GtkImRot structure
 * const gdouble degrees:  shearing angle along the X axis
 * const int    width:    width of image region to shear
 * const int    height:   height of image region to shear
 * const int    x_offset: x-start of region to shear
 * const int    y_offset: y-start of region to shear
 */
{
  gdouble             displacement;
  enum   {LEFT,RIGHT} direction;
  gint                step, y;
  glong               fractional_step, col;

  register guchar    *s,    /* source */
                     *t,    /* target */
                     *fip;  /* finish pointer */

  y_offset--;
  for (y=0; y < (gint) height; y++)
  {
    y_offset++;
    displacement=degrees*(((gdouble) y)-(height-1)/2.0);
    if (displacement == 0.0) 
      continue;
    if (displacement >  0.0) 
      direction = RIGHT;
    else {
      displacement*=(-1.0);
      direction=LEFT;
    }
    step            = (gint) floor(displacement);
    fractional_step = UpShifted(displacement - (gdouble) step);
    if (fractional_step == 0) {
      /* No fractional displacement-- just copy. */
      switch (direction) {
        case LEFT:
          /* Transfer pixels left-to-right. */
          t = ( s = image->data+3*(image->width*y_offset+x_offset) ) - 3*step;
          memcpy(t, s, 3*width);
          /* Set old row to background color. */
	  memset(t + 3*width, 0, 3*step);
          break;
        case RIGHT:
          /* Transfer pixels right-to-left. */
          t = ( s = image->data + 3*(image->width*y_offset+x_offset+width) ) + 3*step;
          t -= 3*width; 
          s -= 3*width;
	  memcpy(t, s, 3*width);
          /* Set old row to background color. */
	  memset(t - 3*step, 0, 3*step);
          break;
      }
      continue;
    }
    /* Fractional displacement. */
    step++;
    switch ( direction ) {
      case LEFT:
        /* Transfer pixels left-to-right. */
        s = image->data + 3*(image->width*y_offset+x_offset);
        t = s - 3*step;
        for ( fip = s + 3 * width; s < fip; s++, t++ ) {
          col=DownShift(*(s-3)*(0x4000-fractional_step) + *s * fractional_step);
          *t = range_limit[col];
        }
        /* Set old row to background color. */
        for ( fip = s, s -= 3; s < fip; s++, t++ ) {
          col=DownShift(*s * (0x4000-fractional_step));
          *t = range_limit[col];
	}
	memset(t, 0, 3*step - 3);
        break;
      case RIGHT:
        /* Transfer pixels right-to-left. */
        s = (image->data + 3*(image->width*y_offset+x_offset+width)) - 1;
        t = s + 3*step;
        /* last pixel to initialize */
        for ( fip = s - 3; s > fip; s--, t-- ) {
          col = DownShift(*s * fractional_step);
          *t  = range_limit[col];
        }
        /* pixel before last until first */
        for ( fip = s - 3 * (width-1); s > fip; s--, t-- ) {
          col = DownShift(*(s+3) * (0x4000-fractional_step) + *s * fractional_step);
          *t  = range_limit[col];
        }
        /* Set old row to background color. */
        s += 3;
        for ( fip = t - 3; t > fip; s--, t-- ) {
          col = DownShift(*s * (0x4000-fractional_step));
          *t  = range_limit[col];
	}
        memset(t - 3*(step - 1) + 1, 0, 3*step - 3);
        break;
    }
  }
}

static void 
_gtk_imrot_shear_vertically(GtkImRot *image, const gdouble degrees,
                            const guint width, const guint height, int x_offset,
                            const int y_offset, register guchar *range_limit)
/* Shear the image in y-direction with given shear angle of 'degrees'.
 * Positive angles shear counter-clockwise (right-hand rule), and negative angles
 * shear clockwise.  Angles are measured relative to a horizontal x-axis.
 * Y shears will increase the height of an image creating black triangles on the
 * top and bottom of the source image.
 * --- Parameters: ---
 * GtkImRot    *image:    address of an GtkImRot structure
 * const gdouble degrees:  shearing angle along the X axis
 * const int    width:    width of image region to shear
 * const int    height:   height of image region to shear
 * const int    x_offset: x-start of region to shear
 * const int    y_offset: y-start of region to shear
 */
{
  gdouble          displacement;
  enum   {UP,DOWN} direction;
  gint             step, y;
  glong            col, fractional_step;

  register guint   col3;
  
  register guchar *s,    /* source */
                  *t,    /* target */
                  *fip,  /* finish pointer */
                  *fop;  /* other finish pointer */

  x_offset--;
  col3 = 3*image->width;
  
  for (y=0; y < (gint) width; y++)
  {
    x_offset++;
    displacement = degrees*(((gdouble) y)-(width-1)/2.0);
    if ( displacement == 0.0 ) continue;
    if ( displacement > 0.0  ) direction = DOWN;
    else {
      displacement *= (-1.0);
      direction     = UP;
    }
    step            = (gint) floor(displacement);
    fractional_step = UpShifted(displacement - (gdouble)step);
    if ( fractional_step == 0 ) {
      /* No fractional displacement-- just copy the pixels. */
      switch (direction) {
        case UP:
          /* Transfer pixels top-to-bottom. */
          s = image->data + 3*(image->width*y_offset+x_offset);
          t = s - step * col3;
          for (fip = t + col3 * height; t < fip; 
               t += col3 - 3, s += col3 - 3) {
            *t = *s;
	    *++t = *++s;
	    *++t = *++s;
          }
          /* Set old column to background color. */
          for (fip = t + col3 * step ; t < fip; t += col3 )
            *t = *(t+1) = *(t+2) = 0;
          break;

        case DOWN:
          /* Transfer pixels bottom-to-top. */
          s = image->data + 3*(image->width*(y_offset+height-1)+x_offset);
          t = s + step * col3;
          for (fip = t - col3 * height; t > fip; 
               t -= col3, s -= col3 ) {
	    *t = *s;
	    *++t = *++s;
	    *++t = *++s;
          }
          /* Set old column to background color. */
          for (fip = t - col3 * step; t > fip; t -= col3 )
            *t = *(t+1) = *(t+2) = 0;
          break;
      }
      continue;
    }
    /* Fractional displacment.*/
    step++;
    switch (direction)
    {
      case UP:
      {
        /* Transfer pixels top-to-bottom. */
        s   = image->data + 3*(image->width*y_offset+x_offset);
        t   = s - step * col3;
        fop = t + height * col3; 
        /* this loop is necessary to get valid *(s-col3) values in the following loop */
        for ( fip = t + 3; t < fip; s++, t++) {
          col = DownShift(*s * fractional_step);
          *t  = range_limit[col];
        }
        for ( s += col3 - 3, t += col3 - 3; t < fop; s += col3 - 3, t += col3 - 3 )
        {
          for ( fip = t + 3; t < fip; s++, t++) {
            col = DownShift(*(s-col3)*(0x4000-fractional_step) + *s * fractional_step);
            *t  = range_limit[col];
	  }
        }
        /* Set old column to background color. */
        s -= col3;
        for ( fip = t + 3; t < fip; t++, s++ ) {
          col = DownShift(*s * (0x4000-fractional_step));
          *t = range_limit[col];
	}
        t += col3 - 3;
        for (fip = t + col3 * (step-1); t < fip; t += col3 )
          *t = *(t+1) = *(t+2) = 0;
        break;
      }
      case DOWN:
      {
        /* Transfer pixels bottom-to-top. */
        s = image->data + 3*(image->width*(y_offset+height)+x_offset) - col3;
        t = s + step * col3;
        for (fop = t - col3*height; fop < t; s -= col3 + 3, t -= col3 + 3 )
        {
	  for ( fip = t + 3; t < fip; s++, t++ ) {
            col = DownShift(*(s+col3) * (0x4000-fractional_step) + *s * fractional_step);
            *t  = range_limit[col];
          }
        }
        /* Set old column to background color. */
        s += col3; /* last pixel */
        for ( fip = t + 3; t < fip; t++, s++ ) {
          col = DownShift(*s * (0x4000-fractional_step));
          *t  = range_limit[col];
	}
	t -= col3 + 3;
        for ( fip = t - col3 * (step-1); t > fip; t -= col3)
          *t = *(t+1) = *(t+2) = 0;
        break;
      }
    }
  }
}

gint
_gtk_imrot_border(GtkImRot *image, const guint b_width, const guint b_height)
{
  guchar  *bordered;
  gint     height, width;

  register guchar  *s,    /* source */
                   *t,    /* target */
                   *fip;  /* finish pointer */

  g_return_val_if_fail( image, -1);

  width    = image->width + (b_width  << 1);
  height   = image->height    + (b_height << 1);
  g_return_val_if_fail ( (width  - b_width  >= image->width) && 
                         (height - b_height >= image->height), -1 );
  bordered = g_new(guchar, width*height*3);

  s = image->data;
  t = bordered + 3*(width * b_height + b_width);
  for ( fip = s + 3*image->width*image->height; s < fip; s += 3*image->width, t += 3*width)
    memcpy(t, s, 3*image->width);

  image->width = width;
  image->height    = height;
  g_free(image->data);
  image->data = bordered;

  return 0;
}

static void
_gtk_imrot_crop(GtkImRot *image, const gdouble x_shear,
                const gdouble y_shear, const guint width, const guint height)
{
  typedef struct Point {
    gdouble x, y;
  } Point;

  gdouble        x_max, x_min, y_max, y_min;
  Point         corners[4];
  gint          x0, y0, w0, h0;
  register gint i;

    /* Calculate the rotated image size. */
  w0 = width;
  h0 = height;
  corners[0].x = (-((gint) w0)/2.0);
  corners[0].y = (-((gint) h0)/2.0);
  corners[1].x = ((gint)   w0)/2.0;
  corners[1].y = (-((gint) h0)/2.0);
  corners[2].x = (-((gint) w0)/2.0);
  corners[2].y = ((gint)   h0)/2.0;
  corners[3].x = ((gint)   w0)/2.0;
  corners[3].y = ((gint)   h0)/2.0;
  for (i=0; i < 4; i++)
  {
    corners[i].x += x_shear*corners[i].y;
    corners[i].y += y_shear*corners[i].x;
    corners[i].x += x_shear*corners[i].y;
    corners[i].x += (image->width - 1)/2.0;
    corners[i].y += (image->height - 3)/2.0;
  }
  x_min = corners[0].x;
  y_min = corners[0].y;
  x_max = corners[0].x;
  y_max = corners[0].y;
  for ( i = 1; i < 4; i++ ) {
    if ( x_min > corners[i].x ) x_min = corners[i].x;
    if ( y_min > corners[i].y ) y_min = corners[i].y;
    if ( x_max < corners[i].x ) x_max = corners[i].x;
    if ( y_max < corners[i].y ) y_max = corners[i].y;
  }
  x_min = floor((gdouble) x_min);
  x_max = ceil((gdouble) x_max);
  y_min = floor((gdouble) y_min);
  y_max = ceil((gdouble) y_max);
      /* Do not crop sheared image. */
  w0 = (guint) (x_max-x_min)-1;
  h0 = (guint) (y_max-y_min)-1;
  x0 = (gint) x_min+(((gint) (x_max-x_min) - w0) >> 1)+1;
  y0 = (gint) y_min+(((gint) (y_max-y_min) - h0) >> 1)+2;
  
  image->data = gtk_imrot_crop_image_buffer(image->data, image->width, x0, y0, w0, h0, 3);

  image->width  = w0;
  image->height = h0;
}
