/*  Inti-GConf: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

//! @file inti/gconf/value.h
//! @brief GConfEntry and GConfValue C++ wrapper interface.

#ifndef INTI_GCONF_VALUE_H
#define INTI_GCONF_VALUE_H

#ifndef GCONF_GCONF_VALUE_H
extern "C"
{
#include <gconf/gconf-value.h>
}
#endif

#ifndef INTI_STRING_H
#include <inti/utf-string.h>
#endif

#ifndef CPP_VECTOR_H
#include <vector>
#endif

namespace Inti {

namespace G {
class Error;
}

namespace GConf {

class Schema;

//! @enum ValueType
//! ValueType is used to indicate the type of a GConf::Value.

enum ValueType
{
	VALUE_INVALID = GCONF_VALUE_INVALID, //!< Indicates errors.
	VALUE_STRING = GCONF_VALUE_STRING, //!< String Value (String)
	VALUE_INT = GCONF_VALUE_INT, //!< Integer Value (int)
	VALUE_FLOAT = GCONF_VALUE_FLOAT, //!< Floating point value (double)
	VALUE_BOOL = GCONF_VALUE_BOOL, //!< Boolean value (bool)
	VALUE_SCHEMA = GCONF_VALUE_SCHEMA, //!< Schema value (GConf::Schema)
	VALUE_LIST = GCONF_VALUE_LIST, //!< List of values (int, bool, double, String or Schema)
	VALUE_PAIR = GCONF_VALUE_PAIR //!< Pair of Values (GConf::Value)
};

//! @class Value value.h inti/gconf/value.h
//! @brief A GConfValue C++ wrapper class.
//!
//! Value stores one of the value types GConf understands; GConf uses GConf::Value to pass values
//! around because it doesn't know the type of its values at compile time. A Value should always
//! be initialized before use. That is, you should not use a Value unless you have called one of
//! the "setter" functions.

class Value
{
	GConfValue *value_;

public:
//! @name Constructors
//! @{

	Value();
	//!< Constructs an empty Value.

	Value(ValueType type);
	//!< Constructs a new Value with the specified type.
	//!< @param type The type of the new Value.
	//!<
	//!< <BR>The type is immutable after construction; values have a fixed type. You must initialize
	//!< the Value after creation; that is, you must set its value with one of the "setter" functions.

	Value(ValueType type, const String& value_str, G::Error *error= 0);
	//!< Constructs a new Value with the specified type and value.
	//!< @param type The type of the new Value.
	//!< @param value_str A string that specifies the value; must be an int, bool, double or string.
	//!< @param error A G::Error object, or null to ignore errors.
	//!<
	//!< <BR>This constructor is for convenience. It only works on simple types (int, bool, float
	//!< and String). The value passed as a string is converted into a value of the specified type.
	//!< The bool value can be true/false, 1/0, or yes/no and is not case sensitive.

	explicit Value(const GConfValue *src);
	//!< Construct a new Value from an existing GConfValue.

	Value(const Value& src);
	//!< Copy constructor.

	~Value();
	//!< Destructor.

	Value& operator=(const Value& src);
	//!< Assignment operator.

//! @}
//! @name Accessors
//! @{

	GConfValue* gconf_value() const { return value_; }
	//!< Get a pointer to the GConfValue.

	bool is_set() const;
	//!< Returns true if the value was set, and false if it was not set or was unset.

	ValueType get_type() const;
	//!< Get the value type for the Value.

	ValueType get_list_type() const;
	//!< Get the value type of the list elements in a Value with type VALUE_LIST.

	int get_int() const;
	//!< Get an integer for a Value with type VALUE_INT.

	bool get_bool() const;
	//!< Get a bool for a Value with type VALUE_BOOL.

	double get_float() const;
	//!< Get a double for a Value with type VALUE_FLOAT.

	String get_string() const;
	//!< Get a String for a Value with type VALUE_STRING.
	//!< If the Value is not initialized (i.e. no one has called set_string()) then the string
	//!< may be null, but of course you should not try to use an uninitialized Value.

	Schema get_schema() const;
	//!< Get a Schema for a Value with type VALUE_SCHEMA. If the Value is uninitialized, 
	//!< it may return null; but of course you should have initialized the Value. The
	//!< GConf library will not return values with a null schema.

	Value get_car() const;
	//!< Get the first member (car) of a Value with type GCONF_VALUE_PAIR.
	//!< @return A Value object.
	//!<
	//!< <BR>The car is another Value, with a primitive type (bool, int, float, String, Schema).

	Value get_cdr() const;
	//!< Get the second member (cdr) of a Value with type GCONF_VALUE_PAIR.
	//!< @return A Value object.
	//!<
	//!< <BR>The cdr is another Value, with a primitive type (bool, int, float, String, Schema).

	bool get_list(std::vector<int>& list) const;
	//!< Requests the list (VALUE_INT) stored in the Value.
	//!< @param list A reference to a vector of int, to hold the list elements.
	//!< @return <EM>true</EM> on success, <EM>false</EM> if an error occurs.

	bool get_list(std::vector<bool>& list) const;
	//!< Requests the list (VALUE_BOOL) stored in the Value.
	//!< @param list A reference to a vector of bool, to hold the list elements.
	//!< @return <EM>true</EM> on success, <EM>false</EM> if an error occurs.

	bool get_list(std::vector<double>& list) const;
	//!< Requests the list (VALUE_FLOAT) stored in the Value.
	//!< @param list A reference to a vector of double, to hold the list elements.
	//!< @return <EM>true</EM> on success, <EM>false</EM> if an error occurs.

	bool get_list(std::vector<String>& list) const;
	//!< Requests the list (VALUE_STRING) stored in the Value.
	//!< @param list A reference to a vector of String, to hold the list elements.
	//!< @return <EM>true</EM> on success, <EM>false</EM> if an error occurs.

	bool get_list(std::vector<Schema>& list) const;
	//!< Requests the list (VALUE_SCHEMA) stored in the Value.
	//!< @param list A reference to a vector of Schema, to hold the list elements.
	//!< @return <EM>true</EM> on success, <EM>false</EM> if an error occurs.

//! @}
//! @name Methods
//! @{

	void set_int(int value);
	//!< Sets the integer value of a Value with type VALUE_INT.
	//!< @param value The new integer value.

	void set_bool(bool value);
	//!< Sets the bool value of a Value with type VALUE_BOOL.
	//!< @param value The new boolean value.

	void set_float(double value);
	//!< Sets the double value of a Value with type VALUE_FLOAT.
	//!< @param value The new float value.

	void set_string(const String& str);
	//!< Sets the String value of a Value with type VALUE_STRING.
	//!< @param str The new String value.

	void set_schema(const Schema& sc);
	//!< Sets the Schema value of a Value with type VALUE_SCHEMA.
	//!< @param sc The new Schema value.

	void set_car(const Value& car);
	//!< Sets the value of the first field (car) of a Value with type VALUE_PAIR.
	//!< @param car A reference to a Value object.

	void set_cdr(const Value& cdr);
	//!< Sets the value of the second field (cdr) of a Value with type VALUE_PAIR.
	//!< @param cdr A reference to a Value object.

	void set_list(std::vector<int>& list);
 	//!< Sets the list (VALUE_INT) for the value.
	//!< @param list A reference to a vector of int that contains the list elements.

	void set_list(std::vector<bool>& list);
 	//!< Sets the list (VALUE_BOOL) for the value.
	//!< @param list A reference to a vector of bool that contains the list elements.

	void set_list(std::vector<double>& list);
 	//!< Sets the list (VALUE_FLOAT) for the value.
	//!< @param list A reference to a vector of float that contains the list elements.

	void set_list(std::vector<String>& list);
 	//!< Sets the list (VALUE_STRING) for the value.
	//!< @param list A reference to a vector of String that contains the list elements.

	void set_list(std::vector<Schema>& list);
 	//!< Sets the list (VALUE_SCHEMA) for the value.
	//!< @param list A reference to a vector of Schema that contains the list elements.

	String to_string();
	//!< Creates a human-readable string representation of a Value.
	//!< @return A String representing the value.
	//!<
	//!< <BR>This is intended for debugging and the like; the string representation is not
	//!< suitable for reliable machine parsing (that is, you shouldn't use this function
	//!< to save a value to a file or anything like that). The exact nature of the string
	//!< representation may change in future versions.

//! @}
};

//! @class Entry value.h inti/gconf/value.h
//! @brief A GConfEntry C++ wrapper class.
//!
//! An Entry stores an entry from a GConf "directory," including a key-value pair, the
//! name of the schema applicable to this entry, whether the value is a default value,
//! and whether GConf can write a new value at this key. key should be an absolute key,
//! not a relative key. (Note that internally GConf breaks this rule sometimes; but in
//! the public interface, key is always an absolute key.) To access the key and value,
//! use get_key() and get_value(). For example if the value "10" is stored at the key
//! "/foo/bar/baz", the Entry will store "baz" and "10".

class Entry
{
	GConfEntry *entry_;

public:
//! @name Constructors
//! @{

	Entry(const String& key, const Value& value);
	//!< Constructs a new Entry with <EM>key</EM> and <EM>value</EM>.
	//!< @param key The full path to the key, starting with '/'.
	//!< @param value The value for <EM>key</EM>.

	Entry(const GConfEntry *src);
	//!< Construct a new Entry from an exisitng GConfEntry.
	//!< @param src A GConfEntry to copy.

	Entry(const Entry& src);
	//!< Copy constructor.

	~Entry();
	//!< Destructor.

	Entry& operator=(const Entry& src);
	//!< Assignment operator.

//! @}
//! @name Accessors
//! @{

	GConfEntry* gconf_entry() const { return entry_; }
	//!< Get a pointer to the GConfEntry.

	bool is_set() const;
	//!< Returns true if the value for the entry was set, and false if it was unset.

	String get_key() const;
	//!< Get the key field of an Entry.

	Value get_value() const;
	//!< Get the value field of an Entry.
	//!< @return A Value object.

	String get_schema_name() const;
	//!< Get the name of the schema applicable to this entry.

	bool get_is_default() const;
	//!< Returns true if the value is the default.

	bool get_is_writable() const;
	//!< Returns true if the value is writable, and false if its read-only.

//! @}
//! @name Methods
//! @{
	
	void set_value(const Value& value);
	//!< Sets the value field for the Entry.
	//!< @param value The new Value.

	void set_schema_name(const String& name);
	//!< Sets the name of the schema applicable to this entry.
	//!< @param name The name of the schema.

	void set_is_default(bool is_default);
	//!< Sets whether the value is the default.

	void set_is_writable(bool is_writable);
	//!< Sets whether the value is writable, or read-only.

//! @}
};

} // namespace GConf

} // namespace Inti

#endif // INTI_GCONF_VALUE_H

