/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  pixbuf.cc - GdkPixbuf C++ wrapper interface
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "pixbuf.h"
#include "../gdk/bitmap.h"
#include "../gdk/color.h"
#include "../gdk/drawable.h"
#include "../gdk/gc.h"
#include "../gdk/image.h"
#include "../gdk/pixmap.h"
#include "../glib/error.h"
#include <gdk/gdkpixbuf.h>

using namespace Inti;

/*  Gdk::PixbufFormat
 */

Gdk::PixbufFormat::PixbufFormat(GdkPixbufFormat *format)
: format_(format)
{
}

Gdk::PixbufFormat::~PixbufFormat()
{
}

String
Gdk::PixbufFormat::get_name() const
{
	char *name = gdk_pixbuf_format_get_name(format_);
	String s(name);
	g_free(name);
	return s;
}

String 
Gdk::PixbufFormat::get_description() const
{
	char *description = gdk_pixbuf_format_get_description(format_);
	String s(description);
	g_free(description);
	return s;
}

std::vector<String> 
Gdk::PixbufFormat::get_mime_types() const
{
	std::vector<String> mime_types;

	int i = 0;
	char **tmp_mime_types = gdk_pixbuf_format_get_mime_types(format_);

	while (tmp_mime_types[i] && *tmp_mime_types[i])
	{
		String s(tmp_mime_types[i]);
		mime_types.push_back(s);
		++i;
	}

	g_strfreev(tmp_mime_types);
	return mime_types;
}

std::vector<String> 
Gdk::PixbufFormat::get_extensions() const
{
	std::vector<String> extensions;

	int i = 0;
	char **tmp_extensions = gdk_pixbuf_format_get_mime_types(format_);

	while (tmp_extensions[i] && *tmp_extensions[i])
	{
		String s(tmp_extensions[i]);
		extensions.push_back(s);
		++i;
	}

	g_strfreev(tmp_extensions);
	return extensions;
}

bool 
Gdk::PixbufFormat::is_writable() const
{
	return gdk_pixbuf_format_is_writable(format_);
}

/*  Gdk::Pixbuf
 */

Gdk::Pixbuf::Pixbuf(GdkPixbuf *pixbuf, bool reference)
: G::Object((GObject*)pixbuf, reference)
{
}
	
Gdk::Pixbuf::Pixbuf(const Pixbuf& src, int src_x, int src_y, int width, int height)
: G::Object((GObject*)gdk_pixbuf_new_subpixbuf(src.gdk_pixbuf(), src_x, src_y, width, height))
{
}

Gdk::Pixbuf::Pixbuf(const String& filename, G::Error *error)
: G::Object((GObject*)gdk_pixbuf_new_from_file(filename.c_str(), *error))
{
}

Gdk::Pixbuf::Pixbuf(const unsigned char *data, int width, int height, int rowstride, bool has_alpha,
GdkPixbufDestroyNotify destroy_fn, gpointer destroy_fn_data, int bits_per_sample, Colorspace colorspace)
: G::Object((GObject*)gdk_pixbuf_new_from_data(data, (GdkColorspace)colorspace, has_alpha, bits_per_sample, width, height,
  rowstride, destroy_fn, destroy_fn_data))
{
}

Gdk::Pixbuf::Pixbuf(const char **data)
: G::Object((GObject*)gdk_pixbuf_new_from_xpm_data(data))
{
}

Gdk::Pixbuf::Pixbuf(int data_length, const unsigned char *data, bool copy_pixels, G::Error *error)
: G::Object((GObject*)gdk_pixbuf_new_from_inline(data_length, data, copy_pixels, *error))
{
}

Gdk::Pixbuf::~Pixbuf()
{
}
	
Gdk::Pixbuf::operator GdkPixbuf* () const
{
	return this ? gdk_pixbuf() : 0; 
}
	
Pointer<Gdk::Pixbuf> 
Gdk::Pixbuf::create(int width, int height, bool has_alpha, int bits_per_sample, Colorspace colorspace)
{
	return G::Object::wrap<Pixbuf>(gdk_pixbuf_new((GdkColorspace)colorspace, has_alpha, bits_per_sample, width, height));
}

Pointer<Gdk::Pixbuf>
Gdk::Pixbuf::create(const Drawable& drawable, int x, int y, int width, int height, Colormap *colormap)
{
	return G::Object::wrap<Pixbuf>(gdk_pixbuf_get_from_drawable(0, drawable.gdk_drawable(), *colormap,
	                               x, y, 0, 0, width, height));
}

Pointer<Gdk::Pixbuf>
Gdk::Pixbuf::create(const Image& image, int x, int y, int width, int height, Colormap *colormap)
{
	return G::Object::wrap<Pixbuf>(gdk_pixbuf_get_from_image(0, image.gdk_image(), *colormap, x, y, 0, 0, width, height));
}

bool
Gdk::Pixbuf::get_formats(std::vector<Pointer<PixbufFormat> >& formats)
{
	g_return_val_if_fail(formats.empty(), false);

	GSList *first = gdk_pixbuf_get_formats();
	GSList *next = first;

	while (next != 0)
	{
		Pointer<PixbufFormat> format(new PixbufFormat((GdkPixbufFormat*)next->data));
		formats.push_back(format);
		next = g_slist_next(next);
	}

	g_slist_free(first);
	return !formats.empty();
}

Gdk::Colorspace
Gdk::Pixbuf::get_colorspace() const
{
	return (Colorspace)gdk_pixbuf_get_colorspace (gdk_pixbuf());
}

int
Gdk::Pixbuf::get_n_channels() const
{
	return gdk_pixbuf_get_n_channels(gdk_pixbuf());
}

bool
Gdk::Pixbuf::get_has_alpha() const
{
	return gdk_pixbuf_get_has_alpha(gdk_pixbuf());
}

int
Gdk::Pixbuf::get_bits_per_sample() const
{
	return gdk_pixbuf_get_bits_per_sample(gdk_pixbuf());
}

unsigned char*
Gdk::Pixbuf::get_pixels() const
{
	return gdk_pixbuf_get_pixels(gdk_pixbuf());
}

int
Gdk::Pixbuf::get_width() const
{
	return gdk_pixbuf_get_width(gdk_pixbuf());
}

int
Gdk::Pixbuf::get_height() const
{
	return gdk_pixbuf_get_height(gdk_pixbuf());
}

int
Gdk::Pixbuf::get_rowstride() const
{
	return gdk_pixbuf_get_rowstride(gdk_pixbuf());
}

bool
Gdk::Pixbuf::get_from_drawable(const Drawable& src, int src_x, int src_y, int dest_x, int dest_y, int width, int height, Colormap *colormap)
{
	return gdk_pixbuf_get_from_drawable(gdk_pixbuf(), src.gdk_drawable(), *colormap, src_x, src_y, dest_x, dest_y, width, height);
}

bool
Gdk::Pixbuf::get_from_image(const Image& src, int src_x, int src_y, int dest_x, int dest_y, int width, int height, Colormap *colormap)
{
	return gdk_pixbuf_get_from_image(gdk_pixbuf(), src.gdk_image(), *colormap, src_x, src_y, dest_x, dest_y, width, height);
}

String
Gdk::Pixbuf::get_option(const String& key) const
{
	return gdk_pixbuf_get_option(gdk_pixbuf(), key.c_str());
}

void
Gdk::Pixbuf::fill(unsigned int pixel)
{
	gdk_pixbuf_fill(gdk_pixbuf(), pixel);
}

namespace { // collect_save_options was copied from gdk-pixbuf-io.c

void collect_save_options (va_list opts, char ***keys, char ***vals)
{
	char *key;
	char *val;
	int count = 0;
	*keys = 0;
	*vals = 0;

	char *next = va_arg (opts, char*);
	while (next)
	{
		key = next;
		val = va_arg (opts, char*);

		++count;

		// woo, slow
		*keys = (char**)g_realloc (*keys, sizeof(char*) * (count + 1));
		*vals = (char**)g_realloc (*vals, sizeof(char*) * (count + 1));

		(*keys)[count-1] = g_strdup (key);
		(*vals)[count-1] = g_strdup (val);

		(*keys)[count] = 0;
		(*vals)[count] = 0;

		next = va_arg (opts, char*);
	}
}

} // collect_save_options

Pointer<Gdk::Pixbuf>
Gdk::Pixbuf::copy() const
{
	return G::Object::wrap_new<Pixbuf>(gdk_pixbuf_copy(gdk_pixbuf()), true);
}

bool
Gdk::Pixbuf::save(const String& filename, const char *type, G::Error *error, ...)
{
	char **keys = 0;
	char **values = 0;

	va_list args;
	va_start (args, error);
	collect_save_options (args, &keys, &values);
	va_end (args);

	bool result = gdk_pixbuf_savev (gdk_pixbuf(), filename.c_str(), type, keys, values, *error);
	
	g_strfreev (keys);
	g_strfreev (values);
	return result;
}

bool
Gdk::Pixbuf::save(const String& filename, const char *type, char **option_keys, char **option_values, G::Error *error)
{
	return gdk_pixbuf_savev(gdk_pixbuf(), filename.c_str(), type, option_keys, option_values, *error);
}

Pointer<Gdk::Pixbuf>
Gdk::Pixbuf::add_alpha(bool substitute_color, unsigned char red, unsigned char green, unsigned char blue)
{
	GdkPixbuf *pixbuf = gdk_pixbuf_add_alpha(gdk_pixbuf(), substitute_color, red, green, blue);
	return pixbuf ? G::Object::wrap_new<Gdk::Pixbuf>(pixbuf, true) : 0;
}

void
Gdk::Pixbuf::copy_area(const Pixbuf& src, int src_x, int src_y, int width, int height, int dest_x, int dest_y)
{
	gdk_pixbuf_copy_area(src.gdk_pixbuf(), src_x, src_y, width, height, gdk_pixbuf(), dest_x, dest_y);
}

void
Gdk::Pixbuf::copy_area(const Pixbuf& src, const Rectangle& src_rect, int dest_x, int dest_y)
{
	gdk_pixbuf_copy_area(src.gdk_pixbuf(), src_rect.x(), src_rect.y(), src_rect.width(),
	                     src_rect.height(), gdk_pixbuf(), dest_x, dest_y);
}

void
Gdk::Pixbuf::saturate_and_pixelate(const Pixbuf& src, float saturation, bool pixelate)
{
	gdk_pixbuf_saturate_and_pixelate(src.gdk_pixbuf(), gdk_pixbuf(), saturation, pixelate);
}

void
Gdk::Pixbuf::saturate_and_pixelate(float saturation, bool pixelate)
{
	gdk_pixbuf_saturate_and_pixelate(gdk_pixbuf(), gdk_pixbuf(), saturation, pixelate);
}

Pointer<Gdk::Pixbuf>
Gdk::Pixbuf::scale_simple(int width, int height, InterpType interp_type)
{
	return G::Object::wrap_new<Pixbuf>(gdk_pixbuf_scale_simple(gdk_pixbuf(), width, height,
	                                   (GdkInterpType)interp_type), true);
}

void
Gdk::Pixbuf::scale(const Pixbuf& src, int dest_x, int dest_y, int dest_width, int dest_height, 
                   double offset_x, double offset_y, double scale_x, double scale_y, InterpType interp_type)
{
	gdk_pixbuf_scale(src.gdk_pixbuf(), gdk_pixbuf(), dest_x, dest_y, dest_width, dest_height, 
	                 offset_x, offset_y, scale_x, scale_y, (GdkInterpType)interp_type);
}

void
Gdk::Pixbuf::scale(const Pixbuf& src, const Rectangle& dest_rect, double offset_x, double offset_y,
                   double scale_x, double scale_y, InterpType interp_type)
{
	gdk_pixbuf_scale(src.gdk_pixbuf(), gdk_pixbuf(), dest_rect.x(), dest_rect.y(), dest_rect.width(), 
	                 dest_rect.height(), offset_x, offset_y, scale_x, scale_y, (GdkInterpType)interp_type);
}

void
Gdk::Pixbuf::composite(const Pixbuf& src, int dest_x, int dest_y, int dest_width, int dest_height, double offset_x, 
                       double offset_y, double scale_x, double scale_y, InterpType interp_type, int overall_alpha)
{
	gdk_pixbuf_composite(src.gdk_pixbuf(), gdk_pixbuf(), dest_x, dest_y, dest_width, dest_height,
	                     offset_x, offset_y, scale_x, scale_y, (GdkInterpType)interp_type, overall_alpha);
}

void
Gdk::Pixbuf::composite(const Pixbuf& src, const Rectangle& dest_rect, double offset_x, double offset_y,
                       double scale_x, double scale_y, InterpType interp_type, int overall_alpha)
{
	gdk_pixbuf_composite(src.gdk_pixbuf(), gdk_pixbuf(), dest_rect.x(), dest_rect.y(), 
	                     dest_rect.width(), dest_rect.height(), offset_x, offset_y, 
			     scale_x, scale_y, (GdkInterpType)interp_type, overall_alpha);
}

void
Gdk::Pixbuf::composite_color(const Pixbuf& src, int dest_x, int dest_y, int dest_width, int dest_height, 
                             double offset_x, double offset_y, double scale_x, double scale_y, 
			     InterpType interp_type, int overall_alpha, int check_x, int check_y,
			     int check_size, unsigned int color1, unsigned int color2)
{
	gdk_pixbuf_composite_color(src.gdk_pixbuf(), gdk_pixbuf(), dest_x, dest_y, dest_width, dest_height, 
	                           offset_x, offset_y, scale_x, scale_y, (GdkInterpType)interp_type, 
				   overall_alpha, check_x, check_y, check_size, color1, color2);
}

void
Gdk::Pixbuf::composite_color(const Pixbuf& src, const Rectangle& dest_rect, double offset_x, double offset_y,
                             double scale_x, double scale_y, InterpType interp_type, int overall_alpha,
                             const Point& check_offset, int check_size, unsigned int color1, unsigned int color2)
{
	gdk_pixbuf_composite_color(src.gdk_pixbuf(), gdk_pixbuf(), dest_rect.x(), dest_rect.y(),
	                           dest_rect.width(), dest_rect.height(), offset_x, offset_y,
	                           scale_x, scale_y, (GdkInterpType)interp_type, overall_alpha,
	                           check_offset.x(), check_offset.y(), check_size, color1, color2);
}

Pointer<Gdk::Pixbuf>
Gdk::Pixbuf::composite_color_simple(int width, int height, InterpType interp_type, int overall_alpha,
                                         int check_size, unsigned int color1, unsigned int color2)
{
	return G::Object::wrap_new<Pixbuf>(gdk_pixbuf_composite_color_simple(gdk_pixbuf(), 
	                                   width, height, (GdkInterpType)interp_type, 
					   overall_alpha, check_size, color1, color2), true);
}

void
Gdk::Pixbuf::render_threshold_alpha(Bitmap& bitmap, int src_x, int src_y, int dest_x, int dest_y,
                                    int width, int height, int alpha_threshold)
{
	gdk_pixbuf_render_threshold_alpha(gdk_pixbuf(), bitmap.gdk_bitmap(), src_x, src_y, 
	                                  dest_x, dest_y, width, height, alpha_threshold);
}

void
Gdk::Pixbuf::render_pixmap_and_mask(Pointer<Pixmap> *pixmap_return, Pointer<Bitmap> *mask_return, 
                                    int alpha_threshold, Colormap *colormap)
{
	GdkPixmap *pixmap = 0;
	GdkBitmap *mask = 0;
	GdkColormap *cmap = colormap ? colormap->gdk_colormap() : gdk_rgb_get_colormap();
	gdk_pixbuf_render_pixmap_and_mask_for_colormap(gdk_pixbuf(), cmap, &pixmap, &mask, alpha_threshold);

	if (pixmap_return)
		*pixmap_return = G::Object::wrap_new<Pixmap>(pixmap, true);

	if (mask_return)
		*mask_return = G::Object::wrap_new<Bitmap>(mask, true);
}

