/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  gc.cc - GdkGC C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "gc.h"
#include "private/gc_p.h"
#include "color.h"
#include "bitmap.h"
#include "region.h"
#include "screen.h"

using namespace Inti;

/* Gdk::GCValues
 */
 
Gdk::GCValues::GCValues()
: mask_(0)
{
	values_.tile = 0;
	values_.stipple = 0;
	values_.clip_mask = 0;
}

Gdk::GCValues::GCValues(const GCValues& src)
: values_(src.values_), mask_(src.mask_)
{
	if (src.tile_set() && src.values_.tile)
		g_object_ref(src.values_.tile);

	if (src.stipple_set() && src.values_.stipple)
		g_object_ref(src.values_.stipple);

	if (src.clip_mask_set() && src.values_.clip_mask)
		g_object_ref(src.values_.clip_mask);
}

Gdk::GCValues::~GCValues()
{
	unset_tile();
	unset_stipple();
	unset_clip_mask();
}

Gdk::GCValues& 
Gdk::GCValues::operator=(const GCValues& src)
{
	if (src.tile_set() && src.values_.tile)
		g_object_ref(src.values_.tile);

	if (src.stipple_set() && src.values_.stipple)
		g_object_ref(src.values_.stipple);

	if (src.clip_mask_set() && src.values_.clip_mask)
		g_object_ref(src.values_.clip_mask);

	unset_tile();
	unset_stipple();
	unset_clip_mask();

	values_ = src.values_;
	mask_ = src.mask_;
	return *this;
}

GdkGCValues*
Gdk::GCValues::gdk_gc_values() const
{
	return const_cast<GdkGCValues*>(&values_);
}

GdkGCValuesMask 
Gdk::GCValues::gdk_gc_values_mask() const
{
	return (GdkGCValuesMask)mask_;
}

Gdk::Color 
Gdk::GCValues::foreground() const
{
	return Color(const_cast<GdkColor*>(&values_.foreground));
}

bool 
Gdk::GCValues::foreground_set() const
{
	return (mask_ & GDK_GC_FOREGROUND) != 0;
}

Gdk::Color 
Gdk::GCValues::background() const
{
	return Color(const_cast<GdkColor*>(&values_.background));
}

bool 
Gdk::GCValues::background_set() const
{
	return (mask_ & GDK_GC_BACKGROUND) != 0;
}

Gdk::Function 
Gdk::GCValues::function() const
{
	return (Function)values_.function;
}

bool 
Gdk::GCValues::function_set() const
{
	return (mask_ & GDK_GC_FUNCTION) != 0;
}

Gdk::Fill 
Gdk::GCValues::fill() const
{
	return (Fill)values_.fill;
}

bool 
Gdk::GCValues::fill_set() const
{
	return (mask_ & GDK_GC_FILL) != 0;
}
	
Gdk::Pixmap*
Gdk::GCValues::tile() const
{
	return tile_set() ? G::Object::wrap<Pixmap>(values_.tile) : 0;
}

bool 
Gdk::GCValues::tile_set() const
{
	return (mask_ & GDK_GC_TILE) != 0;
}

Gdk::Bitmap* 
Gdk::GCValues::stipple() const
{
	return stipple_set() ? G::Object::wrap<Bitmap>(values_.stipple) : 0;
}

bool 
Gdk::GCValues::stipple_set () const
{
	return (mask_ & GDK_GC_STIPPLE) != 0;
}

Gdk::Bitmap* 
Gdk::GCValues::clip_mask() const
{
	return clip_mask_set() ? G::Object::wrap<Bitmap>(values_.clip_mask) : 0;
}

bool 
Gdk::GCValues::clip_mask_set() const
{
	return (mask_ & GDK_GC_CLIP_MASK) != 0;
}

Gdk::SubwindowMode 
Gdk::GCValues::subwindow_mode() const
{
	return (SubwindowMode)values_.subwindow_mode;
}

bool 
Gdk::GCValues::subwindow_mode_set() const
{
	return (mask_ & GDK_GC_SUBWINDOW) != 0;
}

int 
Gdk::GCValues::ts_x_origin() const
{
	return values_.ts_x_origin;
}

bool 
Gdk::GCValues::ts_x_origin_set() const
{
	return (mask_ & GDK_GC_TS_X_ORIGIN) != 0;
}

int 
Gdk::GCValues::ts_y_origin() const
{
	return values_.ts_y_origin;
}

bool 
Gdk::GCValues::ts_y_origin_set() const
{
	return (mask_ & GDK_GC_TS_Y_ORIGIN) != 0;
}

int 
Gdk::GCValues::clip_x_origin() const
{
	return values_.clip_x_origin;
}

bool 
Gdk::GCValues::clip_x_origin_set() const
{
	return (mask_ & GDK_GC_CLIP_X_ORIGIN) != 0;
}

int 
Gdk::GCValues::clip_y_origin() const
{
	return values_.clip_y_origin;
}

bool 
Gdk::GCValues::clip_y_origin_set () const
{
	return (mask_ & GDK_GC_CLIP_Y_ORIGIN) != 0;
}

bool 
Gdk::GCValues::graphics_exposures() const
{
	return values_.graphics_exposures;
}

bool 
Gdk::GCValues::graphics_exposures_set() const
{
	return (mask_ & GDK_GC_EXPOSURES) != 0;
}

int 
Gdk::GCValues::line_width() const
{
	return values_.line_width;
}

bool 
Gdk::GCValues::line_width_set() const
{
	return (mask_ & GDK_GC_LINE_WIDTH) != 0;
}

Gdk::LineStyle 
Gdk::GCValues::line_style() const
{
	return (LineStyle)values_.line_style;
}

bool 
Gdk::GCValues::line_style_set() const
{
	return (mask_ & GDK_GC_LINE_STYLE) != 0;
}

Gdk::CapStyle 
Gdk::GCValues::cap_style () const
{
	return (CapStyle)values_.cap_style;
}

bool 
Gdk::GCValues::cap_style_set() const
{
	return (mask_ & GDK_GC_CAP_STYLE) != 0;
}

Gdk::JoinStyle 
Gdk::GCValues::join_style() const
{
	return (JoinStyle)values_.join_style;
}

bool
Gdk::GCValues::join_style_set() const
{
	return (mask_ & GDK_GC_JOIN_STYLE) != 0;
}

void
Gdk::GCValues::reset()
{
	unset_tile();
	unset_stipple();
	unset_clip_mask();
	mask_ = 0;
}

void
Gdk::GCValues::set_foreground(const Color& color)
{
	values_.foreground = *color.gdk_color();
	mask_ |= GDK_GC_FOREGROUND;
}

void
Gdk::GCValues::unset_foreground()
{
	mask_ &= ~GDK_GC_FOREGROUND;
}

void
Gdk::GCValues::set_background(const Color& color)
{
	values_.background = *color.gdk_color();
	mask_ |= GDK_GC_BACKGROUND;
}

void
Gdk::GCValues::unset_background()
{
	mask_ &= ~GDK_GC_BACKGROUND;
}

void
Gdk::GCValues::set_function(Function function)
{
	values_.function = (GdkFunction)function;
	mask_ |= GDK_GC_FUNCTION;
}

void
Gdk::GCValues::unset_function()
{
	mask_ &= ~GDK_GC_FUNCTION;
}

void
Gdk::GCValues::set_fill(Fill fill)
{
	values_.fill = (GdkFill)fill;
	mask_ |= GDK_GC_FILL;
}

void
Gdk::GCValues::unset_fill()
{
	mask_ &= ~GDK_GC_FILL;
}

void
Gdk::GCValues::set_tile(Pixmap& tile)
{
	unset_tile();
	values_.tile = tile.gdk_pixmap();
	tile.ref();
	mask_ |= GDK_GC_TILE;
}

void
Gdk::GCValues::unset_tile()
{
	if (tile_set())
	{
		g_object_unref(values_.tile);
		mask_ &= ~GDK_GC_TILE;
		values_.tile = 0;
	}
}

void
Gdk::GCValues::set_stipple(Bitmap& stipple)
{
	unset_stipple();
	values_.stipple = stipple.gdk_pixmap();
	stipple.ref();
	mask_ |= GDK_GC_STIPPLE;
}

void
Gdk::GCValues::unset_stipple()
{
	if (stipple_set())
	{
		g_object_unref(values_.stipple);
		mask_ &= ~GDK_GC_STIPPLE;
		values_.stipple = 0;
	}
}

void
Gdk::GCValues::set_clip_mask(Bitmap& mask)
{
	unset_clip_mask();
	values_.clip_mask = mask.gdk_pixmap();
	mask.ref();
	mask_ |= GDK_GC_CLIP_MASK;
}

void
Gdk::GCValues::unset_clip_mask()
{
	if (clip_mask_set())
	{
		g_object_unref(values_.clip_mask);
		mask_ &= ~GDK_GC_CLIP_MASK;
		values_.clip_mask = 0;
	}
}

void
Gdk::GCValues::set_subwindow_mode(SubwindowMode mode)
{
	values_.subwindow_mode = (GdkSubwindowMode)mode;
	mask_ |= GDK_GC_SUBWINDOW;
}

void
Gdk::GCValues::unset_subwindow_mode()
{
	mask_ &= ~GDK_GC_SUBWINDOW;
}

void
Gdk::GCValues::set_ts_origin(int x_origin, int y_origin)
{
	set_ts_x_origin(x_origin);
	set_ts_y_origin(y_origin);
}

void
Gdk::GCValues::set_ts_x_origin(int origin)
{
	values_.ts_x_origin = origin;
	mask_ |= GDK_GC_TS_X_ORIGIN;
}

void
Gdk::GCValues::unset_ts_x_origin()
{
	mask_ &= ~GDK_GC_TS_X_ORIGIN;
}

void
Gdk::GCValues::set_ts_y_origin(int origin)
{
	values_.ts_y_origin = origin;
	mask_ |= GDK_GC_TS_Y_ORIGIN;
}

void
Gdk::GCValues::unset_ts_y_origin()
{
	mask_ &= ~GDK_GC_TS_Y_ORIGIN;
}

void
Gdk::GCValues::set_clip_origin(int x_origin, int y_origin)
{
	set_clip_x_origin(x_origin);
	set_clip_y_origin(y_origin);
}

void
Gdk::GCValues::set_clip_x_origin(int origin)
{
	values_.clip_x_origin = origin;
	mask_ |= GDK_GC_CLIP_X_ORIGIN;
}

void
Gdk::GCValues::unset_clip_x_origin()
{
	mask_ &= ~GDK_GC_CLIP_X_ORIGIN;
}

void
Gdk::GCValues::set_clip_y_origin(int origin)
{
	values_.clip_y_origin = origin;
	mask_ |= GDK_GC_CLIP_Y_ORIGIN;
}

void
Gdk::GCValues::unset_clip_y_origin()
{
	mask_ &= ~GDK_GC_CLIP_Y_ORIGIN;
}

void
Gdk::GCValues::set_graphics_exposures(bool exposures)
{
	values_.graphics_exposures = exposures;
	mask_ |= GDK_GC_EXPOSURES;
}

void
Gdk::GCValues::unset_graphics_exposures()
{
	mask_ &= ~GDK_GC_EXPOSURES;
}

void
Gdk::GCValues::set_line_width(int width)
{
	values_.line_width = width;
	mask_ |= GDK_GC_LINE_WIDTH;
}

void
Gdk::GCValues::unset_line_width()
{
	mask_ &= ~GDK_GC_LINE_WIDTH;
}

void
Gdk::GCValues::set_line_style(LineStyle style)
{
	values_.line_style = (GdkLineStyle)style;
	mask_ |= GDK_GC_LINE_STYLE;
}

void
Gdk::GCValues::unset_line_style()
{
	mask_ &= ~GDK_GC_LINE_STYLE;
}

void
Gdk::GCValues::set_cap_style(CapStyle style)
{
	values_.cap_style = (GdkCapStyle)style;
	mask_ |= GDK_GC_CAP_STYLE;
}

void
Gdk::GCValues::unset_cap_style()
{
	mask_ &= ~GDK_GC_CAP_STYLE;
}

void
Gdk::GCValues::set_join_style(JoinStyle style)
{
	values_.join_style = (GdkJoinStyle)style;
	mask_ |= GDK_GC_JOIN_STYLE;
}

void
Gdk::GCValues::unset_join_style()
{
	mask_ &= ~GDK_GC_JOIN_STYLE;
}

/*  Gdk::GC
 */

Gdk::GC::GC()
: G::Object((GObject*)GCClass::create())
{
}

Gdk::GC::GC(GdkGC *gc, bool reference)
: G::Object((GObject*)gc, reference)
{
}

Gdk::GC::GC(Drawable& drawable)
: G::Object((GObject*)gdk_gc_new(drawable.gdk_drawable()))
{
}

Gdk::GC::GC(Drawable& drawable, const GCValues& values)
: G::Object((GObject*)gdk_gc_new_with_values(drawable.gdk_drawable(), values.gdk_gc_values(), values.gdk_gc_values_mask()))
{
}

Gdk::GC::~GC()
{
}

GdkGCClass*
Gdk::GC::gdk_gc_class() const 
{
	return get_class<GdkGCClass>(); 
}

Gdk::GC::operator GdkGC* () const 
{ 
	return this ? gdk_gc() : 0; 
}
	
int
Gdk::GC::clip_x_origin() const
{
	return gdk_gc()->clip_x_origin;
}

int
Gdk::GC::clip_y_origin() const
{
	return gdk_gc()->clip_y_origin;
}

int
Gdk::GC::ts_x_origin() const
{
	return gdk_gc()->ts_x_origin;
}

int
Gdk::GC::ts_y_origin() const
{
	return gdk_gc()->ts_y_origin;
}

void
Gdk::GC::get_values(GCValues& values) const
{
	values.unset_clip_mask();
	values.unset_stipple();
	values.unset_tile();

	GdkGCValues *tmp_values = values.gdk_gc_values();
	gdk_gc_get_values(gdk_gc(), tmp_values);

	values.mask_ = GDK_GC_FOREGROUND | GDK_GC_BACKGROUND | GDK_GC_FONT |
	               GDK_GC_FUNCTION | GDK_GC_FILL | GDK_GC_SUBWINDOW |
	               GDK_GC_TS_X_ORIGIN | GDK_GC_TS_Y_ORIGIN | GDK_GC_CLIP_X_ORIGIN |
	               GDK_GC_CLIP_Y_ORIGIN | GDK_GC_EXPOSURES | GDK_GC_LINE_WIDTH |
	               GDK_GC_LINE_STYLE | GDK_GC_CAP_STYLE | GDK_GC_JOIN_STYLE;

	if (tmp_values->stipple)
	{
		g_object_ref(tmp_values->stipple);
		values.mask_ |= GDK_GC_STIPPLE;
    	}

	if (tmp_values->tile)
	{
		g_object_ref(tmp_values->tile);
		values.mask_ |= GDK_GC_TILE;
    	}

	if (tmp_values->clip_mask)
	{
		g_object_ref(tmp_values->clip_mask);
		values.mask_ |= GDK_GC_CLIP_MASK;
	}
}

Gdk::Colormap*
Gdk::GC::get_colormap() const
{
	GdkColormap *colormap = gdk_gc_get_colormap(gdk_gc());
	return colormap ? G::Object::wrap<Colormap>(colormap) : 0;
}

Gdk::Screen* 
Gdk::GC::get_screen() const
{
	return G::Object::wrap<Screen>(gdk_gc_get_screen(gdk_gc()));
}

void
Gdk::GC::set_values(const GCValues& values)
{
	gdk_gc_set_values(gdk_gc(), values.gdk_gc_values(), values.gdk_gc_values_mask());
}

void 
Gdk::GC::set_foreground(const Color& color)
{
	gdk_gc_set_foreground(gdk_gc(), color.gdk_color());
}

void
Gdk::GC::set_background(const Color& color)
{
	gdk_gc_set_background(gdk_gc(), color.gdk_color());
}
 
void
Gdk::GC::set_function(Function function)
{
	gdk_gc_set_function(gdk_gc(), (GdkFunction)function);
}

void 
Gdk::GC::set_fill(Fill fill)
{
	gdk_gc_set_fill(gdk_gc(), (GdkFill)fill);
}

void
Gdk::GC::set_tile(Pixmap& tile)
{
	gdk_gc_set_tile(gdk_gc(), tile.gdk_pixmap());
}

void
Gdk::GC::set_stipple(Pixmap& stipple)
{
	gdk_gc_set_stipple(gdk_gc(), stipple.gdk_pixmap());
}

void
Gdk::GC::set_ts_origin(int x, int y)
{
	gdk_gc_set_ts_origin(gdk_gc(), x, y);
}

void 
Gdk::GC::set_ts_origin(const Point& point)
{
	set_ts_origin(point.x(), point.y());
}

void
Gdk::GC::set_clip_origin(int x, int y)
{
	gdk_gc_set_clip_origin(gdk_gc(), x, y);
}

void
Gdk::GC::set_clip_origin(const Point& point)
{
	set_clip_origin(point.x(), point.y());
}

void 
Gdk::GC::set_clip_mask(Bitmap& mask)
{
	gdk_gc_set_clip_mask(gdk_gc(), mask.gdk_pixmap());
}

void
Gdk::GC::set_clip(int x, int y, int width, int height)
{
	Rectangle rectangle(x, y, width, height);
	set_clip(rectangle);
}		

void 
Gdk::GC::set_clip(const Rectangle& rectangle)
{
	gdk_gc_set_clip_rectangle(gdk_gc(), rectangle.gdk_rectangle());
}

void 
Gdk::GC::set_clip(const Region& region)
{
	gdk_gc_set_clip_region(gdk_gc(), region.gdk_region());
}

void 
Gdk::GC::set_subwindow(SubwindowMode mode)
{
	gdk_gc_set_subwindow(gdk_gc(), (GdkSubwindowMode)mode);
}

void 
Gdk::GC::set_exposures(bool exposures)
{
	gdk_gc_set_exposures(gdk_gc(), exposures);
}

void
Gdk::GC::set_line_width(int line_width)
{
	GdkGCValues values;
	values.line_width = line_width;
	gdk_gc_set_values(gdk_gc(), &values, GDK_GC_JOIN_STYLE);
}

void
Gdk::GC::set_line_style(LineStyle line_style)
{
	GdkGCValues values;
	values.line_style = (GdkLineStyle)line_style;
	gdk_gc_set_values(gdk_gc(), &values, GDK_GC_JOIN_STYLE);
}

void
Gdk::GC::set_cap_style(CapStyle cap_style)
{
	GdkGCValues values;
	values.cap_style = (GdkCapStyle)cap_style;
	gdk_gc_set_values(gdk_gc(), &values, GDK_GC_JOIN_STYLE);
}

void
Gdk::GC::set_join_style(JoinStyle join_style)
{
	GdkGCValues values;
	values.join_style = (GdkJoinStyle)join_style;
	gdk_gc_set_values(gdk_gc(), &values, GDK_GC_JOIN_STYLE);
}

void
Gdk::GC::set_line_attributes(int line_width, LineStyle line_style, CapStyle cap_style, JoinStyle join_style)
{
	gdk_gc_set_line_attributes(gdk_gc(), line_width, (GdkLineStyle)line_style, (GdkCapStyle)cap_style, (GdkJoinStyle)join_style);
}

void
Gdk::GC::set_dashes(int dash_offset, const std::vector<gint8>& dash_list)
{
	gdk_gc_set_dashes(gdk_gc(), dash_offset, const_cast<gint8*>(&dash_list[0]), dash_list.size());
}

void 
Gdk::GC::offset(int x_offset, int y_offset)
{
	gdk_gc_offset(gdk_gc(), x_offset, y_offset);
}

void 
Gdk::GC::set_colormap(Colormap *colormap)
{
	gdk_gc_set_colormap(gdk_gc(), colormap->gdk_colormap());
}

void
Gdk::GC::set_rgb_fg_color(const Color& color)
{
	return gdk_gc_set_rgb_fg_color(gdk_gc(), color.gdk_color());
}

void
Gdk::GC::set_rgb_bg_color(const Color& color)
{
	return gdk_gc_set_rgb_bg_color(gdk_gc(), color.gdk_color());
}

/*  Gdk::GCClass
 */

void
Gdk::GCClass::init(GdkGCClass *g_class)
{
	G::ObjectClass::init((GObjectClass*)g_class);
	g_class->get_values = &get_values_proxy;
	g_class->set_values = &set_values_proxy;
	g_class->set_dashes = &set_dashes_proxy;
}

GType
Gdk::GCClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GDK_TYPE_GC, (GClassInitFunc)init);
	}
	return type;
}

void*
Gdk::GCClass::create()
{
	return g_object_new(get_type(), 0);
}

void
Gdk::GCClass::get_values_proxy(GdkGC *gc, GdkGCValues *values)
{
	GC *tmp_gc = G::Object::pointer<GC>(gc);
	if (tmp_gc)
		tmp_gc->do_get_values(values);
	else
	{
		GdkGCClass *g_class = G::TypeInstance::class_peek_parent<GdkGCClass>(GDK_GC_GET_CLASS(gc));
		if (g_class->get_values)
			g_class->get_values(gc, values);
	}
}

void
Gdk::GCClass::set_values_proxy(GdkGC *gc, GdkGCValues *values, GdkGCValuesMask mask)
{
	GC *tmp_gc = G::Object::pointer<GC>(gc);
	if (tmp_gc)
		tmp_gc->do_set_values(values, mask);
	else
	{
		GdkGCClass *g_class = G::TypeInstance::class_peek_parent<GdkGCClass>(GDK_GC_GET_CLASS(gc));
		if (g_class->set_values)
			g_class->set_values(gc, values, mask);
	}
}

void
Gdk::GCClass::set_dashes_proxy(GdkGC *gc, gint dash_offset, gint8 dash_list[], gint n)
{
	GC *tmp_gc = G::Object::pointer<GC>(gc);
	if (tmp_gc)
		tmp_gc->do_set_dashes(dash_offset, dash_list, n);
	else
	{
		GdkGCClass *g_class = G::TypeInstance::class_peek_parent<GdkGCClass>(GDK_GC_GET_CLASS(gc));
		if (g_class->set_dashes)
			g_class->set_dashes(gc, dash_offset, dash_list, n);
	}
}

/*  Overridable GdkGC methods
 */

void
Gdk::GC::do_get_values(GdkGCValues *values)
{
	GdkGCClass *g_class = class_peek_parent<GdkGCClass>(gdk_gc_class());
	if (g_class->get_values)
		g_class->get_values(gdk_gc(), values);
}

void
Gdk::GC::do_set_values(GdkGCValues *values, GdkGCValuesMask mask)
{
	GdkGCClass *g_class = class_peek_parent<GdkGCClass>(gdk_gc_class());
	if (g_class->set_values)
		g_class->set_values(gdk_gc(), values, mask);
}

void
Gdk::GC::do_set_dashes(int dash_offset, gint8 dash_list[], int n)
{
	GdkGCClass *g_class = class_peek_parent<GdkGCClass>(gdk_gc_class());
	if (g_class->set_dashes)
		g_class->set_dashes(gdk_gc(), dash_offset, dash_list, n);
}

