/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  treeselection.cc - GtkTreeSelection C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "treeview.h"
#include "private/treeselection_p.h"
#include "treemodel.h"

using namespace Inti;

/*  Gtk::TreeSelection
 */

Gtk::TreeSelection::TreeSelection(GtkTreeSelection *tree_selection, bool reference)
: G::Object((GObject*)tree_selection, reference)
{
}

Gtk::TreeSelection::~TreeSelection()
{
}
	
GtkTreeSelectionClass*
Gtk::TreeSelection::gtk_tree_selection_class() const 
{ 
	return get_class<GtkTreeSelectionClass>(); 
}
	
Gtk::TreeSelection::operator GtkTreeSelection* () const 
{
	return this ? gtk_tree_selection() : 0; 
}

Gtk::SelectionMode
Gtk::TreeSelection::get_mode() const
{
	return (SelectionMode)gtk_tree_selection_get_mode(gtk_tree_selection());
}

Gtk::TreeView*
Gtk::TreeSelection::get_tree_view() const
{
	return G::Object::pointer<TreeView>(gtk_tree_selection_get_tree_view(gtk_tree_selection()));
}

bool
Gtk::TreeSelection::any_selected() const
{
	return gtk_tree_selection_get_selected(gtk_tree_selection(), 0, 0);
}

bool
Gtk::TreeSelection::get_selected(TreeIter *iter, TreeModel **model) const
{
	GType type = gtk_tree_selection()->type;
	g_return_val_if_fail(type == GTK_SELECTION_SINGLE || type == GTK_SELECTION_BROWSE, false);

	GtkTreeModel *tmp_model = 0;
	bool result = gtk_tree_selection_get_selected(gtk_tree_selection(), &tmp_model, *iter);
	if (model && tmp_model)
	{
		G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(tmp_model));
		*model = dynamic_cast<Gtk::TreeModel*>(object);
	}
	return result;
}

bool
Gtk::TreeSelection::get_selected_rows(std::vector<Pointer<TreePath> >& rows, TreeModel **model)
{
	g_return_val_if_fail(rows.empty(), false);

	GtkTreeModel *tmp_model = 0;
	GList *first = gtk_tree_selection_get_selected_rows(gtk_tree_selection(), &tmp_model);
	GList *next = first;

	while (next != 0)
	{
		Pointer<TreePath> tmp_row(G::Boxed::wrap<TreePath>(GTK_TYPE_TREE_PATH, (GtkTreePath*)next->data, false));
		rows.push_back(tmp_row);
		next = g_list_next(next);
	}

	if (model && tmp_model)
	{
		G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(tmp_model));
		*model = dynamic_cast<TreeModel*>(object);
	}

	g_list_free(first);
	return !rows.empty();
}

int 
Gtk::TreeSelection::count_selected_rows() const
{
	return gtk_tree_selection_count_selected_rows(gtk_tree_selection());
}

bool
Gtk::TreeSelection::path_is_selected(const TreePath& path) const
{
	return gtk_tree_selection_path_is_selected(gtk_tree_selection(), path.gtk_tree_path());
}

bool 
Gtk::TreeSelection::iter_is_selected(const TreeIter& iter) const
{
	return gtk_tree_selection_iter_is_selected(gtk_tree_selection(), iter.gtk_tree_iter());
}

void
Gtk::TreeSelection::set_mode(SelectionMode type)
{
	gtk_tree_selection_set_mode(gtk_tree_selection(), (GtkSelectionMode)type);
}

namespace { // TreeSelection slot callbacks

gboolean select_slot_callback(GtkTreeSelection*, GtkTreeModel *model, GtkTreePath *path, gboolean path_currently_selected, gpointer data)
{
	Gtk::TreeSelection::SelectSlot *slot = (Gtk::TreeSelection::SelectSlot*)data;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(model));
	Gtk::TreeModel *tmp_model = dynamic_cast<Gtk::TreeModel*>(object);
	Gtk::TreePath tmp_path(path);
	return slot->call(tmp_model, tmp_path, path_currently_selected);
}

void foreach_slot_calback(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	Gtk::TreeSelection::ForeachSlot *slot = (Gtk::TreeSelection::ForeachSlot*)data;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(model));
	Gtk::TreeModel *tmp_model = dynamic_cast<Gtk::TreeModel*>(object);
	Gtk::TreePath tmp_path(path);
	Gtk::TreeIter tmp_iter(iter);
	slot->call(tmp_model, tmp_path, tmp_iter);
}

} // TreeSelection slot callbacks

void
Gtk::TreeSelection::set_select_function(const SelectSlot *select)
{
	gtk_tree_selection_set_select_function(gtk_tree_selection(), &select_slot_callback, (void*)select, 0);
}

void
Gtk::TreeSelection::selected_foreach(const ForeachSlot *each)
{
	gtk_tree_selection_selected_foreach(gtk_tree_selection(), &foreach_slot_calback, (void*)each);
}

void
Gtk::TreeSelection::select_path(const TreePath& path)
{
	gtk_tree_selection_select_path(gtk_tree_selection(), path.gtk_tree_path());
}

void
Gtk::TreeSelection::unselect_path(const TreePath& path)
{
	gtk_tree_selection_unselect_path(gtk_tree_selection(), path.gtk_tree_path());
}

void
Gtk::TreeSelection::select_iter(const TreeIter& iter)
{
	gtk_tree_selection_select_iter(gtk_tree_selection(), iter.gtk_tree_iter());
}

void
Gtk::TreeSelection::unselect_iter(const TreeIter& iter)
{
	gtk_tree_selection_unselect_iter(gtk_tree_selection(), iter.gtk_tree_iter());
}

void
Gtk::TreeSelection::select_all()
{
	gtk_tree_selection_select_all(gtk_tree_selection());
}

void 
Gtk::TreeSelection::unselect_all()
{
	gtk_tree_selection_select_all(gtk_tree_selection());
}

void 
Gtk::TreeSelection::select_range(const TreePath& start_path, const TreePath& end_path)
{
	gtk_tree_selection_select_range(gtk_tree_selection(), start_path.gtk_tree_path(), end_path.gtk_tree_path());
}

void 
Gtk::TreeSelection::unselect_range(const TreePath& start_path, const TreePath& end_path)
{
	gtk_tree_selection_unselect_range(gtk_tree_selection(), start_path.gtk_tree_path(), end_path.gtk_tree_path());
}

/*  Gtk::TreeSelectionClass
 */

void
Gtk::TreeSelectionClass::init(GtkTreeSelectionClass *g_class)
{
	G::ObjectClass::init((GObjectClass*)g_class);
	g_class->changed = &changed_proxy;
}

GType
Gtk::TreeSelectionClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_TREE_SELECTION, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::TreeSelectionClass::create()
{
	return g_object_new(get_type(), 0);
}

void
Gtk::TreeSelectionClass::changed_proxy(GtkTreeSelection *selection)
{
	TreeSelection *tmp_selection = G::Object::pointer<TreeSelection>(selection);
	if (tmp_selection)
		tmp_selection->on_changed();
	else
	{
		GtkTreeSelectionClass *tmp_class = GTK_TREE_SELECTION_GET_CLASS(selection);
		GtkTreeSelectionClass *g_class = G::TypeInstance::class_peek_parent<GtkTreeSelectionClass>(tmp_class);
		if (g_class->changed)
			g_class->changed(selection);
	}
}

/*  Signal handlers
 */

void
Gtk::TreeSelection::on_changed()
{
	GtkTreeSelectionClass *g_class = class_peek_parent<GtkTreeSelectionClass>(gtk_tree_selection_class());
	if (g_class->changed)
		g_class->changed(gtk_tree_selection());
}

/*  Signals
 */

const Gtk::TreeSelection::ChangedSignalType Gtk::TreeSelection::changed_signal("changed");

