/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/pango/fontset.h
//! @brief A PangoFontset C++ wrapper interface.
//!
//! Provides Fontset, an object that represents a set of Font to use when rendering text..

#ifndef INTI_PANGO_FONTSET_H
#define INTI_PANGO_FONTSET_H

#ifndef INTI_G_OBJECT_H
#include <inti/glib/object.h>
#endif

#ifndef __PANGO_FONT_H__
#include <pango/pango-font.h>
#endif

#ifndef __PANGO_FONTSET_H__
#include <pango/pango-fontset.h>
#endif

#ifndef _CPP_VECTOR
#include <vector>
#endif

namespace Inti {

namespace Pango {

class Font;
class FontMetrics;

//! @class Fontset fontset.h inti/pango/fontset.h
//! @brief A PangoFontset C++ wrapper class.
//!
//! A Fontset represents a set of Font to use when rendering text. It is the result
//! of resolving a FontDescription against a particular Context. It has operations
//! for finding the component font for a particular Unicode character, and for
//! finding a composite set of metrics for the entire fontset.

class Fontset : public G::Object
{
	friend class G::Object;

	Fontset(const Fontset&);
	Fontset& operator=(const Fontset&);

protected:
//! @name Constructors
//! @{
	
	explicit Fontset(PangoFontset *fontset, bool reference = true);
	//!< Construct a new Fontset from an existing PangoFontset.
	//!< @param fontset A pointer to a PangoFontset.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>fontset</EM> can be a newly created PangoFontset or an existing
	//!< PangoFontset. (see G::Object::Object).

//! @}

public:
//! @name Constructors
//! @{

	virtual ~Fontset();
	//!< Destructor.

//! @}
//! @name Accessors
//! @{

	PangoFontset* pango_fontset() const { return (PangoFontset*)instance; }
	//!< Get a pointer to the PangoFontset structure.

	operator PangoFontset* () const;
	//!< Conversion operator; safely converts a Fontset to a PangoFontset pointer.

	Pointer<Font> get_font(unsigned int wc) const;
	//!< Returns the font in the fontset that contains the best glyph for the unicode character wc.
	//!< @param wc A unicode character.
	//!< @return A smart pointer to a Font.

	Pointer<FontMetrics> get_metrics() const;
	//!< Get overall metric information for the fonts in the fontset.
	//!< @return A smart pointer to a FontMetrics object.

//! @}
};

} // namespace Pango

} // namespace Inti

#endif // INTI_PANGO_FONTSET_H

