#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdkx.h>

#include "guiutils.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


#include "images/icon_browse_20x20.xpm"


/* GUI DND local callback data structure, this is used for passing
 * data for local DND callbacks such as GUIDNDDragMotionCB().   
 */
typedef struct {

	GdkDragAction	default_action_same,
			default_action;

} gui_dnd_cb_struct;

/* Last DND icon set widget, pixmap, and mask. Used by GUISetDNDIcon(). */
static GtkWidget *gui_last_dnd_widget = NULL;
static GdkPixmap *gui_last_dnd_pixmap = NULL;
static GdkBitmap *gui_last_dnd_mask = NULL;


/* Stored as object user data on any button or toggle button that we 
 * create. This is used for mapping or unmapping child widgets packed
 * into these buttons.
 */
typedef struct {

	GtkWidget *button;

	GtkWidget	*label_normal,
			*label_highlight,
			*label_pressed;		/* Or toggled. */

	GtkWidget	*pixmap_normal,
			*pixmap_highlight,
			*pixmap_pressed;	/* Or toggled. */

} gui_button_data_struct;

/* Tool bar data structure, passed as object user data on the tool bar's
 * main hbox.
 */
typedef struct {

	GtkWidget **button;
	int total_buttons;

        /* Size of buttons, coordinate values in structure are ignored. */
        GtkAllocation   size_all,
                        size_label_only,
                        size_pixmap_only;

} gui_tool_bar_data_struct;

/* Our tooltips group, note that this will never get deallocated but
 * isn't a big problem since we only need one and it'll be destroyed
 * when the application exits.
 */
static GtkTooltips *gui_tooltips = NULL;
/* Global record marking if tooltips are enabled or disabled. */
static gbool gui_tooltips_state;

/* Record of a window's icon and related resources passed to the WM. */
typedef struct {

	/* The window referancing these resources. */
	GdkWindow	*window;

	/* The icon window used by the WM. */
	GdkWindow	*icon_window;
	GdkPixmap	*icon_pixmap;	/* The background pixmap. */
	GdkBitmap	*icon_mask;	/* The mask (NULL if none). */

} gui_wm_icon_ref_struct;
static gui_wm_icon_ref_struct **wm_icon_ref = NULL;
static int total_wm_icon_refs = 0;



static char *GUIGetKeyName(int key);
static gboolean GUIDNDDragMotionCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y, guint t,
	gpointer data
);
static void GUIDNDDragEndCB(
	GtkWidget *widget, GdkDragContext *dc,
	gpointer data
);
static void GUIButtonDestroyCB(GtkObject *object, gpointer data);

static gint GUIBannerExposeCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void GUIBannerDestroyCB(GtkObject *object, gpointer data);
static void GUIComboActivateCB(GtkWidget *widget, gpointer data);
static void GUIComboDestroyCB(GtkObject *object, gpointer data);
static void GUIMenuActivateCB(gpointer data);
static void GUIMenuDestroyCB(GtkObject *object, gpointer data);
static void GUIPullOutDestroyCB(GtkObject *object, gpointer data);
static void GUIPullOutDrawHandleDrawCB(
	GtkWidget *widget, GdkRectangle *area, gpointer data
);
static gint GUIPullOutDrawHandleCB(
        GtkWidget *widget, GdkEvent *event, gpointer data
);
static gint GUIPullOutPullOutBtnCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static gint GUIPullOutCloseCB(GtkWidget *widget, GdkEvent *event, gpointer data);


void GUIResizeBuffer(
	gint bpp,
	const gpointer src,
	gint src_width, gint src_height, gint src_bpl,
	gpointer tar,
	gint tar_width, gint tar_height, gint tar_bpl
);
void GUIStyleDeallocUnref(GtkStyle *style);
void GUIRCStyleDeallocUnref(GtkRcStyle *rcstyle);

int GUICListGetCellGeometry(
	void *w, int column, int row,
	int *x, int *y, int *width, int *height
);
void GUIGetWindowRootPosition(
	void *w, int *x, int *y
);

void *GUICreateCursorFromData(
	u_int8_t **cursor_data,
	double hot_x, double hot_y,
	int width, int height
);
void *GUICreateMenuItemIcon(u_int8_t **icon);
void GUISetDNDIcon(
	GtkWidget *widget,		/* A DND source widget. */
	GdkColormap *colormap,
	GdkPixmap *pixmap,		/* Icon pixmap. */
	GdkBitmap *mask,		/* Icon mask. */
	int hot_x, int hot_y
);
static gint GUIWMIconListMatch(GdkWindow *window);
static void GUIWMIconListAdd(
	GdkWindow *w, GdkWindow *icon_window,
	GdkPixmap *icon_pixmap, GdkBitmap *icon_mask
);
static void GUIWMIconListRemove(gpointer data);
void GUISetWMIcon(GdkWindow *w, u_int8_t **icon);

static void GUICreateGlobalTooltips(void);
void GUISetWidgetTip(GtkWidget *w, const gchar *tip);
void GUIShowTipsNow(GtkWidget *w);
void GUISetGlobalTipsState(gbool state);

void GUIButtonChangeLayout(void *button, int show_pixmap, int show_label);
void *GUIButtonPixmapLabelHV(
	u_int8_t **icon, const char *label, void **label_rtn,
        int orient
);
void *GUIButtonPixmapLabelH(
        u_int8_t **icon, const char *label, void **label_rtn
);
void *GUIButtonPixmapLabelV(
        u_int8_t **icon, const char *label, void **label_rtn
);
void *GUIButtonPixmap(u_int8_t **icon);

void *GUIToggleButtonPixmapLabelHV(
        u_int8_t **icon, const char *label, void **label_rtn,
        int orient   
);
void *GUIToggleButtonPixmapLabelH(
        u_int8_t **icon, const char *label, void **label_rtn
);
void *GUIToggleButtonPixmapLabelV(
        u_int8_t **icon, const char *label, void **label_rtn
);
void *GUIToggleButtonPixmap(u_int8_t **icon);

void *GUIPromptBarOrBrowse(
        u_int8_t **icon, const char *label,
        void **label_rtn, void **entry_rtn, void **browse_rtn
);
void *GUIPromptBarWithBrowse(
        u_int8_t **icon, const char *label,
        void **label_rtn, void **entry_rtn, void **browse_rtn,
        void *browse_client_data, void (*browse_cb)(void *, void *)
);
void *GUIPromptBar(
        u_int8_t **icon, const char *label,
        void **label_rtn, void **entry_rtn
);

void GUIDNDSetSrc(
	void *w, void *drag_type, int ndrag_types,
	unsigned int actions, unsigned int buttons,
	void (*begin_cb)(GtkWidget *, GdkDragContext *, gpointer),
	void (*request_data_cb)(GtkWidget *, GdkDragContext *,
		GtkSelectionData *, guint, guint, gpointer),
	void (*delete_data_cb)(GtkWidget *, GdkDragContext *, gpointer),
	void (*end_cb)(GtkWidget *, GdkDragContext *, gpointer),
	gpointer client_data
);
void GUIDNDSetTar(
        void *w, void *drag_type, int ndrag_types,
        unsigned int actions, unsigned int default_action_same,
	unsigned int default_action,
	void (*recieved_data_cb)(GtkWidget *, GdkDragContext *,
		gint, gint, GtkSelectionData *, guint, guint,
		gpointer
	),
	gpointer client_data
);
void GUIDNDSetDragIcon(void *pixmap, void *mask);

GtkWidget *GUIBannerCreate(
	const gchar *label, const gchar *font_name,
	GdkColor color_fg, GdkColor color_bg,
	gint align,		/* One of GTK_JUSTIFY_*. */
	gbool expand
);
void *GUIComboCreate(
        const char *label,
        const char *text,       /* Initial text. */
        void *list,             /* Initial glist of items for combo list. */
        int max_items,          /* Maximum items allowed. */
        void **combo_rtn,	/* Combo widget return. */
        void *client_data,
        void (*func_cb)(GtkWidget *w, void *),
        void (*list_change_cb)(GtkWidget *, void *, GList *)
);
void GUIComboActivateValue(void *w, const char *value);
void GUIComboAddItem(void *w, const char *value);
void *GUIComboGetList(void *w);
void GUIComboSetList(void *w, void *list);
void GUIComboClearAll(void *w);

void *GUIMenuBarCreate(void **accel_group);
void *GUIMenuCreate(void);
void *GUIMenuItemCreate(
        void *menu, int type,	/* One of GUI_MENU_ITEM_TYPE_*. */
        void *accel_group,
        u_int8_t **icon, const char *label,
        int accel_key, unsigned int accel_mods,
        void **functional_widget_rtn,
        void *client_data,
        void (*func_cb)(GtkWidget *w, void *)
);
unsigned int GUISetMenuItemEnterCB(
        void *w,
        int (*enter_cb)(void *, void *, void *),
        void *client_data
);
void *GUIMenuAddToMenuBar(
        void *menu_bar, void *menu,
        const char *menu_bar_item_label,
        int align	/* One of GUI_MENU_BAR_ALIGN_*. */
);
void *GUIMenuAddToMenuBarPixmapH(
        void *menu_bar, void *menu,
        const char *menu_bar_item_label, const u_int8_t **icon,
        int align       /* One of GUI_MENU_BAR_ALIGN_*. */
);

void *GUIPullOutCreateH(
	void *parent_box,
        int homogeneous, int spacing,           /* Of client vbox. */
        int expand, int fill, int padding,      /* Of holder hbox. */
	int toplevel_width, int toplevel_height,
        void *pull_out_client_data,
        void (*pull_out_cb)(void *, void *),
        void *push_in_client_data,
        void (*push_in_cb)(void *, void *)
);
void *GUIPullOutGetToplevelWindow(  
        void *client_box,    
        int *x, int *y, int *width, int *height
);
void GUIPullOutPullOut(void *client_box);
void GUIPullOutPushIn(void *client_box); 


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *	Returns a statically allocated string containing the name
 *	of the given key.
 */
static char *GUIGetKeyName(int key)
{
	switch(key)
	{
          case GDK_BackSpace:
	    return("BackSpace");
          case GDK_Tab:
            return("Tab");
          case GDK_Linefeed:
            return("LineFeed");
          case GDK_Clear:
            return("Clear");
          case GDK_Return:
            return("Return");
          case GDK_Pause:
            return("Pause");
          case GDK_Scroll_Lock:
            return("ScrollLock");
          case GDK_Sys_Req:
            return("SysReq");
          case GDK_Escape:
            return("Escape");
          case GDK_Delete:
            return("Delete");
          case GDK_Home:
            return("Home");
          case GDK_Left:
            return("Left");
          case GDK_Right:
            return("Right");
          case GDK_Down:
            return("Down");
          case GDK_Up:
            return("Up");
          case GDK_Page_Up:
            return("PageUp");
          case GDK_Page_Down:
            return("PageDown");
          case GDK_End:
            return("End");
          case GDK_Begin:
            return("Begin");
          case GDK_Select:
            return("Select");
          case GDK_Print:
            return("Print");
          case GDK_Execute:
	    return("Execute");
          case GDK_Insert:
            return("Insert");
          case GDK_Undo:
            return("Undo");
          case GDK_Redo:
            return("Redo");
          case GDK_Menu:
            return("Menu");
	  case GDK_Find:
            return("Find");
          case GDK_Cancel:
            return("Cancel");
          case GDK_Help:
            return("Help");
          case GDK_Break:
            return("Break");
          case GDK_Num_Lock:
            return("NumLock");

          case GDK_KP_Space:
            return("KPSpace");
          case GDK_KP_Tab:
            return("KBTab");
          case GDK_KP_Enter:
            return("KPEnter");
          case GDK_KP_F1:
            return("KPF1");
          case GDK_KP_F2:
            return("KPF2");
          case GDK_KP_F3:
            return("KPF3");
          case GDK_KP_F4:
            return("KPF4");
          case GDK_KP_Home:
            return("KPHome");
          case GDK_KP_Left:
            return("KPLeft");
          case GDK_KP_Up:
            return("KPUp");
          case GDK_KP_Right:
            return("KPRight");
          case GDK_KP_Down:
            return("KPDown");
          case GDK_KP_Page_Up:
            return("KPPageUp");
          case GDK_KP_Page_Down:
            return("KPPageDown");
          case GDK_KP_End:
            return("KPEnd");
          case GDK_KP_Begin:
            return("KPBegin");
          case GDK_KP_Insert:
            return("KPInsert");
          case GDK_KP_Delete:
            return("KPDelete");
          case GDK_KP_Equal:
            return("KPEqual");
          case GDK_KP_Multiply:
            return("KPMultiply");
          case GDK_KP_Add:
            return("KPAdd");
          case GDK_KP_Separator:
            return("KPSeparator");
          case GDK_KP_Subtract:
            return("KPSubtract");
          case GDK_KP_Decimal:
            return("KPDecimal");
          case GDK_KP_Divide:
            return("KPDivide");
          case GDK_KP_0:
            return("KP0");
          case GDK_KP_1:
            return("KP1");
          case GDK_KP_2:
            return("KP2");
          case GDK_KP_3:
            return("KP3");
          case GDK_KP_4:
            return("KP4");
          case GDK_KP_5:
            return("KP5");
          case GDK_KP_6:
            return("KP6");
          case GDK_KP_7:
            return("KP7");
          case GDK_KP_8:
            return("KP8");
          case GDK_KP_9:
            return("KP9");

          case GDK_F1:
            return("F1");
          case GDK_F2:
            return("F2");
          case GDK_F3:
            return("F3");
          case GDK_F4:
            return("F4");
          case GDK_F5:
            return("F5");
          case GDK_F6:
            return("F6");
          case GDK_F7:
            return("F7");
          case GDK_F8:
            return("F8");
          case GDK_F9:
            return("F9");
          case GDK_F10:
            return("F10");
          case GDK_F11:
            return("F11");
          case GDK_F12:
            return("F12");
          case GDK_F13:
            return("F13");
          case GDK_F14:
            return("F14");
          case GDK_F15:
            return("F15");
          case GDK_F16:
            return("F16");
          case GDK_F17:
            return("F17");
          case GDK_F18:
            return("F18");
          case GDK_F19:
            return("F19");
          case GDK_F20:
            return("F20");
          case GDK_F21:
            return("F21");
          case GDK_F22:
            return("F22");
          case GDK_F23:
            return("F23");
          case GDK_F24:
            return("F24");
          case GDK_F25:
            return("F25");
          case GDK_F26:
            return("F26");
          case GDK_F27:
            return("F27");
          case GDK_F28:
            return("F28");
          case GDK_F29:
            return("F29");
          case GDK_F30:
            return("F30");

          case GDK_Shift_L:
            return("ShiftL");
          case GDK_Shift_R:
            return("ShiftR");
          case GDK_Control_L:
            return("ControlL");
          case GDK_Control_R:
            return("ControlR");
          case GDK_Caps_Lock:
            return("CapsLock");
          case GDK_Shift_Lock:
            return("ShiftLock");
          case GDK_Meta_L:
            return("MetaL");
          case GDK_Meta_R:
            return("MetaR");
          case GDK_Alt_L: 
            return("AltL");
          case GDK_Alt_R:
            return("AltR");
          case GDK_Super_L:
            return("SuperL");
          case GDK_Super_R: 
            return("SuperR");
          case GDK_Hyper_L: 
            return("HyperL");
          case GDK_Hyper_R: 
            return("HyperR");
	  default:
	    return("Unknown");
	}
}

/*
 *	DND drag motion callback, this function will basically
 *	handle the change of default action on the given destination
 *	widget.
 *
 *	The given data is assumed to be a gui_dnd_cb_struct *.
 */
static gboolean GUIDNDDragMotionCB(
        GtkWidget *widget, GdkDragContext *dc,
        gint x, gint y, guint t,   
        gpointer data
)
{
	gbool same;
	GtkWidget *source_widget;
	GdkDragAction action;
	gui_dnd_cb_struct *cb_data = (gui_dnd_cb_struct *)data;
	if((dc == NULL) || (cb_data == NULL))
	    return(FALSE);

	/* Get source widget and check if it is the same as the
	 * destination widget.
	 */
	source_widget = gtk_drag_get_source_widget(dc);
        same = ((source_widget == widget) ? TRUE : FALSE);

	/* Get default action from cb_data structure, if source and
	 * target are the same widget then use default_action_same
	 * otherwise use default_action.
	 */
	action = ((same) ?
	    cb_data->default_action_same : cb_data->default_action
	);


	/* Respond with default drag action (status). First we check
	 * the dc's list of actions. If the list only contains
	 * move or copy then we select just that, otherwise we return
	 * with our default suggested action.
	 * If no valid actions are listed then we respond with 0.
	 */

	/* Only move? */
	if(dc->actions == GDK_ACTION_MOVE)
	    gdk_drag_status(dc, GDK_ACTION_MOVE, t);
	/* Only copy? */
	else if(dc->actions == GDK_ACTION_COPY)
	    gdk_drag_status(dc, GDK_ACTION_COPY, t);
        /* Only link? */
        else if(dc->actions == GDK_ACTION_LINK)
            gdk_drag_status(dc, GDK_ACTION_LINK, t);
	/* Other action, check if listed in our actions list? */
        else if(dc->actions & action)
            gdk_drag_status(dc, action, t);
	/* All else respond with 0. */
	else
	    gdk_drag_status(dc, 0, t);

	/* Is given widget valid and sensitive? */
	if((widget == NULL) ? 0 :
	    (GTK_IS_WIDGET(widget) && GTK_WIDGET_SENSITIVE(widget))
	)
	{
	    /* Set widget into focus as needed. */
	    if(!GTK_WIDGET_HAS_FOCUS(widget))
		gtk_widget_grab_focus(widget);

	    /* Handle rest by widget type. */
	    /* Is widget a ctree? */
            if(GTK_IS_CTREE(widget))
            {
		/* Referance it as a clist. */
                GtkCList *clist = (GtkCList *)widget;
                gint row, column;

		/* Update flags as needed. */
		clist->flags |= GTK_CLIST_DRAW_DRAG_LINE;

		/* Calculate column row from x and y coordinates. */
                if(!gtk_clist_get_selection_info(
                    clist, x, y, &row, &column
                ))
                {
                    row = -1;   
                    column = -1;
                }

                /* Change in drag position? */   
                if(row != clist->focus_row)      
                { 
                    clist->focus_row = row;
                    gtk_widget_queue_draw(widget);
                }
            }
	    /* Is widget a clist? */
	    else if(GTK_IS_CLIST(widget))
	    {
		GtkCList *clist = (GtkCList *)widget;
		gint row, column;

                /* Update flags as needed. */
		clist->flags |= GTK_CLIST_DRAW_DRAG_LINE;

                /* Calculate column row from x and y coordinates. */
                if(!gtk_clist_get_selection_info(   
                    clist, x, y, &row, &column
                ))
                {
                    row = -1;
                    column = -1;
                }
		else
		{
                    row -= 1;   /* Need to offset row. */
		}

                /* Change in drag position? */
                if(row != clist->focus_row)
                {
		    clist->focus_row = row;
                    gtk_widget_queue_draw(widget);
                }
	    }
	}

	return(TRUE);
}

/*
 *      DND drag end callback, this function will reset the target
 *	widget as needed and clean up any additional resources.
 *
 *	The given data is always NULL for now.
 */
static void GUIDNDDragEndCB(
        GtkWidget *widget, GdkDragContext *dc,
        gpointer data
)
{
        /* Is given widget valid and sensitive? */
        if((widget == NULL) ? 0 :
            (GTK_IS_WIDGET(widget) && GTK_WIDGET_SENSITIVE(widget))
        )
        {
            /* Handle rest by widget type. */
            /* Is widget a ctree? */
            if(GTK_IS_CTREE(widget))
            {
		/* Referance as a clist. */
		GtkCList *clist = (GtkCList *)widget;

		clist->flags &= ~GTK_CLIST_DRAW_DRAG_LINE;
		gtk_widget_queue_draw(widget);
	    }
	    /* Is widget a clist? */
            else if(GTK_IS_CLIST(widget))
            {
                /* Referance as a clist. */
                GtkCList *clist = (GtkCList *)widget;

		clist->flags &= ~GTK_CLIST_DRAW_DRAG_LINE;
		gtk_widget_queue_draw(widget);
            }


	}

	return;
}


/*
 *	Button (or toggle button) destroy callback.
 */
static void GUIButtonDestroyCB(GtkObject *object, gpointer data)
{
	gui_button_data_struct *btn_data = (gui_button_data_struct *)data;
	if(btn_data == NULL)
	    return;

	/* No substructures to deallocate. */

	/* Deallocate structure itself. */
	free(btn_data);

	return;
}

/*
 *	Redraw banner widget callback.
 */
static gint GUIBannerExposeCB(
        GtkWidget *widget, GdkEvent *event, gpointer data
)
{
        GdkFont *font;    
	GtkStyle *style;
        GdkWindow *window;
	GtkWidget *w = widget;
	gpointer *cb_data = (gpointer *)data;
	const gchar *label;
	gint align;
        if((w == NULL) || (cb_data == NULL))
            return(TRUE);

        if(GTK_WIDGET_NO_WINDOW(w))
            return(TRUE);
	else
	    window = w->window;
        if(window == NULL)
            return(TRUE);

	/* Parse callback data. */
	label = (const gchar *)cb_data[0];
	align = (gint)cb_data[1];
	font = (GdkFont *)cb_data[2];

	/* Get style. */
	style = gtk_widget_get_style(w);
	if(style == NULL)
	    return(TRUE);

	/* Set backgrond color. */
        gdk_window_set_background(
            window, &style->bg[GTK_STATE_NORMAL]
        );
        gdk_window_clear(window);

	/* Got font and label? */
	if((font != NULL) && (label != NULL))
	{
            gint lbearing, rbearing, width, ascent, descent;

            gdk_string_extents(
                font, label,
                &lbearing, &rbearing, &width, &ascent, &descent
            );

	    switch(align)
	    {
	      case GTK_JUSTIFY_CENTER:
		gdk_draw_string(
		    (GdkDrawable *)window,
		    font, style->fg_gc[GTK_STATE_NORMAL],
		    (widget->allocation.width / 2) - (width / 2),
		    3 + ascent,
		    (const gchar *)label
		);
		break;

              default:
                gdk_draw_string(
                    (GdkDrawable *)window,
                    font, style->fg_gc[GTK_STATE_NORMAL],
                    3,
                    3 + ascent,
                    (const gchar *)label
                );
                break;
	    }
	}

        return(TRUE);
}

/*
 *	Banner widget destroy callback.
 */
static void GUIBannerDestroyCB(GtkObject *object, gpointer data)
{
	gchar *label;
	gint align;
	GdkFont *font;
	gpointer *cb_data = (gpointer *)data;
	if(cb_data == NULL)
	    return;

        /* Parse callback data. */
        label = (gchar *)cb_data[0];
        align = (gint)cb_data[1];
	font = (GdkFont *)cb_data[2];

	/* Free label. */
	cb_data[0] = NULL;
	g_free(label);
	label = NULL;

	/* Unref font. */
	cb_data[2] = NULL;
	if(font != NULL)
	{
	    gdk_font_unref(font);
	    font = NULL;
	}

	/* Free data pointer array. */
	g_free(cb_data);
}


/*
 *      Local combo box activate callback, this function will
 *      take the given data pointer as a dynamically allocated
 *      (void **).
 *
 *	First it will update the glist for the combo box and update
 *	the the combo's list. Then call (if not NULL) the specified
 *	activate callback function.
 */
static void GUIComboActivateCB(GtkWidget *widget, gpointer data)
{
	int max_items;
	GtkWidget *parent, *wlabel;
	GtkCombo *combo;
	GList *glist_in;
	void *client_data;
	void (*func_cb)(GtkWidget *, void *);
        void (*list_change_cb)(GtkWidget *, void *, GList *);
	char *new_value;
	void **cb_data = (void **)data;


	if((widget == NULL) ||
           (cb_data == NULL)
	)
	    return;

	/* Parse callback data, format is as follows;
	 * table	Parent table that holds this combo and label.
	 * label	Label (may be NULL).
	 * combo	This combo widget.
	 * GList	Contains list of strings in combo list.
	 * max_items	Max items allowed in combo list.
	 * client_data	Calling function set callback data.
	 * func_cb	Calling function set callback function.
	 * list_change_cb	Calling function set list change function.
	 * NULL
	 */
	parent = (GtkWidget *)(cb_data[0]);
	wlabel = (GtkWidget *)(cb_data[1]);
	combo = (GtkCombo *)(cb_data[2]);
	glist_in = (GList *)(cb_data[3]);
	max_items = (int)(cb_data[4]);
        client_data = (void *)(cb_data[5]);
        func_cb = cb_data[6];
	list_change_cb = cb_data[7];

	/* Give up if given combo widget is NULL. */
	if(combo == NULL)
	    return;

	/* Get new value string from combo's entry widget. */
	new_value = gtk_entry_get_text(GTK_ENTRY(combo->entry));

	/* Make a duplicate of the recieved value or NULL if it is not
	 * available.
	 */
	new_value = ((new_value != NULL) ?
		g_strdup((const char *)new_value) : NULL
	);

	/* Give up if new_value is NULL. */
	if(new_value == NULL)
	    return;

	/* Add the new value to the combo box list, this will only
	 * be added if value is not already in the list. Excess items
	 * may be truncated if adding this item would cause items to
	 * exceed the set max items on the combo box.
	 *
	 * List change callback will be called if the value was input
	 * to the combo box's list.
	 */
	GUIComboAddItem((void *)combo, new_value);

	/* Call activate function. */
	if(func_cb != NULL)
	{
	    func_cb(
		(GtkWidget *)combo,	/* Pass combo box as widget. */
                client_data		/* Client data. */
	    );
	}

	/* Free new_value which we duplicated. */
	free(new_value);

	return;
}

/*
 *      Local combo destroy callback, this function will free the given
 *      pointer data and it pointed to resources.
 */
static void GUIComboDestroyCB(GtkObject *object, gpointer data)
{
	GtkWidget *parent, *wlabel;
	GtkCombo *combo;
	GList *glist_in;
	int max_items;
        void *client_data;
        void (*func_cb)(GtkWidget *, void *);
        void (*list_change_cb)(GtkWidget *, void *, GList *);
        void **cb_data = (void **)data;


	if(cb_data == NULL)
	    return;

        /* Parse callback data, format is as follows;
         * table        Parent table that holds this combo and label.
         * label        Label (may be NULL).
         * combo        This combo widget.
         * GList        Contains list of strings in combo list.
         * max_items    Max items allowed in combo list.
         * client_data  Calling function set callback data.
         * func_cb      Calling function set callback function.
         * list_change_cb       Calling function set list change function.
         * NULL
         */
        parent = (GtkWidget *)(cb_data[0]);
        wlabel = (GtkWidget *)(cb_data[1]);
        combo = (GtkCombo *)(cb_data[2]);
        glist_in = (GList *)(cb_data[3]);
        max_items = (int)(cb_data[4]);
        client_data = (void *)(cb_data[5]);
        func_cb = cb_data[6];
        list_change_cb = cb_data[7];


	/* Do not call list change callback function on destroy. */

	/* Begin deallocating data referenced on callback data. */

        /* Deallocate glist. */
        if(glist_in != NULL)
        {
            g_list_foreach(glist_in, (GFunc)g_free, NULL);
            g_list_free(glist_in);
            glist_in = NULL;
        }

	/* Free callback data pointer array itself. */
	free(data);
}


/*
 *	Local menu item activate callback, this function will
 *	take the given data pointer as a dynamically allocated
 *	(void **).
 */
static void GUIMenuActivateCB(gpointer data)
{
	int argc = 3;
	GtkWidget *functional_widget;
	void *client_data;
	void (*func_cb)(GtkWidget *, gpointer);
	void **cb_data = (void **)data;
	if(cb_data == NULL)
	    return;


	/* Parse callback data, there should be four pointers in
	 * the (void **)cb_data array. The pointers point to as follows;
	 * w			functional_widget.
	 * client_data		client_data set by calling function.
	 * void *func_cb(GtkWidget *, gpointer)		Calling function's function.
	 * NULL
	 */
	functional_widget = ((argc > 0) ? cb_data[0] : NULL);
	client_data = ((argc > 1) ? cb_data[1] : NULL);
	func_cb = ((argc > 2) ? cb_data[2] : NULL);

	if(func_cb != NULL)
	    func_cb(
		functional_widget,
		client_data
	    );

	return;
}

/*
 *      Local menu item destroy callback, this function will
 *      free the given pointer data
 */
static void GUIMenuDestroyCB(GtkObject *object, gpointer data)
{
	free(data);
	return;
}

/*
 *	Local pull out widget destroy callback, this function will
 *	free the given pointer data and the destroy the toplevel
 *	GtkWindow.
 */
static void GUIPullOutDestroyCB(GtkObject *object, gpointer data)
{
	GtkWidget *client_hbox, *holder_hbox, *toplevel;
	void **cb_data = (void **)data;

	if(cb_data != NULL)
	{
            /* Format is as follows (12 arguments):
             *
             *  client_hbox
             *  holder_hbox
             *  holder_window
             *  holder_window_x
             *  holder_window_y
             *  holder_window_width
             *  holder_window_height
             *  in_place                (1 if true).
             *  pull_out_client_data
             *  pull_out_cb
             *  push_in_client_data
             *  push_in_cb
             */
            client_hbox = (GtkWidget *)cb_data[0];
            holder_hbox = (GtkWidget *)cb_data[1];
	    toplevel = (GtkWidget *)cb_data[2];

	    /* Get given object should be the holder hbox. */
	    if((gpointer)holder_hbox != (gpointer)object)
 	    {
		fprintf(stderr,
 "GUIPullOutDestroyCB(): Invalid object, not holder_hbox.\n"
		);
	    }

	    /* Destroy the toplevel window. */
	    if(toplevel != NULL)
	    {
		gtk_widget_destroy(toplevel);
		cb_data[2] = toplevel = NULL;
	    }
	}

	free(cb_data);
	return;
}

/*
 *	Pull out draw handle for signal "draw".
 */
static void GUIPullOutDrawHandleDrawCB(
        GtkWidget *widget, GdkRectangle *area, gpointer data  
)
{
        GdkWindow *window;
        GtkStyle *style_ptr;

        if((widget == NULL) ? 1 : GTK_WIDGET_NO_WINDOW(widget))
            return;

        window = widget->window;
        if(window == NULL)
            return;

        /* Get widget's style structure. */
        style_ptr = gtk_widget_get_style(widget);
        /* If none then fall back to global style. */
        if(style_ptr == NULL)
            style_ptr = gtk_widget_get_default_style();
        /* No style available, give up. */
        if(style_ptr == NULL)
            return;

        gdk_window_clear(window);
        gtk_draw_handle(
            style_ptr, window,
            GTK_STATE_NORMAL,           /* State type. */
            GTK_SHADOW_OUT,             /* Shadow type. */
            0, 0,
            widget->allocation.width,
            widget->allocation.height,
            GTK_ORIENTATION_HORIZONTAL
        );

        return;
}

/*
 *	Pull out draw handle callback for signal "expose_event".
 *
 *	This redraws the handle graphics on the given widget.
 */
static gint GUIPullOutDrawHandleCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	if((widget == NULL) ? 1 : GTK_WIDGET_NO_WINDOW(widget))
	    return(TRUE);
	gtk_widget_queue_draw(widget);
	return(TRUE);
}

/*
 *	Pull out button callback.
 */
static gint GUIPullOutPullOutBtnCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
        GtkWidget *client_hbox, *holder_hbox, *toplevel;
        void **cb_data = (void **)data;
	gint holder_window_width, holder_window_height;
        void *pull_out_client_data;
        void (*pull_out_cb)(void *, void *);

        if(cb_data != NULL)
        {
            /* Format is as follows (12 arguments):
             *
             *  client_hbox
             *  holder_hbox
             *  holder_window
             *  holder_window_x
             *  holder_window_y
             *  holder_window_width
             *  holder_window_height
             *  in_place                (1 if true).
             *  pull_out_client_data
             *  pull_out_cb
             *  push_in_client_data
             *  push_in_cb
             */
            client_hbox = (GtkWidget *)cb_data[0];
            holder_hbox = (GtkWidget *)cb_data[1];
            toplevel = (GtkWidget *)cb_data[2];

	    holder_window_width = (gint)cb_data[5];
	    holder_window_height = (gint)cb_data[6];

	    pull_out_client_data = cb_data[8];
	    pull_out_cb = cb_data[9];

	    /* Create toplevel window as needed. */
            if(toplevel == NULL)
            {
		GtkWidget *w;

		toplevel = w = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		cb_data[2] = (void *)w;
                gtk_widget_realize(w);
                gtk_widget_set_usize(
		    w,
		    (holder_window_width <= 0) ? -1 : holder_window_width,
		    (holder_window_height <= 0) ? -1 : holder_window_height
		);
		gtk_window_set_policy(
		    GTK_WINDOW(w),
		    TRUE, TRUE, FALSE
		);
		gtk_signal_connect(
		    GTK_OBJECT(w), "delete_event",
		    GTK_SIGNAL_FUNC(GUIPullOutCloseCB),
		    (gpointer)cb_data
		);
	    }

	    /* Reparent client_hbox to toplevel window. */
	    if((client_hbox != NULL) && (toplevel != NULL))
	    {
		gtk_widget_show(toplevel);

		if((GTK_WIDGET_FLAGS(client_hbox) & GTK_NO_REPARENT))
		    fprintf(stderr, "Cannot reparent.\n");
		else
		    gtk_widget_reparent(client_hbox, toplevel);
	    }

	    /* Hide holder hbox. */
	    if(holder_hbox != NULL)
		gtk_widget_hide(holder_hbox);

	    /* Mark in callback data that its been `pulled out'. */
	    cb_data[7] = (void *)0;

            /* Call pull out callback? */
            if(pull_out_cb != NULL)
                pull_out_cb(client_hbox, pull_out_client_data);
        }

        return(TRUE);
}

/*
 *	Close (push in) callback.
 */
static gint GUIPullOutCloseCB(GtkWidget *widget, GdkEvent *event, gpointer data)
{
        GtkWidget *client_hbox, *holder_hbox, *toplevel;
        void **cb_data = (void **)data;
        void *push_in_client_data;
        void (*push_in_cb)(void *, void *);

        if(cb_data != NULL)
        {
            /* Format is as follows (12 arguments):
             *
             *  client_hbox
             *  holder_hbox
             *  holder_window
             *  holder_window_x
             *  holder_window_y
             *  holder_window_width
             *  holder_window_height
             *  in_place                (1 if true).
             *  pull_out_client_data
             *  pull_out_cb
             *  push_in_client_data
             *  push_in_cb
             */
            client_hbox = (GtkWidget *)cb_data[0];
            holder_hbox = (GtkWidget *)cb_data[1];
            toplevel = (GtkWidget *)cb_data[2];

	    push_in_client_data = cb_data[10];
	    push_in_cb = cb_data[11];


            /* Reparent client_hbox to holder_hbox. */
            if((client_hbox != NULL) && (holder_hbox != NULL))
            {
		gtk_widget_show(holder_hbox);

                if((GTK_WIDGET_FLAGS(client_hbox) & GTK_NO_REPARENT))
                    fprintf(stderr, "Cannot reparent.\n");
                else
                    gtk_widget_reparent(client_hbox, holder_hbox);
            }

	    /* Hide toplevel. */
	    if(toplevel != NULL)
		gtk_widget_hide(toplevel);

            /* Mark in callback data that its been `pushed in'. */
            cb_data[7] = (void *)1;

	    /* Call push in callback? */
	    if(push_in_cb != NULL)
		push_in_cb(client_hbox, push_in_client_data);
        }

	return(TRUE);
}

/*
 *	Coppies the source image buffer to the target image buffer.
 */
void GUIResizeBuffer(
        gint bpp,
        const gpointer src,
        gint src_width, gint src_height, gint src_bpl,
        gpointer tar,
        gint tar_width, gint tar_height, gint tar_bpl
)
{
	
	guint8 *tar_buf = (guint8 *)tar;
	const guint8 *src_buf = (const guint8 *)src;
        u_int8_t *src_buf_ptr8, *tar_buf_ptr8;
        u_int16_t *src_buf_ptr16, *tar_buf_ptr16;
        u_int32_t *src_buf_ptr32, *tar_buf_ptr32;

	/* Current positions. */
	gint tar_x_col, tar_y_row;
	gint src_x_col, src_y_row;	/* In units of 256. */

	/* Increments, in units of 256. */
	gint src_dh_x_col_inc, src_dh_y_row_inc;
	gint src_dh_width, src_dh_height;


        if((tar_width < 1) || (tar_height < 1) ||
           (src_width < 1) || (src_height < 1) ||
           (tar_buf == NULL) || (src_buf == NULL) ||
           (bpp < 1)
        )
            return;

	/* Auto calculate bytes per lines as needed. */
	if(src_bpl <= 0)
	    src_bpl = src_width * bpp;
	if(tar_bpl <= 0)
	    tar_bpl = tar_width * bpp;

        /* 8 bits. */
        if(bpp == 1)
        {
            src_dh_x_col_inc = (gint)(src_width * 256) / tar_width;
            src_dh_y_row_inc = (gint)(src_height * 256) / tar_height;

            src_dh_width = src_width * 256;
            src_dh_height = src_height * 256;

            tar_x_col = 0;
            tar_y_row = 0;
            src_x_col = 0 * 256;
            src_y_row = 0 * 256;

            /* Begin copying source buffer to target buffer. */
            while((tar_y_row < tar_height) &&
                  (src_y_row < src_dh_height)
            )
            {
                /* Get buffer position. */
                src_buf_ptr8 = (u_int8_t *)(&src_buf[
                    ((src_y_row >> 8) * src_bpl) +
                    ((src_x_col >> 8) * bpp)
                ]);
                tar_buf_ptr8 = (u_int8_t *)(&tar_buf[
                    (tar_y_row * tar_bpl) +
                    (tar_x_col * bpp)
                ]);
                *tar_buf_ptr8 = *src_buf_ptr8;


                /* Increment colum positions. */
                tar_x_col += 1;
                src_x_col += src_dh_x_col_inc;

                /* Go to next line? */
                if((tar_x_col >= tar_width) ||
                   (src_x_col >= src_dh_width)
                )
                {
                    tar_x_col = 0;
                    src_x_col = 0;

                    tar_y_row += 1;
                    src_y_row += src_dh_y_row_inc;
                }
            }
        }
        /* 16 bits. */
        else if(bpp == 2)
        {
            src_dh_x_col_inc = (gint)(src_width * 256) / tar_width;
            src_dh_y_row_inc = (gint)(src_height * 256) / tar_height;

            src_dh_width = src_width * 256;
            src_dh_height = src_height * 256;

            tar_x_col = 0;
            tar_y_row = 0;
            src_x_col = 0 * 256;
            src_y_row = 0 * 256;

            /* Begin copying source buffer to target buffer. */
            while((tar_y_row < tar_height) &&
                  (src_y_row < src_dh_height)
            )
            {
                /* Get buffer position. */
                src_buf_ptr16 = (u_int16_t *)(&src_buf[
                    ((src_y_row >> 8) * src_bpl) +
                    ((src_x_col >> 8) * bpp)
                ]);
                tar_buf_ptr16 = (u_int16_t *)(&tar_buf[
                    (tar_y_row * tar_bpl) +
                    (tar_x_col * bpp)
                ]);
                *tar_buf_ptr16 = *src_buf_ptr16;


                /* Increment colum positions. */
                tar_x_col += 1;
                src_x_col += src_dh_x_col_inc;

                /* Go to next line? */
                if((tar_x_col >= tar_width) ||
                   (src_x_col >= src_dh_width)
                )
                {
                    tar_x_col = 0;
                    src_x_col = 0;

                    tar_y_row += 1;
                    src_y_row += src_dh_y_row_inc;
                }
            }
        }
        /* 32 bits. */
        else if(bpp == 4)
        {
            src_dh_x_col_inc = (gint)(src_width * 256) / tar_width;
            src_dh_y_row_inc = (gint)(src_height * 256) / tar_height;

            src_dh_width = src_width * 256;
            src_dh_height = src_height * 256;

            tar_x_col = 0;
            tar_y_row = 0;
            src_x_col = 0 * 256;
            src_y_row = 0 * 256;

            /* Begin copying source buffer to target buffer. */
            while((tar_y_row < tar_height) &&
                  (src_y_row < src_dh_height)
            )
            {
                /* Get buffer position. */
                src_buf_ptr32 = (u_int32_t *)(&src_buf[
                    ((src_y_row >> 8) * src_bpl) +
                    ((src_x_col >> 8) * bpp)
                ]);
                tar_buf_ptr32 = (u_int32_t *)(&tar_buf[
                    (tar_y_row * tar_bpl) +
                    (tar_x_col * bpp)
                ]);
                *tar_buf_ptr32 = *src_buf_ptr32;


                /* Increment colum positions. */
                tar_x_col += 1;
                src_x_col += src_dh_x_col_inc;

                /* Go to next line? */
                if((tar_x_col >= tar_width) ||
                   (src_x_col >= src_dh_width)
                )
                {
                    tar_x_col = 0;
                    src_x_col = 0;

                    tar_y_row += 1;
                    src_y_row += src_dh_y_row_inc;
                }
            }
        }
}


/*
 *      Deallocates and unrefs all resources on the given GtkStyle and
 *      detaches and unrefs the structure itself.
 */
void GUIStyleDeallocUnref(GtkStyle *style)
{
        GtkStyle *s = style;
        if(s != NULL)
        {
            gtk_style_detach(s);	/* Detach and unref style. */
        }
}

/*
 *	Deallocates and unrefs all resources on the given GtkRcStyle and
 *	unrefs the structure itself.
 */
void GUIRCStyleDeallocUnref(GtkRcStyle *rcstyle)
{
	GtkRcStyle *rcs = rcstyle;
	if(rcs != NULL)
	{
	    gint i;

	    g_free(rcs->name);
	    rcs->name = NULL;

	    g_free(rcs->font_name);
	    rcs->font_name = NULL;

	    g_free(rcs->fontset_name);
	    rcs->fontset_name = NULL;

	    for(i = 0; i < 5; i++)
	    {
		g_free(rcs->bg_pixmap_name[i]);
		rcs->bg_pixmap_name[i] = NULL;
	    }

	    gtk_rc_style_unref(rcs);
	}
}


/*
 *	Returns the geometry of the given cell if it is valid on
 *	the given clist w. Geometry is relative to the clist's 
 *	clist_window.
 *
 *	Returns non-zero on error.
 */
int GUICListGetCellGeometry(
        void *w, int column, int row,
        int *x, int *y, int *width, int *height
)
{
	GtkAdjustment *hadj, *vadj;
	GdkWindow *clist_window;
	GtkCList *clist;
	GList *glist;
	GtkCListRow *row_ptr;
	gint i, cx, cy, cwidth, cheight;
	GtkCListColumn *column_ptr;


	if(x != NULL)
	    (*x) = 0;
	if(y != NULL)
            (*y) = 0;
        if(width != NULL)
            (*width) = 0;
        if(height != NULL)
            (*height) = 0;


	if(w == NULL)
	    return(-1);

	if(!GTK_IS_CLIST((GtkWidget *)w))
	    return(-1);

	clist = GTK_CLIST((GtkWidget *)w);
	hadj = clist->hadjustment;
	vadj = clist->vadjustment;
	clist_window = clist->clist_window;
	if(clist_window == NULL)
	    return(-1);

	/* Given row and column in bounds? */
	if((column < 0) || (column >= clist->columns))
	    return(-1);
	if((row < 0) || (row >= clist->rows))
	    return(-1);

	/* Get cy and cheight. */
	cy = 0;
	cheight = 0;
	glist = clist->row_list;
	for(i = 0; glist != NULL; i++)
	{
	    row_ptr = (GtkCListRow *)glist->data;
	    if(row_ptr != NULL)
	    {
		GtkCell *cell_ptr = row_ptr->cell;

		if((cell_ptr == NULL) ? 0 : (cell_ptr->vertical > 0))
		    cheight = cell_ptr->vertical;
		else
		    cheight = clist->row_height;

                if(i == row)
                    break;

		cy += cheight;
	    }

	    glist = glist->next;
	}

	/* Get cx and cwidth. */
	cx = 0;
	cwidth = 0;
	if(clist->column != NULL)
	{
	    for(i = 0; i < clist->columns; i++)
	    {
		column_ptr = &clist->column[i];
		if(column_ptr == NULL)
		    continue;

		cwidth = column_ptr->width;

                if(i == column)
                    break;

		cx += cwidth;
	    }
	}

	/* Offset cx and cy with scroll adjustments. */
	if(hadj != NULL)
	    cx = cx - (hadj->value - hadj->lower);
        if(vadj != NULL)
            cy = cy - (vadj->value - vadj->lower);

	/* Update returns. */
        if(x != NULL)
            (*x) = cx;
        if(y != NULL)
            (*y) = cy;
        if(width != NULL)
            (*width) = cwidth;
        if(height != NULL)
            (*height) = cheight;

	return(0);
}

/*
 *	Returns the coordinate position for GdkWindow w relative to the
 *	root (desktop) window.
 */
void GUIGetWindowRootPosition(
        void *w, int *x, int *y
)
{
	GdkWindow *cur_w = (GdkWindow *)w;
	GdkWindow *root_w;
	int cx, cy;


	if(x != NULL)
	    (*x) = 0;
	if(y != NULL)
	    (*y) = 0;

        if(cur_w == NULL)
            return;

	root_w = (GdkWindow *)GDK_ROOT_PARENT();
        if(root_w == NULL)
            return;

	while(cur_w != root_w)
	{
	    gdk_window_get_position(cur_w, &cx, &cy);
	    if(x != NULL)
		(*x) = ((*x) + cx);
            if(y != NULL)
                (*y) = ((*y) + cy);
	    cur_w = gdk_window_get_parent(cur_w);
	}

	return;
}

/*
 *	Converts the coordinates x and y relative to GdkWindow w to
 *	be relative to the root (desktop) window.
 */
void GUIGetPositionRoot(
        void *w, int x, int y, int *rx, int *ry
)
{
        GdkWindow *cur_w = (GdkWindow *)w;
        GdkWindow *root_w;
        int cx, cy;


        if(rx != NULL)
            (*rx) = 0;
        if(ry != NULL)
            (*ry) = 0;

        if(cur_w == NULL)
            return;

        root_w = (GdkWindow *)GDK_ROOT_PARENT();
	if(root_w == NULL)
	    return;

        while((cur_w != root_w) && (cur_w != NULL))
        {
            gdk_window_get_position(cur_w, &cx, &cy);
	    x += cx;
	    y += cy;
            cur_w = gdk_window_get_parent(cur_w);
        }

        if(rx != NULL)
            (*rx) = x;
        if(ry != NULL)
            (*ry) = y;

	return;
}

/*
 *	Create a cursor (GdkCursor) from xpm data.
 *This function problem dosen't really work, so don't use it just yet!!
 */
void *GUICreateCursorFromData(
	u_int8_t **cursor_data,
	double hot_x, double hot_y,
	int width, int height	/* Both can be 0. */
)
{
	GdkCursor *cur;
	GdkWindow *window;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GtkStyle *style;
	gint w, h;


	if(cursor_data == NULL)
	    return(NULL);

	window = (GdkWindow *)GDK_ROOT_PARENT();
        if(window == NULL)  
            return(NULL);

        style = gtk_widget_get_default_style();
        pixmap = gdk_pixmap_create_from_xpm_d(
            window,
            &mask,
            &style->bg[GTK_STATE_NORMAL],
            (gchar **)cursor_data
        );

        gdk_window_get_size((GdkWindow *)pixmap, &w, &h);

	cur = gdk_cursor_new_from_pixmap(
	    pixmap,
	    (GdkPixmap *)mask,
	    &style->fg[GTK_STATE_NORMAL],
	    &style->bg[GTK_STATE_NORMAL],
	    (gint)(hot_x * MAX(w - 1, 0)),
	    (gint)(hot_y * MAX(h - 1, 0))
	);

	return(cur);                                          
}


/*
 *      Used by GUISetWMIcon() and GUIWMIconListRemove().
 *
 *	Checks if the given GdkWindow w is in the WM icon referances list.
 *
 *	The index is returned on a successful match or -1 on no match.
 */
static gint GUIWMIconListMatch(GdkWindow *window)
{
	gint i;
        gui_wm_icon_ref_struct *ref_ptr;


	if(window == NULL)
	    return(-1);

        for(i = 0; i < total_wm_icon_refs; i++)
        {
	    ref_ptr = wm_icon_ref[i];
            if(ref_ptr == NULL)
                continue;

	    if(ref_ptr->window != window)
		continue;

	    return(i);
        }

	return(-1);
}

/*
 *	Used by GUISetWMIcon().
 *
 *	Adds a new WM icon referance to our wm_icon_ref list.
 */
static void GUIWMIconListAdd(
        GdkWindow *w, GdkWindow *icon_window,
        GdkPixmap *icon_pixmap, GdkBitmap *icon_mask
)
{
	int i;
	gui_wm_icon_ref_struct *ref_ptr;


	/* Look for available pointer. */
	for(i = 0; i < total_wm_icon_refs; i++)
	{
	    if(wm_icon_ref[i] == NULL)
		break;
	}
	if(i >= total_wm_icon_refs)
	{
	    i = total_wm_icon_refs;
	    total_wm_icon_refs = i + 1;
	    wm_icon_ref = (gui_wm_icon_ref_struct **)realloc(
		wm_icon_ref,
		total_wm_icon_refs * sizeof(gui_wm_icon_ref_struct *)
	    );
	    if(wm_icon_ref == NULL)
	    {
		total_wm_icon_refs = 0;
		return;
	    }
	}

	wm_icon_ref[i] = ref_ptr = (gui_wm_icon_ref_struct *)calloc(
	    1, sizeof(gui_wm_icon_ref_struct)
	);
	if(ref_ptr != NULL)
	{
	    ref_ptr->window = w;
	    ref_ptr->icon_window = icon_window;
	    ref_ptr->icon_pixmap = icon_pixmap;
	    ref_ptr->icon_mask = icon_mask;
	}

	return;
}

/*
 *      Used by GUISetWMIcon() and also as a GDestroyNotify callback
 *	function.
 *
 *	Removes the given window from the WM icon referances list if
 *	found. Any resources on the matched WM icon referance will be
 *	unref'ed.
 */
static void GUIWMIconListRemove(gpointer data)
{
	gint i;
	gui_wm_icon_ref_struct *ref_ptr;
	GdkWindow *w = (GdkWindow *)data;


	i = GUIWMIconListMatch(w);
	if(i < 0)
	    return;

	ref_ptr = wm_icon_ref[i];
	if(ref_ptr != NULL)
	{
	    ref_ptr->window = NULL;

	    if(ref_ptr->icon_window != NULL)
	    {
		gdk_window_unref(ref_ptr->icon_window);
		ref_ptr->icon_window = NULL;
	    }

            if(ref_ptr->icon_pixmap != NULL)
            {
                gdk_pixmap_unref(ref_ptr->icon_pixmap);
                ref_ptr->icon_pixmap = NULL; 
            }

            if(ref_ptr->icon_mask != NULL)
            {
                gdk_bitmap_unref(ref_ptr->icon_mask);
                ref_ptr->icon_mask = NULL;
            }

	    /* Deallocate structure irself and reset pointer in array to
	     * NULL.
	     */
	    free(ref_ptr);
	    wm_icon_ref[i] = ref_ptr = NULL;
	}

/* Reclaim pointers? */

	return;
}

/*
 *      Sets the specified window manager icon data to window w.
 *
 *	Window w must be realized first.
 */
void GUISetWMIcon(GdkWindow *w, u_int8_t **icon)
{
	gint i;
        GdkPixmap *pixmap;
        GdkBitmap *mask;
        GtkStyle *style;


	if(w == NULL)
	    return;

        style = gtk_widget_get_default_style();
        if(style == NULL)
	    return;

        pixmap = gdk_pixmap_create_from_xpm_d(
	    w, &mask, &style->bg[GTK_STATE_NORMAL], (gchar **)icon
	);

        gdk_window_set_icon(w, NULL, pixmap, mask);
/*	gdk_window_set_icon_name(w, "WMHints Test Icon"); */

        /* Do not unref pixmap and mask, they will be recorded on the
	 * wm_icon_ref list and unref'ed later when window w's icon is
	 * updated or when window w is destroyed.
	 */

	/* Check if the given window w is in our list of WM icon
	 * referances.
	 */
	i = GUIWMIconListMatch(w);
	if(i < 0)
	{
	    /* Not in list, add a new item to our list. */
	    GUIWMIconListAdd(w, NULL, pixmap, mask);

	    /* Since window was not in list, it implies it also does not
	     * have our destroy notify callback set.
	     */
	    gdk_drawable_set_data(
		(GdkDrawable *)w,
		"wm_icon",		/* Key? */
		(gpointer)w,
		(GDestroyNotify)GUIWMIconListRemove
	    );
	}
	else
	{
	    gui_wm_icon_ref_struct *ref_ptr = wm_icon_ref[i];
	    if(ref_ptr != NULL)
	    {
		/* Unref old referances and update new referances. */

		if(ref_ptr->icon_window != NULL)
		    gdk_window_unref(ref_ptr->icon_window);
		ref_ptr->icon_window = NULL;

                if(ref_ptr->icon_pixmap != NULL)
                    gdk_pixmap_unref(ref_ptr->icon_pixmap);
                ref_ptr->icon_pixmap = NULL;

                if(ref_ptr->icon_mask != NULL)
                    gdk_bitmap_unref(ref_ptr->icon_mask);
                ref_ptr->icon_mask = NULL;
	    }
	}
}


/*
 *	Returns a fixed widget containing the icon.
 *	If icon is NULL then a fixed widget will be returned with
 *	no pixmap.
 * 
 *	The returned widget will not be automatically shown but it's
 *	pixmap will be.
 */
void *GUICreateMenuItemIcon(u_int8_t **icon)
{
	GdkWindow *window;
        GtkWidget *w, *pixmap;
        GdkPixmap *gdk_pixmap;
        GdkBitmap *mask;
        GtkStyle *style;
        gint width, height;


        w = gtk_fixed_new();
        if(w == NULL)
            return(NULL);

        window = (GdkWindow *)GDK_ROOT_PARENT();
        if(window == NULL)
            return(NULL);

        style = gtk_widget_get_style(w);

        /* If no icon data specified, then use default empty icon. */
	if(icon == NULL)
	{
	    gchar *bitmap_data;

	    width = GUI_MENU_ITEM_DEF_HEIGHT;
	    height = width;
	    gdk_pixmap = NULL;
	    pixmap = NULL;

	    /* Allocate tempory bitmap data. */
	    bitmap_data = (gchar *)calloc(
		((width / 8) + 1) * height,
		sizeof(gchar)
	    );
	    if(bitmap_data != NULL)
		mask = gdk_bitmap_create_from_data(
		    window, bitmap_data,
		    width, height
		);
	    else
		mask = NULL;
	    free(bitmap_data);	/* Free our allocated bitmap data. */

	    gdk_pixmap = gdk_pixmap_new(
		window, width, height, -1
	    );
            pixmap = gtk_pixmap_new(gdk_pixmap, mask);
	}
	else
	{
	    /* Create icon pixmap. */
            gdk_pixmap = gdk_pixmap_create_from_xpm_d(
	        window,
	        &mask,
	        &style->bg[GTK_STATE_NORMAL],
	        (gchar **)icon
	    );
	    pixmap = gtk_pixmap_new(gdk_pixmap, mask);

	    /* Get size of newly loaded pixmap. */
	    gdk_window_get_size((GdkWindow *)gdk_pixmap, &width, &height);
	}

	/* Limit size (max with height only). */
	if(width > GUI_MENU_ITEM_DEF_HEIGHT)
	    width = GUI_MENU_ITEM_DEF_HEIGHT;
	if(height > GUI_MENU_ITEM_DEF_HEIGHT)
	    height = GUI_MENU_ITEM_DEF_HEIGHT;

        /* Adjust size of fixed widget to fit pixmap. */
        gtk_widget_set_usize(w, width, height);

        /* Put pixmap into fixed widget. */
	if(pixmap != NULL)
            gtk_fixed_put(GTK_FIXED(w), pixmap, 0, 0);

	/* Unref gdk pixmap so when no other resources are using it, it
	 * will be destroyed.
	 */
	gdk_pixmap_unref(gdk_pixmap);

	/* Mask available? */
	if(mask != NULL)
	{
            gtk_widget_shape_combine_mask(w, mask, 0, 0);

	    /* Unref bitmap mask so when no other resources are using
	     * it, it will be destroyed.
	     */
	    gdk_bitmap_unref(mask);
	}

	if(pixmap != NULL)
	{
	    /* Make pixmap eligable for showing since its a child of the
	     * returned fixed widget.
	     */
            gtk_widget_show(pixmap);
	}

        return(w);
}

/*
 *	Sets the potential DND icon for the next DND operation (if it
 *	occures) with respect to the given pixmap and mask.
 *
 *	The widget is used as referance to note who this DND icon is
 *	being set to (it can be NULL).
 */
void GUISetDNDIcon(
        GtkWidget *widget,              /* A DND source widget. */ 
        GdkColormap *colormap,
        GdkPixmap *pixmap,              /* Icon pixmap. */
        GdkBitmap *mask,                /* Icon mask. */
        int hot_x, int hot_y
)
{


	/* No way to set DND icon if pixmap is NULL. */
	if(pixmap == NULL)
	    return;

	/* Check for change from previous set. */
	if(pixmap == gui_last_dnd_pixmap)
	{
	    /* Pixmap is the same, is the mask the same? */
	    if(mask == gui_last_dnd_mask)
	    {
		/* No icon change, so just skip. */
		return;
	    }
	}

	/* Get system colormap if given colormap is NULL. */
	if(colormap == NULL)
	    colormap = gdk_colormap_get_system();
	/* If no colormap available then give up. */
	if(colormap == NULL)
	    return;

	/* Need to increase referance count on pixmap and mask
	 * pair since the call to gtk_drag_set_default_icon() will
	 * reduce the referance count by one after one drag and
	 * drop operation has completed.
	 */
	gdk_pixmap_ref(pixmap);
	if(mask != NULL)
	    gdk_bitmap_ref(mask);
	gtk_drag_set_default_icon(
	    colormap,
	    pixmap, mask,
	    (gint)hot_x, (gint)hot_y
	);

	/* Record pixmap and mask. */
	gui_last_dnd_pixmap = pixmap;
	gui_last_dnd_mask = mask;

	/* Record referance widget. */
	gui_last_dnd_widget = widget;

	return;
}


/*
 *	Creates the global gui_tooltips widget for controlling the
 *	enabled/disabled state of all tooltips set by these functions.
 *
 *	When the global gui_tooltips widget is first created, the
 *	tooltips will be enabled and gui_tooltips_state will be reset
 *	to TRUE.
 */
static void GUICreateGlobalTooltips(void)
{
	/* Need to create global gui_tooltips widget? */
        if(gui_tooltips == NULL)
        {
	    GtkWidget *w;

            gui_tooltips = (GtkTooltips *)w = gtk_tooltips_new();
	    if(w != NULL)
            {
                gtk_tooltips_enable(GTK_TOOLTIPS(w));
                gui_tooltips_state = TRUE;
            }
        }
}

/*
 *	Sets the tip for the given widget w, if the widget w already
 *	has a tip set then it will be changed to the new value.
 *
 *	If tip is NULL then no tip will be set on the given widget.
 */
void GUISetWidgetTip(GtkWidget *w, const gchar *tip)
{
	if(w == NULL)
	    return;

	/* Create global tooltips as needed. */
	GUICreateGlobalTooltips();
        if(gui_tooltips == NULL)
            return;

	if(tip != NULL)
	{
	    gtk_tooltips_set_tip(
		gui_tooltips,		/* Our tooltips group. */
		w,
		tip, NULL
	    );
	}
	else
	{
            gtk_tooltips_set_tip(
                gui_tooltips,           /* Our tooltips group. */
                w,
                NULL, NULL
            );
	}
}

/*
 *	Makes the tooltip associated with the given widget shown
 *	immediatly if the widget has a tip defined. Also requires
 *	that the global gui_tooltips_state be TRUE, otherwise
 *	this function does nothing.
 */
void GUIShowTipsNow(GtkWidget *w)
{
	GdkEventCrossing e;
	GdkWindow *window;


	if((w == NULL) || !gui_tooltips_state)
	    return;

	if(GTK_WIDGET_NO_WINDOW(w))
	    return;
	else
	    window = w->window;
	if(window == NULL)
	    return;

	/* Send a fake enter notify event to make widget think
	 * its time to show the tooltips. Note that the widget
	 * should be watching for the enter_notify_event.
	 */
	e.type = GDK_ENTER_NOTIFY;
	e.window = window;
	e.send_event = 1;		/* True if we're sending event. */
	e.subwindow = window;
	e.time = GDK_CURRENT_TIME;
	e.x = 0.0;
	e.y = 0.0;
	e.x_root = 0.0;
	e.y_root = 0.0;
	e.mode = GDK_CROSSING_NORMAL;
	e.detail = GDK_NOTIFY_ANCESTOR;
	e.focus = TRUE;			/* Focus. */
	e.state = 0;			/* Key modifiers. */
	gdk_event_put((GdkEvent *)&e);
}

/*
 *	Enables/disables global tooltips state.
 */
void GUISetGlobalTipsState(gbool state)
{
        /* Create global tooltips as needed. */
        GUICreateGlobalTooltips();
        if(gui_tooltips == NULL)
	    return;

	if(state)
	    gtk_tooltips_enable(gui_tooltips);
	else
	    gtk_tooltips_disable(gui_tooltips);

	/* Update global tool tips state. */
	gui_tooltips_state = ((state) ? TRUE : FALSE);
}


/*
 *	Changes the layout, showing or hiding the pixmap and label.
 *	The given button must be one returned from GUIButtonPixmap*()
 *	or GUIToggleButtonPixmap*().
 */
void GUIButtonChangeLayout(void *button, int show_pixmap, int show_label)
{
	GtkWidget *w;
	gui_button_data_struct *btn_data;


	if(button == NULL)
	    return;

	btn_data = (gui_button_data_struct *)gtk_object_get_user_data(
	    GTK_OBJECT(button)
	);
	if(btn_data == NULL)
	    return;

	w = (GtkWidget *)btn_data->label_normal;
	if(w != NULL)
	{
	    if(show_label)
		gtk_widget_show(w);
	    else
		gtk_widget_hide(w);
	}

        w = (GtkWidget *)btn_data->pixmap_normal;
        if(w != NULL)
        {
            if(show_pixmap)
                gtk_widget_show(w); 
            else
                gtk_widget_hide(w);
        }

	return;
}

/*
 *      Creates a push button widget with the specified icon
 *      data created as a pixmap placed into it and the label.
 *
 *	If orient is set to 1 then the orientation will be horizontal,
 *	otherwise vertical.
 */
void *GUIButtonPixmapLabelHV(
	u_int8_t **icon, const char *label, void **label_rtn,
	int orient
)
{
        GtkWidget *w, *labelwid = NULL, *pixmapwid = NULL;
        GtkWidget *button = gtk_button_new();
	gint width = 0, height = 0;
	gui_button_data_struct *btn_data;


	if(label_rtn != NULL)
	    (*label_rtn) = NULL;

	if(button == NULL)
	    return(button);

	/* Allocate button data. */
	btn_data = (gui_button_data_struct *)calloc(
	    1, sizeof(gui_button_data_struct)
	);
	if(btn_data != NULL)
	    btn_data->button = button;

	/* Store button data as user data on button as object. */
	gtk_object_set_user_data(GTK_OBJECT(button), (gpointer)btn_data);

	/* Connect destroy event to button so that the button data can be
	 * deallocated.
	 */
	gtk_signal_connect(
	    GTK_OBJECT(button), "destroy",
	    GTK_SIGNAL_FUNC(GUIButtonDestroyCB),
	    (gpointer)btn_data
	);


	/* Box in button for parenting. */
	if(orient == 1)
	    w = gtk_hbox_new(FALSE, 0);
	else
	    w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(button), w);
        gtk_widget_show(w);

	/* Create icon pixmap? */
	if(icon != NULL)
	{
	    GtkStyle *style = gtk_widget_get_style(button);
	    GdkWindow *window = (GdkWindow *)GDK_ROOT_PARENT();

	    if((style != NULL) && (window != NULL))
	    {
		GdkBitmap *mask;
		GdkPixmap *pixmap;

	        pixmap = gdk_pixmap_create_from_xpm_d(
		    window,
		    &mask,
		    &style->bg[GTK_STATE_NORMAL],
		    (gchar **)icon
		);
		pixmapwid = gtk_pixmap_new(pixmap, mask);
		if(btn_data != NULL)
		    btn_data->pixmap_normal = pixmapwid;
		gdk_window_get_size((GdkWindow *)pixmap, &width, &height);
		gtk_box_pack_start(GTK_BOX(w), pixmapwid, TRUE, FALSE, 0);
		gtk_widget_show(pixmapwid);

		gdk_pixmap_unref(pixmap);
		if(mask != NULL)
		    gdk_bitmap_unref(mask);
	    }
	}

	if(label != NULL)
	{
	    labelwid = gtk_label_new(label);
            if(btn_data != NULL)
                btn_data->label_normal = labelwid;
            if(label_rtn != NULL)
                (*label_rtn) = labelwid;
	    gtk_label_set_justify(
		GTK_LABEL(labelwid),
                ((orient == 1) ? GTK_JUSTIFY_LEFT : GTK_JUSTIFY_CENTER)
	    );
            gtk_box_pack_start(GTK_BOX(w), labelwid, TRUE, FALSE, 0);
            gtk_widget_show(labelwid);
	}
	else
	{
	    /* If label is NULL then match size of button to exact
	     * size of pixmap.
	     */
	    if((width != 0) && (height != 0))
	    {
		gtk_widget_set_usize(button, width + 2, height + 2);
	    }
	}

	return(button);
}

/*              
 *      Creates a push button widget with the specified icon
 *      data created as a pixmap placed into it and the label          
 *      separated horizontally.
 */
void *GUIButtonPixmapLabelH(
	u_int8_t **icon, const char *label, void **label_rtn
)
{
	return(GUIButtonPixmapLabelHV(icon, label, label_rtn, 1));
}

/*      
 *      Creates a push button widget with the specified icon
 *      data created as a pixmap placed into it and the label
 *      separated vertically.
 */
void *GUIButtonPixmapLabelV(
	u_int8_t **icon, const char *label, void **label_rtn
)
{
        return(GUIButtonPixmapLabelHV(icon, label, label_rtn, 0));
}

/*
 *      Creates a push button widget with the specified icon
 *      data created as a pixmap placed into it.
 */
void *GUIButtonPixmap(u_int8_t **icon)
{
        return(
            GUIButtonPixmapLabelH(icon, NULL, NULL)
        );
}


/*
 *      Creates a toggle button widget with the specified icon
 *      data created as a pixmap placed into it and the label.
 *
 *      If orient is set to 1 then the orientation will be horizontal,
 *      otherwise vertical.
 */
void *GUIToggleButtonPixmapLabelHV(
	u_int8_t **icon, const char *label, void **label_rtn, int orient
)
{
        GtkWidget *w, *labelwid = NULL, *pixmapwid = NULL;
        GtkWidget *button = gtk_toggle_button_new();
        gint width = 0, height = 0;
	gui_button_data_struct *btn_data;


        if(label_rtn != NULL)
            (*label_rtn) = NULL;

        if(button == NULL)
            return(button);


        /* Allocate button data. */
        btn_data = (gui_button_data_struct *)calloc(
            1, sizeof(gui_button_data_struct)
        );
        if(btn_data != NULL)
            btn_data->button = button;
 
        /* Store button data as user data on button as object. */
        gtk_object_set_user_data(GTK_OBJECT(button), (gpointer)btn_data);

        /* Connect destroy event to button so that the button data can be
         * deallocated. 
         */
        gtk_signal_connect(
            GTK_OBJECT(button), "destroy",
            GTK_SIGNAL_FUNC(GUIButtonDestroyCB),
            (gpointer)btn_data
        );   


        /* Box in button for parenting. */
        if(orient == 1)
            w = gtk_hbox_new(FALSE, 0);
        else
            w = gtk_vbox_new(FALSE, 0);
        gtk_container_add(GTK_CONTAINER(button), w);
        gtk_widget_show(w);

        /* Create icon pixmap? */
        if(icon != NULL)
        {
            GtkStyle *style = gtk_widget_get_style(button);
            GdkWindow *window = (GdkWindow *)GDK_ROOT_PARENT();

            if((style != NULL) && (window != NULL))
            {
                GdkBitmap *mask;
                GdkPixmap *pixmap;

                pixmap = gdk_pixmap_create_from_xpm_d(
                    window,
                    &mask,
                    &style->bg[GTK_STATE_NORMAL],
                    (gchar **)icon
                );
                pixmapwid = gtk_pixmap_new(pixmap, mask);
		if(btn_data != NULL)
		    btn_data->pixmap_normal = pixmapwid;
                gdk_window_get_size((GdkWindow *)pixmap, &width, &height);
                gtk_box_pack_start(
                    GTK_BOX(w), pixmapwid, TRUE, FALSE, 0
                );
                gtk_widget_show(pixmapwid);

		gdk_pixmap_unref(pixmap);
		if(mask != NULL)
		    gdk_bitmap_unref(mask);
            }
        }

        if(label != NULL)
        {
            labelwid = gtk_label_new(label);
	    if(btn_data != NULL)
		btn_data->label_normal = labelwid;
            if(label_rtn != NULL)
                (*label_rtn) = labelwid;
            gtk_label_set_justify(
		GTK_LABEL(labelwid),
		((orient == 1) ? GTK_JUSTIFY_LEFT : GTK_JUSTIFY_CENTER)
	    );
            gtk_box_pack_start(
                GTK_BOX(w), labelwid, TRUE, FALSE, 0
            );
            gtk_widget_show(labelwid);
        }
        else
        {
            /* If label is NULL then match size of button to exact
             * size of pixmap.
             */
            if((width != 0) && (height != 0))
            {
                gtk_widget_set_usize(button, width + 2, height + 2);
            }
        }

        return(button);
}

/*
 *      Creates a toggle button widget with the specified icon
 *      data created as a pixmap placed into it and the label
 *	oriented horizontally.
 */
void *GUIToggleButtonPixmapLabelH(
	u_int8_t **icon, const char *label, void **label_rtn
)
{
        return(GUIToggleButtonPixmapLabelHV(icon, label, label_rtn, 1));
}

/*
 *      Creates a toggle button widget with the specified icon
 *      data created as a pixmap placed into it and the label
 *      oriented vertically.
 */
void *GUIToggleButtonPixmapLabelV(
	u_int8_t **icon, const char *label, void **label_rtn
)
{
        return(GUIToggleButtonPixmapLabelHV(icon, label, label_rtn, 0));
}

/*
 *      Creates a toggle button widget with the specified icon
 *      data created as a pixmap placed into it.
 */             
void *GUIToggleButtonPixmap(u_int8_t **icon)
{
        return(GUIToggleButtonPixmapLabelHV(icon, NULL, NULL, 0)); 
}


/*
 *	Creates a prompt bar, consisting of an icon, label and text
 *	entry widgets.
 *
 *	If browse_rtn is not NULL then a browse button will be created
 *	as well.
 */
void *GUIPromptBarOrBrowse(
        u_int8_t **icon, const char *label,
	void **label_rtn, void **entry_rtn, void **browse_rtn
)
{
	gint border_minor = 2;
	GtkWidget *parent, *w;


	/* Reset returns. */
        if(label_rtn != NULL)
            (*label_rtn) = NULL;
	if(entry_rtn != NULL)
	    (*entry_rtn) = NULL;
	if(browse_rtn != NULL)
	    (*browse_rtn) = NULL;

	/* Create prompt bar parent. */
	w = gtk_table_new(
	    1,
	    ((icon == NULL) ? 0 : 1) +
	    ((label == NULL) ? 0 : 1) +
            1 +
	    ((browse_rtn == NULL) ? 0 : 1),
	    FALSE
	);
	/* Do not show parent to prompt bar. */
	parent = w;

	/* Create icon. */
	if(icon != NULL)
	{
	    gint attach_x = 0;
	    gint width, height;
            GtkStyle *style = gtk_widget_get_style(parent);
            GdkWindow *window = (GdkWindow *)GDK_ROOT_PARENT();
	    GtkWidget *pixmapwid;

	    w = gtk_fixed_new();
            gtk_table_attach(
		GTK_TABLE(parent), w,
                attach_x, attach_x + 1,
                0, 1,
                0,
                0,
                border_minor / 2, 0
            );
	    gtk_widget_show(w);

            if((style != NULL) && (window != NULL))
            {
                GdkBitmap *mask;
                GdkPixmap *pixmap;

                pixmap = gdk_pixmap_create_from_xpm_d(
                    window,
                    &mask,
                    &style->bg[GTK_STATE_NORMAL],
                    (gchar **)icon
                );
                pixmapwid = gtk_pixmap_new(pixmap, mask);
                gdk_window_get_size((GdkWindow *)pixmap, &width, &height);
		gtk_fixed_put(GTK_FIXED(w), pixmapwid, 0, 0);
		gtk_widget_set_usize(w, width, height);
                gtk_widget_show(pixmapwid);

                gdk_pixmap_unref(pixmap);
                if(mask != NULL)
                    gdk_bitmap_unref(mask);
            }
	}

	/* Create label? */
	if(label != NULL)
	{
            gint attach_x = ((icon == NULL) ? 0 : 1);
	    GtkWidget *parent2;

	    w = gtk_hbox_new(FALSE, 0);
	    gtk_table_attach(
                GTK_TABLE(parent), w,  
                attach_x, attach_x + 1,
                0, 1,
                GTK_FILL,
                0,  
                border_minor / 2, 0
            );
	    gtk_widget_show(w);
	    parent2 = w;

	    w = gtk_label_new(label);
	    if(label_rtn != NULL)
		(*label_rtn) = w;
	    gtk_box_pack_end(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	    gtk_widget_show(w);
	}

	/* Create text entry? */
	if(1)
	{
            gint attach_x = ((icon == NULL) ? 0 : 1) +
		((label == NULL) ? 0 : 1);

	    w = gtk_entry_new();
	    if(entry_rtn != NULL)
		(*entry_rtn) = w;
	    gtk_table_attach(
		GTK_TABLE(parent), w,
                attach_x, attach_x + 1,
                0, 1,
                GTK_SHRINK | GTK_EXPAND | GTK_FILL,
                0,
                border_minor / 2, 0
            );
            gtk_widget_show(w);
	}

	/* Create browse button? */
	if(browse_rtn != NULL)
	{
            gint attach_x = ((icon == NULL) ? 0 : 1) +
                ((label == NULL) ? 0 : 1) +
		1;

	    w = (GtkWidget *)GUIButtonPixmap(
		(u_int8_t **)icon_browse_20x20_xpm
	    );
	    if(w != NULL)
	    {
		(*browse_rtn) = w;
		gtk_widget_set_usize(w, 20 + 2, 20 + 2);
                gtk_table_attach(
		    GTK_TABLE(parent), w,
                    attach_x, attach_x + 1,
                    0, 1,
                    0,
                    0,
                    border_minor / 2, 0
                );
                gtk_widget_show(w);
	    }
	}

	return(parent);
}

/*
 *	Creates prompt bar with browse button.
 */
void *GUIPromptBarWithBrowse(
        u_int8_t **icon, const char *label,
        void **label_rtn, void **entry_rtn, void **browse_rtn,
	void *browse_client_data, void (*browse_cb)(void *, void *)
)
{
	GtkWidget *parent;

	parent = GUIPromptBarOrBrowse(
	    icon, label, label_rtn, entry_rtn, browse_rtn
	);

	/* Set signal callback for the browse button. */
	if((browse_rtn != NULL) && (browse_cb != NULL))
	{
	    GtkWidget *w = (GtkWidget *)(*browse_rtn);

	    if(w != NULL)
	    {
                gtk_signal_connect(
                    GTK_OBJECT(w), "clicked",
                    GTK_SIGNAL_FUNC(browse_cb),
                    (gpointer)browse_client_data
                );
	    }
	}

	return(parent);
}

/*
 *	Creates a new prompt bar.
 */
void *GUIPromptBar(
        u_int8_t **icon, const char *label,
        void **label_rtn, void **entry_rtn
)
{
	return(GUIPromptBarOrBrowse(
            icon, label, label_rtn, entry_rtn, NULL
        ));
}


/*
 *	Sets up widget as a DND drag source.
 */
void GUIDNDSetSrc(
        void *w, void *drag_type, int ndrag_types,
        unsigned int actions, unsigned int buttons,
        void (*begin_cb)(GtkWidget *, GdkDragContext *, gpointer),
        void (*request_data_cb)(GtkWidget *, GdkDragContext *,
                GtkSelectionData *, guint, guint, gpointer),
        void (*delete_data_cb)(GtkWidget *, GdkDragContext *, gpointer),
        void (*end_cb)(GtkWidget *, GdkDragContext *, gpointer),
        gpointer client_data
)
{
	GtkWidget *widget = (GtkWidget *)w;
	GtkTargetEntry *target_entry = (GtkTargetEntry *)drag_type;
	gint total_target_entries = (gint)ndrag_types;


	if((widget == NULL) ||
           (target_entry == NULL) ||
           (total_target_entries <= 0)
	)
	    return;

	/* Widget must have window. */
	if(GTK_WIDGET_NO_WINDOW(widget))
	{
	    fprintf(stderr,
 "GUIDNDSetSrc(): Widget does not have window.\n"
	    );
	    return;
	}

	/* Set this widget as a DND drag source. */
	gtk_drag_source_set(
	    widget,
	    buttons,
	    target_entry, total_target_entries,
	    actions
	);

	/* Set drag begin callback. */
	if(begin_cb != NULL)
	    gtk_signal_connect(
		GTK_OBJECT(widget), "drag_begin",
		GTK_SIGNAL_FUNC(begin_cb), client_data
	    );

	/* Set drag data request/get callback. */
	if(request_data_cb != NULL)
	    gtk_signal_connect(
		GTK_OBJECT(widget), "drag_data_get",
		GTK_SIGNAL_FUNC(request_data_cb), client_data
	    );

	/* Set frag data delete callback. */
	if(delete_data_cb != NULL)
	    gtk_signal_connect(
		GTK_OBJECT(widget), "drag_data_delete",
		GTK_SIGNAL_FUNC(delete_data_cb), client_data
	    );

	/* Set drag end callback. */
        if(end_cb != NULL)
            gtk_signal_connect(
                GTK_OBJECT(widget), "drag_end",
                GTK_SIGNAL_FUNC(end_cb), client_data
            );

	/* Set our own drag_end callback to be called after the one for
	 * the client function.
	 */
	gtk_signal_connect(
	    GTK_OBJECT(widget), "drag_end",
	    GTK_SIGNAL_FUNC(GUIDNDDragEndCB),
	    (gpointer)NULL		/* No client data for now. */
	);

	return;
}

/*
 *	Sets up a widget as DND drop target.
 */
void GUIDNDSetTar(
        void *w, void *drag_type, int ndrag_types,
        unsigned int actions, unsigned int default_action_same,
	unsigned int default_action,
        void (*recieved_data_cb)(GtkWidget *, GdkDragContext *,  
                gint, gint, GtkSelectionData *, guint, guint,  
                gpointer     
        ),
        gpointer client_data
)
{
	static char init_cb_structs = 1;
        GtkWidget *widget = (GtkWidget *)w;
        GtkTargetEntry *target_entry = (GtkTargetEntry *)drag_type;
        gint total_target_entries = (gint)ndrag_types;
	static gui_dnd_cb_struct	move_same_cb_data,
					move_cb_data,
					copy_cb_data;
	gui_dnd_cb_struct *cb_data_ptr = NULL;


        if((widget == NULL) || 
           (target_entry == NULL) ||
           (total_target_entries <= 0)
        )
            return;

        /* Widget must have window. */
        if(GTK_WIDGET_NO_WINDOW(widget))
        {   
            fprintf(stderr,
 "GUIDNDSetTar(): Widget does not have window.\n"
            );
            return;
        }

	/* Set widget as a DND drop target. */
	gtk_drag_dest_set(
	    widget,
/*	    GTK_DEST_DEFAULT_MOTION | */
	    GTK_DEST_DEFAULT_HIGHLIGHT |
	    GTK_DEST_DEFAULT_DROP,
            target_entry, total_target_entries,
            actions
        );

        if(recieved_data_cb != NULL)
            gtk_signal_connect(
                GTK_OBJECT(widget), "drag_data_received",
                GTK_SIGNAL_FUNC(recieved_data_cb), client_data
            );


	/* Need to initialize DND callback data structures? */
	if(init_cb_structs)
	{
	    move_same_cb_data.default_action_same = GDK_ACTION_MOVE;
	    move_same_cb_data.default_action = GDK_ACTION_COPY;

            move_cb_data.default_action_same = GDK_ACTION_MOVE;
            move_cb_data.default_action = GDK_ACTION_MOVE;

            copy_cb_data.default_action_same = GDK_ACTION_COPY;
	    copy_cb_data.default_action = GDK_ACTION_COPY;

	    init_cb_structs = 0;	/* Mark as no longer needing init. */
	}

	/* Set local drag motion callback data. */
	/* Move only on same? */
	if((default_action_same == GDK_ACTION_MOVE) &&
           (default_action == GDK_ACTION_COPY)
	)
	    cb_data_ptr = &move_same_cb_data;
	/* Always move? */
	else if((default_action_same == GDK_ACTION_MOVE) &&
                (default_action == GDK_ACTION_MOVE)
        )
            cb_data_ptr = &move_cb_data;
	/* Always copy? */
	else if((default_action_same == GDK_ACTION_COPY) &&
                (default_action == GDK_ACTION_COPY)
	)
	    cb_data_ptr = &copy_cb_data;

	/* Set our local drag motion callback. */
	if(cb_data_ptr != NULL)
	    gtk_signal_connect(
		GTK_OBJECT(widget), "drag_motion",
		GTK_SIGNAL_FUNC(GUIDNDDragMotionCB),
		(gpointer)cb_data_ptr
	    );

	return;
}

/*
 *	Defines the drag icon used in the next (if any) drag and drop.
 *	The referance counts for the given GdkPixmap and GdkBitmap
 *	pair will be increased by one and then passed to
 *	gtk_drag_set_default_icon() where it will manage the referance
 *	count and unref when there is no longer need for it so the
 *	calling function need not ref or unref based on a call to
 *	this function.
 */
void GUIDNDSetDragIcon(void *pixmap, void *mask)
{
	gint w = 15, h = 15;
	GdkPixmap *p = (GdkPixmap *)pixmap;
	GdkBitmap *m = (GdkBitmap *)mask;


	if((p == NULL) || (m == NULL))
	    return;

	gdk_window_get_size((GdkWindow *)p, &w, &h);
        gdk_pixmap_ref(p);
        gdk_bitmap_ref(m);
        gtk_drag_set_default_icon(
	    gdk_colormap_get_system(),
	    p, m,
	    (gint)(w / 2), (gint)(h / 2)
	);

	return;
}


/*
 *	Creates a new banner widget.
 */
GtkWidget *GUIBannerCreate(
        const gchar *label, const gchar *font_name,
        GdkColor color_fg, GdkColor color_bg,
        gint align,             /* One of GTK_JUSTIFY_*. */
        gbool expand
)
{
	GdkFont *font = NULL;
	GtkRcStyle *rcstyle;
	GtkWidget *w = gtk_drawing_area_new();
	gpointer *cb_data = (gpointer *)calloc(4, sizeof(gpointer));


#define ALPHABET_STR	"\
ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz"

	/* Set up callback data. */
	if(cb_data != NULL)
	{
	    cb_data[0] = ((label == NULL) ?
		NULL : g_strdup(label)
	    );
	    cb_data[1] = (void *)align;
	    cb_data[2] = font = ((font_name != NULL) ?
		gdk_font_load(font_name) : NULL
	    );
	    cb_data[3] = NULL;
	}

	/* Connect expose_event signal for redrawing. */
        gtk_signal_connect(
            GTK_OBJECT(w), "expose_event",
            GTK_SIGNAL_FUNC(GUIBannerExposeCB),
	    (gpointer)cb_data
        );
	gtk_widget_add_events(w, GDK_EXPOSURE_MASK);
	/* Destroy signal, for deallocaing coppied label. */
        gtk_signal_connect(
            GTK_OBJECT(w), "destroy",
            GTK_SIGNAL_FUNC(GUIBannerDestroyCB),
            (gpointer)cb_data
        );

	/* Set style. */
	rcstyle = gtk_rc_style_new();
	if(font_name != NULL)
	{
	    g_free(rcstyle->font_name);
	    rcstyle->font_name = g_strdup(font_name);
	}
        rcstyle->color_flags[GTK_STATE_NORMAL] = GTK_RC_FG | GTK_RC_BG;
	rcstyle->color_flags[GTK_STATE_INSENSITIVE] = GTK_RC_FG | GTK_RC_BG;
	memcpy(&rcstyle->fg[GTK_STATE_NORMAL], &color_fg, sizeof(GdkColor));
        memcpy(&rcstyle->bg[GTK_STATE_NORMAL], &color_bg, sizeof(GdkColor));
	/* Modify foreground and background colors for insensitive. */
#define LOWER_VALUE(c)	((((gint)(c) - (gint)((guint16)-1 / 2)) * 0.75) + \
(gint)((guint16)-1 / 2))
	color_fg.red = LOWER_VALUE(color_fg.red);
	color_fg.green = LOWER_VALUE(color_fg.green);
	color_fg.blue = LOWER_VALUE(color_fg.blue);
        color_bg.red = LOWER_VALUE(color_bg.red);
        color_bg.green = LOWER_VALUE(color_bg.green);
        color_bg.blue = LOWER_VALUE(color_bg.blue);
#undef LOWER_VALUE
        memcpy(&rcstyle->fg[GTK_STATE_INSENSITIVE], &color_fg, sizeof(GdkColor));
        memcpy(&rcstyle->bg[GTK_STATE_INSENSITIVE], &color_bg, sizeof(GdkColor));

	gtk_widget_modify_style(w, rcstyle);

	GUIRCStyleDeallocUnref(rcstyle);
	rcstyle = NULL;


	/* Get new style values from widget. */
        if(font != NULL)
        {
	    gint width, height;


	    if(expand)
		width = -1;
	    else if(font != NULL)
		width = gdk_string_width(
		    font, (label == NULL) ? ALPHABET_STR : label
		);
	    else
		width = -1;

	    if(font != NULL)
		height = gdk_string_height(
		    font, (label == NULL) ? ALPHABET_STR : label) + 6;
	    else
		height = -1;
  
	    gtk_widget_set_usize(w, width, height);
        }

#undef ALPHABET_STR

	return(w);
}

/*
 *	Creates a new combo box and label, the combo box and label
 *	will be placed inside of a table.
 *
 *	The return value wil be the table widget or NULL on failure.
 *
 *	The given GList of strings list will be duplicated and the one
 *	passed to this list will not be modified.
 */
void *GUIComboCreate(
        const char *label,
        const char *text,       /* Initial text. */
        void *list,             /* Initial glist of items for combo list. */
        int max_items,          /* Maximum items allowed. */
	void **combo_rtn,	/* Combo widget return. */
        void *client_data,
        void (*func_cb)(GtkWidget *w, void *),
	void (*list_change_cb)(GtkWidget *, void *, GList *)
)
{
	GtkWidget *parent, *wlabel = NULL, *w;
        GtkCombo *combo;
	const char *cstrptr;
	void **cb_data;


	/* Reset return values. */
	if(combo_rtn != NULL)
	    (*combo_rtn) = NULL;

	/* Create parent table widget. */
	parent = gtk_table_new(
	    1,
	    ((label == NULL) ? 0 : 1) +
	    1,
	    FALSE
	);
	if(parent == NULL)
	    return(NULL);
	/* Do not show the table parent. */

	/* Create label? */
	if(label != NULL)
	{
	    gint x = 0;

	    wlabel = gtk_label_new(label);
            gtk_table_attach(GTK_TABLE(parent), wlabel,
                x, x + 1,
                0, 1,
                0,
                0,
                2, 2
            );
	    gtk_label_set_justify(GTK_LABEL(wlabel), GTK_JUSTIFY_RIGHT);
	    gtk_widget_show(wlabel);
	}

	/* Create combo. */
	if(1)
	{
	    gint i;
	    gint x = ((label == NULL) ? 0 : 1);
	    GList *glist_in, *glist_out;


	    w = gtk_combo_new();
	    combo = GTK_COMBO(w);
	    if(combo_rtn != NULL)
		(*combo_rtn) = w;
            gtk_table_attach(GTK_TABLE(parent), w,
                x, x + 1,
                0, 1,
                GTK_SHRINK | GTK_EXPAND | GTK_FILL,
                0,
                2, 2
            );
/*
            combo->value_in_list = 0;
 */
	    gtk_combo_disable_activate(combo);
	    if(text == NULL)
	        gtk_entry_set_text(GTK_ENTRY(combo->entry), "");
	    else
                gtk_entry_set_text(GTK_ENTRY(combo->entry), text);
            gtk_entry_set_editable(GTK_ENTRY(combo->entry), TRUE);
            gtk_combo_set_case_sensitive(combo, FALSE);


	    /* Begin creating output glist, a duplicate input glist. */
	    glist_out = NULL;
	    glist_in = (GList *)list;

	    /* Is input glist given? */
	    if(glist_in == NULL)
	    {
		/* Input glist is NULL so create a bunch of empty strings
		 * totalling that specified by max_items for the output
		 * glist.
		 */
		for(i = 0; i < max_items; i++)
		    glist_out = g_list_append(glist_out, g_strdup(""));
	    }
	    else
	    {
		/* Input glist is given, make a duplicate of it as the
		 * output glist.
		 */
		i = 0;
		while((i < max_items) && (glist_in != NULL))
		{
		    cstrptr = (const char *)glist_in->data;
		    glist_out = g_list_append(
			glist_out,
			g_strdup((cstrptr == NULL) ? "" : cstrptr)
		    );
		    glist_in = glist_in->next;
		    i++;
		}
	    }
	    glist_in = NULL;	/* Input glist should no longer be used. */

	    /* Set new output glist to combo box. */
	    if(glist_out != NULL)
		gtk_combo_set_popdown_strings(combo, glist_out);

	    /* Do not delete the output glist, it will be passed to the
	     * callback data below.
	     */

	    /* Allocate callback data, 9 pointers of format;
	     * table		Parent table that holds this combo and label.
	     * label		Label (may be NULL).
	     * combo		This combo widget.
	     * GList		Contains list of strings in combo list.
	     * max_items	Max items allowed in combo list.
	     * client_data	Calling function set callback data.
	     * func_cb		Calling function set callback function.
	     * list_change_cb	Calling function set list change function.
	     * NULL
	     */
	    cb_data = (void **)calloc(9, sizeof(void *));
	    cb_data[0] = (void *)parent;
	    cb_data[1] = (void *)wlabel;
	    cb_data[2] = (void *)combo;
	    cb_data[3] = (void *)glist_out;
	    cb_data[4] = (void *)MAX(max_items, 0);
            cb_data[5] = client_data;
            cb_data[6] = (void *)func_cb;
	    cb_data[7] = (void *)list_change_cb;
	    cb_data[8] = NULL;

	    /* Record pointer to cb_data pointer array in the
	     * combo object's user data.
	     */
	    gtk_object_set_user_data(GTK_OBJECT(combo), cb_data);

	    /* Set signal handlers for combo's entry widget to local
	     * callbacks.
	     */
            gtk_signal_connect(
                GTK_OBJECT(combo), "destroy",
                GTK_SIGNAL_FUNC(GUIComboDestroyCB),
                (gpointer)cb_data
            );
            gtk_signal_connect(
                GTK_OBJECT(combo->entry), "activate",
                GTK_SIGNAL_FUNC(GUIComboActivateCB),
                (gpointer)cb_data
            );
	    gtk_widget_show(w);
	}

	return(parent);
}

/*
 *	Adds the specified value to the combo widget w's list and current
 *	value for the combo widget's entry widget.
 *
 *	The item will only be added to the list if it is not NULL
 *	and does not already exist in the list.
 *
 *	Excess items may be truncated if adding this item would exceed
 *	the combo list's maximum allowed items.
 */
void GUIComboActivateValue(void *w, const char *value)
{
	void **cb_data;
	GtkCombo *combo = (GtkCombo *)w;
	if((combo == NULL) || (value == NULL))
	    return;

	cb_data = (void **)gtk_object_get_user_data(GTK_OBJECT(combo));
	if(cb_data == NULL)
	    return;

	/* Set new value on combo's entry widget. */
	gtk_entry_set_text(
	    GTK_ENTRY(combo->entry), value
	);

	/* Call activate callback as if this was an actual activate
	 * signal.
	 */
	GUIComboActivateCB(GTK_WIDGET(combo), (gpointer)cb_data);

	return;
}

/*
 *	Adds the given value to the beginning of the glist for the
 *	combo box w. Older items in the list may be truncated if adding
 *	a new value would exceed the set maximum items for the combo box.
 *
 *	If value is already in the list then no operation will be
 *	performed.
 *
 *	The combo box's set list change callback will be called if it is
 *	not NULL and the new value was different and added succesfully.
 */
void GUIComboAddItem(void *w, const char *value)
{
        int i, max_items;
        gbool status;
	const char *cstrptr;
        GtkWidget *parent, *wlabel;
        GtkCombo *combo;
        GList *glist_in, *glist_next_in;
        void *client_data;
        void (*func_cb)(GtkWidget *, void *);
        void (*list_change_cb)(GtkWidget *, void *, GList *);
        void **cb_data;


        if((w == NULL) || (value == NULL))
	    return;

	combo = GTK_COMBO(w);


	/* Get object callback data. */
        cb_data = (void **)gtk_object_get_user_data(GTK_OBJECT(combo));
        if(cb_data == NULL)
            return;

        /* Parse callback data, format is as follows;
         * table        Parent table that holds this combo and label.
         * label        Label (may be NULL).
         * combo        This combo widget.
         * GList        Contains list of strings in combo list.
         * max_items    Max items allowed in combo list.
         * client_data  Calling function set callback data.
         * func_cb      Calling function set callback function.
         * list_change_cb       Calling function set list change function.
         * NULL
         */
        parent = (GtkWidget *)(cb_data[0]);
        wlabel = (GtkWidget *)(cb_data[1]);
        combo = (GtkCombo *)(cb_data[2]);
        glist_in = (GList *)(cb_data[3]);
        max_items = (int)(cb_data[4]);
        client_data = (void *)(cb_data[5]);
        func_cb = cb_data[6];
        list_change_cb = cb_data[7];


        /* Check if new value is already in the input glist. */
        status = TRUE;
        i = 0;
        glist_next_in = glist_in;
	/* Iterate from 0 to max_items or when the input glist is
	 * NULL (whichever occures first).
	 */
        while((i < max_items) && (glist_next_in != NULL))
        {
            cstrptr = (char *)glist_next_in->data;
            if(cstrptr != NULL)
            {
 /* Check if case sensitive? */
                /* Compare list item value with given value. */
                if(!strcasecmp(cstrptr, value))
                {
		    /* Given value matches a value in the list, so set
		     * status to FALSE indicating there is a value already
		     * in the list.
		     */
                    status = FALSE;
                    break;
                }
            }
            i++;
            glist_next_in = glist_next_in->next;
        }
	/* Variable status will be set to FALSE if the value is already
	 * in the list.
	 */

	/* Check if max_items allows us to add a new item to the list and
	 * if status is TRUE (indicating value is not already in the
	 * list).
	 */
        if((max_items > 0) && status)
        { 
            /* Create new output glist. */
            GList *glist_out = NULL;

            /* Add first item in output glist to be the new value fetched
	     * from the combo's entry.
             */
            glist_out = g_list_append(
                glist_out, g_strdup((const char *)value)
            );
        
            /* Now copy old input glist items to output glist, starting
	     * with i = 1 since we already have one item in the output
	     * glist.
             */
            i = 1;
            glist_next_in = glist_in;
            while((i < max_items) && (glist_next_in != NULL)) 
            {
                cstrptr = (const char *)glist_next_in->data;
                glist_out = g_list_append(
                    glist_out,
                    g_strdup((cstrptr == NULL) ? "" : cstrptr)
                );
                glist_next_in = glist_next_in->next;
                i++;
            }

            /* Set new output glist to the combo box's list. */
            if(glist_out != NULL)
                gtk_combo_set_popdown_strings(combo, glist_out);

            /* Free old input glist since its no longer being used. */
            if(glist_in != NULL)
            {
                g_list_foreach(glist_in, (GFunc)g_free, NULL);
                g_list_free(glist_in);
                glist_in = NULL;
            }

            /* Update input glist to be that of the output glist. */
            glist_in = glist_out;

            /* Record new glist on callback data. */
            cb_data[3] = (void *)glist_in;

            /* Call list change function and pass the new glist. */
            if(list_change_cb != NULL)
            {
                list_change_cb(
                    (GtkWidget *)combo, /* Pass combo box as widget. */
                    client_data,        /* Client data. */
                    glist_in            /* New glist. */
                );
            }
        }

	return;
}

/*
 *	Returns the GList for the combo widget created by
 *	GUIComboCreate().
 */
void *GUIComboGetList(void *w)
{
	void **cb_data;
	GtkCombo *combo = (GtkCombo *)w;
	if(combo == NULL)
	    return(NULL);

	cb_data = (void **)gtk_object_get_user_data(GTK_OBJECT(combo));
	if(cb_data == NULL)
	    return(NULL);

        /* Parse callback data, format is as follows;
         * table            Parent table that holds this combo and label.
         * label            Label (may be NULL).
         * combo            This combo widget.
         * GList            Contains list of strings in combo list.
         * max_items        Max items allowed in combo list.
         * client_data      Calling function set callback data.
         * func_cb          Calling function set callback function.
         * list_change_cb   Calling function set list change function.
         * NULL
         */

	return(cb_data[3]);
}

/*
 *      Sets the new glist for the combo widget w.
 *
 *	If the pointers for the new list and old list base match
 *	then the new list will simply be updated to the combo with
 *	a call to gtk_combo_set_popdown_strings(). If the base pointers
 *	do not match then the new list will be added and the old list will
 *	be destroyed.
 *
 *	If the given list base pointer is NULL then the current list will
 *	be destroyed.
 *
 *	In any case the given list should no longer be referenced by
 *	the calling function after this call.
 */
void GUIComboSetList(void *w, void *list)
{
	GList *glist;
	const char *cstrptr;
	void **cb_data;
        GtkCombo *combo = (GtkCombo *)w;
        if(combo == NULL)
            return;

        cb_data = (void **)gtk_object_get_user_data(GTK_OBJECT(combo));

        /* Parse callback data, format is as follows;
         * table            Parent table that holds this combo and label.
         * label            Label (may be NULL).
         * combo            This combo widget.
         * GList            Contains list of strings in combo list.
         * max_items        Max items allowed in combo list.
         * client_data      Calling function set callback data.
         * func_cb          Calling function set callback function.
         * list_change_cb   Calling function set list change function.
         * NULL
         */
	glist = (GList *)cb_data[3];

	/* Is given list NULL? */
	if(list == NULL)
	{
            gint i, max_items = (gint)cb_data[4];
	    GList	*glist_new,
			*glist_old = glist;

	    /* Create a new glist containing just empty strings. */
	    glist_new = NULL;
	    for(i = 0; i < max_items; i++)
                glist_new = g_list_append(glist_new, g_strdup(""));

            /* Was new glist created successfully? */
            if(glist_new != NULL)
	    {
		/* Set new glist to combo. */
                gtk_combo_set_popdown_strings(combo, glist_new);

		/* If old glist exists, then delete it. */
		if(glist_old != NULL)
		{
		    g_list_foreach(glist_old, (GFunc)g_free, NULL);
		    g_list_free(glist_old);
		    glist_old = NULL;
		}

		/* Update pointer to new glist. */
		cb_data[3] = (void *)glist_new;
		glist = glist_new;
	    }
	}
	/* Given list matches current list's base pointer values? */
	else if((void *)list == (void *)glist)
	{
	    /* Just update list on combo then. */
	    gtk_combo_set_popdown_strings(combo, glist);

	    /* No need to change pointer on callback data. */

	    /* No need to deallocate given list. */
	}
	else
	{
	    /* New and current list base pointers do not match and
	     * current glist may be NULL.
	     */
	    gint i, max_items;
	    GList	*glist_new, *glist_in,
			*glist_old = glist;

	    /* Make a copy the given list as glist_new and limit the
	     * number of items to max_items.
	     */
	    i = 0;
	    max_items = (gint)cb_data[4];
	    glist_new = NULL;		/* New glist. */
	    glist_in = (GList *)list;	/* Input glist. */
	    while((i < max_items) && (glist_in != NULL))
	    {
		cstrptr = (const char *)glist_in->data;
		glist_new = g_list_append(
		    glist_new,
		    g_strdup((cstrptr == NULL) ? "" : cstrptr)
		);
		glist_in = glist_in->next;
		i++;
	    }

	    /* Destroy the given input list, it is no longer needed. */
	    glist_in = (GList *)list;	/* Input glist. */
	    list = NULL;		/* Mark input list as NULL. */
	    if(glist_in != NULL)
	    {
                g_list_foreach(glist_in, (GFunc)g_free, NULL);
                g_list_free(glist_in);
                glist_in = NULL;
	    }

	    /* Is new coppied glist valid? */
	    if(glist_new != NULL)
	    {
                /* Set new glist to combo. */
                gtk_combo_set_popdown_strings(combo, glist_new);

                /* If old glist exists, then delete it. */
                if(glist_old != NULL)
                { 
                    g_list_foreach(glist_old, (GFunc)g_free, NULL);
                    g_list_free(glist_old);
                    glist_old = NULL;
                }

                /* Update pointer to new glist on callback data. */
                cb_data[3] = (void *)glist_new;
                glist = glist_new;
	    }
	}
}

/*
 *	Resets all values in the combo widget.
 */
void GUIComboClearAll(void *w)
{
	GtkCombo *combo = (GtkCombo *)w;
	if(combo == NULL)
	    return;

	/* Clear text entry. */
 	gtk_entry_set_text(GTK_ENTRY(combo->entry), "");
	gtk_entry_set_position(GTK_ENTRY(combo->entry), 0);

	/* Clear combo's glist by setting a new one as NULL. */
	GUIComboSetList(combo, NULL);
}


/*
 *	Creates a new menu bar.
 */
void *GUIMenuBarCreate(void **accel_group)
{
	if(accel_group != NULL)
	    *accel_group = gtk_accel_group_new();

	return(gtk_menu_bar_new());
}

/*
 *	Creates a new menu.
 */
void *GUIMenuCreate(void)
{
	return(gtk_menu_new());
}

/*
 *	Creates a new menu item and appends it to the menu.
 *	Returns to the pointer to the new menu item.
 *
 *	The functional_widget_rtn can be NULL, if it is not NULL
 *	it will point to the functinoal widget for the menu item.
 *
 *	For instance if the menu item type was GUI_MENU_ITEM_TYPE_LABEL
 *	then functional_widget_rtn will point to the label widget.
 */
void *GUIMenuItemCreate(
	void *menu, int type,	/* One of GUI_MENU_ITEM_TYPE_*. */
	void *accel_group,
	u_int8_t **icon, const char *label,
	int accel_key, unsigned int accel_mods,
	void **functional_widget_rtn,
	void *client_data,
	void (*func_cb)(GtkWidget *w, void *)
)
{
	GtkWidget *w, *menu_item = NULL, *parent, *parent2;


	if(menu == NULL)
	    return(NULL);

	/* Reset functional widget ID. */
	if(functional_widget_rtn != NULL)
	    (*functional_widget_rtn) = NULL;

	/* Create by type. */
	switch(type)
	{
	  case GUI_MENU_ITEM_TYPE_LABEL:
	  case GUI_MENU_ITEM_TYPE_SUBMENU:
            /* Create new menu item. */
            w = gtk_menu_item_new();
	    if(!GTK_WIDGET_NO_WINDOW(w))
		gtk_widget_add_events(w, GDK_ENTER_NOTIFY_MASK);
            gtk_menu_append(GTK_MENU(menu), w);
            gtk_widget_show(w);
            menu_item = w;
            parent = w;

	    /* Create an hbox to hold icon and label(s). */
	    w = gtk_table_new(
		1,
		((accel_key) ? 3 : 2),
		FALSE
	    ); 
	    gtk_container_add(GTK_CONTAINER(parent), w);
	    gtk_widget_show(w);  
	    parent = w;

	    /* Create icon, if icon is NULL then an empty fixed
	     * widget would have been returned.
	     */
	    w = (GtkWidget *)GUICreateMenuItemIcon(icon);
	    if(w != NULL)
	    {
                gtk_table_attach(GTK_TABLE(parent), w,
                    0, 1,
                    0, 1,
                    GTK_SHRINK,
                    GTK_SHRINK,
                    0, 0
                );
		gtk_widget_show(w);
	    }

	    /* Create label. */
	    w = gtk_hbox_new(FALSE, 0);
	    gtk_table_attach(GTK_TABLE(parent), w,
                1, 2,
                0, 1,
                GTK_SHRINK | GTK_EXPAND | GTK_FILL,
                GTK_SHRINK,
                0, 0
            );
            gtk_widget_show(w);
	    parent2 = w;

	    w = gtk_label_new((label == NULL) ? "(null)" : label);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	    gtk_widget_show(w);

	    /* Record label as functional widget. */
	    if(functional_widget_rtn != NULL)
	        (*functional_widget_rtn) = w;

	    /* Set callback for menu item? */
	    if(func_cb != NULL)
	    {
                /* Allocate callback data, this will be free()'ed
                 * when the menu item is destroyed.
                 */
	        void **cb_data = (void **)calloc(4, sizeof(void *));
	        if(cb_data != NULL)
		{
		    cb_data[0] = w;	/* Functional widget. */
		    cb_data[1] = client_data;
		    cb_data[2] = (void *)func_cb;
		    cb_data[3] = NULL;
		}
                gtk_signal_connect(
                    GTK_OBJECT(menu_item), "destroy",
                    GTK_SIGNAL_FUNC(GUIMenuDestroyCB),
                    (gpointer)cb_data
                ); 
		gtk_signal_connect_object(
		    GTK_OBJECT(menu_item), "activate",
		    GTK_SIGNAL_FUNC(GUIMenuActivateCB),
		    (gpointer)cb_data
		);
	    }

            /* Create accelerator label and add accelerator? */
	    if(accel_key)
	    {
		char text[256];
		char text2[80];

		text[0] = ' ';		/* Add one space. */
		text[1] = '\0';
		if(accel_mods & GDK_LOCK_MASK)
                    strcat(text, "Lock+");
		if(accel_mods & GDK_CONTROL_MASK)
                    strcat(text, "Ctrl+");
		if(accel_mods & GDK_SHIFT_MASK)
		    strcat(text, "Shift+");

		if(accel_key & 0xffffff00)
		{
		    strcat(text, GUIGetKeyName(accel_key));
		}
		else
		{
		    sprintf(text2, "%c", toupper(accel_key));
		    strcat(text, text2);
		}

		w = gtk_label_new(text);
		gtk_table_attach(GTK_TABLE(parent),
                    w,
                    2, 3,
                    0, 1,
                    GTK_SHRINK,
                    GTK_SHRINK,
                    0, 0
                );
                gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
                gtk_widget_show(w);

		if(accel_group != NULL)
		    gtk_widget_add_accelerator(
			menu_item, "activate", (GtkAccelGroup *)accel_group,
			accel_key, accel_mods,
			GTK_ACCEL_VISIBLE
		    );
	    }
	    break;

	  case GUI_MENU_ITEM_TYPE_CHECK:
            /* Create new check menu item. */
            w = gtk_check_menu_item_new_with_label(
		(label == NULL) ? "(null)" : label
	    );
            gtk_menu_append(GTK_MENU(menu), w);
            gtk_widget_show(w);
            menu_item = w;
            parent = w;

	    /* Record check menu item as the functional widget. */
            if(functional_widget_rtn != NULL)
                *functional_widget_rtn = w;

            /* Set callback for the check menu item? */
            if(func_cb != NULL)
            {
                /* Allocate callback data, this will be free()'ed
		 * when the menu item is destroyed.
                 */
                void **cb_data = (void **)calloc(4, sizeof(void *));
                if(cb_data != NULL)
                {
                    cb_data[0] = w;     /* Functional widget. */
                    cb_data[1] = client_data;
                    cb_data[2] = (void *)func_cb;
                    cb_data[3] = NULL;
                }
                gtk_signal_connect(
                    GTK_OBJECT(menu_item), "destroy",
                    GTK_SIGNAL_FUNC(GUIMenuDestroyCB),
                    (gpointer)cb_data 
                );
                gtk_signal_connect_object(
                    GTK_OBJECT(menu_item), "activate",
                    GTK_SIGNAL_FUNC(GUIMenuActivateCB),
                    (gpointer)cb_data
                );
            }

            /* Add accelerator? */
            if((accel_key) && (accel_group != NULL))
            {
                gtk_widget_add_accelerator(
                    menu_item, "activate", (GtkAccelGroup *)accel_group,
                    accel_key, accel_mods,
                    GTK_ACCEL_VISIBLE
                );
            }
            break;

	  case GUI_MENU_ITEM_TYPE_SEPARATOR:
            /* Create new menu item. */
            w = gtk_menu_item_new();
            gtk_menu_append(GTK_MENU(menu), w);
            gtk_widget_show(w);
            menu_item = w;
            parent = w;
                
 	    /* Create horizontal separator. */
            w = gtk_hseparator_new();
            gtk_container_add(GTK_CONTAINER(parent), w);
	    GTK_WIDGET_UNSET_FLAGS(GTK_WIDGET(menu_item), GTK_SENSITIVE);
            gtk_widget_show(w);

	    /* Record horizontal separator as the functional widget. */
            if(functional_widget_rtn != NULL)
                *functional_widget_rtn = w;
	    break;
	}



	return(menu_item);
}


/*
 *	Sets the hint message for the given menu item widget.
 *
 *	Since menu item widgets can't have tooltips like other widgets,
 *	this one calls a callback when the pointer enters the menu
 *	item widget and the call back will decide on what to do.
 */
unsigned int GUISetMenuItemEnterCB(
	void *w,
	int (*enter_cb)(void *, void *, void *),
	void *client_data
)
{
	if((w != NULL) && (enter_cb != NULL))
	{
	    return(gtk_signal_connect(
		GTK_OBJECT((GtkWidget *)w), "enter_notify_event",
		GTK_SIGNAL_FUNC(enter_cb),
		client_data
	    ));
	}
	else
	{
	    return(0);
	}
}

/*
 *	Adds the menu to the menu_bar, giving it the label value of
 *	menu_bar_item_label. Returns the pointer to the menu bar label
 *	widget or NULL on error.
 */
void *GUIMenuAddToMenuBar(
	void *menu_bar, void *menu,
	const char *menu_bar_item_label,
	int align	/* One of GUI_MENU_BAR_ALIGN_*. */
)
{
	GtkWidget *w = NULL;

	if((menu_bar == NULL) ||
           (menu == NULL)
	)
	    return(w);

	w = gtk_menu_item_new_with_label(
	    (menu_bar_item_label == NULL) ? "(null)" : menu_bar_item_label
	);
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(w), GTK_WIDGET(menu));
	gtk_menu_bar_append(GTK_MENU_BAR(menu_bar), GTK_WIDGET(w));
	gtk_widget_show(w);

	if(align == GUI_MENU_BAR_ALIGN_RIGHT)
	    gtk_menu_item_right_justify(GTK_MENU_ITEM(w));

	return(w);
}

/*
 *	Same as GUIMenuAddToMenuBar() except a pixmap is placed to the
 *	left of the menu item on the menu bar.
 */
void *GUIMenuAddToMenuBarPixmapH(
        void *menu_bar, void *menu,
        const char *menu_bar_item_label, const u_int8_t **icon,
        int align	/* One of GUI_MENU_BAR_ALIGN_*. */
)
{
	return(GUIMenuAddToMenuBar(
	    menu_bar, menu, menu_bar_item_label, align
	));
}


/*
 *	Links the menu item to the sub menu.
 */
void GUIMenuItemSetSubMenu(
        void *menu_item, void *sub_menu
)
{
	if(menu_item == NULL)
	    return;

	if(sub_menu == NULL)
	    gtk_menu_item_remove_submenu((GtkMenuItem *)menu_item);
	else
	    gtk_menu_item_set_submenu(
		(GtkMenuItem *)menu_item,
		(GtkWidget *)sub_menu
	    );

	return;
}

/*
 *	Creates a pull out widget horizontally, returning the hbox which
 *	the calling function can pack child widgets into.
 *
 *	The given parent must be a hbox or a vbox.
 *
 *	The toplevel_width and toplevel_height specify the size of the
 *	toplevel window that will be used when this widget is `pulled
 *	out'.
 *
 *	gtk_widget_show() will be called for you, the client function need
 *	not call it.
 */
void *GUIPullOutCreateH(  
        void *parent_box,
	int homogeneous, int spacing,		/* Of client vbox. */
	int expand, int fill, int padding,	/* Of holder hbox. */
        int toplevel_width, int toplevel_height,
	void *pull_out_client_data,
	void (*pull_out_cb)(void *, void *),
	void *push_in_client_data,
        void (*push_in_cb)(void *, void *)
)
{
	void **cb_data;
	GtkWidget *pull_out_btn, *holder_hbox, *client_hbox;

	if(parent_box == NULL)
	    return(NULL);

	/* Create a hbox to place into the given parent box. This hbox
	 * will hold the parenting hbox plus some other widgets.
	 */
	holder_hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(
	    (GtkBox *)parent_box, holder_hbox,
	    expand, fill, padding
	);
	gtk_widget_show(holder_hbox);

	/* Create pull out button and parent it to the holder hbox.
	 * Note that this is not really a button but an event box.
	 */
	pull_out_btn = gtk_event_box_new();
	gtk_box_pack_start(
            GTK_BOX(holder_hbox), pull_out_btn, FALSE, FALSE, 0
        );
	gtk_widget_set_usize(pull_out_btn, 10, -1);
        gtk_widget_show(pull_out_btn);


	/* Create the client hbox, which will be parented to the above
	 * holder hbox when `placed in' and reparented to a toplevel
	 * GtkWindow when `pulled out'.
	 */
	client_hbox = gtk_hbox_new(homogeneous, spacing);
	gtk_box_pack_start(
	    GTK_BOX(holder_hbox), client_hbox, TRUE, TRUE, 0
	);
        gtk_widget_show(client_hbox);


	/* Set callbacks and callback data. */

	/* Allocate and set up callback data. */
	cb_data = (void **)calloc(13, sizeof(void *));
	if(cb_data != NULL)
	{
	    /* Format is as follows (13 arguments):
	     *
	     *	client_hbox
	     *	holder_hbox
	     *	holder_window
	     *	holder_window_x
	     *	holder_window_y
	     *	holder_window_width
	     *	holder_window_height
	     *	in_place		(1 if true).
	     *  pull_out_client_data
	     *	pull_out_cb
	     *	push_in_client_data
	     *	push_in_cb
	     */
	    cb_data[0] = (void *)client_hbox;
	    cb_data[1] = (void *)holder_hbox;
	    cb_data[2] = NULL;
	    cb_data[3] = 0;
	    cb_data[4] = 0;
	    cb_data[5] = (void *)MAX(toplevel_width, 0);
	    cb_data[6] = (void *)MAX(toplevel_height, 0);
	    cb_data[7] = (void *)1;		/* Initially `pushed in'. */
	    cb_data[8] = (void *)pull_out_client_data;
	    cb_data[9] = (void *)pull_out_cb;
            cb_data[10] = (void *)push_in_client_data;
            cb_data[11] = (void *)push_in_cb;
	    cb_data[12] = NULL;

	    /* Set callback data pointer on the client hbox. */
	    gtk_object_set_user_data(GTK_OBJECT(client_hbox), cb_data);
	}

        /* Connect destroy event to holder_hbox so the callback data
	 * and related resources can be deallocated when the client
	 * functions destroy the holder_hbox.
	 */
        gtk_signal_connect(
            GTK_OBJECT(holder_hbox), "destroy",
            GTK_SIGNAL_FUNC(GUIPullOutDestroyCB),
            (gpointer)cb_data
        );

	gtk_signal_connect(
            GTK_OBJECT(pull_out_btn), "button_press_event",
            GTK_SIGNAL_FUNC(GUIPullOutPullOutBtnCB),
            (gpointer)cb_data
        );
        gtk_signal_connect_after(
            GTK_OBJECT(pull_out_btn), "expose_event",
            GTK_SIGNAL_FUNC(GUIPullOutDrawHandleCB),
            NULL                /* No callback data. */
        );
        gtk_signal_connect(
            GTK_OBJECT(pull_out_btn), "draw",
            GTK_SIGNAL_FUNC(GUIPullOutDrawHandleDrawCB),
            NULL                /* No callback data. */
        );

	return(client_hbox);
}

/*
 *	Returns the pointer to the toplevel window (if any) of the
 *	client_hbox and the geometry of that window.
 *
 *	The client box should be one created by GUIPullOutCreate*().
 */
void *GUIPullOutGetToplevelWindow(
        void *client_box,
	int *x, int *y, int *width, int *height
)
{
	void **cb_data;
	GtkWidget *w = NULL;

	if(x != NULL)
	    (*x) = 0;
	if(y != NULL)
	    (*y) = 0;
	if(width != NULL)
	    (*width) = 0;
	if(height != NULL)
	    (*height) = 0;

	if(client_box == NULL)
	    return(w);

        cb_data = (void **)gtk_object_get_user_data((GtkObject *)client_box);
	if(cb_data != NULL)
	{
            /* Format is as follows (13 arguments):
             *
             *  client_hbox
             *  holder_hbox
             *  holder_window
             *  holder_window_x
             *  holder_window_y
             *  holder_window_width
             *  holder_window_height
             *  in_place                (1 if true).
             *  pull_out_client_data
             *  pull_out_cb
             *  push_in_client_data
             *  push_in_cb   
             */
            w = (GtkWidget *)cb_data[2];

	    if((w != NULL) ? !GTK_WIDGET_NO_WINDOW(w) : 0)
	    {
		GdkWindow *window = w->window;
		gint rx, ry, rwidth, rheight, rdepth;

		gdk_window_get_geometry(
		    window,
		    &rx, &ry, &rwidth, &rheight,
		    &rdepth
		);

		if(x != NULL)
		    (*x) = rx;
		if(y != NULL)
		    (*y) = ry;
		if(width != NULL)
		    (*width) = rwidth;
		if(height != NULL)
		    (*height) = rheight;
	    }
	}

	return(w);
}

/*
 *	Pulls out the pull out box.
 */
void GUIPullOutPullOut(void *client_box)
{
	void **cb_data;

	if(client_box == NULL)
	    return;

	cb_data = (void **)gtk_object_get_user_data((GtkObject *)client_box);
	if(cb_data != NULL)
	{
            /* Format is as follows (13 arguments):
             *
             *  client_hbox
             *  holder_hbox
             *  holder_window
             *  holder_window_x
             *  holder_window_y
             *  holder_window_width
             *  holder_window_height
             *  in_place                (1 if true).
             *  pull_out_client_data
             *  pull_out_cb
             *  push_in_client_data
             *  push_in_cb
             */

	    /* In place (pushed in)? */
	    if((int)cb_data[7])
	    {
		GUIPullOutPullOutBtnCB(
		    NULL,
		    NULL,
		    (gpointer)cb_data
		);
	    }
	}

	return;
}

/*
 *      Pushes in the pull out box.
 */
void GUIPullOutPushIn(void *client_box)
{
	GtkWidget *w;
        void **cb_data;

        if(client_box == NULL)
            return;

        cb_data = (void **)gtk_object_get_user_data((GtkObject *)client_box);
        if(cb_data != NULL)
        {
            /* Format is as follows (13 arguments):
             *
             *  client_hbox
             *  holder_hbox
             *  holder_window
             *  holder_window_x
             *  holder_window_y
             *  holder_window_width
             *  holder_window_height
             *  in_place                (1 if true).
             *  pull_out_client_data
             *  pull_out_cb
             *  push_in_client_data
             *  push_in_cb
             */

	    w = (GtkWidget *)cb_data[2];

            /* Not in place (pulled out)? */
            if(!((int)cb_data[7]))
            {
		GUIPullOutCloseCB(
		    (GtkWidget *)cb_data[2],
		    NULL,
		    (gpointer)cb_data
		);
            }
        }

        return;
}            

