/***************************************************************************
 $RCSfile: mediumddv-07.cpp,v $
                             -------------------
    cvs         : $Id: mediumddv-07.cpp,v 1.2 2004/01/13 22:26:26 cstim Exp $
    begin       : Fri Nov 16 2001
    copyright   : (C) 2001 by Martin Preuss
    email       : openhbci@aquamaniac.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

/*
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef __declspec
# if BUILDING_DLL
#  define DLLIMPORT __declspec (dllexport)
# else /* Not BUILDING_DLL */
#  define DLLIMPORT __declspec (dllimport)
# endif /* Not BUILDING_DLL */
#else
# define DLLIMPORT
#endif



#include <openssl/bn.h>
#include <openssl/ripemd.h>


#include <list>
#include <openhbci/error.h>
#include <openhbci/pointer.h>
#include <openhbci/hbcistring.h>
#include <openhbci/date.h>
#include <openhbci/value.h>
#include <openhbci/balance.h>
#include <openhbci/auth.h>
#include <openhbci/interactor.h>
#include <openhbci/hbci.h>
#include <openhbci/user.h>
#include <openhbci/account.h>
#include <openhbci/bpdjob.h>
#include <openhbci/bank.h>
#include <openhbci/cryptkey.h>
#include <openhbci/deskey.h>
#include <openhbci/medium.h>
#include <mediumddv.h>

// for bad_cast exception
#include <typeinfo>

namespace HBCI {


MediumDDV::MediumDDV(const Hbci *hbci,
                     const string &cardnumber)
:Medium(hbci)
,HBCICard()
,_ismounted(false)
,_mountcount(0)
,_cardnumber(cardnumber)
,_country(280)
{
}


MediumDDV::~MediumDDV(){
  //fprintf(stderr,"~MediumDDV\n");
}


const string& MediumDDV::mediumName() const {
  return _cardnumber;
}


const string& MediumDDV::serialNumber() const {
  return _cardnumber;
}


Error MediumDDV::verify(const string &data, const string &signature){
  string hash;
  string mac;

  // make RIPEMD-160
  hash=ripe(data);
  // create MAC by the chip card
  if (!hash2MAC(hash,mac))
    return Error("MediumDDV::verify",
		 "Error on hash2MAC()",
		 0);
  // compare both strings
  if (mac==signature)
    return Error();
  return Error("MediumDDV::verify",
	       "Bad signature()",
	       0);
}


string MediumDDV::sign(const string &data){
  string hash;
  string mac;

  // make RIPEMD-160
  hash=ripe(data);
  // create MAC by the chip card
  if (!hash2MAC(hash,mac))
    throw Error("MediumDDV::sign",
		"Error on hash2MAC()",
		0);
  // return MAC
  // that's all
  return mac;
}


string MediumDDV::createMessageKey() const {
    string kL;
    string kR;
    string key;

    // create left
    if (!const_cast<MediumDDV*>(this)->getRandom(kL))
        throw Error("MediumDDV::createMessageKey",
                        "Error on getRandom()",
                        0);
    // create right
    if (!const_cast<MediumDDV*>(this)->getRandom(kR))
        throw Error("MediumDDV::createMessageKey",
                        "Error on getRandom()",
                        0);
    key=kL + kR;
    return key;
}


string MediumDDV::encryptKey(const string &srckey){
    string kL;
    string kR;
    string kLenc;
    string kRenc;
    string dstkey;

    // check sizes
    if (srckey.length()!=16)
        throw Error("MediumDDV::encryptKey",
                        "Bad length of srckey",
                        0);
    // split key
    kL=srckey.substr(0,8);
    kR=srckey.substr(8);
    // encode left
    cryptBlock(kL,kLenc);

    // encode right
    cryptBlock(kR,kRenc);
    dstkey=kLenc + kRenc;
    return dstkey;
}


string MediumDDV::decryptKey(const string &srckey){
    return encryptKey(srckey);
}


Error MediumDDV::_checkCard(){
  Error err;
  HBCICard::CardData scid;
  string cardnumber;

  try {
    // try to read the card identification data
    if (Hbci::debugLevel()>1)
      fprintf(stderr,"Getting card id.\n");
    scid=getCardId();
    if (Hbci::debugLevel()>1)
      fprintf(stderr,"Reading card number.\n");
    cardnumber=scid.cardNumber();
  }
  catch (CTError xerr) {
    Error lerr;

    if (hbci()->debugLevel()>0)
      fprintf(stderr,"Exception: %s\n",
	      xerr.errorString().c_str());
    lerr=Error(xerr.where(),
	       ERROR_LEVEL_NORMAL,
	       HBCI_ERROR_CODE_MEDIUM,
	       ERROR_ADVISE_DONTKNOW,
	       xerr.info(),
	       xerr.explanation());
    return Error("MediumDDV::_checkCard()", lerr);
  }

  // if cardnumber read is not the one we want
  if (Hbci::debugLevel()>1)
    fprintf(stderr,"Card is open, checking card number.\n");
  if (cardnumber==_cardnumber || _cardnumber.empty()) {
    // set new card number if ours is empty
    if (_cardnumber.empty())
      _cardnumber=cardnumber;
  }
  else {
    if (hbci()->debugLevel()>1)
      fprintf(stderr,"False card\n");
    return Error("MediumDDV::_checkCard",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_WRONG_MEDIUM,
		 ERROR_ADVISE_RETRY,
		 "wrong card");
  }
  // everything's fine
  return Error();
}


Error MediumDDV::_enterPin(Pointer<User> user,
			   const string &pin,
			   bool hasKeypad) {
  Error herr;
  CTError err;
  string localpin;

  if (hasKeypad) {
    if (Hbci::debugLevel()>1)
      fprintf(stderr,"Terminal has a keypad, will ask for pin.\n");
    // tell the user about pin verification
    _hbci->interactor().ref().
      msgStartInputPinViaKeypad(user);
    err=verifyPin();
    // tell the user about end of pin verification
    _hbci->interactor().ref().
      msgFinishedInputPinViaKeypad(user);
    if (!err.isOk()) {
      if (Hbci::debugLevel()>1)
	fprintf(stderr,"Bad pin.\n");
      _hbci->interactor().ref().msgStateResponse("Error: "+
						 err.errorString());
      if (err.code()==k_CTERROR_OK) {
	int ec;

	if (err.subcode1()==0x63) {
	  switch (err.subcode2()) {
	  case 0xc0: ec=HBCI_ERROR_CODE_PIN_WRONG_0; break;
	  case 0xc1: ec=HBCI_ERROR_CODE_PIN_WRONG_1; break;
	  case 0xc2: ec=HBCI_ERROR_CODE_PIN_WRONG_2; break;
	  default:   ec=HBCI_ERROR_CODE_PIN_WRONG; break;
	  } // switch
	  return Error("MediumDDV::_enterPin",
		       ERROR_LEVEL_CRITICAL,
		       ec,
		       ERROR_ADVISE_RETRY,
		       "Bad pin",
		       "verifyPin (1)");
	}
	else if (err.subcode1()==0x69 &&
		 err.subcode2()==0x83) {
	  ec=HBCI_ERROR_CODE_CARD_DESTROYED;
	  return Error("MediumDDV::_enterPin",
		       ERROR_LEVEL_CRITICAL,
		       ec,
		       ERROR_ADVISE_ABORT,
		       "Card destroyed",
		       "verifyPin (2)");
	}
	else if (err.subcode1()==0x64 &&
		 err.subcode2()==0x01) {
	  ec=HBCI_ERROR_CODE_PIN_ABORTED;
	  return Error("MediumDDV::_enterPin",
		       ERROR_LEVEL_NORMAL,
		       ec,
		       ERROR_ADVISE_ABORT,
		       "User aborted",
		       "verifyPin (3)");

	}
      }
      return Error("MediumDDV::_enterPin",
		   ERROR_LEVEL_NORMAL,
		   HBCI_ERROR_CODE_PIN_ABORTED,
		   ERROR_ADVISE_RETRY,
		   err.errorString(),
		   "verifyPin (4)");
    }
  } // if hasKeyPad
  else {
    localpin=pin;
    if (Hbci::debugLevel()>1)
      fprintf(stderr,"Terminal has NO keypad, will ask for pin.\n");
    if (localpin.length()<4) {
      herr=_hbci->authentificator().ref().getSecret(user,
						    _cardnumber,
						    localpin);
      if (!herr.isOk()) {
	if (Hbci::debugLevel()>1)
	  fprintf(stderr,"No pin entered.\n");
	_hbci->interactor().ref().msgStateResponse("Error: "+
						   herr.errorString());
	return Error("MediumDDV::_enterPin",herr);
      } // if user aborted
    } // if pin is not long enough

    // verify the pin
    if (Hbci::debugLevel()>1)
      fprintf(stderr,"Verifying cardholder pin.\n");
    err=verifyPin(localpin); // CARDHOLDER pin
    if (!err.isOk()) {
      if (Hbci::debugLevel()>1)
	fprintf(stderr,"Bad pin.\n");
      _hbci->interactor().ref().msgStateResponse("Error: "+err.errorString());
      if (err.code()==k_CTERROR_OK) {
	int ec;

	if (err.subcode1()==0x63) {
	  switch (err.subcode2()) {
	  case 0xc0: ec=HBCI_ERROR_CODE_PIN_WRONG_0; break;
	  case 0xc1: ec=HBCI_ERROR_CODE_PIN_WRONG_1; break;
	  case 0xc2: ec=HBCI_ERROR_CODE_PIN_WRONG_2; break;
	  default:   ec=HBCI_ERROR_CODE_PIN_WRONG; break;
	  } // switch
	  return Error("MediumDDV::_enterPin",
		       ERROR_LEVEL_CRITICAL,
		       ec,
		       ERROR_ADVISE_RETRY,
		       "Bad pin",
		       "verifyPin (5)");
	}
	else if (err.subcode1()==0x69 &&
		 err.subcode2()==0x83)
	  ec=HBCI_ERROR_CODE_CARD_DESTROYED;
	return Error("MediumDDV::_enterPin",
		     ERROR_LEVEL_CRITICAL,
		     ec,
		     ERROR_ADVISE_ABORT,
		     "Card destroyed",
		     "verifyPin (6)");
      }

      return Error("MediumDDV::_enterPin",
		   ERROR_LEVEL_CRITICAL,
		   HBCI_ERROR_CODE_PIN_ABORTED,
		   ERROR_ADVISE_ABORT,
		   err.errorString(),
		   "verifyPin (7)");
    }
  } // if no keyPad

  return Error();
}


// This is for LibChipCard v0.7 and better
Error MediumDDV::mountMedium(const string &pin){
  try {
    Error herr;
    CTError err;
    string localpin;
    unsigned int rflags;
    unsigned int rstatus;
    int rv;
    string cardnumber;
    int requestid;
    HBCICard::CardData scid;
    bool haveacard;
    bool skipask;

    if (isMounted()) {
      _mountcount++;
      return Error();
    }

    if (Hbci::debugLevel()>1)
      fprintf(stderr,"Checking if any card is inserted.\n");
    _hbci->interactor().ref().abort(false);

    rv=startWaitForCard(requestid,
			false,
			0,
			0,
			CHIPCARD_STATUS_INSERTED |
			CHIPCARD_STATUS_PROCESSOR,
			CHIPCARD_STATUS_INSERTED |
			CHIPCARD_STATUS_PROCESSOR|
			CHIPCARD_STATUS_LOCKED_BY_OTHER);
    if (rv!=CHIPCARD_SUCCESS) {
      if (Hbci::debugLevel()>1)
	fprintf(stderr, "1:Could not mount DDV card: %d\n",rv);
      return Error("MediumDDV::mountMedium()",
		   ERROR_LEVEL_NORMAL,
		   HBCI_ERROR_CODE_NO_CARD,
		   ERROR_ADVISE_ABORT,
		   "startWaitForCard.");
    }

    // check with a short timeout
    rv=checkWaitForCard(requestid,2,rflags,rstatus);
    if (rv!=CHIPCARD_SUCCESS &&
	rv!=CHIPCARD_ERROR_NO_MESSAGE) {
      if (Hbci::debugLevel()>1)
	fprintf(stderr, "2:Could not mount DDV card: %d\n",rv);
      stopWaitForCard(requestid);
      return Error("MediumDDV::mountMedium()",
		   ERROR_LEVEL_NORMAL,
		   HBCI_ERROR_CODE_NO_CARD,
		   ERROR_ADVISE_ABORT,
		   "checkWaitForCard.");
    }

    haveacard=(rv==CHIPCARD_SUCCESS);
    skipask=false;
    // wait for correct card to be inserted
    while(1) {
      while(!haveacard) {
	if (!skipask) {
	  // no card inserted, ask user to do so
	  if (Hbci::debugLevel()>1)
	    fprintf(stderr,"Card is not inserted, asking user.\n");
	  if (!_hbci->interactor().ref().
	      msgInsertMediumOrAbort(owner(),MediumTypeCard)) {
	    if (Hbci::debugLevel()>1)
	      fprintf(stderr,"User aborted.\n");
	    stopWaitForCard(requestid);
	    return Error("MediumDDV::mountMedium()",
			 ERROR_LEVEL_NORMAL,
			 HBCI_ERROR_CODE_PIN_ABORTED,
			 ERROR_ADVISE_ABORT,
			 "No card, user aborted.");
	  }
	} // if !skipask
	skipask=false;
	fprintf(stderr,"Checking again.\n");
	rv=checkWaitForCard(requestid,10,rflags,rstatus);
	if (rv!=CHIPCARD_SUCCESS) {
	  if (rv!=CHIPCARD_ERROR_NO_MESSAGE) {
	    stopWaitForCard(requestid);
	    if (Hbci::debugLevel()>1)
	      fprintf(stderr, "3:Could not mount DDV card: %d\n",rv);
	    return Error("MediumDDV::mountMedium()",
			 ERROR_LEVEL_NORMAL,
			 HBCI_ERROR_CODE_NO_CARD,
			 ERROR_ADVISE_ABORT,
			 "checkWaitForCard.");
	  }
	}
	else
	  break;
      } // while waiting for a card to be inserted
      // open card
      err=openCard();
      if (!err.isOk()) {
	stopWaitForCard(requestid);
	if (Hbci::debugLevel()>1)
	  fprintf(stderr, "4:Could not mount DDV card: %d\n",rv);
	return Error("MediumDDV::mountMedium()",
		     ERROR_LEVEL_NORMAL,
		     HBCI_ERROR_CODE_NO_CARD,
		     ERROR_ADVISE_ABORT,
		     "openCard.");
      }
      // check card
      herr=_checkCard();
      if (!herr.isOk()) {
	// error checking the card, so close it
	err=closeCard();
	if (!err.isOk()) {
	  stopWaitForCard(requestid);
	  return Error("MediumDDV::mountMedium()",
		       ERROR_LEVEL_NORMAL,
		       HBCI_ERROR_CODE_NO_CARD,
		       ERROR_ADVISE_ABORT,
		       err.errorString());
	}
	if (herr.code()==HBCI_ERROR_CODE_WRONG_MEDIUM) {
	  // false card, ask user to insert the correct one
	  if (!_hbci->interactor().ref().
	      msgInsertCorrectMediumOrAbort(owner(),
					    MediumTypeCard)) {
	    stopWaitForCard(requestid);
	    return Error("MediumDDV::mountMedium()",
			 ERROR_LEVEL_NORMAL,
			 HBCI_ERROR_CODE_PIN_ABORTED,
			 ERROR_ADVISE_RETRY,
			 "user aborted");
	  }
	  skipask=true;
	}
	else {
	  stopWaitForCard(requestid);
	  return Error("MediumDDV::mountMedium()",herr);
	}
      } // if checkcard is not ok
      else {
	break;
      }
      haveacard=false;
    } // while waiting for correct card

    // card is ok, we dont want any other card
    stopWaitForCard(requestid);
    while(1) {
      herr=_enterPin(owner(),pin,rflags & CHIPCARD_READERFLAGS_KEYPAD);
      if (!herr.isOk()) {
	  //if (herr.advise()!=ERROR_ADVISE_RETRY) {
	  // no retry, so return
	  closeCard();
	  return Error("MediumDDV::mountMedium()",herr);
	  //}
      }
      else
	break;
    } // while waiting for pin input

    _mountcount++;
    return Error();
  } // try
  catch (CTError xerr) {
    Error lerr;

    lerr=Error(xerr.where(),
	       ERROR_LEVEL_NORMAL,
	       HBCI_ERROR_CODE_MEDIUM,
	       ERROR_ADVISE_DONTKNOW,
	       xerr.info(),
	       xerr.explanation());
    return Error("MediumDDV::mountMedium()", lerr);
  }
}


Error MediumDDV::unmountMedium(const string&){
    CTError err;

    if (--_mountcount>0)
      return Error();
    _mountcount=0;
    err=closeCard();
    if (!err.isOk(0x62))
      // just check for IO error, do not check SW1/2
      return Error("MediumDDV::unmountMedium()",
		   "Could not unmount chip card.",
		   0);
    return Error();
}


bool MediumDDV::isMounted() {
  if (_mountcount<1)
    return false;
  if (!const_cast<MediumDDV*>(this)->isConnected()) {
    _mountcount=0;
    return false;
  }

  return true;
}


string MediumDDV::mediumId() const {
    string result;
    const_cast<MediumDDV*>(this)->readCID();
    result="@";
    result+=String::num2string(_cid.length());
    result+="@";
    result+=_cid;
    return result;
}


void MediumDDV::readCID(){
  if (!getCID(_cid).isOk())
    throw Error("MediumDDV::readCID()",
		"Could not read chip card.",
		0);
}


unsigned int MediumDDV::nextSEQ(){
    unsigned int seq;

    if (!readSEQ(seq))
        throw Error("MediumDDV::nextSEQ()",
                        "Could not read chip card.",
                        0);
    seq++;
    if (!writeSEQ(seq))
        throw Error("MediumDDV::nextSEQ()",
                        "Could not write chip card.",
                        0);
    return seq;
}


int MediumDDV::signKeyNumber() const {
    return 0;
}


int MediumDDV::signKeyVersion() const {
    return 0;
}


int MediumDDV::cryptKeyNumber() const {
    return 0;
}


int MediumDDV::cryptKeyVersion() const {
    return 0;
}


Error MediumDDV::selectContext(int country,
			       const string &instcode,
			       const string &userid){
  HBCICard::instituteData sid;
  int i;
  int validentries;

  validentries=0;
  // if the context is the same as the current then do nothing
  if (_instcode==instcode && _userid==userid)
    return Error();

  // try to find the context on chip card
  for (i=1; i<6; i++) {
    try {
      sid=getInstituteData(i);
    }
    catch (CTError xerr) {
      Error lerr;

      if (hbci()->debugLevel()>0)
	fprintf(stderr,"Exception: %s\n",
		xerr.errorString().c_str());
      lerr=Error(xerr.where(),
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_MEDIUM,
		 ERROR_ADVISE_DONTKNOW,
		 xerr.info(),
		 xerr.explanation());
      return Error("MediumDDV::selectContext()", lerr);
    }

    // fist check if the entry is valid
    if (sid.country()!=0 && sid.code()!="00000000") {
      validentries++;
      if (sid.code()==instcode)
	if (sid.user()==userid) {
	  // ok, we found it, set new context
	  _country=country;
	  _instcode=instcode;
	  _userid=userid;
	  _addr=sid.address();
	  return Error();
	}
    } // if valid
  } // for
  // ok, we did not find the context, now check why
  if (!validentries) {
#if DEBUGMODE>0
    fprintf(stderr,
	    "Warning: Your card has no valid institute entries.\n");
#endif
    _country=country;
    _instcode=instcode;
    _userid=userid;
    return Error();
  }
  return Error("MediumDDV::selectContext",
	       ERROR_LEVEL_NORMAL,
	       HBCI_ERROR_CODE_INVALID,
	       ERROR_ADVISE_DONTKNOW,
	       "no matching entry found");
}


Error MediumDDV::getContext(int num,
			    int &countrycode,
			    string &instcode,
			    string &userid,
			    string &server) const {
  HBCICard::instituteData sid;

  if (num<1)
    return Error("MediumDDV::selectContext",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_INVALID,
		 ERROR_ADVISE_DONTKNOW,
		 "bad context number");
  if (!const_cast<MediumDDV*>(this)->isMounted()){
    return Error("MediumDDV::selectContext",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_INVALID,
		 ERROR_ADVISE_DONTKNOW,
		 "medium not mounted");
  }
  try {
    sid=const_cast<MediumDDV*>(this)->getInstituteData(num);
  }
  catch (CTError xerr) {
    Error lerr;

    if (hbci()->debugLevel()>0)
      fprintf(stderr,"Exception: %s\n",
	      xerr.errorString().c_str());
    lerr=Error(xerr.where(),
	       ERROR_LEVEL_NORMAL,
	       HBCI_ERROR_CODE_MEDIUM,
	       ERROR_ADVISE_DONTKNOW,
	       xerr.info(),
	       xerr.explanation());
    return Error("MediumDDV::getContext()", lerr);
  }

  countrycode=sid.country();
  instcode=sid.code();
  userid=sid.user();
  server=sid.address();
  return Error();
}

Error MediumDDV::changePIN() {
  return Error("MediumDDV::changePIN",
	       "can't change PIN on DDV cards",
	       0);
}


Error MediumDDV::changeContext(int context, int country,
			       const string instcode,
			       const string userid, const string custid,
			       const string server) {
  Error err;

  if (context<1)
    return Error("MediumDDV::changeContext",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_INVALID,
		 ERROR_ADVISE_DONTKNOW,
		 "bad context number");

  err=mountMedium();
  if (!err.isOk())
    return Error("MediumDDV::changeContext", err);

  HBCICard::instituteData sid;

  try {
	sid=const_cast<MediumDDV*>(this)->getInstituteData(context);

	if (0 != country)
	  sid.setCountry(country);
	if (! instcode.empty())
	  sid.setCode(instcode);
	if (! userid.empty())
	  sid.setUser(userid);
	if (! server.empty())
	  sid.setAddress(server);
	
	const_cast<MediumDDV*>(this)->putInstituteData(context, sid);
  }
  catch (CTError xerr) {
    Error lerr;
    unmountMedium();

    if (hbci()->debugLevel()>0)
      fprintf(stderr,"Exception: %s\n",
	      xerr.errorString().c_str());
    lerr=Error(xerr.where(),
	       ERROR_LEVEL_NORMAL,
	       HBCI_ERROR_CODE_MEDIUM,
	       ERROR_ADVISE_DONTKNOW,
	       xerr.info(),
	       xerr.explanation());
    return Error("MediumDDV::getContext()", lerr);
  }

  return unmountMedium();
}


MediumDDV::CallBackResult MediumDDV::doCallBack(bool first){
  if (!_hbci->interactor().ref().keepAlive())
    return CallBackAbort;
  return CallBackContinue;
}


} // namespace HBCI

HBCI_Medium *HBCI_MediumDDV_Medium(HBCI_MediumDDV *h)
{
    return h;
}
HBCI_MediumDDV *HBCI_Medium_MediumDDV(HBCI_Medium *h)
{
    try {
	return dynamic_cast<HBCI_MediumDDV*>(h);
    } 
    catch (bad_cast) {
	fprintf(stderr, 
		"HBCI_Medium_MediumDDV: Caught bad_cast exception.\n");
	return 0l;
    }
}



