/***************************************************************************
 $RCSfile: bankimpl.h,v $
 -------------------
 cvs         : $Id: bankimpl.h,v 1.6 2003/04/04 23:40:49 cstim Exp $
 begin       : Wed Jun 12 2002
 copyright   : (C) 2002 by Martin Preuss
 email       : openhbci@aquamaniac.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef HBCIBANKIMPL_H
#define HBCIBANKIMPL_H

namespace HBCI {
  class BankImpl;
}

#include <string>
#include <list>

#include <openhbci/pointer.h>
#include <openhbci/hbci.h>
#include <openhbci/account.h>
#include <openhbci/user.h>
#include <openhbci/bpdjob.h>
#include <openhbci/date.h>
#include <openhbci/bank.h>
#include <openhbci/bankparams.h>

namespace HBCI {
/**
 * @short Implementation of the abstract base class Bank.
 *
 * This class represents a full institute. It contains a list of
 * its customers as well as a list of its accounts. There is also a list
 * of institute messages which this institute has send to us.
 *
 * This class should only be visible inside OpenHBCI -- all external
 * applications are only supposed to use the interface class
 * Bank.
 *
 * @author Martin Preuss<martin@libchipcard.de>
 */
class DLLIMPORT BankImpl: public Bank, public bankParams {
private:
    const Hbci *_hbci;
    int _hbciVersion;
    list<Pointer<Account> > _accounts;
    list<Pointer<User> > _users;
    list<instituteMessage> _messages;

public:
    BankImpl(const Hbci *hbci,
             int country = 280,
             const string &code = "",
             const string &server = "",
             int hbciversion = HBCI_VERSION_201);

    /**
     * @author Martin Preuss <martin@libchipcard.de>
     */
    BankImpl(const Hbci *hbci, 
		 const bankParams &p,
		 int hbciversion=HBCI_VERSION_201);
    virtual ~BankImpl();

    BankImpl &operator=(const bankParams &p);

    /** Returns the central HBCI object that controls everything. */
    const Hbci *hbci() const { return _hbci;};

    /**
     * Get the HBCI version.
     * This is the version of the HBCI protocol to be used with this bank.
     */
    int hbciVersion() const { return _hbciVersion;};

    /**
     * Set the HBCI protocol version to be used with this bank.
     */
    void setHbciVersion(int v) { _hbciVersion=v;};

    /**
     * @author Martin Preuss <martin@libchipcard.de>
     */
    Pointer<User> findUser(const string &userid) const;

    /**
     * Searches for a customer in the banks internal list.
     * No wildcards or jokers allowed.
     * @param id userid to look for
     * @author Martin Preuss <martin@libchipcard.de>
     */
    virtual Pointer<Customer>
        findCustomer(const string &id) const;

    /**
     * @author Martin Preuss <martin@libchipcard.de>
     */
    Pointer<Account> findAccount(const string &nr,
                                         const string &subid="") const;

    /**
     * @author Martin Preuss <martin@libchipcard.de>
     */
    const list<Pointer<Account> > &accounts() const;

    /**
     * @author Martin Preuss <martin@libchipcard.de>
     */
    const list<Pointer<User> > &users() const;

    /**
     * @author Martin Preuss <martin@libchipcard.de>
     */
    void addAccount(Pointer<Account> a);

    /**
     * @author Martin Preuss <martin@libchipcard.de>
     */
    void addUser(Pointer<User> c);

    /**
     * Removes a customer from the internal list. The object is
     * not destroyed, but you should not use it anymore.
     * @author Martin Preuss <martin@libchipcard.de>
     */
    void removeUser(Pointer<User> c);

    /**
     * Removes an account from the internal list. The object is
     * not destroyed, but you should not use it anymore.
     * @author Martin Preuss <martin@libchipcard.de>
     */
    void removeAccount(Pointer<Account> a);

    const list<instituteMessage> &messages() const 
	{ 
	    return _messages;
	};
    void addInstituteMessage(const instituteMessage &msg)
	{
	    _messages.push_back(msg); 
	};
    void deleteInstituteMessage(const instituteMessage &msg) 
	{
	    _messages.remove(msg);
	};


    /**
     * Return the version of the "Bank Parameter Daten" (BPD).
     *
     * The BPD contains information about the institute, such as address of
     * the server, country code, bank code etc. If these data change on the
     * server side then the server will increment this number. Upon next
     * dialog initialisation this library checks the versions and receives
     * the new BPD if necessary.
     */
    int bpdVersion() const { return bankParams::version();};
    void setBPDVersion(int i) { bankParams::setVersion(i);};

    int countryCode() const { return bankParams::countryCode();};
    void setCountryCode(int i) { bankParams::setCountryCode(i);};

    const string &bankCode() const { return bankParams::instituteCode();};
    void setBankCode(const string &s) { bankParams::setInstituteCode(s);};

    /** Returns the name of this Bank. This name is the one that the
     * bank sends back upon initally connecting. */
    const string &name() const { return bankParams::name();};
    void setName(const string &s) { bankParams::setName(s);};

    const list<int> &languages() const { return bankParams::languages();};
    void addLanguage(int l) { bankParams::addLanguage(l);};

    list<int> supportedVersions() const 
	{ 
	    return bankParams::supportedVersions();
	};
    void addVersion(int i) { bankParams::addVersion(i);};

    int maxMessageSize() const { return bankParams::maxMessageSize();};
    void setMaxMessageSize(int i) { bankParams::setMaxMessageSize(i);};

    int maxDifferentActions() const 
	{ 
	    return bankParams::maxDifferentActions();
	};
    void setMaxDifferentActions(int i) 
	{ 
	    bankParams::setMaxDifferentActions(i);
	};

    /**
     *
     */
    int language() const { return bankParams::language();};
    void setLanguage(int l) {bankParams::setLanguage(l);};

    /**
     * Supported type of protocolstack:
     * 1 == T-Online (ETSI 300 072); 2 == TCP/IP
     */
    int type() const { return bankParams::type();};
    void setType(int i) { bankParams::setType(i);};

    /**
     * Communication address. Depends on type of protocol stack: 
     * If T-Online, addr is the address of the Gateway as a number.
     * If TCP/IP, addr is the IP address of the HBCI server in 
     * dotted notation, e.g. "123.123.123.123".
     */
    const string &addr() const { return bankParams::addr();};
    void setAddr(const string &s) { bankParams::setAddr(s);};

    /**
     * Suffix of the communication address. Depends on type of protocol stack:
     * If T-Online, suffix is the region code of the service.
     * If TCP/IP, this is ignored.
     */
    const string &suffix() const { return bankParams::suffix();};
    void setSuffix(const string &s) { bankParams::setSuffix(s);};

    /**
     * Filter mechanism. If the transport stack needs a conversion
     * from 8bit- to 7bit-characters (as is the case for TCP/IP,
     * according to HBCI spec), this parameter specifies which filter
     * to use. Can be:
     * "MIM" == MIME Base 64
     * "UUE" == Uuencode / Uudecode
     */
    const string &filter() const { return bankParams::filter();};
    void setFilter(const string &s) { bankParams::setFilter(s);};

    /** Version number of the filter, if one was specified.
     */
    int filterVersion() const { return bankParams::filterVersion();};
    void setFilterVersion(int i) { bankParams::setFilterVersion(i);};

    /**
     * List of supported jobs.
     */
    list<bpdJob> supportedJobs() const
	{
	    return bankParams::supportedJobs();
	};
    void addJob(bpdJob j) { bankParams::addJob(j); };

    /**
     * Searches for a job which matches the given criteria.
     */
    const bpdJob *findJob(const string &segname, 
			  int minv=0, int maxv=9999) const
	{
	    return bankParams::findJob(segname,minv,maxv);
	};


};
} /* namespace HBCI */


#endif /* HBCIBANKIMPL_H */



