/***************************************************************************
                          hbci.h  -  description
                             -------------------
    begin                : Sun Nov 18 2001
    copyright            : (C) 2001 by Martin Preuss
    email                : openhbci@aquamaniac.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


#ifndef HBCI_H
#define HBCI_H

/** @file hbci.h
 *
 * @short Definition of the central object HBCI::Hbci and its C
 * wrapper @ref HBCI_Hbci. */

#ifdef __cplusplus
#include <string>
#include <list>
#include <iostream>
#include <openhbci/dllimport.h>
#include <openhbci/pointer.h>
namespace HBCI {
  class Hbci;
};
/** @ingroup HBCI_Hbcig 
 * @short The C accessible type of HBCI::Hbci. */
typedef struct HBCI::Hbci HBCI_Hbci;
#include <openhbci/auth.h>
#else /* __cplusplus */
typedef struct HBCI_Hbci HBCI_Hbci;
#endif /* __cplusplus */
#include <openhbci/interactor.h>


enum {
    HBCI_VERSION_201=201,
    HBCI_VERSION_210=210,
    HBCI_VERSION_220=220
};
enum {
    HBCI_SECURITY_DDV=1,
    HBCI_SECURITY_RDH=2
};
enum HBCI_Language {
    HBCI_LANGUAGE_GERMAN=1,
    HBCI_LANGUAGE_ENGLISCH=2,
    HBCI_LANGUAGE_FRENCH=3
};
enum {
    HBCI_SYNC_SYSTEMID=0,
    HBCI_SYNC_MSGNUMBER=1,
    HBCI_SYNC_SIGNATUREID=2
};

#ifdef __cplusplus
#define HBCI_SYSTEM_NAME    "openhbci"

using namespace std;

namespace HBCI {
/** @short The central user interaction class for all %HBCI connections
 * through OpenHBCI.
 *
 * This is the central user interaction class for all %HBCI
 * connections through OpenHBCI. You need exactly one object of this
 * class when you use OpenHBCI. 
 *
 * This class contains objects for user-interaction (Interactor,
 * Auth), system version and name, and the configuration mode flags
 * for OpenHBCI (readonly, retrievalonly).
 * 
 * @author Martin Preuss<openhbci@aquamaniac.de> */
class DLLIMPORT Hbci {
private:
    bool _readonly;
    bool _retrievalOnly;
    Pointer<Interactor> _interactor;
    Pointer<Auth> _auther;
    string _systemName;
    string _systemVersion;
    static int _debugLevel;

public:
    /**
     *  Constructor.
     *
     * @author Martin Preuss<openhbci@aquamaniac.de>
     *
     * @param readonly if true then no jobs are allowed which actually
     * do something with your account at the
     * bank. (e.g. JobSingleTransfer)
     *
     * @param retrievalOnly If set to true, OpenHBCI will only store
     * the minimum of data per account required for subsequent
     * retrieval of account data. I.e. it will not store the balance,
     * transactions, and standing orders of an account.
     */
    Hbci(bool readonly=false,
         bool retrievalOnly=false);
    /**  Default Destructor. */
    virtual ~Hbci();


    /** @name User Interaction
     *
     * Methods to get and set the user interaction classes.  */
    /*@{*/
    /**
     *  Returns the Auth currently to be used.
     *
     * This returns a pointer to an object which can be used for
     * authentification.
     * This is used by the media to ask for the pin.
     * When creating the HBCI-object this one gets a new authentificator
     * assigned (Auth).
     * As you can see this is a perfect example of the value of Pointer.
     * When assigning a new Auth object you don't have to care whether
     * already is one and how it was created. Simply do a <br>
     * authentificator=new Auth()<br>
     * To reassign it, if there already were one, it gets deleted
     * automatically by Pointer ;-)
     */
    Pointer<Auth> authentificator() const { return _auther;};
    /**
     *  Set a new HBCIAuth to be used.
     */
    void setAuthentificator(Pointer<Auth> a) { _auther=a;};

    /**
     *  Returns the HBCIInteractor currently to be used.
     */
    Pointer<Interactor> interactor() const { return _interactor;};
    /** 
     *  Set a new Interactor to be used.
     *
     * The Interactor is a base class that provides methods for
     * <i>all</i> occasions where the core OpenHBCI needs user interaction.
     */
    void setInteractor(Pointer<Interactor> i) { _interactor=i;};
    /*@}*/

    /** @name HBCI configuration
     *
     * Methods that return the basic HBCI configuration. */
    /*@{*/
    /**
     *  Returns true if OpenHBCI is in read-only mode.
     *
     * Returns true if OpenHBCI is in read-only mode, i.e. no jobs are
     * allowed which actually do something with your account at the
     * bank. (e.g. JobSingleTransfer)
     *
     * @author Martin Preuss<openhbci@aquamaniac.de>
     * @return true if read-only, false otherwise
     */
    bool isReadOnly() const { return _readonly; };

    /**
     *  Returns true if OpenHBCI is in retrieval-only mode.
     *
     * Returns true if OpenHBCI is in retrieval-only mode, i.e.
     * OpenHBCI will only store the minimum of data per account
     * required for subsequent retrieval of account data. I.e. it will
     * not store the balance, transactions, and standing orders of an
     * account.
     */
    bool isRetrievalOnly() const { return _retrievalOnly; };
    /*@}*/

    /** @name Version Information */
    /*@{*/
    /**
     *  Gives you the library version.
     *
     * Gives you the library version. This might be used by your
     * application to determine the capabilities of the current
     * openHBCI version.
     *
     * @author Martin Preuss<openhbci@aquamaniac.de>
     * @param major reference to an int variable to receive the major version
     * @param minor reference to an int variable to receive the minor version
     * @param patchlevel reference to an int variable to receive the
     * patchlevel
     */
    static void libraryVersion(int &major,
			       int &minor,
			       int &patchlevel);

    /**
     *  Gives you the library version.
     *
     * Gives you the library version. This might be used by your
     * application to determine the capabilities of the current
     * openHBCI version.
     *
     * @author Martin Preuss<openhbci@aquamaniac.de>
     * @param major reference to an int variable to receive the major version
     * @param minor reference to an int variable to receive the minor version
     * @param patchlevel reference to an int variable to receive the
     * patchlevel
     * @param build reference to an int variable to receive the build number
     */
    static void libraryVersion(int &major,
			       int &minor,
			       int &patchlevel,
			       int &build);

    /*@}*/

    /** @name System identification
     *
     * Methods that get and set the name under which OpenHBCI
     * identifies itself to the bank server. (FIXME: is this correct?)
     * */
    /*@{*/
    /**
     *  Returns the system name.
     *
     * This is the system name that OpenHBCI tells the bank.
     */
    const string &systemName() const { return _systemName;};
    /** 
     *  Set the system name.
     *
     * Use this to define the system name. It defaults to
     * HBCI_SYSTEM_NAME which is probably set to "openhbci". */
    void setSystemName(const string &n);

    /**
     *  Returns the system version as a string.
     *
     * This is the system version that OpenHBCI tells the bank when
     * communicating.  */
    const string &systemVersion() const { return _systemVersion;};
    /**
     *  Set the system version string.
     *
     * Use this to define the system version. It defaults to
     * HBCI_SYSTEM_VERSION.
     */
    void setSystemVersion(const string &s);
    /*@}*/

    /** Debug message settings */
    /*@{*/
    /**
     * Returns the current debugLevel. The higher this value the more debug
     * messages you see. Default is zero.
     */
    DLLIMPORT static int debugLevel();

    /**
     * Set the current debugLevel. The higher this value the more debug
     * messages you see. Defaults to zero.
     */
    DLLIMPORT static void setDebugLevel(int l);
    /*@}*/
};
} /* namespace HBCI */

extern "C" {
#endif /* __cplusplus */
    /** @defgroup HBCI_Hbcig HBCI_Hbci functions */
    /*@{*/
    /** @name User Interaction */
    /*@{*/
    /** 
     *  Set a new Interactor to be used.
     *
     * The Interactor is a base class that provides methods for
     * <i>all</i> occasions where the core OpenHBCI needs user
     * interaction.
     * @param h The object to perform this action on.
     * @param inter New interactor to be used.
     * @param autoDelete If TRUE, then this object takes ownership of
     * the HBCI_Interactor object and will delete it if the life time is
     * over. If FALSE, this object will <i>not</i> delete the
     * HBCI_Interactor object; instead, the caller is still responsible for
     * its deletion.  
     */
    extern void HBCI_Hbci_setInteractor(HBCI_Hbci *h, 
					HBCI_Interactor *inter, 
					int autoDelete);
    /**
     *  Returns the HBCIInteractor currently to be used.
     *
     * @param h The object to perform this action on.
     */
    extern HBCI_Interactor *HBCI_Hbci_interactor(const HBCI_Hbci *h);
    /*@}*/
    /** @name System Identification */
    /*@{*/
    /**
     *  Returns the system name.
     *
     * This is the system name that OpenHBCI tells the bank.
     * @param h The object to perform this action on.
     */
    extern const char *HBCI_Hbci_systemName(const HBCI_Hbci *h);
    /** 
     *  Set the system name.
     *
     * Use this to define the system name. It defaults to
     * HBCI_SYSTEM_NAME which is probably set to "openhbci". 
     * @param h The object to perform this action on.
     * @param n New system name.
     */
    extern void HBCI_Hbci_setSystemName(HBCI_Hbci *h, const char *n);
    /**
     *  Returns the system version as a string.
     *
     * This is the system version that OpenHBCI tells the bank when
     * communicating.  
     * @param h The object to perform this action on.
     */
    extern const char *HBCI_Hbci_systemVersion(const HBCI_Hbci *h);
    /** 
     *  Set the system version as a string.
     *
     * This is the system version that OpenHBCI tells the bank when
     * communicating.  
     * @param h The object to perform this action on.
     * @param n New system version.
     */
    extern void HBCI_Hbci_setSystemVersion(HBCI_Hbci *h, const char *n);
    /*@}*/
    /** @name Version Information */
    /*@{*/
    /**
     *  Gives you the library version.
     *
     * Gives you the library version. This might be used by your
     * application to determine the capabilities of the current
     * openHBCI version.
     *
     * @author Martin Preuss<openhbci@aquamaniac.de>
     * @param major reference to an int variable to receive the major version
     * @param minor reference to an int variable to receive the minor version
     * @param plevel reference to an int 
     * variable to receive the patchlevel
     */
    extern void HBCI_Hbci_libraryVersion(int *major, 
					 int *minor, int *plevel);
    /**
     *  Gives you the library version.
     *
     * Gives you the library version. This might be used by your
     * application to determine the capabilities of the current
     * openHBCI version.
     *
     * @author Martin Preuss<openhbci@aquamaniac.de>
     * @param major reference to an int variable to receive the major version
     * @param minor reference to an int variable to receive the minor version
     * @param plevel reference to an int 
     * variable to receive the patchlevel
     * @param pbuild reference to an int
     * variable to receive the build
     */
    extern void HBCI_Hbci_libraryVersion_build(int *major, 
					       int *minor, int *plevel,
					       int *pbuild);

    /**
     * Set the current debugLevel. The higher this value the more debug
     * messages you see. Defaults to zero.
     */
    extern void HBCI_Hbci_setDebugLevel(int l);
    /*@}*/
    /*@}*/
#ifdef __cplusplus
}

#endif /* __cplusplus */

#endif
