/***************************************************************************
                          DateTime.cpp  -  description
                             -------------------
    begin                : Sat 19 Feb 2000
    copyright            : (C) 2000 by Martin Preuss
    email                : martin@aquamaniac.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/



#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef __declspec
# if BUILDING_DLL
#  define DLLIMPORT __declspec (dllexport)
# else /* Not BUILDING_DLL */
#  define DLLIMPORT __declspec (dllimport)
# endif /* Not BUILDING_DLL */
#else
# define DLLIMPORT
#endif

#include <time.h>
#include <string>
#include <stdio.h>
#include "datetime.h"
#include "error.h"

namespace HBCI {

const char DateTime::_daynames[][16]={
#ifdef LANG_de
    "Sonntag","Montag","Dienstag","Mittwoch",
    "Donnerstag","Freitag","Samstag"
#endif
#ifdef LANG_en
    "Sunday","Monday","Tuesday","Wednesday",
    "Thursday","Friday","Saturday"
#endif
};

const char DateTime::_shortdaynames[][4]={
#ifdef LANG_de
    "Son","Mon","Die","Mit",
    "Don","Fre","Sam"
#endif
#ifdef LANG_en
    "Sun","Mon","Tue","Wed",
    "Thu","Fri","Sat"
#endif
};


const char DateTime::_monthnames[][16]={
    "",
#ifdef LANG_de
    "Januar","Februar","Mrz","April",
    "Mai","Juni","Juli","August",
    "September","Oktober","November","Dezember"
#endif
#ifdef LANG_en
    "January","February","March","April",
    "May","June","July","August",
    "September","October","November","December"
#endif
};


const char DateTime::_shortmonthnames[][4]={
    "",
#ifdef LANG_de
    "Jan","Feb","Mr","Apr",
    "Mai","Jun","Jul","Aug",
    "Sep","Okt","Nov","Dez"
#endif
#ifdef LANG_en
    "Jan","Feb","Mar","Apr",
    "May","Jun","Jul","Aug",
    "Sep","Oct","Nov","Dec"
#endif
};


DateTime::DateTime()
:_seconds(0)
,_minutes(0)
,_hours(0)
,_year(1972)
,_month(2)
,_day(17)
,_weekday(4)
{
}


DateTime::DateTime(unsigned short year,
           unsigned short month,
           unsigned short day,
           unsigned short hour,
           unsigned short minute,
           unsigned short second)
:_seconds(second)
,_minutes(minute)
,_hours(hour)
,_year(year)
,_month(month)
,_day(day)
,_weekday(0)
{
}


DateTime::DateTime(const struct tm &t){
    _importTime(t);
}


DateTime::DateTime(const time_t &tm) {
    struct tm *tcal;

    tcal=gmtime(&tm);
    _importTime(*tcal);
}


DateTime::~DateTime() {
}


void DateTime::_importTime(const struct tm &tin) {
    _seconds=tin.tm_sec;
    _minutes=tin.tm_min;
    _hours=tin.tm_hour;
    _day=tin.tm_mday;
    _month=tin.tm_mon+1;
    _year=tin.tm_year+1900;
    _weekday=tin.tm_wday;
}


struct tm DateTime::exportTimeTM() const{
    struct tm tout;

    tout.tm_sec=_seconds;
    tout.tm_min=_minutes;
    tout.tm_hour=_hours;
    tout.tm_mday=_day;
    tout.tm_mon=_month-1;
    tout.tm_year=_year-1900;
    tout.tm_wday=_weekday;
    return tout;
}


time_t DateTime::exportTime() const{
    struct tm tcal;
    time_t t;

    tcal=exportTimeTM();
    t=mktime(&tcal);
    if (t==-1)
        throw Error("DateTime::exportTime",
                        ERROR_LEVEL_NORMAL,
                        0,
                        ERROR_ADVISE_DONTKNOW,
                        strerror(errno),
                        "mktime()");
    return t;
}



DateTime DateTime::currentTime(bool gmt) {
    time_t temptime;
    DateTime tm;
    struct tm *tcal;

    if (-1==time(&temptime))
        throw  Error("DateTime::_getTime",
                         ERROR_LEVEL_NORMAL,
                         0,
                         ERROR_ADVISE_DONTKNOW,
                         strerror(errno),
                         "time()");
    if (gmt)
        tcal=gmtime(&temptime);
    else
        tcal=localtime(&temptime);
    if (!tcal)
        throw Error("DateTime::_getTime",
                        ERROR_LEVEL_NORMAL,
                        0,
                        ERROR_ADVISE_DONTKNOW,
                        strerror(errno),
                        "localtime()");
    return DateTime(*tcal);
}


bool DateTime::isValid() const{
    bool retval;

    retval=true;

    if (_month>12  || _month<1) retval=false;
    if (_day>31    || _day<1) retval=false;
    if (_hours>23) retval=false;
    if (_minutes>59) retval=false;
    if (_seconds>59) retval=false;
    if (_weekday>6) retval=false;
    return retval;
}


int DateTime::compareTime(const DateTime &t2) const{
    int i,j;

    i=_year;
    j=t2._year;
    if (i<j) return -1;
    if (i>j) return 1;
    i=_month;
    j=t2._month;
    if (i<j) return -1;
    if (i>j) return 1;
    i=_day;
    j=t2._day;
    if (i<j) return -1;
    if (i>j) return 1;
    i=_hours;
    j=t2._hours;
    if (i<j) return -1;
    if (i>j) return 1;
    i=_minutes;
    j=t2._minutes;
    if (i<j) return -1;
    if (i>j) return 1;
    i=_seconds;
    j=t2._seconds;
    if (i<j) return -1;
    if (i>j) return 1;
    return 0;
}



void  DateTime::_appendNum(string &b, int num, const char *fs){
    char buffer[64];
    int i;

    i=sprintf(buffer,fs,num);
    b+=buffer;
}


string DateTime::timeString(string fs) const{
    unsigned int pos;
	string p;

    pos=0;
    while (pos<fs.length()) {
        if (fs.at(pos)!='%')
            p+=fs.at(pos);
        else {
            pos++;
            if (pos>=fs.length())
                return "";
            switch (fs.at(pos)) {
            case 'h': _appendNum(p,_hours,"%.2d"); break;
            case 'm': _appendNum(p,_minutes,"%.2d"); break;
            case 's': _appendNum(p,_seconds,"%.2d"); break;
            case 'D': _appendNum(p,_day,"%.2d"); break;
            case 'M': p+=_monthnames[_month]; break;
            case 'S': p+=_shortmonthnames[_month]; break;
            case 'N': _appendNum(p,_month,"%.2d"); break;
            case 'Y': _appendNum(p,_year,"%.4d"); break;
            case 'W': p+=_daynames[_weekday]; break;
            case 'w': p+=_shortdaynames[_weekday]; break;
            //case 'Z': p+=timeZoneName(); break;
            //case 'z': _appendNum(p,timeZone()/3600,"%03d"); break;
            default: return "";
            }
        } // else
        pos++;
    } // while
    return p;
}


DateTime DateTime::addSeconds(long seconds) const{
    time_t secs;

    secs=exportTime();
    secs+=seconds;
    return DateTime(secs);
}

} /* namespace HBCI */




