/***************************************************************************
 $RCSfile: statusreport.h,v $
                             -------------------
    cvs         : $Id: statusreport.h,v 1.4 2003/05/21 12:01:42 cstim Exp $
    begin       : Sun May 17 2003
    copyright   : (C) 2003 by Martin Preuss
    email       : openhbci@aquamaniac.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

/** @file statusreport.h 
    @brief MessageReference, StatusReport and their C wrappers.
 */
#ifndef STATUSREPORT_H
#define STATUSREPORT_H

#include <openhbci/dllimport.h>

#ifdef __cplusplus
#include <string>
#include <list>
using namespace std;
#include <openhbci/date.h>


namespace HBCI {

/** @brief Fully qualified reference to a specific message that was
 * sent some time earlier.
 *
 * If the bank gives us feedback about the status of a particular
 * message (e.g. transfer order), this MessageReference identifies the
 * order (HBCI::OutboxJob) which is meant.
 */
class DLLIMPORT MessageReference {
private:
  string _dialogId;
  int _msgNumber;
public:
  MessageReference();
  /** Constructor with dialog id and message number. */
  MessageReference(const string &did, int msgnum);
  /** Constructor for a HBCI data element group (DEG). This is just
      the same string as the one returned by toString(). (FIXME: is
      this correct?) */
  MessageReference(const string &deg);
  /** Destructor */
  ~MessageReference();
  /** Returns the dialog id. */
  const string &dialogId() const { return _dialogId;};
  /** Returns the message number. */
  int messageNumber() const { return _msgNumber;};
  /** Returns the HBCI-representation of a message reference in one
      string. */
  string toString() const;
  /** Equality */
  bool operator==(const MessageReference &ref) const;
  /** Comparison. Returns -1 when this MessageReference is smaller
      than the other, returns 0 when equal, and returns 1 when this
      one is greater than the other. The dialogId are compared
      lexicographically, and the messageNumber numerically. */
  int cmp(const MessageReference &other) const;
};
} /* namespace */
typedef HBCI::MessageReference HBCI_MessageReference;
extern "C" {
#else /* __cplusplus */
  typedef struct HBCI_MessageReference HBCI_MessageReference;
#endif /* __cplusplus */

  /** @name HBCI_MessageReference */
  /*@{*/
  /** Constructor with dialog id and message number. */
  extern HBCI_MessageReference *
  HBCI_MessageReference_new(const char *did, int msgnum);
  /** Constructor for a HBCI data element group (DEG). This is just
      the same string as the one returned by toString(). (FIXME: is
      this correct?) */
  extern HBCI_MessageReference *
  HBCI_MessageReference_new_deg(const char *deg);
  /** Destructor */
  extern void HBCI_MessageReference_delete(HBCI_MessageReference *h);
  /** Returns the dialog id. */
  extern const char *
  HBCI_MessageReference_dialogId(const HBCI_MessageReference *h);
  /** Returns the message number. */
  extern int 
  HBCI_MessageReference_messageNumber(const HBCI_MessageReference *h);
  /** Returns the HBCI-representation of a message reference in one
   * string.
   *
   * @note The returned char array is owned by the caller and must be
   * free'd when no longer needed */
  extern char * 
  HBCI_MessageReference_toString(const HBCI_MessageReference *h);
  /** Equality */
  extern int HBCI_MessageReference_equal(const HBCI_MessageReference *h1,
					 const HBCI_MessageReference *h2);
  /** Comparison. Returns -1 when MessageReference h1 is smaller than
      h2, returns 0 when equal, and returns 1 when h1 is greater than
      h2. The dialogId are compared lexicographically, and the
      messageNumber numerically. */
  extern int HBCI_MessageReference_cmp(const HBCI_MessageReference *h1,
					 const HBCI_MessageReference *h2);
  /*@}*/

#ifdef __cplusplus
} /* extern C */

namespace HBCI {

/** @brief Status information about one particular message.
 *
 * This Statusreport contains the bank's information about what
 * happened with the message (e.g. transfer order) references by the
 * messageReference(). */
class DLLIMPORT StatusReport {
private:
  Date _date;
  Time _time;
  MessageReference _msgref;
  int _segref;
  int _code;
  string _ged;
  string _text;

public:
  StatusReport();
  ~StatusReport();

  /** @name Date and Time */
  /*@{*/
  /** Returns the date when this status report was valid. */
  const Date &date() const { return _date;};
  void setDate(const Date &d) { _date=d;};

  /** Returns the time when this status report was valid. */
  const Time &time() const { return _time;};
  void setTime(const Time &t) { _time=t;};
  /*@}*/

  /** @name Result information */
  /*@{*/
  /** Returns the HBCI result code of this status report. */
  int result() const { return _code;};
  void setResult(int r) { _code=r;};

  /** Returns the clear text message of this status report. */
  const string& resultText() const { return _text;};
  void setResultText(const string &s) { _text=s;};

  /** Returns a long string containing all information of this status
      report; mainly intended for debugging. */
  string dump() const;
  /*@}*/

  /** @name Reference to the original HBCI::OutboxJob */
  /*@{*/
  /** Returns the message reference to find the message that this
      status report belongs to, i.e. the reference that was given
      previously in OutboxJob::messageReference(). */
  const MessageReference &messageReference() const { return _msgref;};
  void setMessageReference(const MessageReference &r) { _msgref=r;};

  /** Returns the segment number inside the referred message (by
      messageReference() ) that this status report belongs to. This is
      the segment number from
      OutboxJob::segmentForStatusReport(). (FIXME: is this
      correct?) */
  int segment() const { return _segref;};
  void setSegment(int s) { _segref=s;};

  /** Returns the message group this status report belongs to (FIXME:
      is this correct?) */
  const string& groupReference() const { return _ged;};
  void setGroupReference(const string &s) { _ged=s;};
  /*@}*/
};

} /* namespace */
typedef HBCI::StatusReport HBCI_StatusReport;
typedef std::list<HBCI::StatusReport> list_HBCI_StatusReport;
extern "C" {
#else /* __cplusplus */
  typedef struct HBCI_StatusReport HBCI_StatusReport;
  typedef struct list_HBCI_StatusReport list_HBCI_StatusReport;
#endif /* __cplusplus */

  /** @name HBCI_StatusReport */
  /*@{*/
  /** Returns the date when this status report was valid. */
  extern const HBCI_Date *HBCI_StatusReport_date(const HBCI_StatusReport *h);

  /** Returns the time when this status report was valid. */
  extern const HBCI_Time *HBCI_StatusReport_time(const HBCI_StatusReport *h);

  /** Returns the HBCI result code of this status report. */
  extern int HBCI_StatusReport_result(const HBCI_StatusReport *h);

  /** Returns the message group this status report belongs to (FIXME:
      is this correct?) */
  extern const char *
  HBCI_StatusReport_groupReference(const HBCI_StatusReport *h);

  /** Returns the clear text message of this status report. */
  extern const char *HBCI_StatusReport_resultText(const HBCI_StatusReport *h);

  /** Returns the message reference to find the message that this
      status report belongs to. */
  extern const HBCI_MessageReference *
  HBCI_StatusReport_messageReference(const HBCI_StatusReport *h);

  /** Returns the segment number inside the referred message (by
      messageReference() ) that this status report belongs to. */
  extern int HBCI_StatusReport_segment(const HBCI_StatusReport *h);
  /** Returns a long string containing all information of this status
   * report; mainly intended for debugging.
   *
   * @note The returned char array is owned by the caller and must
   * be free'd when no longer needed! */
  extern char *HBCI_StatusReport_dump(const HBCI_StatusReport *h);
  /*@}*/

  /** @name list_HBCI_StatusReport */
  /*@{*/
  /** Returns the size of the list. */
  extern unsigned int 
  list_HBCI_StatusReport_size(const list_HBCI_StatusReport *l);
  /** Callback function for list_HBCI_StatusReport_foreach. */
  typedef void *(*list_HBCI_StatusReport_cb) (const HBCI_StatusReport* value, 
					      void *user_data);
  /** Traverses the list, calling the callback function 'func' on
   * each list element.  Traversal will stop when 'func' returns a
   * non-NULL value, and the routine will return with that
   * value. Otherwise the routine will return NULL. 
   * @param l The list to traverse.
   * @param func The function to be called with each list element.
   * @param user_data A pointer passed on to the function 'func'.
   * @return The non-NULL pointer returned by 'func' as soon as it
   * returns one. Otherwise (i.e. 'func' always returns NULL)
   * returns NULL.
   * @author Christian Stimming <stimming@tuhh.de> */
  extern void *list_HBCI_StatusReport_foreach(const list_HBCI_StatusReport *l,
					      list_HBCI_StatusReport_cb func,
					      void *user_data);
  /*@}*/

#ifdef __cplusplus
} /* extern C */
#endif /* __cplusplus */

#endif


