#ifndef S11N_NS_DATA_ALGO_H_INCLUDED
#define S11N_NS_DATA_ALGO_H_INCLUDED
////////////////////////////////////////////////////////////////////////
// data_node_algo.h:
//
// Defines some helper algorithms for working with Serializables and
// Data Nodes.
//
// License: Public Domain
// Author: stephan@s11n.net
////////////////////////////////////////////////////////////////////////

#include <string>
#include <list>
#include <iterator>
#include <algorithm> // for_each
#include <cassert>

#include <S11N_NS/debuggering_macros.h> // COUT/CERR
#include <S11N_NS/to_string.h> // to/from_string()
#include <S11N_NS/functor.h> // object_reference_wrapper, same_name, et al
#include <S11N_NS/pointer_stripper.h> // pointer_stripper class

#include <S11N_NS/abstract_creator.h> // abstract_creator class


////////////////////////////////////////////////////////////////////////
// NO DEPS ON data_node.h ALLOWED!
////////////////////////////////////////////////////////////////////////

namespace S11N_NS {



        /**
           Adds ch as a child of parent. Parent takes over ownership
           of ch.

           ContainerType must support:

           children().push_back( ChildType * );

        */
        template <typename NodeType, typename ChildType>
        void add_child( NodeType & parent, ChildType * ch )
        {
                parent.children().push_back( ch );
        }

 
 

        /**
           Creates a new node, named nodename, as a child of parent.

           Returns a reference to the new child, which parent now
           owns.

           NodeType must support:

           name( string );

           children().push_back( NodeType * );

        */
        template <typename NodeType>
        NodeType & create_child( NodeType & parent,
                                  const std::string nodename )
        {
                NodeType * n = new NodeType;
                n->name( nodename );
                parent.children().push_back( n );
                return *n;
        }



        





        /**
           Each child in parent.children() which has the given name is
           copied into the target container.

           Returns the number of items added to target.

           DestContainerT must support an insert iterator which will
           insert the pointer type contained in the list returned by
           parent.children(). i.e., it must hold (const
           SomeSerializableType *).

           Ownership of the children does not change by calling this
           function. Normally they are owned by the parent node
           (unless the client explicitely did something about that.
        */
        template <typename NodeT, typename DestContainerT>
        size_t find_children_by_name( const NodeT & parent,
                               const std::string & name,
                               DestContainerT & target )
        {
                size_t c = target.size();
                S11N_NS::copy_if( parent.children().begin(),
                                         parent.children().end(),
                                         std::insert_iterator<DestContainerT>( target,
                                                                               target.begin() ),
                                         S11N_NS::same_name<NodeT>( name )
                                         );
                // you gotta love the STL, man.
                return target.size() - c;
        }

        /**
           Finds the FIRST child in parent with the given name and
           returns a pointer to it, or 0 if no such child is found.

           Ownership of the children does not change by calling this
           function: parent still owns it.
        */
        template <typename NodeT>
        const NodeT *
        find_child_by_name( const NodeT & parent, const std::string & name )
        {
                typedef typename NodeT::child_list_type::const_iterator CIT;
                CIT it = std::find_if( parent.children().begin(),
                                       parent.children().end(),
                                       S11N_NS::same_name<NodeT>( name )
                                       );
                return (parent.children().end() == it) ? 0 : *it;
        }

        /**
           A non-const overload of find_child_by_name(). Functionally
           identical to the const form, except for the constness of
           the parent argument and return value.

           Ownership of the returned pointer is not changed by calling
           this function (normally parent owns it, but clients may
           change that without affecting this function). When in
           doubt, i.e. during "normal usage", do NOT delete the returned
           pointer, because the parent node owns it.
        */
        template <typename NodeT>
        NodeT *
        find_child_by_name( NodeT & parent, const std::string & name )
        {
                typedef typename NodeT::child_list_type::iterator IT;
                IT it = std::find_if( parent.children().begin(),
                                       parent.children().end(),
                                       S11N_NS::same_name<NodeT>( name )
                                       );
                return (parent.children().end() == it) ? 0 : *it;
        }


} // namespace S11N_NS

#endif // S11N_NS_DATA_ALGO_H_INCLUDED
