/* m17n-gui.h -- header file for the GUI API of the m17n library.
   Copyright (C) 2003, 2004
     National Institute of Advanced Industrial Science and Technology (AIST)
     Registration Number H15PRO112

   This file is part of the m17n library.

   The m17n library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   The m17n library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the m17n library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.  */

#ifndef _M17N_GUI_H_
#define _M17N_GUI_H_

#ifndef _M17N_H_
#include <m17n.h>
#endif

#ifdef __cplusplus
extern "C"
{
#endif

#if !defined (FOR_DOXYGEN) || defined (DOXYGEN_INTERNAL_MODULE)

extern void m17n_init_win (void);
#undef M17N_INIT
#define M17N_INIT() m17n_init_win ()

extern void m17n_fini_win (void);
#undef M17N_FINI
#define M17N_FINI() m17n_fini_win ()

#endif

/***en @defgroup m17nGUI GUI API */
/***ja @defgroup m17nGUI GUI API */
/*=*/

/*** @ingroup m17nGUI */
/***en @defgroup m17nFrame Frame */
/***ja @defgroup m17nFrame ե졼 */
/*=*/

/*** @ingroup m17nFrame */
/***en
    @brief Type of frames.

    The type #MFrame is for a @e frame object.  Each frame holds
    various information about the corresponding physical display/input
    device.

    The internal structure of the type #MFrame is concealed from an
    application program, and its contents depend on the window system
    in use.  In the m17n-X library, it contains the information about
    @e display and @e screen in the X Window System.  */

/***ja
    @brief ե졼η.

    #MFrame ϡ@e ե졼 ֥ѤηǤ롣ġΥե졼ϡ
    줬бʪŪɽϥǥХγƼݻ롣

    #MFrame ¤ϡץꥱץफϸʤ
    ޤƤϻѤ륦ɥƥ˰¸롣ޤm17n-X 
    ֥ˤե졼ϡX ɥ @e display  @e screen 
    ˴ؤġ
      */

typedef struct MFrame MFrame;

/*=*/

extern MSymbol Mdevice;

extern MSymbol Mfont;
extern MSymbol Mfont_width;
extern MSymbol Mfont_ascent;
extern MSymbol Mfont_descent;
extern MFrame *mframe_default;

extern MSymbol Mdisplay;
extern MSymbol Mscreen;
extern MSymbol Mdrawable;
extern MSymbol Mwidget;
extern MSymbol Mdepth;
extern MSymbol Mcolormap;

extern MFrame *mframe (MPlist *plist);

extern void *mframe_get_prop (MFrame *frame, MSymbol key);

/* end of frame module */
/*=*/

/*** @ingroup m17nGUI  */
/***en @defgroup m17nFont Font */
/***ja @defgroup m17nFont ե */
/*=*/

/*** @ingroup m17nFont */
/***en
    @brief Type of fonts.

    The type #MFont is the structure defining fonts.  It contains
    information about the following properties of a font: foundry,
    family, weight, style, stretch, adstyle, registry, size, and
    resolution.

    This structure is used both for specifying a font in a fontset
    and for storing information about available system fonts.

    The internal structure is concealed from an application program.  */

/***ja
    @brief եȤη.

    #MFont ϥեȻѤι¤ΤǤꡢեȤΥץѥƥǤ
     foundry, family, weight, style, stretch, adstyle, registry,
    size, resolution ˴ؤޤࡣ

    ι¤ΤϥեȥåΥեȤꤹݤȡѲǽʥ
    ƥեȤξǼݤξѤ롣

    ¤ϥץꥱץफϸʤ  */

/***
    @seealso
    mfont (), mfont_from_name (), mfont_find ().  */

typedef struct MFont MFont;

/*=*/

extern MSymbol Mx, Mfreetype, Mxft;

extern MPlist *mfont_freetype_path;

extern MFont *mfont ();

extern MFont *mfont_copy (MFont *font);

extern MFont *mfont_parse_name (char *name, MSymbol format);

extern char *mfont_unparse_name (MFont *font, MSymbol format);

/* These two are obsolete (from 1.1.0).  */
extern char *mfont_name (MFont *font);
extern MFont *mfont_from_name (char *name);

extern MSymbol Mfoundry;
extern MSymbol Mfamily;
extern MSymbol Mweight;
extern MSymbol Mstyle;
extern MSymbol Mstretch;
extern MSymbol Madstyle;
extern MSymbol Mregistry;
extern MSymbol Msize;
extern MSymbol Mresolution;

extern MSymbol Mfontconfig;

extern void *mfont_get_prop (MFont *font, MSymbol key);

extern int mfont_put_prop (MFont *font, MSymbol key, void *val);

extern int mfont_set_encoding (MFont *font,
			       MSymbol encoding_name, MSymbol repertory_name);


/*=*/

/***en
    @brief Find a font.

    The mfont_find () function returns a pointer to the available font
    that matches best with the specification $SPEC in frame $FRAME.

    $SCORE, if not NULL, must point to a place to store the score
    value which indicates how well the found font matches $SPEC.  The
    smaller score means a better match.

    $LIMITED_SIZE, if nonzero, forces the font selector to find a
    font not greater than the #Msize property of $SPEC.  */

/***ja
    @brief եȤõ.

    ؿ mfont_find () ϡե졼 $FRAME ǥե $SPEC ˤ
    ȤפѲǽʥեȤؤΥݥ󥿤֤  

    $SCORE  NULL Ǥ뤫ĤäեȤ $SPEC ˤɤۤɹ
    Ƥ뤫򼨤¸ؤΥݥ󥿤Ǥ롣
    ۤɤäƤ뤳Ȥ̣롣

    $LIMITED_SIZE  0 ǤʤС$SPEC Υץѥƥ #Msize 礭
    ʤեȤõ롣
*/

extern MFont *mfont_find (MFrame *frame, MFont *spec,
			  int *score, int limited_size);

extern MSymbol *mfont_selection_priority ();

extern int mfont_set_selection_priority (MSymbol *keys);

extern int mfont_resize_ratio (MFont *font);

extern MPlist *mfont_list (MFrame *frame, MFont *font, MSymbol language,
			   int maxnum);


/* end of font module */
/*=*/

/*** @ingroup m17nGUI  */
/***en @defgroup m17nFontset Fontset */
/***ja @defgroup m17nFontset եȥå */
/*=*/
/*** @addtogroup m17nFontset
     @{   */
typedef struct MFontset MFontset;

extern MFontset *mfontset (char *name);

extern MSymbol mfontset_name (MFontset *fontset);

extern MFontset *mfontset_copy (MFontset *fontset, char *name);

extern int mfontset_modify_entry (MFontset *fontset,
				  MSymbol language, MSymbol script,
				  MSymbol charset,
				  MFont *spec, MSymbol layouter_name,
				  int how);

extern MPlist *mfontset_lookup (MFontset *fontset, MSymbol script,
				MSymbol language, MSymbol charset);
/*** @}   */
/* end of fontset module */
/*=*/

/*** @ingroup m17nGUI */
/***en @defgroup m17nFace Face */
/***ja @defgroup m17nFace ե */
/*=*/

/*** @ingroup m17nFace */
/***en
    @brief Type of faces.

    The type #MFace is the structure of face objects.  The internal
    structure is concealed from an application program.  */

/***ja
    @brief եη.

    #MFace ϥե֥ȤΤι¤ΤǤ롣¤
    ץꥱץफϸʤ  */

typedef struct MFace MFace;
/*=*/

extern MSymbol Mforeground;
extern MSymbol Mbackground;
extern MSymbol Mvideomode;
extern MSymbol Mnormal;
extern MSymbol Mreverse;
extern MSymbol Mhline;
extern MSymbol Mbox;
extern MSymbol Mfontset;
extern MSymbol Mratio;
extern MSymbol Mhook_func;
extern MSymbol Mhook_arg;

/* Predefined faces.  */
extern MFace *mface_normal_video;
extern MFace *mface_reverse_video;
extern MFace *mface_underline;
extern MFace *mface_medium;
extern MFace *mface_bold;
extern MFace *mface_italic;
extern MFace *mface_bold_italic;
extern MFace *mface_xx_small;
extern MFace *mface_x_small;
extern MFace *mface_small;
extern MFace *mface_normalsize;
extern MFace *mface_large;
extern MFace *mface_x_large;
extern MFace *mface_xx_large;
extern MFace *mface_black;
extern MFace *mface_white;
extern MFace *mface_red;
extern MFace *mface_green;
extern MFace *mface_blue;
extern MFace *mface_cyan;
extern MFace *mface_yellow;
extern MFace *mface_magenta;

/* etc */
extern MSymbol Mface;

extern MFace *mface ();

extern MFace *mface_copy (MFace *face);

extern MFace *mface_merge (MFace *dst, MFace *src);

extern MFace *mface_from_font (MFont *font);

/*=*/

/*** @ingroup m17nFace */
/***en
    @brief Type of horizontal line spec of face.

    The type #MFaceHLineProp is to specify the detail of #Mhline
    property of a face.  The value of the property must be a pointer
    to an object of this type.  */
/***ja
    @brief եοʿѷ.

    #MFaceHLineProp ϥե #Mhline ץѥƥξܺ٤ꤹ뷿
    Ǥ롣ΥץѥƥͤϤηΥ֥ȤǤʤƤϤʤʤ
      */

typedef struct
{
  /***en Type of the horizontal line.  */
  /***ja ʿΥ.  */
  enum MFaceHLineType
    {
      MFACE_HLINE_BOTTOM,      
      MFACE_HLINE_UNDER,
      MFACE_HLINE_STRIKE_THROUGH,
      MFACE_HLINE_OVER,
      MFACE_HLINE_TOP
    } type;

  /***en Width of the line in pixels.  */
  /***ja ʥԥñ̡.  */
  unsigned width;

  /***en Color of the line.  If the value is Mnil, foreground color of
      a merged face is used.  */
  /***ja ο.  Mnil ʤС礷եʿȤ롣  */
  
  MSymbol color;
} MFaceHLineProp;
/*=*/

/*** @ingroup m17nFace */
/***en
    @brief Type of box spec of face.

    The type #MFaceBoxProp is to specify the detail of #Mbox property
    of a face.  The value of the property must be a pointer to an
    object of this type.  */
/***ja
    @brief եΰϤȻѷ.

    #MFaceBoxProp ϥե #Mbox ץѥƥξܺ٤ꤹ뷿Ǥ
    롣ΥץѥƥͤϤηΥ֥ȤǤʤƤϤʤʤ
      */

typedef struct
{
  /***en Width of the box line in pixels.  */
  /***ja ʥԥñ̡.  */
  unsigned width;

  MSymbol color_top;
  MSymbol color_bottom;
  MSymbol color_left;
  MSymbol color_right;

  unsigned inner_hmargin;
  unsigned inner_vmargin;
  unsigned outer_hmargin;
  unsigned outer_vmargin;

} MFaceBoxProp;
/*=*/

/*** @ingroup m17nFace */
/***en
    @brief Type of hook function of face.

    The type #MFaceHookFunc is to specify the #Mhook property of a
    face.  The value of the property must be function of this
    type.  */
/***ja
    @brief եΥեåؿη.

    #MFaceHookFunc ϥե #Mhook ץѥƥꤹ뷿Ǥ롣
    ΥץѥƥͤϡηδؿǤʤƤϤʤʤ
      */
typedef void (*MFaceHookFunc) (MFace *face, void *arg, void *info);
/*=*/

extern void *mface_get_prop (MFace *face, MSymbol key);

extern int mface_put_prop (MFace *face, MSymbol key, void *val);

extern void mface_update (MFrame *frame, MFace *face);

/* end of face module */
/*=*/

/*** @ingroup m17nGUI */
/***en @defgroup m17nDraw Drawing */
/***ja @defgroup m17nDraw ɽ */
/*=*/

/*** @ingroup m17nDraw */
/***en
    @brief Window system dependent type for a window.

    The type #MDrawWindow is for a window; a rectangular area that
    works in several ways like a miniature screen.

    What it actually points depends on a window system.  A program
    that uses the m17n-X library must coerce the type @c Drawable to
    this type.  */
/***ja 
    @brief ɥƥ˰¸롢ɥη.

    #MDrawWindow ϥɥʤĤǥ꡼Υߥ˥
    ȤƯΰѤηǤ롣

    ºݤ˲ؤϥɥƥ˰¸롣 m17n X 饤֥
    Ѥץ @c Drawable 򤳤ηѴʤƤϤʤ
    ʤ */

typedef void *MDrawWindow;
/*=*/

/*** @ingroup m17nDraw */
/***en
    @brief Window system dependent type for a region.

    The type #MDrawRegion is for a region; an arbitrary set of pixels
    on the screen (typically a rectangular area).

    What it actually points depends on a window system.  A program
    that uses the m17n-X library must coerce the type @c Region to
    this type.  */
/***ja
    @brief ɥƥ˰¸롢ΰη.

    #MDrawRegion ΰ衢ʤ꡼ǤդΥԥν
   ŵŪˤ϶ΰѤηǤ롣

    ºݤ˲ؤϥɥƥ˰¸롣 m17n X 饤֥
    Ѥץ @c Region 򤳤ηѴʤƤϤʤ
    ʤ  */

typedef void *MDrawRegion;
/*=*/

/*** @ingroup m17nDraw */
/***en
    @brief Type of a text drawing control.

    The type #MDrawControl is the structure that controls how to draw
    an M-text.  */
/***ja
    @brief ƥɽη.

    #MDrawControl ϡM-text ɤɽ뤫椹빽¤ΤǤ롣
      */


typedef struct
{
  /***en If nonzero, draw an M-text as image, i.e. with background
      filled with background colors of faces put on the M-text.
      Otherwise, the background is not changed.  */
  /***ja 0 ǤʤС M-text Ȥơʤطʤ M-text 
      եǻꤵƤطʿɽ롣Ǥʤ
      ʤѤʤ  */
  unsigned as_image : 1;

  /***en If nonzero and the first glyph of each line has negative
      lbearing, shift glyphs horizontally to right so that no pixel is
      drawn to the left of the specified position.  */
  /***ja 0 ǤʤƹԤκǽΥդlbearing ʤСդ
      ʿ˱ˤ餷ơꤷ֤꺸˥ԥ뤬ʤ褦
      롣  */
  unsigned align_head : 1;

  /***en If nonzero, draw an M-text two-dimensionally, i.e., newlines
      in M-text breaks lines and the following characters are drawn in
      the next line.  If <format> is non-NULL, and the function
      returns nonzero line width, a line longer than that width is
      also broken.  */
  /***ja 0 ǤʤСM-text 򣲼ŪˡʤM-text  
      newline ǲԤ³ʸϼιԤɽ롣⤷ <format>  
      NULL Ǥʤδؿ 0 Ǥʤ֤СĹԤ
      Ԥ롣  */
  unsigned two_dimensional : 1;

  /***en If nonzero, draw an M-text to the right of a specified
      position.  */
  /***ja 0 ǤʤСM-text ꤷ֤αɽ롣  */
  unsigned orientation_reversed : 1;

  /***en If nonzero, reorder glyphs correctly for bidi text.  */ 
  /***ja 0 ʤСbidi ƥѤ˥դ󤹤롣  */
  unsigned enable_bidi : 1;

  /***en If nonzero, don't draw characters whose general category (in
      Unicode) is Cf (Other, format).  */
  /***ja 0 ǤʤС˥ɤ֤̥ƥ꤬ Cf (Other,
      format) Ǥʸɽʤ  */
  unsigned ignore_formatting_char : 1;

  /***en If nonzero, draw glyphs suitable for a terminal.  Not yet
      implemented.  */
  /***ja 0 ǤʤСüѤΥդɽ롣̤  */
  unsigned fixed_width : 1;

  /***en If nonzero, draw glyphs with anti-aliasing if a backend font
      driver supports it.  */
  /***ja 0 ǤʤСꥢǥդɽ롣ʥХå
      ɤΥեȥɥ饤Фꥢǽ򥵥ݡȤƤ
      Τߡ */
  unsigned anti_alias : 1;

  /***en If nonzero, disable the adjustment of glyph positions to
      avoid horizontal overlapping at font boundary.  */
  /***ja 0 ǤʤСեȶǤοʿΥդνŤʤ
      뤿Υհ֤Ĵ̵ˤ롣  */
  unsigned disable_overlapping_adjustment : 1;

  /***en If nonzero, the values are minimum line ascent and descent
      pixels.  */
  /***ja 0 ǤʤСͤϹԤ ascent  descent κǾͤ򼨤  */
  unsigned int min_line_ascent;
  unsigned int min_line_descent;

  /***en If nonzero, the values are maximum line ascent and descent
      pixels.  */
  /***ja 0 ǤʤСͤϹԤ ascent  descent κͤ򼨤  */
  unsigned int max_line_ascent;
  unsigned int max_line_descent;

  /***en If nonzero, the value specifies how many pixels each line can
      occupy on the display.  The value zero means that there is no
      limit.  It is ignored if <format> is non-NULL.  */
  /***ja 0 ǤʤСͤϤΥǥץ쥤ǳƹԤ뤳ȤΤǤ
      ԥ򼨤 0 ϸꤵʤȤ̣롣<format> 
      NULL Ǥʤ̵뤵롣   */
  unsigned int max_line_width;

  /***en If nonzero, the value specifies the distance between tab
      stops in columns (the width of one column is the width of a
      space in the default font of the frame).  The value zero means
      8.  */
  /***ja 0 ǤʤСͤϥ֥ȥå״֤εΥ򥳥ñ̡ʥ
      ե졼ΥǥեȥեȤˤʸǤˤǼ 
      0  8 ̣롣 */
  unsigned int tab_width;

  /***en If non-NULL, the value is a function that calculates the
      indentation and width limit of each line based on the line
      number LINE and the coordinate Y.  The function store the
      indentation and width limit at the place pointed by INDENT and
      WIDTH respectively.

      The indentation specifies how many pixels the first glyph of
      each line is shifted to the right (if the member
      <orientation_reversed> is zero) or to the left (otherwise).  If
      the value is negative, each line is shifted to the reverse
      direction.

      The width limit specifies how many pixels each line can occupy
      on the display.  The value 0 means that there is no limit.

      LINE and Y are reset to 0 when a line is broken by a newline
      character, and incremented each time when a long line is broken
      because of the width limit.

      This has an effect only when <two_dimensional> is nonzero.  */
  /***ja 0 ǤʤСͤϴؿǤꡢδؿϹֹ LINE Ⱥɸ Y 
      ˴ŤƳƹԤΥǥȤȺ׻줾INDENT 
      WIDTH ǻؤ¸롣

      ǥȤϡƹԤκǽΥդʥ 
      <orientation_reversed>  0 λˤ뤤Ϻʤʳλˤ˲
      ԥ뤺餹ꤹ롣ͤʤеˤ餹

      ϡƹԤǥץ쥤뤳ȤΤǤԥκ
      ͤǤ롣ͤ 0 ξ¤ʤȤ̣롣

      LINE  Y ϲʸˤäƹԤޤäݤˤ 0 ˥ꥻåȤ졢
      ˤäƹԤޤäˤ 1 Ť䤵롣

       <two_dimensional>  0 ǤʤˤΤͭǤ롣  */
  void (*format) (int line, int y, int *indent, int *width);

  /***en If non-NULL, the value is a function that calculates a line
      breaking position when a line is too long to fit within the
      width limit.  POS is the position of the character next to the
      last one that fits within the limit.  FROM is the position of the
      first character of the line, and TO is the position of the last
      character displayed on the line if there were not width limit.
      LINE and Y are the same as the arguments to <format>.

      The function must return a character position to break the
      line.

      The function should not modify MT.

      The mdraw_default_line_break () function is useful for such a
      script that uses SPACE as a word separator.  */
  /***ja NULL ǤʤСͤϹԤ˼ޤʤ˹Ԥ
      ֤׻ؿǤ롣POS Ϻ˼ޤǸʸμʸ
      ΰ֤Ǥ롣FROM ϹԤκǽʸΰ֡TO Ϻꤵ
      ƤʤФιԤɽǸʸΰ֤Ǥ롣LINE  
      Y  <format> ΰƱǤ롣

      δؿϹԤʸ֤֤ʤƤϤʤʤޤMT 
      ѹƤϤʤʤ

      ؿ mdraw_default_line_break () ϡζڤȤѤ
      륹ץѤȤѤ뤳ȤǤ롣  */
  int (*line_break) (MText *mt, int pos, int from, int to, int line, int y);

  int with_cursor;

  /***en Specifies the character position to display a cursor.  If it
      is greater than the maximum character position, the cursor is
      displayed next to the last character of an M-text.  If the value
      is negative, even if <cursor_width> is nonzero, cursor is not
      displayed.  */
  /***ja ɽʸ֤򼨤ʸ֤礭
      СM-text κǸʸ٤ɽ롣ʤС
      <cursor_width>  0 ǤʤƤ⥫ɽʤ
        */
  int cursor_pos;

  /***en If nonzero, display a cursor at the character position
      <cursor_pos>.  If the value is positive, it is the pixel width
      of the cursor.  If the value is negative, the cursor width is
      the same as the underlining glyph(s).  */
  /***ja 0 ǤʤС<cursor_pos> ˥ɽ롣ͤʤС
      Ϥ͡ʥԥñ̡ˤǤ롣ʤС
      륰դƱǤ롣  */
  int cursor_width;

  /***en If nonzero and <cursor_width> is also nonzero, display double
      bar cursors; at the character position <cursor_pos> and at the
      logically previous character.  Both cursors have one pixel width
      with horizontal fringes at upper or lower positions.  */
  /***ja If 0 Ǥʤ <cursor_width>  0 ǤʤСС
      ʸ<cursor_pos> ŪˤˤʸΣɽ
      롣Ȥ⣱ԥǡ岼˿ʿξ꤬Ĥ*/
  int cursor_bidi;

  /***en If nonzero, on drawing partial text, pixels of surrounding
      texts that intrude into the drawing area are also drawn.  For
      instance, some CVC sequence of Thai text (C is consonant, V is
      upper vowel) is drawn so that V is placed over the middle of two
      Cs.  If this CVC sequence is already drawn and only the last C
      is drawn again (for instance by updating cursor position), the
      right half of V is erased if this member is zero.  By setting
      this member to nonzero, even with such a drawing, we can keep
      this CVC sequence correctly displayed.  */
  /***ja 0 ǤʤСƥȤΰʬɽݤˡΥƥ
      Τɽΰ˿ʬɽ롣ȤСʸ
      λҲ-첻-ҲȤ󥹤ΤĤϡ첻ĤλҲ
      δ֤˾ˤΤ褦롣Τ褦ʥ󥹤Ǥ
      ƤꡢǸλҲľʤȤС֤
      ݤʤɡˤΥФ 0 ǤС첻αȾʬä
      ޤ 0 ʳˤ뤳ȤˤäơΤ褦ʺݤˤҲ-
      -Ҳ󥹤ɽ³뤳ȤǤ롣  */
  int partial_update;

  /***en If nonzero, don't cache the result of any drawing information
      of an M-text.  */
  /***ja 0 ǤʤСM-text ɽ˴ؤ򥭥å夷ʤ
       */
  int disable_caching;

  /* If non-NULL, limit the drawing effect to the specified region.  */
  MDrawRegion clip_region;

} MDrawControl;

/*=*/

/*** @ingroup m17nDraw */
/***en
    @brief Type of metric for gylphs and texts.

    The type #MDrawMetric is for a metric of a glyph and a drawn text.
    It is also used to represent a rectangle area of a graphic
    device.  */
/***ja
    @brief դȥƥȤˡη.

    #MDrawMetric ϥդɽ줿ƥȤˡѤηǤ롣ޤ
    ɽǥХζΰɽΤˤѤ롣 */

typedef struct {
  int x, y;
  unsigned int width, height;
} MDrawMetric;

/*=*/

/*** @ingroup m17nDraw */
/***en
    @brief Type of information about a glyph.

    The type #MDrawGlyphInfo is the structure that contains
    information about a glyph.  It is used by mdraw_glyph_info ().  */
/***ja
    @brief դ˴ؤη.

    #MDrawGlyphInfo ϥդ˴ؤޤ๽¤ΤǤ롣
    mdraw_glyph_info () ϤѤ롣  */

typedef struct
{
  /***en Character range corresponding to the glyph.  */
  /***ja դбʸϰ.  */
  int from, to;

  /***en Character ranges corresponding to the line of the glyph.  */
  /***ja  դбʸϰ.  */
  int line_from, line_to;

  /***en X/Y coordinates of the glyph.  */
  /***ja դ X/Y ɸ.  */
  int x, y;

  /***en Metric of the glyph.  */
  /***ja դˡ.  */
  MDrawMetric metrics;

  /***en Font used for the glyph.  Set to NULL if no font is found for
      the glyph.  */
  /***ja դ˻ȤեȡĤʤ NULL
      the glyph.  */
  MFont *font;

  /***en Character ranges corresponding to logically previous and next
      glyphs.  Note that we do not need the members prev_to and
      next_from because they must be the same as the members from and
      to respectively.  */
  /***ja ŪΥդбʸϰϡ prev_to 
      next_from ϡ줾 from to ƱǤʤƤϤʤʤ
      פǤ뤳Ȥա  */
  int prev_from, next_to;

  /***en Character ranges corresponding to visually left and right
      glyphs. */
  /***ja ɽκΥդбʸϰϡ  */
  int left_from, left_to;
  int right_from, right_to;

  /***en Logical width of the glyph.  Nominal distance to the next
      glyph.  */
  /***ja դŪΥդȤܾ̾εΥ  */
  int logical_width;
} MDrawGlyphInfo;

/*=*/

/*** @ingroup m17nDraw */
/***en
    @brief Type of information about a glyph metric and font.

    The type #MDrawGlyph is the structure that contains information
    about a glyph metric and font.  It is used by the function
    mdraw_glyph_list ().  */
/***ja
    @brief դˡȥեȤ˴ؤη.

    #MDrawGlyph ϥդˡȥեȤ˴ؤޤ๽¤ΤǤ
    롣mdraw_glyph_list () ϤѤ롣  */

typedef struct
{
  /***en Character range corresponding to the glyph.  */
  /***ja դбʸϰ.  */
  int from, to;

  /***en Font glyph code of the glyph.  */
  /***ja եΥեɡ  */
  int glyph_code;

  /***en Logical width of the glyph.  Nominal distance to the next
      glyph.  */
  /***ja դŪΥդȤܾ̾εΥ  */
  int x_advance, y_advance;

  /***en X/Y offset relative to the glyph position.  */
  /***ja դΰ֤Ф X/Y եå.  */
  int x_off, y_off;

  /***en Metric of the glyph.  */
  /***ja դˡ.  */
  int lbearing, rbearing, ascent, descent;

  /***en Font used for the glyph.  Set to NULL if no font is found for
      the glyph.  */
  /***ja դ˻ȤեȡĤʤ NULL
      the glyph.  */
  MFont *font;

  /***en Type of the font.  One of Mx, Mfreetype, Mxft.  */
  /***ja եȤΥסMxMfreetypeMxft Τ줫  */
  MSymbol font_type;

  /***en Pointer to the font structure.  The actual type is
      (XFontStruct *) if <font_type> member is Mx, FT_Face if
      <font_type> member is Mfreetype, and (XftFont *) if <font_type>
      member is Mxft.  */
  /***ja եȤι¤ΤؤΥݥ󥿡ºݤη <font_type> Ф
      Mx ʤ (XFontStruct *) Mfreetype ʤ FT_FaceMxft ʤ
      (XftFont *) */
  void *fontp;

} MDrawGlyph;

/*=*/

/***en
    @brief Type of text items.

    The type #MDrawTextItem is for @e textitem objects.
    Each textitem contains an M-text and some other information to
    control the drawing of the M-text.  */

/***ja
    @brief textitem η.

    #MDrawTextItem  @e ƥȥƥ ֥ѤηǤ롣
    ƥȥƥϡ1Ĥ M-text ȡɽ椹뤿γƼ
    ޤǤ롣

    @latexonly \IPAlabel{MTextItem} @endlatexonly  */

typedef struct
{
  /***en M-text. */
  /***ja M-text. */
  MText *mt;                      

  /***en Optional change in the position (in the unit of pixel) along
      the X-axis before the M-text is drawn.  */
  /***ja M-text ɽ˹ԤʤXΰĴ (ԥñ) */
  int delta;                     

  /***en Pointer to a face object.  Each property of the face, if not
      Mnil, overrides the same property of face(s) specified as a text
      property in <mt>.  */
  /***ja ե֥ȤؤΥݥ󥿡եγƥץѥƥ 
      Mnil Ǥʤ <mt> ǻꤵ줿եƱץѥƥͥ
      */
  MFace *face;

  /***en Pointer to a draw control object.  The M-text <mt> is drawn
      by mdraw_text_with_control () with this control object.  */
  /***ja ɽ楪֥ȤؤΥݥ󥿡 mdraw_text_with_control () 
      ϤΥ֥ȤѤ M-text <mt> ɽ롣  */
  MDrawControl *control;

} MDrawTextItem;

/*=*/

extern int mdraw_text (MFrame *frame, MDrawWindow win, int x, int y,
		       MText *mt, int from, int to);

extern int mdraw_image_text (MFrame *frame, MDrawWindow win, int x, int y,
			     MText *mt, int from, int to);

extern int mdraw_text_with_control (MFrame *frame, MDrawWindow win,
				    int x, int y, MText *mt, int from, int to,
				    MDrawControl *control);

extern int mdraw_coordinates_position (MFrame *frame,
				       MText *mt, int from, int to,
				       int x, int y, MDrawControl *control);

extern int mdraw_text_extents (MFrame *frame,
			       MText *mt, int from, int to,
			       MDrawControl *control,
			       MDrawMetric *overall_ink_return,
			       MDrawMetric *overall_logical_return,
			       MDrawMetric *overall_line_return);

extern int mdraw_text_per_char_extents (MFrame *frame,
					MText *mt, int from, int to,
					MDrawControl *control,
					MDrawMetric *ink_array_return,
					MDrawMetric *logical_array_return,
					int array_size,
					int *num_chars_return,
					MDrawMetric *overall_ink_return,
					MDrawMetric *overall_logical_return);

extern int mdraw_glyph_info (MFrame *frame, MText *mt, int from, int pos,
			     MDrawControl *control, MDrawGlyphInfo *info);

extern int mdraw_glyph_list (MFrame *frame, MText *mt, int from, int to,
			     MDrawControl *control, MDrawGlyph *glyphs,
			     int array_size, int *num_glyphs_return);

extern void mdraw_text_items (MFrame *frame, MDrawWindow win, int x, int y,
			      MDrawTextItem *items, int nitems);

extern void mdraw_per_char_extents (MFrame *frame, MText *mt,
				    MDrawMetric *array_return,
				    MDrawMetric *overall_return);

extern int mdraw_default_line_break (MText *mt, int pos,
				     int from, int to, int line, int y);

extern void mdraw_clear_cache (MText *mt);

/* end of drawing module */
/*=*/

/*** @ingroup m17nGUI */
/***en @defgroup m17nInputMethodWin Input Method (GUI) */
/***ja @defgroup m17nInputMethodWin ϥ᥽å (GUI) */
/*=*/

extern MInputDriver minput_gui_driver;

/*=*/
/*** @ingroup m17nInputMethodWin */
/***en 
    @brief Type of the argument to the function minput_create_ic ().

    The type #MInputGUIArgIC is for the argument $ARG of the function
    minput_create_ic () to create an input context of an internal
    input method.  */

/***ja
    @brief ؿ minput_create_ic () ΰη.

    #MInputGUIArgIC ϡؿ minput_create_ic () ϥ᥽åɤ
    ϥƥȤݤΡ $ARG ѤηǤ롣  */

typedef struct
{
  /***en Frame of the client.  */
  /***ja 饤ȤΥե졼  */
  MFrame *frame;

  /***en Window on which to display the preedit and status text.  */
  /***ja preedit ƥȤ status ƥȤɽ륦ɥ  */
  MDrawWindow client;

  /***en Window that the input context has a focus on.  */
  /***ja ϥƥȤե򤪤Ƥ륦ɥ  */
  MDrawWindow focus;
} MInputGUIArgIC;

/*=*/

extern MSymbol minput_event_to_key (MFrame *frame, void *event);

/* end of input module */
/*=*/
/* end of window modules */
/*=*/

extern MFace *mdebug_dump_face (MFace *face, int indent);
extern MFont *mdebug_dump_font (MFont *font);
extern MFontset *mdebug_dump_fontset (MFontset *fontset, int indent);

#ifdef __cplusplus
}
#endif

#endif /* _M17N_GUI_H_ */

/*
  Local Variables:
  coding: euc-japan
  End:
*/
