/*
 * medussa - a distributed cracking system
 * Copyright (C) 1999 Kostas Evangelinos <kos@bastard.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * $Id: obfuscate.c,v 1.4 2003/02/05 04:38:37 kos Exp $
 *
 * This is an attempt to conceal the linearity of what is essentially a binary
 * generator for use with portscanners etc. Works as follows:
 *
 * 1. A linear sequence of numbers is generated as normal
 * 2. Each number is broken down into bytes
 * 3. Each byte is replaced with the corresponding obfuscating table entry
 * 
 * The table is generated at init time and will be the same for the same 
 * random seed value. Hence, multiple nodes can use the same seed value and
 * not clash with each other's keyspace, since there'sa one-to-one mapping
 * between real values and obfuscated ones.
 */

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <stdlib.h>

#include "common.h"
#include "keyspace.h"
#include "xmalloc.h"
#include "generator.h"
#include "obfuscate.h"

key_index_t *
obfuscate_minindex(obfuscate_t *b) {
  return &b->minindex;
}

key_index_t *
obfuscate_maxindex(obfuscate_t *b) {
  return &b->maxindex;
}

key_index_t *
obfuscate_curindex(obfuscate_t *b) {
  return &b->index;
}

int
obfuscate_minlen(obfuscate_t *b) {
  return b->len;
}

int
obfuscate_maxlen(obfuscate_t *b) {
  return b->len;
}

int
obfuscate_destroy(obfuscate_t *k) {
  keyspace_destroy(k->divisor);
  keyspace_destroy(k->temp);
  keyspace_destroy(k->minindex);
  keyspace_destroy(k->maxindex);
  keyspace_destroy(k->index);
  xfree(k->table);
  xfree(k);
  return 0;
}

static int fill_col(char *table) {
  char beenthere[256];
  int i, j;
  unsigned char c;
  
  for(i=0; i<256; i++)
    beenthere[i] = 0;
  j=0;
  while(j<256) {
    c = rand()&0xff;
    if(beenthere[(int)c])
      continue;
    table[j] = c;
    beenthere[(int)c] = 1;
    j++;
  }
  
  return 0;
}

obfuscate_t *
obfuscate_init(char *params) {
  obfuscate_t *k;
  int i;
  
  k = (obfuscate_t *)xcalloc(1, sizeof(obfuscate_t));

  keyspace_init(k->minindex);
  keyspace_init(k->maxindex);
  keyspace_init(k->index);
  keyspace_init(k->divisor);
  keyspace_init(k->temp);
  keyspace_fromint(0x100, k->divisor);

  param_get_int(params, "len", &k->len, OBFUSCATE_DEFLEN);
  param_get_int(params, "seed", &k->seed, OBFUSCATE_DEFSEED);
  
  k->minlen = k->maxlen = k->len;
  keyspace_fromint(0, k->minindex);
  keyspace_fromint(2, k->maxindex);
  keyspace_pow(k->maxindex, k->len*8, k->maxindex);
  keyspace_set(k->index, k->minindex);  
  obfuscate_set(k, k->index);

  k->table = xcalloc(k->len, OBFUSCATE_TABLECOL_L);
  srand(k->seed);
  for(i=0; i<k->len; i++)
    fill_col(k->table + (OBFUSCATE_TABLECOL_L*i));
  
  return k;
}

int
obfuscate_fetch(obfuscate_t *k, kchar *buf, int userlen, int *len) {
  int i;
  int j;

  for(i=0; i<k->len; i++) {
    keyspace_pow(k->temp, i, k->divisor);
    keyspace_div(k->temp, k->index, k->temp);
    j = keyspace_toint(k->temp) & 0xff;
    buf[i] = *(k->table + (kchar)j + (OBFUSCATE_TABLECOL_L*i));
  }
  
  *len = k->len;
  keyspace_inc(k->index);
  return 0;
}

int
obfuscate_set(obfuscate_t *k, key_index_t index) {

  if(keyspace_cmp(index, k->minindex) < 0 || keyspace_cmp(index, k->maxindex) > 0)
    return 1;

  keyspace_set(k->index, index);
  return 0;
}

int 
obfuscate_done(obfuscate_t *k) {
  return (keyspace_cmp(k->index, k->maxindex) == 0);
}

generator_impl_t obfuscate_impl = {
  "obfuscate",
  (gen_init)obfuscate_init,
  (gen_set)obfuscate_set,
  (gen_fetch)obfuscate_fetch,
  (gen_done)obfuscate_done,
  (gen_destroy)obfuscate_destroy,
  (gen_minindex)obfuscate_minindex,
  (gen_maxindex)obfuscate_maxindex,
  (gen_curindex)obfuscate_curindex,
  (gen_minlen)obfuscate_minlen,
  (gen_maxlen)obfuscate_maxlen
};
