///////////////////////////////////////////////////////////////////////////////
// $Id: mapdrag.cpp,v 1.2 2004/04/04 18:35:22 btrummer Exp $
//
// Package:   MOAGG Edit - Level Editor for MOAGG
// Copyright: Kevin Krammer, 2004
//
///////////////////////////////////////////////////////////////////////////////
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.
//
///////////////////////////////////////////////////////////////////////////////


/*! \file    mapdrag.cpp
    \author  Kevin Krammer, kevin.krammer@gmx.at
    \brief   Declaration of the map drag object class
*/

// Qt includes
#include <qbuffer.h>
#include <qimage.h>
#include <qpixmap.h>

// local includes
#include "mapdocument.h"
#include "mapdrag.h"

///////////////////////////////////////////////////////////////////////////////

MapDrag::MapDrag(MapDocument* doc, QWidget* source, const char* name)
	: QDragObject(source, name),
	  m_doc(doc)
{
	m_imageDrag = new QImageDrag(QImage(16, 16, 32));
}

///////////////////////////////////////////////////////////////////////////////

MapDrag::~MapDrag()
{
	delete m_doc;
	delete m_imageDrag;
}

///////////////////////////////////////////////////////////////////////////////

const char* MapDrag::format(int i) const
{
	switch(i)
	{
		case 0: return "application/x-moagg-map";
		default:
			i -= 1;
			return m_imageDrag->format(i);
	}
}

///////////////////////////////////////////////////////////////////////////////

bool MapDrag::provides(const char* format) const
{
	if(strncmp(format, "application/x-moagg-map", 23) == 0) return true;

	return m_imageDrag->provides(format);
}

///////////////////////////////////////////////////////////////////////////////

QByteArray MapDrag::encodedData(const char* format) const
{
	if (m_doc == 0) return QByteArray();

	if (strncmp(format, "application/x-moagg-map", 23) == 0)
	{
		QBuffer buffer;
		buffer.open(IO_WriteOnly);
		if (m_doc->writeToDevice(&buffer))
			return buffer.buffer();
	}
	else if (m_imageDrag->provides(format))
	{
		QPixmap pixmap(16 * m_doc->numCols(), 16 * m_doc->numRows());
		uint y = 0;
		for (int r = 0; r < m_doc->numRows(); ++r, y += 16)
		{
			uint x = 0;
			for (int c = 0; c < m_doc->numCols(); ++c, x += 16)
			{
				QPixmap* pix = m_doc->pixmap(r, c);
				if (pix != 0)
					bitBlt(&pixmap, x, y, pix);
			}
		}
		QImage image;
		image = pixmap;
		m_imageDrag->setImage(image);

		return m_imageDrag->encodedData(format);
	}

	return QByteArray();
}

///////////////////////////////////////////////////////////////////////////////

bool MapDrag::canDecode(QMimeSource* source)
{
	// sanity check
	if (source == 0) return false;

	return source->provides("application/x-moagg-map");
}

///////////////////////////////////////////////////////////////////////////////

bool MapDrag::decode(QMimeSource* source, MapDocument* doc)
{
	// sanity checks
	if (source == 0 || doc == 0) return false;

	QBuffer buffer(source->encodedData("application/x-moagg-map"));
	if (!buffer.open(IO_ReadOnly)) return false;

	bool result = doc->readFromDevice(&buffer);

	buffer.close();

	return result;
}

// End of file

