/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: mresolver.c,v 1.11 2004/08/27 18:32:51 ostborn Exp $
*/

#include <sys/types.h>
#include <stdlib.h>
#include <string.h>


#include "misc.h"
#include "mresolver.h"
#include "datatypes/query/datatype.h"

/* Returns reverse domain string for a IPv4 address.
 * If parameter isn't a valid IPv4 address then it returns NULL.
 */
const char* resolver_reverse_ip(const char *ip) {
	static char ip_buf[30];
	int n[4];
	int i = 0;
	int x = 1;
	int l;
	char *p;

	if (!ip || *ip < '0' || *ip > '9' || (l = strlen(ip)) > 15 || l < 7) return NULL;

	n[0] = n[1] = n[2] = n[3] = 0;

	p = (char *)&ip[l];

	do {
		p--;
		if (*p == '.') {
			i++;
			if (i>3) return NULL;
			x = 1;
		} else if (*p >= '0' || *p <= '9') {
			n[i] += (*p - '0') * x;
			if (n[i] > 255) return NULL;
			x *= 10;
		} else {
			return NULL;
		}
	} while (p != ip);

	if (i < 3) return NULL;

	snprintf(ip_buf, sizeof(ip_buf), "%d.%d.%d.%d.in-addr.arpa.", n[0], n[1], n[2], n[3]);
	return ip_buf;
}

int resolver_start(mconfig *conf, mlogrec *rec) {
#ifdef HAVE_LIBADNS
	mlogrec_web *recweb = NULL;
	
	if (conf->enable_resolver &&
	    rec->ext_type == M_RECORD_TYPE_WEB &&
	    (recweb = rec->ext) != NULL && 
	    recweb->req_host_ip->used) {
		const char *conv_ip;

		/* start the resolver for the IP */
		conv_ip = resolver_reverse_ip(recweb->req_host_ip->ptr);

		if (conv_ip) {
			mdata *data = NULL;

			if (!mhash_in_hash(conf->query_hash, recweb->req_host_ip->ptr)) {
				adns_query *query = malloc(sizeof(adns_query));
				char *key;

				adns_submit(*(conf->adns),
					    conv_ip,
					    adns_r_ptr,
					    adns_qf_quoteok_cname|adns_qf_cname_loose,
					    NULL,
					    query
					    );

				/* put ip and query into a hash */
				key = splaytree_insert(conf->strings, recweb->req_host_ip->ptr);
				data = mdata_Query_create(key, query);
				mhash_insert_sorted(conf->query_hash, data);
			}
			/* don't free query !! it will be removed by mhash_free */
		}
	}
#endif
	return 0;
}

int resolver_finish(mconfig *conf, mlogrec *rec) {
#ifdef HAVE_LIBADNS
	/* get the resolved data */
	if (conf->enable_resolver &&
	    rec->ext_type == M_RECORD_TYPE_WEB) {
		mlogrec_web * recweb = rec->ext;

		if (!recweb->req_host_name->used && recweb->req_host_ip->used) {
			/* resolve the name if neccesary */

			adns_answer *answer = NULL;
			mdata *query;

			if (conf->debug_resolver)
				fprintf(stderr, "%s.%d: resolve %-15s -- ", __FILE__, __LINE__, recweb->req_host_ip->ptr);

			if ((query = mhash_get_data(conf->query_hash, recweb->req_host_ip->ptr))) {
				if (!query->data.query.hostname) {
					adns_wait(*(conf->adns), query->data.query.query, &answer, NULL);

					if (answer->status == adns_s_ok) {
						query->data.query.hostname = strdup(*answer->rrs.str);

						if (conf->debug_resolver)
							fprintf(stderr, "cache miss ");
					} else {
						if (conf->debug_resolver)
							fprintf(stderr, "error      ");

						query->data.query.hostname = strdup(recweb->req_host_ip->ptr);
					}
					free(answer);
				} else {
					if (conf->debug_resolver)
						fprintf(stderr, "cache hit  ");
				}

				if (conf->debug_resolver)
					fprintf(stderr, "--> %s\n", query->data.query.hostname);

				if (!is_ip(query->data.query.hostname)) {
					buffer_copy_string(recweb->req_host_name, query->data.query.hostname);
				}
			}
		}
	}
#endif
	return 0;
}
