/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <math.h>
#include <unistd.h>
#include <string.h>
#include <lcms.h>

#include "MRI.h"
#include "MRW_Loader.h"
#include "FindFile.h"
#include "options.h"

char	*progname;

void
Usage()
{
	fprintf (stderr, "Usage: %s [options...] < mrwfile > pnmfile\n", progname);
	fprintf (stderr, "Command line options:\n");
	PrintOptions (stderr);
	exit (1);
}
int
main (int argc, char *argv[])
{
	MRI *mri;
	MRI_balance	balanceSpec;
	char	buffer[2048];
	struct link *head;
	char		*errmsg;
	double lumascale;

	progname = argv[0];

	InitOptions();
	sprintf (buffer, "%s/mrwtoppm.rc", DATAPREFIX);
	ProcessOptionFile (buffer, "mrwtoppm");
	if (getenv ("HOME")) {
		sprintf (buffer, "%s/.mrwtoppm/mrwtoppm.rc", getenv("HOME"));
		ProcessOptionFile (buffer, "mrwtoppm");
	}
	if (!ProcessCmdLineOptions (argc-1, argv+1))
		Usage();

	mri = MRW_Loader (stdin, &errmsg);
	if (mri == (MRI *)0) {
		fprintf (stderr, "%s: cannot read image from standard input (%s).\n", progname, errmsg);
		Usage ();
	}

	fprintf (stderr, "%s: Info: Raw image from camera %s\n", progname, MRI_GetCameraName (mri));
	fprintf (stderr, "%s: Info: CCD size = %dx%d\n", progname, MRI_GetHeight(mri), MRI_GetWidth(mri));

	/* Apply camera adjustment (if requested). */
	if (strcmp (cameraDetails, "none") != 0) {
		char *cameraDetailsFile;
		if ((cameraDetailsFile = FindFile (cameraDetails, cameraDetailsPath, ".mrw")) != (char *)0) {
			fprintf (stderr, "%s: Info: Adjusting for camera %s\n", progname, cameraDetails);
			MRI_ApplyNoiseMap (mri, cameraDetailsFile);
		}
		else {
			fprintf (stderr, "%s: Error: Could not find camera details '%s'. Output not adjusted.\n", progname, cameraDetails);
		}
	}

	if (cameraProfileName != NULL)
	     mri->colorSpace = strdup (cameraProfileName);

	if ((MRI_GetWidth(mri) % 2) == 1 || (MRI_GetHeight(mri) % 2) == 1) {
		fprintf (stderr, "%s: Limitation: Unable to process odd sized ccd arrays.\n", progname);
		exit (1);
	}

	if (!MRI_GetBalance (&balanceSpec, mri, balance1))
            FindBalanceSpec (&balanceSpec, balance1, colorMapPath);
	MRI_AdjustBalanceLuminance (mri, &balanceSpec, &lumascale);

#ifdef DEBUG
	i = 0; while (getchar() != EOF) i++;
	if (i != 0)
		fprintf (stderr, "%s: Info: %d bytes followed image data\n", progname, i);
#endif
	/* At this point, the entire image has been input.  We now construct the processing pipeline
 	 * to convert the input image into the output. The process starts from the last stage, the
	 * output writer, and builds towards the front of the processing pipeline.
	 */

	/* Before generating the last link in that pipeline, we need to know the order of pixels in the scanlines
	 * coming down the pipeline. The rotate option means that the scanlines can begin with either
	 * a RG..RG scanline (if working from the top of the image), or with a GB...GB scanline (if working from
	 * the bottom). If we also need to reverse the order of pixels in each scanline, we defer doing so until
	 * the scanline is output. (It considerably simplifies the earlier code, which need not consider GR...GR
	 * scanlines as well as the normal order, with only a trivial increase in complexity of the output stage.
	 */

	{ char *cameraProfileFile;
	  char *outputProfileFile;

	  if (strcmp (mri->colorSpace, MRI_NATIVE_PROFILE) == 0)
	      cameraProfileFile = strdup (MRI_NATIVE_PROFILE);
	  else {
	      cameraProfileFile = FindFile (mri->colorSpace, profilePath, NULL);
	      if (cameraProfileFile == (char *)0) {
		  fprintf (stderr, "%s: Error: cannot find camera color space profile: %s\n", progname, mri->colorSpace);
	 	  exit (1);
	      }
	  }
	  if (strcmp (outputProfileName, "__sRGB__") == 0)
		outputProfileFile = strdup ("__sRGB__");
	  else {
		outputProfileFile = FindFile (outputProfileName, profilePath, NULL);
		if (outputProfileFile == (char *)0)
			fprintf (stderr, "%s: Error: cannot find output color space profile: %s\n", progname, outputProfileName);
	  }
	  InitColorProfiles (mri, useLab, cameraProfileFile, outputProfileFile);
	}

	/* Create output filter sequence starting from the rear.
	 * The last link in the pipeline outputs scanlines to the output device.
	 */
	if (xvmini)
		head = MRI_GenXVMiniWriter (mri, Reversed(rotate), stdout);
	else
		head = MRI_GenPPMWriter (outputBits, Reversed(rotate), stdout);

#if 0
	/* Apply additional gamma correction and "contrast" stretching.
	 */
        head = GenMRIEnhance (contrast, gammaVal, head);
#endif

	/* Convert from the internal color space to the output color space.
	 */
	head = GenOutputProfileConverter (mri, head);

	/* Convert from LINETYPE_DOUBLE into LINETYPE_SHORT.
	 */
	head = MRI_GenFtSConverter (head, mri, useLab);

	{ MRI_ImproverOptions io;
	  MRI_ImproverInfo info;
	  io.radius = radius;
	  io.blur = blur;
	  io.iSharpen = iSharpen;
	  io.cSharpen = cSharpen;
	  io.useLab = useLab;
	  io.applyVMF = applyVMF;
	  io.filterColorOnly = filterColorOnly;
	  io.applyMedianFilter = applyMedianFilter;
	  io.toneCurveFile = (char *)0;
	  io.vMedianTolerance = vmedian_tolerance;
	  if (strcmp (toneCurve, "none") != 0) {
		if ((io.toneCurveFile = FindFile (toneCurve, toneCurvePath, ".tc")) != (char *)0) {
			fprintf (stderr, "%s: Info: Applying tone curve %s\n", progname, toneCurve);
		}
		else {
			fprintf (stderr, "%s: Error: Could not find tone curve '%s'. Output not adjusted.\n", progname, toneCurve);
		}
	  }

	  head = MRI_GenImageImprover (head, &io, &info);
	}

	if (useLab) {
#ifdef DEBUG
		fprintf (stderr, "Darkness=%g, contrast=%g, saturation=%g\n", darkness, contrast, saturation);
#endif
		head = MRI_GenContrastAdjuster (darkness, lumascale, contrast, shadows, saturation, head);
	}

	if (shrink == 1)
		fprintf (stderr, "%s: Info: Generating full size output using interpolation method %s\n", progname, imethod);
	else
		fprintf (stderr, "%s: Info: Generating 1/%d size output\n", progname, shrink);
	{ MRI_InterpolatorOptions io;
	io.useLab = useLab;
	io.rotate = rotate;
	io.imethod = imethod;
	io.vmedian_tolerance = vmedian_tolerance;
	io.doInterpolate = shrink == 1;
	head = MRI_GenBayerInterpolator (mri, head, &io, NULL);
	}

	FreeProfileFiles ();

	if (shrink > 1)
		head = MRI_GenSubsampler (shrink, shrink, head);

	head = MRI_GenLimitCorrecter (balanceSpec, head, MRI_RGFirst(mri, rotate));
	head = MRI_GenLimitMasker (head);

	/* Smooth stuck pixels. */
	if (stuckPixelFile != NULL) {
		fprintf (stderr, "%s: Info: Removing stuck pixels given by %s\n", progname, stuckPixelFile);
		head = MRI_GenStuckPixelFilter (stuckPixelFile, rotate, mri, head);
	}

	{ MRI_Region r;
	  int tmp;
	  tmp = MRI_GetWidth (mri);
	  r.width = (tmp / shrink) * shrink;
	  r.x = (tmp - r.width)/2;
	  if (r.x & 1) r.x--;
	  tmp = MRI_GetHeight (mri);
	  r.height = (tmp / shrink) * shrink;
	  r.y = (tmp - r.height)/2;
	  if (r.y & 1) r.y--;
	  fprintf (stderr, "Selected %dx%d+%d+%d\n", r.width, r.height, r.x, r.y);

          MRI_ProcessImageRegion (head, mri, rotate, &r);
	}

        MRI_FlushPipeline (head);

	exit (0);
}
