// pvanalyzer.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/
#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "envelope.h"
#include "localdefs.h"
#include "query.h"
#include "request.h"
#include "pvanalyzer.h"
#include "pvocdata.h"
#include "pvocrequester.h"

class PVAnalysisRequester : public PvocRequester {
	friend class PVAnalyzer;
protected:
	PVAnalysisRequester(const char* title, PhaseVocoder::Info &);
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
private:
	enum FFTSize {
		F_64 = 0x1, F_128 = 0x2, F_256 = 0x4, F_512 = 0x8, F_1024 = 0x10,
		F_2048 = 0x20, F_4096 = 0x40, F_8192 = 0x80, F_16384 = 0x100
	};
	enum OverLap { X4 = 0x1, X2 = 0x2, X1 = 0x4, Xp5 = 0x8 };
	double _frate;
	static int _frameSize;
	static ChoiceValue _fftSize;
	static ChoiceValue _overlapFactor;
};

int PVAnalysisRequester::_frameSize = 0;
ChoiceValue PVAnalysisRequester::_fftSize = PVAnalysisRequester::F_256;
ChoiceValue PVAnalysisRequester::_overlapFactor = PVAnalysisRequester::X1;

PVAnalysisRequester::PVAnalysisRequester(const char* title,
                                         PhaseVocoder::Info &info)
	: PvocRequester(title, info), _frate(0.0) {}

void
PVAnalysisRequester::configureRequest(Request* request) {
	request->appendLabel("All \"0\" values will be set to defaults.");
	request->appendValue("Input frame size (samples):",
			     &_frameSize,
			     PositiveIntegers);
	request->appendValue("Input frame offset (samples):",
			     &_pvocInfo.inputFrameOffset,
			     PositiveIntegers);
	request->appendValue("Input frame rate (Hz):", &_frate,
			     PositiveIntegers);
	request->appendValue("Time Scaling factor:",
			     &_pvocInfo.timeScaleFactor,
			     PositiveNumbers);
	request->appendChoice("FFT Size:",
	        "|64|128|256|512|1024|2048|4096|8192|16384|", &_fftSize, true);
	request->appendChoice("Filter Overlap Factor (x FFT size):",
	                      "|4x|2x|1x|x/2|", &_overlapFactor, true);
	request->appendChoice("Window Type:", "|Hanning|Kaiser|",
	                      &_windowType, true);
}

boolean
PVAnalysisRequester::confirmValues() {
	if(_frate > 0.0)	// if user entered frame rate, reset frame offset
		_pvocInfo.inputFrameOffset = (int(_pvocInfo.samplingRate/_frate));
	int fftpoints = _fftSize << 6; // {1, 2, 4, ..} => {64, 128, 256, ...}
	int frmsize;
	if(_frameSize == 0)	// if nonzero, no need to use overlap factor
		switch(_overlapFactor) {
			case X4:  frmsize = fftpoints * 4;  break;
			case X2:  frmsize = fftpoints * 2;  break;
			case X1:  frmsize = fftpoints;  break;
			case Xp5: frmsize = fftpoints/2;  break;
			default:  break;
		}
	else
	    frmsize = _frameSize;

	_pvocInfo.inputFrameSize = frmsize;
	_pvocInfo.fftSize = fftpoints;
	_pvocInfo.K = (_windowType == Kaiser);	// flag for Kaiser windowing
	return true;
}

//********

PhaseVocoder::Info PVAnalyzer::_savedPvocInfo(0.0, 0.0f, PhaseVocoder::Analysis);

PVAnalyzer::PVAnalyzer(Data* data) : ArrayFunction(data),
	  			     _pvocInfo(_savedPvocInfo),
	  			     _pvoc(nil) {
    // always reset these to match current data
    _pvocInfo.samplingRate = data->sRate();
    _pvocInfo.inputScalingFactor = (data->dataType() == FloatData) ? 1.0 : 1/32767.0;
}

PVAnalyzer::PVAnalyzer(
		Data* data,
		int N, int F, int W, int M, int D, double T, double P,
		PhaseVocoder::Mode mode, boolean useKaiser)
			: ArrayFunction(data, M, D),
			  _pvocInfo(data->sRate(),
		           (data->dataType()==FloatData) ? 1.0 : 1/32767.0,
		           mode,
		           N, F, W, M, D, T, P,
		           useKaiser
		 	 ),		
		 	 _pvoc(nil) {
	initialize();
}

PVAnalyzer::~PVAnalyzer() {
	delete _pvoc;
}

Requester *
PVAnalyzer::createRequester() {
	return new PVAnalysisRequester(
		"Phase Vocoder Analysis of Selected Region:",
		_pvocInfo
	);	
}

void
PVAnalyzer::initialize() {
	_pvoc = new PhaseVocoder(_pvocInfo);
	if(_pvoc->isGood()) {
		setBaseValues(
			_pvoc->roundedInputFrameSize(), 0.0, _pvoc->getInputFrameOffset()
		);
		Super::initialize();
		setAnalysis(
			new PvocData(analysisLength(), _pvoc->analysisChannels(),
			sampRate(), framerate())
		);
	}
}

void
PVAnalyzer::saveConfig() {
    _savedPvocInfo = _pvocInfo;
}

int
PVAnalyzer::operator () (double* in, Data* frame) {
	int status = _pvoc->runAnalysis(in, frame);
	setOffset(_pvoc->getInputFrameOffset());		// update values
	return status;
}

int
PVAnalyzer::initialOffset() { return _pvoc->getStartingOffset(); }

int
PVAnalyzer::analysisLength() {
	return _pvoc->calculateAnalysisLength(target()->length());
}
