// reson.c

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "localdefs.h"
#include "request.h"
#include "reson.h"
#include "valuerequester.h"
#include <math.h>

ResonFilter::ResonFilter(Data* output, double fgain, bool needUndo)
		: SimpleFunction(output, needUndo),
	  	  centerFreq(1000.0), bandWidth(100.0), gain(fgain) {}

void
ResonFilter::set(double ctrFrq, double bWdth) {
	centerFreq = ctrFrq;
	bandWidth = bWdth;
	setCoeffs();
}

//********

class Reson3Requester : public ValueRequester<double> {
	friend class Reson3;
protected:
	Reson3Requester(Reson3 *);
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
private:
	Reson3* client;
};

Reson3Requester::Reson3Requester(Reson3* r)
	: ValueRequester<double>("Apply Resonant Filter to Selected Region:",
	                "Gain Factor:",
	                r->gain), client(r) {}

void
Reson3Requester::configureRequest(Request* request) {
	Range allFreqs(0.0, client->sampRate()/2.0);
	request->appendValue("Center Frequency (Hz.):",
	                     &client->centerFreq, allFreqs);
	request->appendValue("BandWidth (Hz.):",
	                     &client->bandWidth, allFreqs);
	request->appendChoice("Filter gain mode:",
	                      "|Peak Set To Unity|Unity RMS Gain|No Rescaling|",
	                      &client->gainMode, true);
	ValueRequester<double>::configureRequest(request);
}

boolean
Reson3Requester::confirmValues() {
	client->setCoeffs();
	return true;
}

//********

Reson3::Reson3(Data* output, double ctrFreq, double bWdth, GainMode scfac)
		: ResonFilter(output, 1.0, false), gainMode(scfac) {
	initialize();
	set(ctrFreq, bWdth);
}

Reson3::Reson3(Data* output)
	: ResonFilter(output, 1.0, true), gainMode(UnityRMS) {}

Modifier *
Reson3::create(DataEditor *de) {
	return nil;
}

void
Reson3::setCoeffs() {
	coef2 = exp(-M_PI * 2.0 * bandWidth / sampRate());
	double temp = 1.0 - coef2;
	double c = coef2 + 1;
	coef1 = 4.0 * coef2/c * cos(2.0 * M_PI * centerFreq / sampRate());
	coef0 = (gainMode == NoRescale) ? 1.0 :
		(gainMode == UnityRMS) ? sqrt(temp/c*(c*c - coef1*coef1)) :
		temp * sqrt(1.0 - coef1*coef1/(4.0*coef2));
	coef0 *= gain;		// gain factor added
}

void
Reson3::clear() {
	for(int i=0; i<2; i++) past[i] = 0.0;
}

Requester *
Reson3::createRequester() {
	return new Reson3Requester(this);
}

void
Reson3::initialize() {
	clear();
	Super::initialize();
}

void
Reson3::operator () (double *input, int count) {
    for (int n = 0; n < count; n++) {
	double out = coef0 * input[n] + coef1 * past[0] - coef2 * past[1];
	past[1] = past[0];
	past[0] = out;
	input[n] = out;
    }
}

// ********

Reson4::Reson4(Data* output, double centerFreq, double bandWidth)
		: ResonFilter(output, 1.0, false) {
	initialize();
	set(centerFreq, bandWidth);
}

void
Reson4::setCoeffs() {
	coef0 = exp(-M_PI * bandWidth / sampRate());
	coef1 = 1.0 - coef0;
	coef1 *= gain;		// gain factor added
	coef2 = 2.0 * coef0 * cos(2.0 * M_PI * centerFreq / sampRate());
	coef3 = -coef0 * coef0;
}

void
Reson4::restoreState() {
	initialize();
}

void
Reson4::clear() {
	for(int i=0; i<2; i++) inPast[i] = outPast[i] = 0.0;
}

void
Reson4::initialize() {
	clear();
	Super::initialize();
}

void
Reson4::operator () (double *input, int count) {
    for (int n = 0; n < count; n++) {
	double out = coef1*(input[n]-coef0*inPast[1])+coef2*outPast[0]+coef3*outPast[1];
	inPast[1] = inPast[0];
	inPast[0] = input[n];
	outPast[1] = outPast[0];
	outPast[0] = out;
	input[n] = out;
    }
}
