/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_texture.h 
 * @brief Implementation of a texture class.
 * 
 */

#ifndef __GC_TEXTURE_H__
#define __GC_TEXTURE_H__

#include <string>
#include <hash_map>
#include <gl/gl.h>

#include "myx_gc_utilities.h"

using namespace std;

//----------------------------------------------------------------------------------------------------------------------

// Default values for texturing.
const string DefaultTextureWrapMode("clamp");     
const string DefaultTextureMinFilter("nearest"); 
const string DefaultTextureMagFilter("nearest");      
const int DefaultTextureDimensions = 2;
const string DefaultTextureMode("decal");

//----------------------------------------------------------------------------------------------------------------------

class CTextureManager;

// CGCTexture encapsulates a png image used to texture a figure in Generic Canvas. It loads the image data
// and manages it as well as the OpenGL properties for it.
// Note: a texture can be shared amongst several figures however its properties (e.g. minification filter)
//       are set globally and are used for all referencing figures.
class CGCTexture
{
private:
  CTextureManager* FManager; // The manager by which this texture is controlled.
  string FName;              // The identifier of this texture. It is referenced by this name from figures.
  bool FLoaded;              // Set to true once the image data is uploaded to OpenGL.
  string FFilename;          // The name of the png file holding the actual image data.
  GLenum FWrapModeS;         // Specifies how textures are handled at the edges of a figure (horizontal direction).
  GLenum FWrapModeT;         // Specifies how textures are handled at the edges of a figure (vertical direction).
  GLenum FMinFilter;         // Specifies how stretching of texture data is performed when the actual screen image is smaller than the image data.
  GLenum FMagFilter;         // Specifies how stretching of texture data is performed when the actual screen image is larger than the image data.
  GLenum FTarget;            // One-dimensional or two-dimensional textures.
  GLuint FHandle;            // The OpenGL handle for this texture.
  GLint FMode;               // The mode how to apply the texture to the surface.
protected:
  void LoadTexture(void);
public:
  CGCTexture(CTextureManager* Controller, const string& FileName, const string& ID, GLenum WrapModeS, GLenum WrapModeT, 
    GLenum MinFilter, GLenum MagFilter, int Dimensions, GLenum TextureMode);
  ~CGCTexture(void);
  void ActivateTexture(void);
  void DeactivateTexture(void);
};


#pragma warning(disable: 4251) // Disable warning about DLL interface for template classes.

// The list of textures is an associated list of names and CTexture classes.
typedef stdext::hash_map<string, CGCTexture*> CTextures;
typedef stdext::hash_map<string, CGCTexture*>::iterator CTextureIterator;

class CTextureManager
{
private:
  CTextures FTextures;          // A list of CTexture classes.
public:
  ~CTextureManager(void);
  CGCTexture* FindTexture(const string& Name);
  void CreateTextureEntry(const string& Filename, const string& ID, const string& WrapH, const string& WrapV,
    const string& MinificationFilterStr, const string& MagnificationFilterStr, int Dimensions, const string& Mode);
  void ClearTextures(void);
};

//----------------------------------------------------------------------------------------------------------------------

CTextureManager* TextureManager(); // The one (and only) texture manager instance.

#endif // #ifdef __GC_TEXTURE_H__