/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */


#ifdef ENABLE_PHP_MODULES

#include "myx_grt_php.h"
#include "myx_shared_aux_functions.h"

#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
#include <Basetsd.h>
#else
#define LongToPtr(l) ((void*)(l))
#define PtrToLong(p) ((long)(p))
#endif

static MYX_GRT_ERROR php_call_function(MYX_GRT_FUNCTION *function, MYX_GRT_VALUE *value, MYX_GRT_VALUE **retval);
static MYX_GRT_ERROR php_init_module(MYX_GRT_MODULE_LOADER *loader, const char *file, MYX_GRT_MODULE **retmodule);
static MYX_GRT_ERROR php_fetch_messages(MYX_GRT_MODULE_LOADER *loader, MYX_GRT_MSGS *msgs);

static int php_set_ini_entry(char *entry, char *value, int stage);
static int php_ub_write(const char *str, unsigned int str_length TSRMLS_DC);
static void php_log_message(char *message);
static void php_sapi_error(int type, const char* fmt, ...);

static char * php_eval_string(const char *fmt, ...);

// thread "safety" !!!
static void ***tsrm_ls;

/**
 ****************************************************************************
 * @brief Helper function to convert a global object to a php object
 *
 *   This function is called when the GRT module is initialized
 *****************************************************************************/
/*static jobject php_global_object_from_grt_value(MYX_GRT *grt, const char *objectPath, MYX_GRT_VALUE *value)
{
  // this can be used to find a class
  zend_class_entry **ce;

 	if (zend_lookup_class(class_name, class_name_len, &ce TSRMLS_CC) == SUCCESS) {
 		RETURN_BOOL(((*ce)->ce_flags & ZEND_ACC_INTERFACE) == 0);
	} else {
		RETURN_FALSE;
	}
  // --------------------------------


  jobject jobj= NULL;
  MYX_JAVA_LOADER *jloader= find_java_loader(grt);

  if ( (value) && (jloader) )
  {
    if (myx_grt_value_get_type(value) == MYX_STRING_VALUE)
    {
      jobj= (*env)->NewStringUTF(env, myx_grt_value_as_string(value));
    }
    else if (myx_grt_value_get_type(value) == MYX_INT_VALUE)
    {
      jvalue *args= g_new0(jvalue, 1);

      args[0].i= myx_grt_value_as_int(value);
      jobj= (*env)->NewObjectA(env, jloader->java_integer, jloader->java_integer_init, args);
      g_free(args);
    }
    else if (myx_grt_value_get_type(value) == MYX_REAL_VALUE)
    {
      jvalue *args= g_new0(jvalue, 1);

      args[0].d= myx_grt_value_as_real(value);
      jobj= (*env)->NewObjectA(env, jloader->java_double, jloader->java_double_init, args);
      g_free(args);
    }
    else if (myx_grt_value_get_type(value) == MYX_LIST_VALUE)
    {
      jvalue *args= g_new0(jvalue, 2);
      jclass j_class;
      jmethodID j_init;
      char *j_class_name= g_strdup(GRTCLASSPREFIX);

      if ( (myx_grt_list_content_get_type(value) == MYX_STRING_VALUE) && 
        (!myx_grt_list_content_get_struct_name(value)) )
        j_class_name= str_g_append(j_class_name, "GrtStringList");
      else if (myx_grt_list_content_get_struct_name(value))
      {
        j_class_name= str_g_append_and_free(j_class_name, 
          str_g_replace(g_strdup(myx_grt_list_content_get_struct_name(value)), ".", "/"));
        j_class_name= str_g_append(j_class_name, "List");
      }
      else
        j_class_name= str_g_append(j_class_name, "GrtList");
        

      // Find List class
      j_class= (*env)->FindClass(env, j_class_name);
      if (!j_class)
        return NULL;

      // Find List's <init> constructor
      j_init= (*env)->GetMethodID(env, j_class, 
        "<init>", "(Ljava/lang/String;Ljava/lang/String;)V");
      if (!j_init)
        return NULL;

      if (myx_grt_list_content_get_struct_name(value))
        args[0].l= (*env)->NewStringUTF(env, myx_grt_list_content_get_struct_name(value));
      else
        args[0].l= (*env)->NewStringUTF(env, "");
      args[1].l= (*env)->NewStringUTF(env, objectPath);
      jobj= (*env)->NewObjectA(env, j_class, j_init, args);
      g_free(args);

      g_free(j_class_name);
    }
    else if (myx_grt_value_get_type(value) == MYX_DICT_VALUE)
    {
      jvalue *args;
      jclass j_class;
      jmethodID j_init;
      char *j_class_name= g_strdup(GRTCLASSPREFIX);
      
      if (myx_grt_dict_struct_get_name(value))
      {
        // this is a struct instance
        j_class_name= str_g_append_and_free(j_class_name, 
          str_g_replace(g_strdup(myx_grt_dict_struct_get_name(value)), ".", "/"));

        // Find Dict class
        j_class= (*env)->FindClass(env, j_class_name);
        if (!j_class)
          return NULL;
          
        // Find Dict's <init> constructor
        j_init= (*env)->GetMethodID(env, j_class, "<init>", "(Ljava/lang/Object;)V");
        if (!j_init)
          return NULL;

        /// Create new object instance
        args= g_new0(jvalue, 1);
        args[0].l= (*env)->NewStringUTF(env, objectPath);
        jobj= (*env)->NewObjectA(env, j_class, j_init, args);
        g_free(args);
      }
      else
      {
        if ( (myx_grt_dict_content_get_type(value) == MYX_STRING_VALUE) && (!myx_grt_dict_struct_get_name(value)) )
          j_class_name= str_g_append(j_class_name, "GrtStringHashMap");
        else
          j_class_name= str_g_append(j_class_name, "GrtHashMap");

        // Find List class
        j_class= (*env)->FindClass(env, j_class_name);
        if (!j_class)
          return NULL;
          
        // Find List's <init> constructor
        j_init= (*env)->GetMethodID(env, j_class, "<init>", "(Ljava/lang/String;Ljava/lang/String;)V");
        if (!j_init)
          return NULL;

        /// Create new object instance
        args= g_new0(jvalue, 2);
        args[0].l= (*env)->NewStringUTF(env, "");
        args[1].l= (*env)->NewStringUTF(env, objectPath);
        jobj= (*env)->NewObjectA(env, j_class, j_init, args);
        g_free(args);
      }

      g_free(j_class_name);
    }

    return jobj;
  }
  else
    return NULL;
}*/


/**
 ****************************************************************************
 * @brief Init function of the GRT callback module
 *
 *   This function is called when the GRT module is initialized
 *****************************************************************************/
PHP_MINIT_FUNCTION(grt)
{
  //Init
  return SUCCESS;
}

/**
 ****************************************************************************
 * @brief Finalize function of the GRT callback module
 *
 *   This function is called when the PHP engine is shut down
 *****************************************************************************/
void php_grt_module_shutdown()
{
}

/**
 ****************************************************************************
 * @brief Grt callback function to execute another Grt function
 *
 *   Calls the given function and returns the result as XML string
 *****************************************************************************/
PHP_FUNCTION(grtCallGrtFunction)
{
  zval **myx_grt_pointer;
  zval **module;
  zval **function_name;
  zval **arguments;
  MYX_GRT *grt;
  MYX_GRT_ERROR error;
  MYX_GRT_VALUE *result= NULL;
  const char *c_str_module;
  const char *c_str_function_name;
  const char *c_str_arguments= NULL;

  printf("The native function grtSetApplicationPath was called.\n");

  // fetch parameters
  if (ZEND_NUM_ARGS() != 4 || 
    zend_get_parameters_ex(4, &myx_grt_pointer, &module, &function_name, &arguments) == FAILURE) {
		WRONG_PARAM_COUNT;
	}

  // get parameters as strings
  if (Z_TYPE_PP(module) == IS_STRING)
    c_str_module= Z_STRVAL_PP(module);

  if (Z_TYPE_PP(function_name) == IS_STRING)
    c_str_function_name= Z_STRVAL_PP(function_name);

  if (Z_TYPE_PP(arguments) == IS_STRING)
    c_str_arguments= Z_STRVAL_PP(arguments);
    

  if (c_str_module && c_str_function_name)
  {
    // get pointer to grt
    convert_to_long_ex(myx_grt_pointer);
    grt= LongToPtr(Z_LVAL_PP(myx_grt_pointer));

    // do the actual function call
    result= myx_grt_function_get_and_call(grt, c_str_module, c_str_function_name, 0, 
      myx_grt_value_from_xml_global_object(grt, c_str_arguments, strlen(c_str_arguments)), 
      &error);

    if (error != MYX_GRT_NO_ERROR)
    {
      // return an error
      php_error_docref(NULL TSRMLS_CC, E_WARNING, "the function returned and error %d", error);
    }
  }

  // return result as XML string
	Z_STRVAL_P(return_value) = myx_grt_value_to_xml(result);
	Z_STRLEN_P(return_value) = (int) strlen(Z_STRVAL_P(return_value));
	Z_TYPE_P(return_value) = IS_STRING;
}

/**
 ****************************************************************************
 * @brief Grt callback function to execute another Grt function
 *
 *   Calls the given function and returns the result as XML string
 *****************************************************************************/
PHP_FUNCTION(grtGetGrtGlobalById)
{
  zval **myx_grt_pointer;
  zval **id;
  MYX_GRT *grt;
  MYX_GRT_VALUE *value= NULL;
  const char *c_str_id;

  // fetch parameters
  if (ZEND_NUM_ARGS() != 2 || 
    zend_get_parameters_ex(2, &myx_grt_pointer, &id) == FAILURE) {
		WRONG_PARAM_COUNT;
	}

  // get parameters as strings
  if (Z_TYPE_PP(id) == IS_STRING)
    c_str_id= Z_STRVAL_PP(id);

  if (c_str_id)
  {
    // get pointer to grt
    convert_to_long_ex(myx_grt_pointer);
    grt= LongToPtr(Z_LVAL_PP(myx_grt_pointer));

  }
}

function_entry php_grt_functions[] = {
  PHP_FE(grtCallGrtFunction, NULL)
  PHP_FE(grtGetGrtGlobalById, NULL)
  { NULL, NULL, NULL }
};

zend_module_entry php_grt_module_entry = {
  STANDARD_MODULE_HEADER,
  APPLNAME,
  php_grt_functions,
  PHP_MINIT(grt),
  NULL,
  NULL,
  NULL,
  NULL,
  PHP_GRT_VERSION,
  STANDARD_MODULE_PROPERTIES
};


/**
 ****************************************************************************
 * @brief Initializes the PHP loader
 *
 *   Initializes the PHP engine and registers the grt callback module
 *
 * @return  returns 0 if engine was initialized correctly
 *****************************************************************************/
MYX_GRT_MODULE_LOADER *myx_php_init_loader(MYX_GRT *grt, MYX_GRT_ERROR *error)
{
  MYX_GRT_MODULE_LOADER *loader= g_new0(MYX_GRT_MODULE_LOADER, 1);
  MYX_PHP_LOADER *priv= g_new0(MYX_PHP_LOADER, 1);
  
  static char *file_extensions[]= {
    ".php"
  };

  static char *argv[2] = {APPLNAME, NULL};

  *error= MYX_GRT_NO_ERROR;

  loader->grt= grt;
  loader->loader_type= MYX_PHP_MODULE_TYPE;
  loader->priv= priv;
  loader->init_module= php_init_module;
  loader->call_function= php_call_function;
  loader->fetch_messages= php_fetch_messages;
  loader->extensions_num= 1;
  loader->extensions= file_extensions;

  // override PHP/ZE output
  php_embed_module.ub_write= php_ub_write;
  php_embed_module.log_message= php_log_message;
  php_embed_module.sapi_error= php_sapi_error;
  php_embed_module.php_ini_path_override = "./php.ini";

  if (php_embed_init(1, argv PTSRMLS_CC) == FAILURE) 
    goto error;

  // set ini values
  php_set_ini_entry("html_errors", "0", PHP_INI_STAGE_ACTIVATE);
  php_set_ini_entry("display_errors", "1", PHP_INI_STAGE_ACTIVATE);
  php_set_ini_entry("display_startup_errors", "1", PHP_INI_STAGE_ACTIVATE);
  php_set_ini_entry("max_execution_time", "0", PHP_INI_STAGE_ACTIVATE);

  // register the irssi "extension"
  zend_startup_module(&php_grt_module_entry);

  // set the current directory
  php_eval_string("chdir('./php')");

  // include the core PHP classes
  php_eval_string("include 'GrtInit.php'");

  return loader;
  
error:
  *error= MYX_GRT_MODULE_INIT_ERROR;
//errorfree:
  g_free(priv);
  g_free(loader);
  return NULL;
}

/**
 ****************************************************************************
 * @brief Finalizes the PHP engine
 *
 *   The php_grt_module_shutdown function is called explicitly because
 * there are known issues with using the MSHUTDOWN function
 *****************************************************************************/
static void finialize_php(void)
{
  php_grt_module_shutdown();

  php_embed_shutdown(TSRMLS_C);
}

/**
 ****************************************************************************
 * @brief Modifies an PHP ini setting
 *
 *   Modifies the given entry
 *
 * @param entry
 *           name of the entry to modify
 * @param value  
 *           new value to which the entry should be set
 * @param stage  
 *           the stage
 * 
 * @return  returns SUCCESS if the string was processed correctly.
 *****************************************************************************/
static int php_set_ini_entry(char *entry, char *value, int stage)
{
  int retval;

  retval = zend_alter_ini_entry(entry, (uint)strlen(entry)+1, value, (uint)strlen(value)+1,
    PHP_INI_USER, stage);

  return retval;
}

/**
 ****************************************************************************
 * @brief Captures PHP output
 *
 *   This function captures PHP output so that it is not written to stdout
 *
 * @param str
 *           string that was written out
 * @param str_length  
 *           length of the string that was written out
 * 
 * @return  returns SUCCESS if the string was processed correctly.
 *****************************************************************************/
static int php_ub_write(const char *str, unsigned int str_length TSRMLS_DC)
{
  printf("%.*s\n", str_length, str);
  return str_length;
}

/**
 ****************************************************************************
 * @brief Captures PHP log message
 *
 *   This function captures PHP log message so that it is not written to stdout
 *
 * @param message
 *           message that was written out
 *****************************************************************************/
static void php_log_message(char *message)
{
  printf("log_message: %s\n", message);
}

/**
 ****************************************************************************
 * @brief Captures PHP SAPI error message
 *
 *   This function captures PHP SAPI error message so that it is not written 
 * to stdout
 *
 * @param type
 *           type of the message
 * @param fmt
 *           the error string
 *****************************************************************************/
static void php_sapi_error(int type, const char* fmt, ...)
{
  printf("sapi_error: %d, %s\n", type, fmt);
}

/**
 ****************************************************************************
 * @brief Executes one line of PHP code and returns the result as string
 *
 *   This function is called to execute one line of PHP code.
 *
 * @param appl_name
 *           name of the application, this is only used for error messages
 * @param fmt  
 *           a line of PHP code, can contain printf typical output format 
 *           definitions like %d or %s
 * @param ...  
 *           optional arguments to that will substitute the output format 
 *           definitions in fmt
 * 
 * @return  returns a string if the executed code returned a string. 
 *          Otherwise NULL is returned
 *****************************************************************************/
static char * php_eval_string(const char *fmt, ...)
{
  enum eval_ret_t retval;
  char *data= NULL;
  zval *zv= (zval *)malloc(sizeof(zval));
  char *res= NULL;

  va_list ap;
  va_start(ap, fmt);

  zend_first_try {
    vspprintf(&data, 0, fmt, ap);

    retval = zend_eval_string(data, zv, (char*)APPLNAME TSRMLS_CC) == SUCCESS ? EVAL_RET_OK : EVAL_RET_ERROR;

    //check if the returned var is a string
    //if so, return it
    if ((retval == EVAL_RET_OK) && (Z_TYPE_P(zv) == IS_STRING))
        res= g_strdup(Z_STRVAL_P(zv));
  } zend_catch {
    //retval = EVAL_RET_BAIL;
  } zend_end_try();

  if (data)
    efree(data);

  va_end(ap);

  return res;
}



static MYX_GRT_ERROR php_init_module(MYX_GRT_MODULE_LOADER *loader, const char *file, MYX_GRT_MODULE **retmodule)
{
  MYX_PHP_LOADER *jloader= loader->priv;
  MYX_GRT_MODULE *module;
  MYX_PHP_MODULE *pmodule;
  char *class_name;
  char *xml;
  char *module_name= NULL;
  MYX_GRT_VALUE *module_functions= NULL;
  char *extends= NULL;
  char *ptr;
  unsigned int i;
  const char *module_filename= file+6; //remove ./php

  // call class::getModuleinfo();
  class_name = g_path_get_basename(file);
  
  ptr= strchr(class_name, '.');
  if (ptr)
    *ptr= 0;

  ptr= class_name + strlen(class_name) - 4;

  if ( (class_name) && (strlen(class_name) > 4) && 
      (strcmp2(ptr, "Test") == 0))
      return MYX_GRT_BAD_MODULE;

  // include the file
  php_eval_string("include '%s'", module_filename);

  xml= php_eval_string("%s::getModuleInfo();", class_name);

  if (xml)
  {
    MYX_GRT_VALUE *grt_info= myx_grt_value_from_xml(xml, strlen(xml));

    if (!grt_info || myx_grt_value_get_type(grt_info) != MYX_DICT_VALUE)
      g_warning("could not parse xml response data from %s",
                file);
    else
    {
      char *name_start= g_strrstr(myx_grt_dict_item_get_as_string(grt_info, "name"), ".");
      const char *extends= myx_grt_dict_item_get_as_string(grt_info, "extends");

      if (name_start)
        module_name= g_strdup(name_start+1);
      else
        module_name= g_strdup(myx_grt_dict_item_get_as_string(grt_info, "name"));

      module_functions= myx_grt_dict_item_get_value(grt_info, "functions");
      if (module_functions && myx_grt_value_get_type(module_functions)==MYX_LIST_VALUE)
        myx_grt_value_retain(module_functions);
      else
        module_functions= NULL;

      if ((extends) && (extends[0]))
        extends= g_strdup(extends);
    }

    if (!grt_info)
      return MYX_GRT_BAD_MODULE;

    myx_grt_value_release(grt_info);
  }
  else
  {
    // No exception handling needed here because it is handled
    // directly in j_call_static_method

    if (getenv("GRT_VERBOSE"))
      g_warning("Module %s doesn't implement getModuleInfo", file);
    return MYX_GRT_BAD_MODULE;
  }

  if (!module_name || !module_functions)
  {
    if (getenv("GRT_VERBOSE"))
    {
      if (!module_name)
        g_warning("Module info from %s doesn't contain 'name'", file);
      if (!module_functions)
        g_warning("Module info from %s doesn't contain 'functions'", file);
    }
    g_free(module_name);
    g_free(extends);
    if (module_functions)
      myx_grt_value_release(module_functions);

    return MYX_GRT_BAD_MODULE;
  }

  // init internal module descriptor
  module= g_new0(MYX_GRT_MODULE, 1);
  pmodule= g_new0(MYX_PHP_MODULE, 1);

  module->loader= loader;
  module->priv= pmodule;
  module->name= module_name;
  module->path= class_name;
  module->functions_num= myx_grt_list_item_count(module_functions);
  module->functions= g_new0(MYX_GRT_FUNCTION, module->functions_num);
  for (i= 0; i < module->functions_num; i++)
  {
    MYX_GRT_FUNCTION *func= module->functions+i;
    MYX_PHP_FUNCTION *pfunc= g_new0(MYX_PHP_FUNCTION, 1);
    char *tmp= g_strdup(myx_grt_value_as_string(myx_grt_list_item_get(module_functions, i)));
    char *return_type;
    
    func->module= module;

    // do not use myx_grt_parse_function_spec here
    // since we need special handling for the php signature
    func->name= g_strdup(strtok(tmp, ":"));
    pfunc->php_signature= g_strdup(strtok(NULL, ":"));
    func->param_struct_name= NULL;
    return_type= strtok(NULL, ":");
    if ((return_type) && (return_type[0]))
      func->return_struct_name= g_strdup(return_type);
  
    func->priv= pfunc;

    g_free(tmp);
  }
  myx_grt_value_release(module_functions);
  module->extends= extends;

  // php specific module info
  pmodule->classname= g_strdup(class_name);


  *retmodule= module;
  
  if (getenv("GRT_VERBOSE"))
    g_message("Initialized module %s", class_name);

  return MYX_GRT_NO_ERROR;
}

static MYX_GRT_ERROR php_call_function(MYX_GRT_FUNCTION *function, MYX_GRT_VALUE *value, MYX_GRT_VALUE **retval)
{
  MYX_PHP_MODULE *pmodule= function->module->priv;
  MYX_PHP_FUNCTION *pfunc= function->priv;
  char *xml;
  char *value_as_xml= myx_grt_value_to_xml(value);

  //escape "
  if (value_as_xml)
    value_as_xml= str_g_replace(myx_grt_value_to_xml(value), "\"", "\\\"");

  xml= php_eval_string("Grt::callModuleFunction(\"%s\", \"%s\", \"%s\", \"%s\");", pmodule->classname, function->name,
    pfunc->php_signature, value_as_xml);

  if (xml)
    *retval= myx_grt_value_from_xml(xml, strlen(xml));
  else
    *retval= NULL;

  g_free(value_as_xml);
  g_free(xml);

  return MYX_GRT_NO_ERROR;
}

static MYX_GRT_ERROR php_fetch_messages(MYX_GRT_MODULE_LOADER *loader, MYX_GRT_MSGS *msgs)
{
  MYX_GRT_ERROR error;
  MYX_GRT_VALUE *res;

  // fetch messages here and put them in msgs  

  //call getMessages from the GrtBase module

  res= myx_grt_function_get_and_call(loader->grt, "PhpTestModule", "getMessages", 0, NULL, &error);

  //check if the result is ok
  if ((res) && (error == MYX_GRT_NO_ERROR) && (myx_grt_value_get_type(res) == MYX_DICT_VALUE))
  {
    MYX_GRT_VALUE *value= myx_grt_dict_item_get_value(res, "value");

    //check if the "value" is a list of the correct type
    if (value && (myx_grt_value_get_type(value) == MYX_LIST_VALUE) && 
      (strcmp2(myx_grt_list_content_get_struct_name(value), "GrtMessage") == 0))
    {
      unsigned int i;

      //allocate message list
      msgs->msgs_num= myx_grt_list_item_count(value);
      msgs->msgs= g_new0(MYX_GRT_MSG, msgs->msgs_num);

      //loop over all items of the returned list
      for (i= 0; i<msgs->msgs_num; i++)
      {
        MYX_GRT_VALUE *val_msg= myx_grt_list_item_get(value, i);
        MYX_GRT_MSG *msg= msgs->msgs+i;

        msg->msg_type= myx_grt_dict_item_get_as_int(val_msg, "msg_type");
        msg->msg= g_strdup(myx_grt_dict_item_get_as_string(val_msg, "msg"));
        msg->msg_detail= myx_grt_list_as_stringlist(
          myx_grt_dict_item_get_value(val_msg, "details"));
      }
    }
    else
      return MYX_GRT_INTERNAL_ERROR;
  }
  else
    return MYX_GRT_INTERNAL_ERROR;

  return MYX_GRT_NO_ERROR;
}

//-----------------------------------------------------------------------------
// Private Stuff
//-----------------------------------------------------------------------------


#endif
