/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <stdlib.h>

#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>
#ifdef USE_READLINE
#include <readline/readline.h>
#include <readline/history.h>
#endif

#include <myx_grt_public_interface.h>
#include <myx_grt_lua.h>
#include <myx_grt_java.h>

#include <myx_grt_builtin_module_public_interface.h>

#include <myx_aux_functions.h>

static char *read_command(MYX_GRT *env)
{
  char *line= NULL;
  //int status;
#ifndef USE_READLINE
  char linebuf[1024];
#endif
  char *prompt= myx_grt_lua_shell_get_prompt(env);

#ifdef USE_READLINE
  line= readline(prompt);
  g_free(prompt);
  if (!line)
    return NULL;
  if (*line)
    add_history(line);
  line= str_g_append(line, "\n");
#else
  printf(prompt);
  g_free(prompt);
  fflush(stdout);
  if (fgets(linebuf, sizeof(linebuf), stdin) <= 0)
  {
    return NULL;
  }
  line= g_strdup(linebuf);
#endif

  return line;
}



static int generate_java_classes(const char *structs_file, const char *output_path_param)
{
  MYX_GRT_STRUCTS *gstructs;
  MYX_GRT_ERROR error;
  char *filename= g_path_get_basename(structs_file);

  // Check that the struct file begins with "structs."
  if ((!str_beginswith(filename, "structs.")) || (!str_endswith(filename, ".xml")))
  {
    fprintf(stdout, "ERROR: The structs file has to begin with \"structs.\" and "
            "end with \".xml\" but is named %s.\n", filename);
    g_free(filename);
    return -1;
  }

  // Load structs
  gstructs= myx_grt_struct_load_list(structs_file, &error);
  if ((!gstructs) || (error != MYX_GRT_NO_ERROR))
  {
    fprintf(stdout, "ERROR (%d): Cannot load structs from %s.\n", error, structs_file);
    g_free(filename);
    return -1;
  }
  else
  {
    char *output_path= g_strdup(output_path_param);
    char *package_name= g_strdup(filename+8);
    char path_sep[]= {MYX_PATH_SEPARATOR, 0};

    g_free(filename);

    package_name[strlen(package_name)-4]= 0;

    if (output_path[strlen(output_path)-1] != MYX_PATH_SEPARATOR)
      output_path= str_g_append(output_path, path_sep);
    
    myx_mkdir(output_path, 0755);
    
    error= myx_grt_struct_export_java_classes(gstructs, package_name, output_path);
    g_free(package_name);
    
    if (error != MYX_GRT_NO_ERROR)
    {
      fprintf(stdout, "ERROR (%d): Cannot generate java class files in %s.\n", error, output_path);
      return -1;
    }
    g_free(output_path);
  }
  return 0;
}

static int generate_php_classes(const char *structs_file, const char *output_path_param)
{
  MYX_GRT_STRUCTS *gstructs;
  MYX_GRT_ERROR error;
  char *filename= g_path_get_basename(structs_file);

  // Check that the struct file begins with "structs."
  if ((!str_beginswith(filename, "structs.")) || (!str_endswith(filename, ".xml")))
  {
    fprintf(stdout, "ERROR: The structs file has to begin with \"structs.\" and "
            "end with \".xml\" but is named %s.\n", filename);
    g_free(filename);
    return -1;
  }

  // Load structs
  gstructs= myx_grt_struct_load_list(structs_file, &error);
  if ((!gstructs) || (error != MYX_GRT_NO_ERROR))
  {
    fprintf(stdout, "ERROR (%d): Cannot load structs from %s.\n", error, structs_file);
    g_free(filename);
    return -1;
  }
  else
  {
    char *output_path= g_strdup(output_path_param);
    char path_sep[]= {MYX_PATH_SEPARATOR, 0};

    g_free(filename);

    if (output_path[strlen(output_path)-1] != MYX_PATH_SEPARATOR)
      output_path= str_g_append(output_path, path_sep);
    
    myx_mkdir(output_path, 0755);
    
    error= myx_grt_struct_export_php_classes(gstructs, output_path);
    
    if (error != MYX_GRT_NO_ERROR)
    {
      fprintf(stdout, "ERROR (%d): Cannot generate php class files in %s.\n", error, output_path);
      return -1;
    }
    g_free(output_path);
  }
  return 0;
}


MYX_GRT *initialize_grt(const char *classpath)
{
  MYX_GRT *grt= myx_grt_initialize();
  MYX_GRT_MODULE_LOADER *loader;
  MYX_GRT_ERROR error;
  unsigned int c;

  g_message("Scanning for struct definitions in ./xml ...");
  c= myx_grt_scan_for_structs(grt, "./xml", &error);
  if (error != MYX_GRT_NO_ERROR)
  {
    g_warning("Error while scanning for struct definitions (%i).", error);
  }
  else
    g_message("Registered %i struct definition files.", c);

  g_message("Initializing Builtin modules...");
  myx_register_builtin_grt_module_reverse_engineer_mysql(grt);
  myx_register_builtin_grt_module_transformation_mysql(grt);

  // initialize the loaders

  // initialize Java loader
  g_message("Initializing Java loader...");
  loader= myx_java_init_loader(grt, classpath, &error);
  if (!loader)
  {
    g_warning("Error initializing Java module loader (%i).", error);
  }
  else
  {
    if (myx_grt_register_module_loader(grt, loader) < 0)
    {
      g_warning("Could not register Java module loader.");
    }
    else
    {
      // Scan for Java modules
      g_message("Scanning for java plugins in ./java/com/mysql/grt/modules ...");

      c= myx_grt_scan_for_modules(grt, "./java/com/mysql/grt/modules", &error);
      if (error!=MYX_GRT_NO_ERROR)
      {
        g_warning("Error while scanning for java modules (%i).", error);
      }
      else
        g_message("Initialized %i modules.", c);
    }
  }

#ifdef ENABLE_PHP_MODULES
  // initialize PHP loader
  g_message("Initializing PHP loader...");
  loader= myx_php_init_loader(grt, &error);
  if (!loader)
  {
    g_warning("Error initializing PHP module loader (%i).", error);
  }
  else
  {
    if (myx_grt_register_module_loader(grt, loader) < 0)
    {
      g_warning("Could not register PHP module loader.");
    }
    else
    {
      // Scan for PHP modules
      g_message("Scanning for php plugins in ./php/modules ...");

      c= myx_grt_scan_for_modules(grt, "./php/modules", &error);
      if (error!=MYX_GRT_NO_ERROR)
      {
        g_warning("Error while scanning for php modules (%i).", error);
      }
      else
        g_message("Initialized %i modules.", c);
    }
  }
#endif

  // initialize Lua loader
  g_message("Initializing Lua loader...");
  loader= myx_lua_init_loader(grt, &error);
  if (!loader)
  {
    g_warning("Error initializing Lua module loader (%i).", error);
  }
  else
  {
    if (myx_grt_register_module_loader(grt, loader) < 0)
    {
      g_warning("Could not register Lua module loader.");
    }
  }

  myx_grt_init_lua_shell(grt);


  return grt;
}


static void print_messages(MYX_GRT *grt)
{
  MYX_GRT_MSGS *msgs= myx_grt_get_msgs(grt, 0);
  unsigned int i;
 
  if (!msgs) return;

  for (i= 0; i < msgs->msgs_num; i++)
  {
    MYX_GRT_MSG *msg= msgs->msgs+i;
    char *type;

    switch (msg->msg_type)
    {
      case 0: type= ""; break;
      case 1: type= " WARNING"; break;
      case 2: type= " ERROR"; break;
      default: type= NULL; break;
    }

    if (type)
      printf("GRT%s: %s\n", type, msg->msg);
    else
      printf("GRT%i: %s\n", msg->msg_type, msg);
    if (msg->msg_detail)
    {
      unsigned int j;
      for (j= 0; j < msg->msg_detail->strings_num; j++)
      {
        printf("GRT:\t%s\n", msg->msg_detail->strings[j]);
      }
    }
  }
  myx_grt_free_msgs(msgs);
}


static void show_help(const char *argv0)
{
    fprintf(stdout, 
      "Usage: %s [-classpath path] [-d path] [-x] [luafile] \n"
      "       %s -j structsfile outputdir\n\n"    
      "  -classpath .. Sets the java classpath to the given value.\n"
      "  -d path ..... Modules directory\n"
      "  -x .......... Exits the shell after running the specified file\n"
      "  luafile ..... File that is run at startup.\n\n"
      "  -j .......... Generates java classes from the given structs file.\n"
      "\n", argv0, argv0);
}

int main(int argc, char **argv)
{
  MYX_GRT_ERROR status;
  MYX_GRT *grt;
  char *classpath= NULL;
  char *linebuf= NULL;
  char *load_file= NULL;
  char *module_dir= NULL;
  int interactive= 1;
  int c, i;
  
  // ---------------------------------------------------------------------------------------------------------------
  // If the shell is called without arguments, display usage
  /*if (argc == 1)
  {
    show_help(argv[0]);
    exit(1);
  }*/

  // ---------------------------------------------------------------------------------------------------------------
  // Handle java class generation

  if ((argc>1) && (strcmp(argv[1], "-j")==0))
  {    
    if (argc != 4)
    {
      show_help(argv[0]);
      exit(1);
    }
    else
    {
      if (generate_java_classes(argv[2], argv[3])!=0)
        exit(1);
      else
        exit(0);
    }
  }

  // ---------------------------------------------------------------------------------------------------------------
  // Handle php class generation

  if ((argc>1) && (strcmp(argv[1], "-p")==0))
  {    
    if (argc != 4)
    {
      show_help(argv[0]);
      exit(1);
    }
    else
    {
      if (generate_php_classes(argv[2], argv[3])!=0)
        exit(1);
      else
        exit(0);
    }
  }


  // ---------------------------------------------------------------------------------------------------------------
  // Handle module_dir and interactive
  for (i= 1; i < argc; i++)
  {
    if (strcmp(argv[i], "-x")==0)
      interactive= 0;
    else if (strcmp(argv[i], "-d")==0)
    {
      if (argc <= i+1)
      {
        show_help(argv[0]);
        exit(1);
      }
      else
        module_dir= argv[++i];
    }
    else if (strcmp(argv[i], "-classpath")==0)
    {
      if (argc <= i+1)
      {
        show_help(argv[0]);
        exit(1);
      }
      else
        classpath= argv[++i];
    }
    else
      load_file= argv[i];
  }

  // ---------------------------------------------------------------------------------------------------------------
  //Initialize grt
  grt= initialize_grt(classpath);
  if (!grt)
  {
    g_message("ERROR: The GRT environment cannot be initialized.");
    exit(1);
  }

  g_assert(lua_gettop(myx_grt_lua_shell_get_lua(grt))==0);
 

  // ---------------------------------------------------------------------------------------------------------------
  // Scan for plugins
  if (module_dir)
  {
    g_message("Scanning for plugins in %s...", module_dir);
    c= myx_grt_scan_for_modules(grt, module_dir, &status);
    g_message("Initialized %i modules", c);
  }
  g_assert(lua_gettop(myx_grt_lua_shell_get_lua(grt))==0);

  // ---------------------------------------------------------------------------------------------------------------
  // Run script
  if (!interactive)
  {
    if (load_file)
      exit(myx_grt_lua_shell_run_file(grt, load_file));
    else
      exit(0);
  } 
  else if (load_file)
  {
    myx_grt_lua_shell_run_file(grt, load_file);
  }

  // ---------------------------------------------------------------------------------------------------------------
  // Main loop
  for (;;)
  {
    linebuf= read_command(grt);
    if (!linebuf)
    {
      break;
    }

    if(myx_grt_lua_shell_execute(grt, linebuf) == MYX_GRT_SHELL_COMMAND_EXIT)
      break;
 
    print_messages(grt);

    g_free(linebuf);
  }

  myx_grt_finalize(grt);
  
  return 0;
}
