# Copyright (c) 2004 Divmod.
# See LICENSE for details.

import os
import random
import string

from twisted.internet import defer
from twisted.internet import reactor

from nevow import inevow
from nevow import rend
from nevow import loaders
from nevow import static
from nevow.tags import *
from nevow.stan import Proto, Tag
from nevow.flat import flatten

from itertools import count


cn = count().next
cookie = lambda: str(cn())

_hookup = {}

# m = method
# a = argument
# <m n="moveTo">
#   <a v="16" />
#   <a v="16" />
# </m>


m = Proto('m') # method call; contains arguments
a = Proto('a') # argument; has v="" attribute for simple value, or <l> or <d> child for list or dict value
l = Proto('l') # list; has <a> children; <a> children must be simple values currently
d = Proto('d') # dict; has <i> children
i = Proto('i') # dict item; has k="" for key and v="" for simple value (no nested dicts yet)


def squish(it):
    if isinstance(it, Tag):
        return a[it]
    return a(v=it)


class _Remoted(object):
    def __init__(self, cookie, socket):
        self.cookie = cookie
        self.socket = socket

class Text(_Remoted):
    def change(self, text):
        self.socket.call('changeText', self.cookie, text)

    def move(self, x, y):
        self.socket.call('moveText', self.cookie, x, y)

    def listFonts(self):
        if hasattr(self.socket, '_fontList'):
            return defer.succeed(self.socket._fontList)
        cook = cookie()
        self.socket.deferreds[cook] = d = defer.Deferred()
        self.socket.call('listFonts', cook)
        def _cb(l):
            L = l.split(',')
            self.socket._fontList = L
            return L
        return d.addCallback(_cb)

    def font(self, font):
        self.socket.call('font', self.cookie, font)

    def size(self, size):
        self.socket.call('size', self.cookie, size)


class Image(_Remoted):
    def move(self, x, y):
        self.socket.call('moveImage', self.cookie, x, y)

    def scale(self, x, y):
        self.socket.call('scaleImage', self.cookie, x, y)

    def alpha(self, alpha):
        self.socket.call('alphaImage', self.cookie, alpha)

    def rotate(self, angle):
        self.socket.call('rotateImage', self.cookie, angle)


class CanvasSocket(object):
    """An object which represents the client-side canvas. Defines APIs for drawing
    on the canvas. An instance of this class will be passed to your onload callback.
    """
    __implements__ = inevow.IResource,

    def __init__(self):
        self.d = defer.Deferred()

    def locateChild(self, ctx, segs):
        self.cookie = segs[0]
        return (self, ())

    def renderHTTP(self, ctx):
        self.deferreds = {}
        self.buffer = ''
        self.socket = inevow.IRequest(ctx).transport
        ## We be hijackin'
        self.socket.protocol = self
        ## This request never finishes until the user leaves the page
        _hookup[self.cookie](self)
        del _hookup[self.cookie]
        return self.d

    def dataReceived(self, data):
        self.buffer += data
        while '\0' in self.buffer:
            I = self.buffer.index('\0')
            message = self.buffer[:I]
            self.buffer = self.buffer[I+1:]
            self.gotMessage(message)

    def gotMessage(self, message):
        I = message.index(' ')
        self.deferreds[message[:I]].callback(message[I+1:])
        del self.deferreds[message[:I]]

    def connectionLost(self, reason):
        del self.socket

    def done(self):
        """Done drawing; close the connection with the movie
        """
        ## All done with the request object
        self.d.callback('')

    def call(self, method, *args):
        """Call a client-side method with the given arguments. Arguments
        will be converted to strings. You should probably use the other higher-level
        apis instead.
        """
        flatcall = flatten(
            m(n=method)[[
                squish(x) for x in args if x is not None]])
        self.socket.write(flatcall + '\0')

    def line(self, x, y):
        """Draw a line from the current point to the given point.
        
        (0,0) is in the center of the canvas.
        """
        self.call('line', x, y)

    def move(self, x, y):
        """Move the pen to the given point.
        
        (0, 0) is in the center of the canvas.
        """
        self.call('move', x, y)

    def pen(self, width=None, rgb=None, alpha=None):
        """Change the current pen attributes. 

        width: an integer between 0 and 255; the pen thickness, in pixels.
        rgb: an integer between 0x000000 and 0xffffff
        alpha: an integer between 0 and 100; the opacity of the pen
        """
        self.call('pen', width, rgb, alpha)

    def clear(self):
        """Clear the current pen attributes.
        """
        self.call('clear')

    def fill(self, rgb, alpha=100):
        """Set the current fill. Fill will not be drawn until close is called.
        
        rgb: color of fill, integer between 0x000000 and 0xffffff
        alpha: an integer between 0 and 100; the opacity of the fill
        """
        self.call('fill', rgb, alpha)

    def close(self):
        """Close the current shape. A line will be drawn from the end point
        to the start point, and the shape will be filled with the current fill.
        """
        self.call('close')

    def curve(self, controlX, controlY, anchorX, anchorY):
        """Draw a curve
        """
        self.call('curve', controlX, controlY, anchorX, anchorY)

    def gradient(self, type, colors, alphas, ratios, matrix):
        """Draw a gradient. Currently the API for this sucks, see the flash documentation
        for info. Higher level objects for creating gradients will hopefully be developed
        eventually.
        """
        self.call('gradient', type,
            l[[a(v=x) for x in colors]],
            l[[a(v=x) for x in alphas]],
            l[[a(v=x) for x in ratios]],
            d[[i(k=k, v=v) for (k, v) in matrix.items()]])

    def text(self, text, x, y, height, width):
        """Place the given text on the canvas using the given x, y, height and width.
        The result is a Text object which can be further manipulated to affect the text.
        """
        cook = cookie()
        t = Text(cook, self)
        self.call('text', cook, text, x, y, height, width)
        return t

    def image(self, where):
        cook = cookie()
        I = Image(cook, self)
        self.call('image', cook, where)
        print "IMAGE", where
        return I

import pdb

def _getPrefix(c):
    pre = inevow.IRequest(c).path
    if pre.endswith('/'):
        return pre
    return pre + '/'

def canvas(width, height, onload):
    C = cookie()
    _hookup[C] = onload
    return lambda c, d: xml("""
<object classid="clsid:d27cdb6e-ae6d-11cf-96b8-444553540000" codebase="http://download.macromedia.com/pub/shockwave/cabs/flash/swflash.cab#version=7,0,0,0" width="%(width)s" height="%(height)s" id="Canvas" align="middle" style="border: 1px solid black">
<param name="allowScriptAccess" value="sameDomain" />
<param name="movie" value="nevow_canvas_movie?cookie=%(cookie)s&port=%(port)s&prefix=%(prefix)s" />
<param name="quality" value="high" />
<param name="scale" value="noscale" />
<param name="bgcolor" value="#ffffff" />
<embed src="nevow_canvas_movie?cookie=%(cookie)s&port=%(port)s&prefix=%(prefix)s" quality="high" scale="noscale" bgcolor="#ffffff" width="%(width)s" height="%(height)s" name="Canvas" align="middle" allowScriptAccess="sameDomain" type="application/x-shockwave-flash" pluginspage="http://www.macromedia.com/go/getflashplayer" />
</object>
""" % {'height': height, 'width': width, 'cookie': C, 'port': inevow.IRequest(c).transport.server.port, 'prefix':  _getPrefix(c)})


## Some random helpers
rndpt = lambda w, h: (random.randint(-w/2, w/2), random.randint(-h/2, h/2))
rndrct = lambda w, h: rndpt(w, h) + rndpt(w, h)
rndtxt = lambda: ''.join([random.choice(string.letters) for x in range(random.randint(5, 10))])
rndp = lambda: (random.randint(0, 100), )*2
def gotFonts(fnts, T):
    T.font(random.choice(fnts))
def messText(T, w, h):
    """Randomly manipulate a text object
    """
    ch = random.randint(0, 1)
    T.move(*rndpt(w, h))
    T.size(random.randint(9, 48))
    T.listFonts().addCallback(gotFonts, T)
    if random.randint(0, 5): # We're more likely to keep going than to stop
        # Animate changes to this text object
        reactor.callLater(random.random(), messText, T, w, h)


class Canvas(rend.Page):
    """A page which can embed canvases. Simplest usage is to subclass and
    override width, height and onload. Then, putting render_canvas in the
    template will output that canvas there.
    
    You can also embed more than one canvas in a page using the canvas
    helper function, canvas(width, height, onload). The resulting stan
    will cause a canvas of the given height and width to be embedded in
    the page at that location, and the given onload callable to be called
    with a CanvasSocket when the connection is established.
    """
    def __init__(self, words, width=None, height=None, onload=None):
        rend.Page.__init__(self, words)
        if width: self.width = width
        if height: self.height = height
        if onload: self.onload = onload

    child_nevow_canvas_movie = static.File(
        os.path.join(
            os.path.split(__file__)[0], 'Canvas.swf'),
        'application/x-shockwave-flash')

    def child_canvas_socket(self, ctx):
        return CanvasSocket()

    width = 1000
    height = 500

    def onload(self, canvas):
        """Demo of drawing with a CanvasSocket object.
        """
        for x in range(random.randint(10, 20)):
            canvas.pen( 
                random.randint(1, 10),
                random.randint(0, 0xffffff),
                random.randint(0, 100))
            canvas.move(*rndpt(self.width, self.height))
            choice = random.randint(0, 4)
            if choice == 0:
                canvas.line(*rndpt(self.width, self.height))
            elif choice == 1:
                canvas.fill(random.randint(0, 0xffffff), random.randint(0, 100))
                for x in range(random.randint(3, 20)):
                    if random.randint(0, 1):
                        canvas.line(*rndpt(self.width, self.height))
                    else:
                        canvas.curve(*rndrct(self.width, self.height))
                canvas.close()
            elif choice == 2:
                canvas.curve(*rndrct(self.width, self.height))
            elif choice == 3:
                T = canvas.text(random.choice(self.original), *rndpt(self.width, self.height)+(400, 100))
                # This is an example of how you can hold on to drawing objects and continue to
                # draw on them later, because the CanvasSocket holds itself open until done() is called
                messText(T, self.width, self.height)
            else:
                # This demo requires a folder of images which I don't want to put in
                # the nevow source. Hooking this up is left as an exercise for the reader.
                continue
                imgname = random.choice(os.listdir("flsh/images"))
                I = canvas.image('/images/%s' % imgname)
                I.scale(*rndp())
                I.alpha(random.randint(0, 100))
                rotate = random.randint(-180, 180)
                I.rotate(rotate)
                I.move(*rndpt(self.width, self.height))

    # See above comment
    #child_images = static.File('images')

    def render_canvas(self, ctx, data):
        return canvas(self.width, self.height, self.onload)

    docFactory = loaders.stan(html[render_canvas])

