/* $Id: skills.h,v 1.39 2004/10/20 06:01:19 graziano Exp $ */

#ifndef SKILLS_H
#define SKILLS_H

#ifdef __cplusplus
extern "C" {
#endif

/*
** The skills module serves as a nameserver-aware wrapper around the sensor
** experiment modules.  By going through this module, sensor control modules
** can focus on scheduling experiments, rather than on the details of how to
** invoke them and interpret the results.  Concentrating skill control in a
** single module also allows multiple controls to support overlapping skills
** without needing to declare duplicated, and therefore fragile, skill names,
** option values, and the like.
*/

#include "config_nws.h"

#include "register.h"

#ifdef EXPERIMENTAL
	/* we don't have experimental skills right now */
#	define EXPERIMENTAL_SKILLS 0
#	define EXPERIMENTAL_RESOURCES 0
#else
#	define EXPERIMENTAL_SKILLS 0
#	define EXPERIMENTAL_RESOURCES 0
#endif

/* # of known skills and # of resources we are measuring */
#define RESOURCE_COUNT (12 + EXPERIMENTAL_RESOURCES)
#define SKILL_COUNT (9 + EXPERIMENTAL_SKILLS)
  
/* Skills supported by this sensor and the resources they measure. */
typedef enum {cpuMonitor=0, diskMonitor, memoryMonitor, tcpConnectMonitor,
	tcpMessageMonitor, filesystemMonitor, availabilityMonitor, 
	startMonitor, memorySpeedMonitor 
} KnownSkills;


typedef enum {availableCpu=0, currentCpu, freeDisk, freeMemory, connectTimeTcp, 
	bandwidthTcp, latencyTcp, fsreadSpeed, fswriteSpeed ,upTime,
	startTime, memorySpeed 
} MeasuredResources;


/*
 * The result of invoking a skill: a measured resource, the options that apply
 * to the measurement, an indication of whether or not the measurement
 * succeeded, and, if it did, the actual measurement value.  Note that we're
 * assuming here that all resource measurements can be expressed in one double
 * value.  This is presently true, but may need to be revisited.
 */
typedef struct {
	MeasuredResources resource;
	char *options;
	int succeeded;
	double measurement;
} SkillResult;


/**
 * A convenience function that scans #options# for a setting for #name#.
 * if there are multiple settings, a comma separated lists of the
 * settings is returned.  * #options# is a tab separated list of
 * name:value. 
 * Returns the associated value if found, else #defaultValue#.  
 *
 * The return value needs to be freed.
 */
char *
GetOptionValue(const char *options,
               const char *name,
               const char *defaultValue);


/**
 * Returns a unit label for use when registering series of #resource#.
 */
const char *
ResourceLabel(MeasuredResources resource);


/**
 * Returns a string representation of #resource#. 
 */
const char *
ResourceName(MeasuredResources resource);


/**
 * Returns 1 or 0 depending on whether or not the system supports conducting
 * #skill# experiments configured by #options# for the specific #control#.
 *
 * ALL_CONTROLS is the number of control supported by NWS.
 */
#define ALL_CONTROLS 2
#define PERIODIC_CONTROL 0
#define CLIQUE_CONTROL 1
int
SkillAvailableForControl(	KnownSkills skill,
				const char *options,
				int control);
#define SkillAvailable(skill, options) \
	SkillAvailableForControl(skill, options, ALL_CONTROLS)


/*
** Returns a string representation of #skill#.  The return value is volatile
** and will be overwritten by subsequent calls.
*/
const char *
SkillName(KnownSkills skill);


/**
 * Draws from the tab-delimited list #options# the subset that applies to
 * #skill#. If the same option is specified multpile time (ie nice:0
 * nice:1) the resulting option will have all the values comma separated
 * (nice:0,1).  Returns the result in #len#-length #toWhere#.
 */
void
SkillOptions(	KnownSkills skill,
		const char *options,
		char *toWhere,
		int len);


/**
 * Returns the format of #option# (for example 0_to_1_int) or NULL if no
 * option was found. The return value need to be freed.
 */
char *
SkillOptionFormat(const char *option);

/**
 * Returns the system default for #option# and #skill# or NULL if no
 * defaults are available. The return value needs to be freed by the
 * caller.
 */
char *
SkillOptionDefault(	const KnownSkills skill,
			const char *option);

/**
 * Returns the list of options (comma separates) supported by #skill# or
 * NULL in case of error or no options.
 */
const char *
SkillSupportedOptions(const KnownSkills skill);


/**
 * Returns a pointer to the list of resources measured by #skill# in
 * #resources# and its length in #length#. #resources# can be NULL (in
 * which case no MeasuredResources are returned) but #length cannot.
 *
 * Returns 1 on success, 0 on failure.
 */
int
SkillResources(KnownSkills skill,
               const MeasuredResources **resources,
               int *length);


/**
 * Prepares the skill module for use. #options# is a list of options
 * in the form of option:value tab delimited (most notably fork:yes or
 * fork:no will be passed). Returns 1 if successful, else 0. 
 */
int
SkillsInit(	const char *options);


/**
 * Free the memory associate with #results#. There are #howMany# results.
 */
void
FreeSkillResults(	int howMany,
			SkillResult **results);

/**
 * Attempts to conduct #skill# experiments configured by #options#.  Returns
 * the results of the experiments in the array #results# and the length of this
 * array in #length#.  Returns failure results if the skill does not complete
 * within #timeOut# seconds (-1 means automatic timeouts). #results# need
 * to be freed with FreeSkillResults.
 */
void
UseSkill(KnownSkills skill,
         const char *options,
         double timeOut,
         SkillResult **results,
         int *length);


/**
 * This is used only by experiments writer: append a result to
 * #length#-long #results# and adjust #length# accordingly (so you better
 * start #length# from 0).
 * The result (#measurement#) is been taken for #resource#, with the
 * #options#, and if #succeeded# is 1 the experiment succeeded. 
 */
void
AppendResult(	MeasuredResources resource,
		const char *options,
		int succeeded,
		double measurement,
		int *length,
		SkillResult **results);

/**
 * Sends the SKILL_EXPERIMENT to sensor #cookie#, asking it to execute
 * UseSkill. #opts# contains a \t delimited list of options, one of
 * which needs to be a skillName. The function can wait #tout# seconds
 * for the results. If succesful returns 1, and the ResourceName in
 * #resources# and the results in #results# with #howMany# telling how
 * many results there are. Every entry in #resources# needs to be freed,
 * while #d# needs to be allocated by the caller. 
 */
int
RemoteUseSkill(	struct host_cookie *cookie,
		char *opts,
		double timeout,
		char ***resources,
		double *d,
		int *howMany);


#ifdef __cplusplus
}
#endif

#endif
