
/******************************************************************************
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, visit www.shmoo.com/osiris for
**  details.
**
******************************************************************************/

/*****************************************************************************
**
**  File:    md_filter.c
**  Date:    September 17, 2003
**
**  Author:  Brian Wotring
**  Purpose: handle compare filter operations.
**
******************************************************************************/


#include "libosiris.h"
#include "libosirism.h"
#include "libosirisdb.h"
#include "libosirisctl.h"
#include "libfileapi.h"

#include "md_hosts.h"
#include "md_log.h"
#include "md_config.h"
#include "md_compare.h"
#include "md_auth.h"
#include "md_filter.h"

#include "common.h"
#include "logging.h"

#include "config.h"
#include "debug.h"

extern osi_bool name_regex( const char *name, const char *pattern );
extern char cmp_filter_db_path[MAX_PATH_LENGTH];

osi_bool md_filter_create_database( const char *filter_db_path )
{
    int result;
    OSI_DB db;

    if( filter_db_path == NULL )
    {
        return FALSE;
    }

    memset( &db, 0, sizeof( OSI_DB ) );
    osi_strlcpy( db.path, filter_db_path, sizeof( db.path ) );

    result = osi_db_create( &db );

    if( result != OSI_DB_OK )
    {
        return FALSE;
    }

    return TRUE;
}


void md_filter_changes( OSI_DB *filter_db, char *host,
                        unsigned int file_attr, char *path,
                        unsigned int *unfiltered_mask )
{
    int result;
    int result_size;

    OSI_CMP_FILTER filter;

    unsigned int filter_attr;
    unsigned char buffer[sizeof(OSI_CMP_FILTER)];

    OSI_DEBUG( __POS__, "filtering changes for file: %s", path );

    if( unfiltered_mask )
    {
        memset( unfiltered_mask, 0xff, sizeof( unsigned int ) );
    }

    if( ( filter_db == NULL ) || ( host == NULL ) || ( path == NULL ) )
    {
        return;
    } 

    /* go through the filter database. check the least intensive    */
    /* match first, the attribute mask.  Then, check the host, then */
    /* the path.                                                    */

    result = osi_db_get_first_record( filter_db, buffer, sizeof( buffer ),
                                      &result_size );

    if( result != OSI_DB_OK )
    {
        return;
    }

    while( result == OSI_DB_OK )
    {
        unpack_cmp_filter( &filter, buffer, sizeof( buffer ) );
        unwrap_cmp_filter( &filter );

        /* if we are an exclude rule, we swap the bits and do the same */
        /* as an include-only rule.                                    */

        filter_attr = (int)filter.mask;

        if( filter.type == CMP_FILTER_TYPE_INCLUDE_ONLY )
        {
            filter_attr = (unsigned int)(~filter_attr);    
        }

        OSI_DEBUG( __POS__, "applying filter: host=(%s);path=(%s);attr=(%d)",
                   filter.host, filter.path, (int)filter_attr );

        /* xor the changes with the filter to filter out the attributes  */
        /* we don't care about, then AND that result with the changes    */
        /* to see if we anything changed that we care about.  If we care */
        /* about the change, don't proceed to filter it, continue on to  */
        /* the next filter.                                              */

        if( !( filter_attr & file_attr ) )
        {
            OSI_DEBUG( __POS__, "file: (%s) doesn't match this filter.", path );
            goto next_filter;
        }

        OSI_DEBUG( __POS__, "file: (%s) matches current filter.", path );

        /* host doesn't match. */

        if( strlen(filter.host) && name_regex( host, filter.host ) == FALSE )
        {
            OSI_DEBUG( __POS__, "host (%s) doesn't match this filter.", host );
            goto next_filter;
        }

        OSI_DEBUG( __POS__, "host (%s) matches current filter.", path );

        /* path doesn't match. */

        if( strlen(filter.path) && name_regex( path, filter.path ) == FALSE )
        {
            OSI_DEBUG( __POS__, "path (%s) doesn't match filter.", path );
            goto next_filter;
        }

        OSI_DEBUG( __POS__, "path (%s) matches current filter.", path );

        /* we have a match, this change is filtered. */

        /* add the unfiltered elements to the unfiltered mask.  This will   */
        /* be a list of file attributes that changed for this file but were */
        /* not caught by the filters.                                       */

        if( unfiltered_mask )
        {
            if( (*unfiltered_mask) == 0 )
            {
                (*unfiltered_mask) = ( (filter_attr ^ file_attr) & file_attr );
            }

            else
            {
                (*unfiltered_mask) &= ( (filter_attr ^ file_attr) & file_attr );
            }
        }

        OSI_DEBUG( __POS__, "unfiltered mask is: %d", (int)(*unfiltered_mask) );

next_filter:
        result = osi_db_get_next_record( filter_db, buffer, sizeof( buffer ),
                                         &result_size );
    }
}

osi_bool md_filter_open_database( OSI_DB *filter_db )
{
    if( filter_db != NULL )
    {
        memset( filter_db, 0, sizeof( OSI_DB ) );

        osi_strlcpy( filter_db->path, cmp_filter_db_path,
                     sizeof( filter_db->path ) );

        if( osi_db_open( filter_db, OSI_DB_READ ) == OSI_DB_OK )
        {
            return TRUE;
        }
    }

    return FALSE;
}

osi_bool md_filter_close_database( OSI_DB *filter_db )
{
    if( filter_db != NULL )
    {
        osi_db_close( filter_db );
        return TRUE;
    }

    return FALSE;
}
